/* 
 *  'proof' - an interactive proofreader (terminal version)
 *   
 *  Copyright (c) 19{89-91} by Craig Latta. all rights reserved.
 *   
 *  Permission to use, copy, and/or distribute for any purpose and
 *  without fee is hereby granted, provided that both the above copyright
 *  notice and this permission notice appear in all copies and derived works.
 *  Fees for distribution or use of this software or derived works may only
 *  be charged with express written permission of the copyright holder.
 *  this software is provided ``as is'' without express or implied warranty.
 *  
 *  FILE:		motor.c
 *  
 *  DESCRIPTION:	the motor of the left-associative parser used
 *                      to check grammar in 'proof'.
 */

#include  <errno.h>
#include  <stdio.h>
#include  <string.h>
#include  <ctype.h>
#include  <sys/types.h>
#include  <sys/stat.h>
#include  <sys/file.h>
#include  <malloc.h>
#include  <stuff.h>

#include  "gdbm.h"
#include  "hash.h"
#include  "proof.h"

/* Project-global variables (from proof.h) */

char       error[BUFFER_LEN];
HISTENTRY  *history;   /* history array */
int        backtracked, curr_deriv;
CONDITION  curr_condition;


/* 
 * module-global variables: These file locations are suggested, but
 * edit them properly for your site
 */

#define         DEFINE "/usr/local/lib/proof/define"
#define         GLOAT "/usr/local/lib/proof/gloating"
#define		LEX_FILENAME "/usr/local/lib/proof/lexicon"
#define         MOTD "/usr/local/lib/proof/motd"
#define         PEACE "/usr/local/lib/proof/peace"
#define         WORKS "/usr/local/lib/proof/works"

hashtable  	*table;                 /* local hashtable */
GDBM_FILE      	lexicon;               /* remote database */
BUFFER          command[MAX_COMMANDS];
int             comm = 0;


/* functions */

/* 
 *  FUNCTION      comb_definition(DEFINITION  definition)
 *   
 *  PURPOSE       remove redundancies from a definition
 * 
 *  INPUT	  see above
 *  
 *  OUTPUT        none
 * 
 *  SIDE EFFECTS  see above
 */
      
extern void
  comb_definition(definition)
DEFINITION  definition;
{
  int  def_len, index, i, hit;
  

  for (index = 1; index <= definition[0][0]; index++)
    {
      hit = 0;
      for (i = index + 1; i <= definition[0][0]; i++)
	{
	  if (compare_categories(definition[index],
				 definition[i]))
	    {
	      compact_definition(definition, i);
	      hit = 1;
	    }
	}
      if (hit)
	index = 0;
    }

} /* end; compact_definition(DEFINITION  definition) : finished */

/*
 *  FUNCTION      extern void  compact_category(int *category, seg)
 *  
 *  PURPOSE       compact a segment 'seg' out of a segment list
 *  
 *  INPUT         int *category, seg -- see above
 *  
 *  OUTPUT        none
 * 
 *  SIDE EFFECTS  'category' gets compacted.
 */

extern void
  compact_category(category, seg)
CATEGORY  category;
int       seg;
{
  int  seg_index;
  
  if (seg == 0)
    return;

  else 
    {
      category[0]--;

      if (seg == category[0] + 1)
	{
	  return;
	}
      
      for (seg_index = seg; seg_index <=  category[0]; seg_index++)
	category[seg_index] = category[seg_index + 1];
    }
} /* end: compact_category() : finished */


/*
 *  FUNCTION      extern void  compact_definition(int **categories, cat)
 *  
 *  PURPOSE       compact a category 'cat' out of a category list
 *  
 *  INPUT         int **categories, *cat -- see above
 *  
 *  OUTPUT        none
 * 
 *  SIDE EFFECTS  'categories' gets compacted.
 */

extern void
  compact_definition(definition, cat)
DEFINITION  definition;
int         cat;
{
  int  cat_index;
  
  if (cat == 0)
    return;

  definition[0][0]--;
  
  if (cat == definition[0][0])
    return;
  
  for (cat_index = cat; cat_index < definition[0][0]; cat_index++)
    copy_category(definition[cat_index], definition[cat_index + 1]);

} /* end: compact_definition() : finished */


/*
 *  FUNCTION      extern int  compare_categories(int *cat1, *cat2)
 *  
 *  PURPOSE       compare two categories
 *  
 *  INPUT         int *cat1, *cat2 -- two categories to be compared
 *  
 *  OUTPUT        1 if identical, 0 otherwise
 * 
 *  SIDE EFFECTS  none
 */

extern int
  compare_categories(cat1, cat2)
CATEGORY  cat1, cat2;
{
  int  i;
  
  
  if (cat1[0] != cat2[0])
    return(0);
  
  for (i = 0; i <= cat1[0]; i++)
    {
      if (cat1[i] != cat2[i])
	return(0);
    }
  
  return(1);
  
} /* end: compare_categories() : finished */
  

/*
 *  FUNCTION      extern void  copy_category(int  *target, *source)
 *  
 *  PURPOSE       copies from one category array to another
 *  
 *  INPUT         int  *target, *source
 *  
 *  OUTPUT        none
 * 
 *  SIDE EFFECTS  see above
 */

extern void
  copy_category(target, source)
CATEGORY  target, source;
{
  int  sindex;
  
  for (sindex = 1; sindex <= source[0]; sindex++)
    target[sindex] = source[sindex];
  
  target[0] = --sindex;

} /* end: copy_category() : finished */


/*
 *  FUNCTION      extern void  copy_segments(int  **target, **source)
 *  
 *  PURPOSE       copies from one segments array to another
 *  
 *  INPUT         int  **target, **source
 *  
 *  OUTPUT        none
 * 
 *  SIDE EFFECTS  see above
 */

extern void
  copy_segments(target, source)
DEFINITION  target, source;
{
  int  cindex, sindex;
  
  target[0][0] = source[0][0];
  
  for (cindex = 1; cindex <= source[0][0]; cindex++)
    {  
      for (sindex = 0; sindex <= source[cindex][0]; sindex++)
	{
	  target[cindex][sindex] = source[cindex][sindex];
	}
      target[cindex][0] = --sindex;
    }
} /* end: copy_segments() : finished */


/* 
 *  FUNCTION      datum *define_word(char *word)
 *  
 *  PURPOSE       Queries the user about a word, in order to build
 *  and store a lexical entry for it in the internal
 *  and disk-based databases.
 *  
 *  INPUT         char *word     --    an unknown word.
 *  
 *  OUTPUT        Indicates completion of definition, and writes the unknown
 *  word to a file on disk for examination by the lexicon
 *  maintainer, or indicates failure.
 * 
 *  SIDE EFFECTS  See above.
 */

datum *
  define_word(word, merge, query)
char  *word;
int   merge;
int   query;
{
  char		buffer[BUFFER_LEN], next_seg[BUFFER_LEN];
  datum		bufkey, bufrec;
  int		i, stored;
  static datum	fetched;
  
  bufrec.dptr = NEWVEC(char, BUFFER_LEN);
  
  bufkey.dptr = word;
  bufkey.dsize = strlen(word);
  
  if (query) {
    /* 
     * Query user for a lexical definition of an unknown word. Write this 
     * information to the lexical entries file, and to the internal hash table
     * for lexical entries.
     */
    
    /* Should ask clear, minimally-technical questions about the lexeme! */
    
    printf("Enter each low-level categorization (segments separated by spaces) \nfor '%s' on its own line, end with '.' on a line by itself. \nproof: define> ", word);
    gets(buffer);
  }
  else
    strcpy(buffer, command[comm]);
  
  while (buffer[0] != '.') /* user has yet to provide end-of */
    /* definition signal */ {
      
      /* provide translation to segment tokens */
      
      while ((buffer[0] != '\0') && (buffer[0] != ':') && (buffer[0] != '.')) 
	/* there is input left to translate in this categorization */
	{
	  /* find the next segment of the current categorization */
	  for (i = 0; (buffer[i] != ' ') && (buffer[i] != ',') && 
	       (buffer[i] != '\0') && (buffer[i] != '.') && 
	       (buffer[i] != ':'); i++)
	    ;
	  strncpy(next_seg, buffer, i);
	  next_seg[i] = '\0';
	  
	  if (buffer[i] == '\0')  {
	    buffer[0] = '\0';
	  }
	  else /* hack off the newly-read segment */ {
	    if (buffer[i] == '.')
	      buffer[0] = '.';
	    else if (buffer[i] == ':')
	      strcpy(buffer, buffer + i);
	    else
	      strcpy(buffer, buffer + i + 1);
	  }
	  
	  /* Here's my dumb definition interface... */
	  if (strcmp(next_seg, "A") == 0)  {
	    bufrec.dptr[strlen(bufrec.dptr)] = A;
	  }
	  else if (strcmp(next_seg, "AG") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = AG;
	  }
	  else if (strcmp(next_seg, "AV") == 0)	{
	    bufrec.dptr[strlen(bufrec.dptr)] = AV;
	  } 
	  else if (strcmp(next_seg, "B") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = B;
	  }
	  else if (strcmp(next_seg, "C") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = C;
	  }
	  else if (strcmp(next_seg, "_C") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = __C;
	  }
	  else if (strcmp(next_seg, "CA") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = CA;
	  }
	  else if (strcmp(next_seg, "_CA") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = _CA;
	  }
	  else if (strcmp(next_seg, "CH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = CH;
	  }
	  else if (strcmp(next_seg, "C_H") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = C_H;
	  }
	  else if (strcmp(next_seg, "_CH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = _CH;
	  }
	  else if (strcmp(next_seg, "_C_H") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = _C_H;
	  }
	  else if (strcmp(next_seg, "D") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = D;
	  }
	  else if (strcmp(next_seg, "DECL") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = DECL;
	  }
	  else if (strcmp(next_seg, "DO") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = DO;
	  }
	  else if (strcmp(next_seg, "HV") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = HV;
	  }
	  else if (strcmp(next_seg, "INTERROG") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = INTERROG;
	  }
	  else if (strcmp(next_seg, "M") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = M;
	  }
	  else if (strcmp(next_seg, "N") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = N;
	  }
	  else if (strcmp(next_seg, "NM") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = NM;
	  }
	  else if (strcmp(next_seg, "NOM") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = NOM;
	  }
	  else if (strcmp(next_seg, "NP") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = NP;
	  }
	  else if (strcmp(next_seg, "P") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = P;
	  }
	  else if (strcmp(next_seg, "P1") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = P1;
	  }
	  else if (strcmp(next_seg, "P3") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = P3;
	  }
	  else if (strcmp(next_seg, "OB") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB;
	  }
	  else if (strcmp(next_seg, "OB_P3") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_P3;
	  }
	  else if (strcmp(next_seg, "OB_P1") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_P1;
	  }
	  else if (strcmp(next_seg, "OB_PH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_PH;
	  }
	  else if (strcmp(next_seg, "OB_S1") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_S1;
	  } 
	  else if (strcmp(next_seg, "OB_S3") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_S3;
	  }
	  else if (strcmp(next_seg, "PH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = PH;
	  }
	  else if (strcmp(next_seg, "P_H") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = P_H;
	  }
	  else if (strcmp(next_seg, "PNM") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = PNM;
	  }
	  else if (strcmp(next_seg, "S") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = S;
	  }
	  else if (strcmp(next_seg, "S1") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = S1;
	  }
	  else if (strcmp(next_seg, "S3") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = S3;
	  }
	  else if (strcmp(next_seg, "SC") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = SC;
	  }
	  else if (strcmp(next_seg, "SH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = SH;
	  }
	  else if (strcmp(next_seg, "S_H") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = S_H;
	  }
	  else if (strcmp(next_seg, "TO") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = TO;
	  }
	  else if (strcmp(next_seg, "U") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = U;
	  }
	  else if (strcmp(next_seg, "V") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = V;
	  }
	  else if (strcmp(next_seg, "VI") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = VI;
	  }
	  else if (strcmp(next_seg, "WH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = WH;
	  }
	  else if (strcmp(next_seg, "OB_WH") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = OB_WH;
	  }
	  else if (strcmp(next_seg, "W_H") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = W_H;
	  }
	  else if (strcmp(next_seg, "WP") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = WP;
	  }
	  else if (strcmp(next_seg, "WS") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = WS;
	  }
	  else if (strcmp(next_seg, "WU") == 0) {
	    bufrec.dptr[strlen(bufrec.dptr)] = WU;
	  }
	  else 
	    {
	      printf("The segment '%s' is not recognized... please notify the lexicon-maintenence staff.\n", next_seg);
	      /* ignore the entire definition */
	      buffer[0] = '\0';
	    }
	}
      
      /* separate the categorizations */
      bufrec.dptr[strlen(bufrec.dptr)] = CAT_BUFFER;
      
      if (query)
	{
	  printf("proof: define> ");
	  gets(buffer);
	}
      else
	if (buffer[0] != '.')
	  strcpy(buffer, buffer + 1);
    }
  
  if (merge) {
    fetched.dptr = LookupHash(table, word);
    if (fetched.dptr == NULL)
      fetched = gdbm_fetch(lexicon, bufkey);
    if (fetched.dptr != NULL)
      strcat(bufrec.dptr, fetched.dptr);
  }
  
  /* terminate the definition */
  bufrec.dptr[strlen(bufrec.dptr)] = NULL;
  
  /* ndbm complains unless the following condition is met */
  bufrec.dsize = strlen(bufrec.dptr) + 1; 
  
  stored = gdbm_store(lexicon, bufkey, bufrec, GDBM_REPLACE);
  fetched = gdbm_fetch(lexicon, bufkey);
  
  if ((stored == 0) && (fetched.dptr != NULL)) {
    if (i > 0)
      printf("Defined.\n");
    else
      {
	printf("Definition for '%s' unchanged.\n", word);
	FREELOCVEC(bufrec.dptr);
	return(0);
      }
    
    if (merge == NOMERGE)
      curr_condition = VA_NEW;
  }
  
  else
    printf("Error in database / hashtable update.");
  
  /* huh -- why am I using 'memcpy' ? */    
  
  memcpy(buffer, (void *) bufrec.dptr, bufrec.dsize); 
  
  InsertHash(table, bufkey.dptr, bufrec.dptr);
  
  FREELOCVEC(bufrec.dptr);
  
  return(&fetched);
  
} /* end: define_word() : finished */


/* 
 *  FUNCTION      void display_segments(int  segments[MAX_SEGMENTS]
 *                                                   [MAX_SEGMENTS])
 *  
 *  PURPOSE       to display the segments of a sentence start
 *  
 *  INPUT         int  **segments
 *  
 *  OUTPUT        See above.
 *  
 *  SIDE EFFECTS  None.
 */

extern void
  display_segments(segments)
DEFINITION  segments;
{
  int cat_index, seg_index;
  
  for (cat_index = 1; cat_index <= segments[0][0]; cat_index++) {
    printf("     Meaning #%d: ", cat_index);
    for (seg_index = 1; seg_index <= segments[cat_index][0]; seg_index++) {
      if (segments[cat_index][seg_index] == A) {
	printf("A ");
      }
      else if (segments[cat_index][seg_index] == AG)
	{
	  printf("AG ");
	}
      else if (segments[cat_index][seg_index] == AV) {
	printf("AV ");
      }
      else if (segments[cat_index][seg_index] == B) {
	printf("B ");
      }
      else if (segments[cat_index][seg_index] == C) {
	printf("C ");
      }
      else if (segments[cat_index][seg_index] == __C) {
	printf("__C ");
      }
      else if (segments[cat_index][seg_index] == CA) {
	printf("CA ");
      }
      else if (segments[cat_index][seg_index] == _CA) {
	printf("_CA ");
      }
      else if (segments[cat_index][seg_index] == CH) {
	printf("CH ");
      }
      else if (segments[cat_index][seg_index] == C_H) {
	printf("C_H ");
      }
      else if (segments[cat_index][seg_index] == _CH) {
	printf("_CH ");
      }
      else if (segments[cat_index][seg_index] == _C_H) {
	printf("_C_H ");
      }
      else if (segments[cat_index][seg_index] == D) {
	printf("D ");
      }
      else if (segments[cat_index][seg_index] == DECL) {
	printf("DECL ");
      }
      else if (segments[cat_index][seg_index] == DO) {
	printf("DO ");
      }
      else if (segments[cat_index][seg_index] == HV) {
	printf("HV ");
      }
      else if (segments[cat_index][seg_index] == INTERROG) {
	printf("INTERROG ");
      }
      else if (segments[cat_index][seg_index] == M) {
	printf("M ");
      }
      else if (segments[cat_index][seg_index] == N) {
	printf("N ");
      }
      else if (segments[cat_index][seg_index] == NM) {
	printf("NM ");
      }
      else if (segments[cat_index][seg_index] == NOM) {
	printf("NOM ");
      }
      else if (segments[cat_index][seg_index] == NP) {
	printf("NP ");
      }
      else if (segments[cat_index][seg_index] == P) {
	printf("P ");
      }
      else if (segments[cat_index][seg_index] == P1) {
	printf("P1 ");
      }
      else if (segments[cat_index][seg_index] == P3) {
	printf("P3 ");
      }
      else if (segments[cat_index][seg_index] == OB) {
	printf("OB ");
      }
      else if (segments[cat_index][seg_index] == OB_PH) {
	printf("OB_PH ");
      }
      else if (segments[cat_index][seg_index] == OB_P1) {
	printf("OB_P1 ");
      }
      else if (segments[cat_index][seg_index] == OB_P3) {
	printf("OB_P3 ");
      }
      else if (segments[cat_index][seg_index] == OB_S1) {
	printf("OB_S1 ");
      } 
      else if (segments[cat_index][seg_index] == OB_S3) {
	printf("OB_S3 ");
      }
      else if (segments[cat_index][seg_index] == OB_WH) {
	printf("OB_WH ");
      }
      else if (segments[cat_index][seg_index] == PH) {
	printf("PH ");
      }
      else if (segments[cat_index][seg_index] == P_H) {
	printf("P_H ");
      }
      else if (segments[cat_index][seg_index] == PNM) {
	printf("PNM ");
      }
      else if (segments[cat_index][seg_index] == S) {
	printf("S ");
      }
      else if (segments[cat_index][seg_index] == S1) {
	printf("S1 ");
      }
      else if (segments[cat_index][seg_index] == S3) {
	printf("S3 ");
      }
      else if (segments[cat_index][seg_index] == SC) {
	printf("SC ");
      }
      else if (segments[cat_index][seg_index] == SD) {
	printf("SD ");
      }
      else if (segments[cat_index][seg_index] == SH) {
	printf("SH ");
      }
      else if (segments[cat_index][seg_index] == OB_SH) {
	printf("OB_SH ");
      }
      else if (segments[cat_index][seg_index] == S_H) {
	printf("S_H ");
      }
      else if (segments[cat_index][seg_index] == TO) {
	printf("TO ");
      }
      else if (segments[cat_index][seg_index] == U) {
	printf("U ");
      }
      else if (segments[cat_index][seg_index] == V) {
	printf("V ");
      }
      else if (segments[cat_index][seg_index] == VI) {
	printf("VI ");
      }
      else if (segments[cat_index][seg_index] == WH) {
	printf("WH ");
      }
      else if (segments[cat_index][seg_index] == W_H) {
	printf("W_H ");
      }
      else if (segments[cat_index][seg_index] == WP) {
	printf("WP ");
      }
      else if (segments[cat_index][seg_index] == WS) {
	printf("WS ");
      }
      else if (segments[cat_index][seg_index] == WU) {
	printf("WU ");
      }
    }
    printf(".\n");
  }
}


/*
  
  FUNCTION      STATE  *expand(sentence)
  
  PURPOSE       To expand an input sentence to a STATE.
  
  INPUT		char   *sentence   -   an input sentence.
  
  OUTPUT	A STATE pointer.
  
  SIDE EFFECTS  None.
  
  */

STATE *
  expand(sentence)
char  *sentence;
{
  STATE *lexsentence;
  
  lexsentence = NEW(STATE);
  lexsentence->start = NEW(LEXENTRY);
  lexsentence->start->surface[0] = '\0';
  lexsentence->start->segments[0][0] = 0;
  lexsentence->rest = sentence;
  
  return(lexsentence);
} /* end: expand() : finished */


/* 
  
  FUNCTION      void  get_first_surface(char *first_surface, *words)
  
  PURPOSE       To copy the first surface of "words" to "first_surface".
  It is assumed that 'words' is a null-terminated string.
  
  INPUT         char *words  -  pointer to some words.
  
  OUTPUT        Returns the first surface in a set of words if it is
  of length SURFACE_LEN or smaller, else it returns a
  null surface.
  
  SIDE EFFECTS  'words' gets its first word hacked off.
  
  */

void
  get_first_surface(first_surface, words)
char *first_surface, *words;
{
  int          len = 0;


  /* remove leading punctuation and whitespace */
  while ((words[0] == ':') || (words[0] == ' ') || (words[0] == '\n'))
    strcpy(words, words + 1);

  /* end-of-sentence characters have lexical entries, too */
  if ((words[len] == '!') || (words[len] == ';') ||
      (words[len] == '?') || (words[len] == '.'))
    first_surface[len++] = words[len];

  else  
    {
      for (len = 0; ((words[len] != '!') && (words[len] != ';') &&
		     (words[len] != '?') && (words[len] != ' ') && 
		     (words[len] != '.') && (words[len] != '\n') && 
		     (words[len] != '\0') && (words[len] != ':')); len++)
	first_surface[len] = words[len];
    }
  
  first_surface[len] = '\0';
  
  /*
    if the first surface is too long then return a null surface, else
    return the first surface of "words"
    */
  if (len > SURFACE_LEN)
    first_surface[0] = '\0';
  
  if (words[len] == '\0')  {
    /* empty "words" */
    words[0] = '\0';
  }
  else /* hack off leading surface from wordlist */ {
    strcpy(words, words + len); 
  }
  
} /* end: void  get_first_surface(char *first_surface, *words): finished */


/* 
  
  FUNCTION      char *get_last_surface(char *last_surface, *words)
  
  PURPOSE       To get the surface of the last word in a set of words.
  It is assumed that 'words' is null-terminated.
  
  INPUT         char *words  -  pointer to some words.
  
  OUTPUT        Returns the surface of the last word in a set of words,
  with no restriction on length (as opposed to 
  get_first_surface() ).
  
  SIDE EFFECTS  None.
  
  */

char *
  get_last_surface(words)
char  *words;
{
  char  *last_surface;
  int   index;
  
  last_surface = NEWVEC(char, BUFFER_LEN);
  
  for (index = strlen(words); (words[index] != ' ') && (index > 0); index--)
    ;
  
  if (index == 0) /* 'words' is composed of one word only */ {
    strcpy(last_surface, words);
  }
  else /* get last word of a series of words */ {
    strcpy(last_surface, words + index + 1);
  }
  
  return(last_surface);
}


/* 
 *  FUNCTION      LEXENTRY *get_next_word(words)
 *  
 *  PURPOSE       To get the next word of a set of words, in LEXENTRY form.
 *  
 *  INPUT		char *words  -  pointer to some words.
 *  
 *  OUTPUT	Returns a pointer to a LEXENTRY representing the next word,
 *  of a given set of words, else attempts to obtain a lexical
 *  definition for the word from the user.
 *  
 *  SIDE EFFECTS  The given wordlist gets its first word hacked off, via
 *  get_first_surface().
 */   

LEXENTRY *
  get_next_word(words)
char *words;
{
  BUFFER    buffer;
  LEXENTRY  *next_lex;
  char      *result;
  datum     bufkey, bufrec;
  int       i, unstored = 0;
  
  next_lex = NEW(LEXENTRY);
  /* the number of current meanings is 0 */
  next_lex->segments[0][0] = 0;
  
  /* get a pointer to the next word, and hack it from the word list */
  get_first_surface(next_lex->surface, words);
  if (strlen(next_lex->surface) > MAX_LEN) {
    fprintf(stderr, "Long lexeme is: '%s'.\n", next_lex->surface);
    curr_condition = INV_LONG;
    return(next_lex);
  }
  
  /*
   * See if a lexical entry for the next word already exists in the internal 
   * hash table.
   */
  
  result = LookupHash(table, next_lex->surface);
  
  if (result != NULL) {
    if (result[0] != 0)
      {
	/* unpack buffer to a LEXENTRY, return as next_lex */	    
	unpack(result, next_lex);
	return(next_lex);
      }
    else
      unstored = 1;
  }
  
  /* 
   * Read entry for next_lex->surface from the lexical entries file, using
   * dbm(3). If this fails, convert the word to all lower-case, if it is 
   * not in that form already, and try again. If that fails, provide the 
   * user the chance to define the word for the lexicon. If the user then 
   * chooses not to define the word, move on to the next sentence, keeping 
   * track of the unknown word. If the fetch succeeds, load a lexical entry, 
   * and insert that into the local hash table, using Jordan Lampe's hashing 
   * functions. 
   */
  
  if (!unstored) /* damn... I guess this is a stupid hack */ {
    bufkey.dptr = next_lex->surface;
    bufkey.dsize = strlen(next_lex->surface);
    
    bufrec = gdbm_fetch(lexicon, bufkey);
    
    if (bufrec.dptr == NULL)
      {
	/* 
	 * convert the capital letters in the next word to lower-case, 
	 * get rid of puncuation marks 
	 */
	strcpy(buffer, next_lex->surface);
	
	for (i = 0; i < strlen(buffer); i++) {
	  if (('A' <= buffer[i]) && (buffer[i] <= 'Z'))
	    buffer[i] += 32;
	}
	
	bufkey.dptr = buffer;
	bufrec = gdbm_fetch(lexicon, bufkey);
      }
    else
      if (bufrec.dptr[0] == 0)
	unstored = 1;
  }      
  
  if ((unstored) || (bufrec.dptr == NULL))  
    /* possible second disk search failed */ 
    {
      if (strncmp(command[comm], "undefine", 8) == 0)
	{
	  printf("The lexeme '%s' is currently undefined.\n", 
		 next_lex->surface);
	  return(next_lex);
	}  
      
      if (strncmp(command[comm], "define", 6) != 0)
	{
	  printf("The lexeme '%s' is not defined in the lexicon. Define? [y/n] > ", next_lex->surface);
	  gets(buffer);
	  
	  if ((buffer[0] == 'n') || (buffer[0] == 'N')) {
	    /* indicate failure */
	    curr_condition = INV_UNKNOWN;
	    sprintf(error, "The lexeme '%s' remained undefined.\n", 
		    next_lex->surface);
	    return(next_lex);
	  }
	}
      bufrec = *define_word(next_lex->surface, NOMERGE, QUERY);
    }
  
  /* the next word is now defined */
  
  /* huh -- why am I using 'memcpy' ? */    
  memcpy(buffer, (void *) bufrec.dptr, bufrec.dsize); 
  
  buffer[bufrec.dsize] = '\0';
  
  InsertHash(table, next_lex->surface, buffer);
  
  /* expand to a LEXENTRY */
  unpack(buffer, next_lex);
  return(next_lex);
  
} /* end: get_next_word() : finished */


/* 
 *  FUNCTION      void print_status(LEXENTRY *start, LEXENTRY *next)
 *  
 *  PURPOSE       Print the parse status 
 *  
 *  INPUT         LEXENTRY *start, *next  -- the start and next of a parse.
 *  
 *  OUTPUT        None.
 *  
 *  SIDE EFFECTS  None.
 */

void
  print_status(start, next)
LEXENTRY *start, *next;
{
  printf("The start is '%s':\n",start->surface);
  display_segments(start->segments);
  printf("The next lexeme is '%s':\n", 
	 next->surface);
  display_segments(next->segments);
}


/*
  
  FUNCTION      void read_it(STATE *sentence)
  
  PURPOSE       To apply the function combine() to the evaluated versions of
  a sentence start and the next word, call the result a
  sentence start and apply combine() to it and the next word,
  etc. This continues until either:
  1) There is no next word, in which case the last generated
  sentence start is returned, or
  2) combine() fails, in which case an error message and the
  successful part of the derivation are returned (if I can 
  cram it into a STATE).
  
  INPUT         STATE    *sentence  -  the current sentence start and the 
  rest of the sentence. 
  
  OUTPUT        See above.
  
  SIDE EFFECTS  None.
  
  */

void
  read_it(sentence)
STATE    *sentence;
{
  LEXENTRY  *next_word; 
  /* 
   * space is allocated for "next_word" by the
   * call to get_next_word() below 
   */
  BUFFER    buffer;
  int       i, j;
  

  /* space was allocated for "sentence" by expand() */

  if ((sentence->rest[0] < 41) || (sentence->rest[0] > 90))
    {
      strcpy(error, "The sentence '");
      strcat(error, sentence->rest);
      strcat(error, "' does not begin with a capitalized word, and is therefore  invalid.\n");
      curr_condition = INVALID;
      return;
    }
  if ((sentence->rest[strlen(sentence->rest) - 1] != '.') &&
      (sentence->rest[strlen(sentence->rest) - 1] != '?') && 
      (sentence->rest[strlen(sentence->rest) - 1] != '!'))
    {
      strcpy(error, "The sentence '");
      strcat(error, sentence->rest);
      strcat(error, "' does not end with a period, exclamation mark, or          question mark, and is therefore invalid.\n");
      curr_condition = INVALID;
      return;
    }
  
  /* assume sentence to be correct, initially */
  curr_condition = VALID; 
  
  /* use the next (first) word's segments. */
  sentence->start = get_next_word(sentence->rest);
  if (curr_condition == INV_UNKNOWN || 
      curr_condition == INV_LONG)
    return;

  /*
   * keep attempting to parse the input, while more input exists, or 
   * backtracking may be necessary
   */
  
  while ((sentence->rest[0] != '\0') || (curr_condition == INVALID))
    {
      /* save current start segments for possible future backtracking. */
      copy_segments(history[curr_deriv].segments, sentence->start->segments);
      history[0].segments[0][0] = curr_deriv;

      /* backtrack if necessary */
      while ((curr_condition != VALID) && 
	     (curr_condition != VA_NEW))
	{
	  if (curr_deriv == 1) 
	    /* 
	     * there isn't room to backtrack,
	     * hence the current sentence is not
	     * grammatically correct
	     */
	    return;
	  else 
	    {
	      backtracked++;
	      curr_deriv--;

	      /* 
	       * refresh last valid start surface and segments
	       */

	      strcpy(sentence->start->surface, history[curr_deriv].start);
	      strcpy(sentence->rest, history[curr_deriv].rest);
	      copy_segments(sentence->start->segments,
			    history[curr_deriv].segments);
	      sentence->start->segments[0][0] =
		history[curr_deriv].segments[0][0];
	      
	      next_word = get_next_word(history[curr_deriv].next);
	      printf("\n      Backtracking!\n");
	      combine(sentence->start, next_word);
	    }
	}
      
      if (!backtracked)
	{
	  next_word = get_next_word(sentence->rest);
	  if (curr_condition == INV_UNKNOWN || 
	      curr_condition == INV_LONG)
	    return;
	  
	  combine(sentence->start, next_word);
	}
      
      /* 
       * if at least one rule was successful, modify the sentence 
       * start's surface and the history
       */
      
      if (curr_condition == VALID)
	{
	  strcpy(history[curr_deriv].start, sentence->start->surface);
	  strcpy(history[curr_deriv].rest, sentence->rest);
	  strcpy(history[curr_deriv].next, next_word->surface);
	  strcat(sentence->start->surface, " ");
	  strcat(sentence->start->surface, next_word->surface);
	  curr_deriv++;
	  backtracked = 0;
	}
    }

  if (curr_condition != INVALID)
    {
      printf("\n[END]: ");
      i = history[curr_deriv - 1].rules[0];
      j = history[curr_deriv - 1].rules[i];
      switch(j)
	{
	case ADD_MINUS_NOM:
	  printf("add minus nominative\n");
	  break;
	  
	case ADD_MINUS_VERB:
	  printf("add minus verb\n");
	  break;

	case COMPLETE:
	  printf("complete\n");
	  break;
	  
	case DET_PLUS_ADJ:
	  printf("determiner plus adjective\n");
	  break;

	case DET_PLUS_NOUN:
	  printf("determiner plus noun\n");
	  break;
      
	case FVERB_PLUS_MAIN:
	  printf("finite verb plus main clause\n");
	  break;
      
	case FVERB_PLUS_NOM:
	  printf("finite verb plus nominative\n");
	  break;

	case NOM_PLUS_FVERB:
	  printf("nominative plus finite verb\n");
	  break;
      
	case NOUN_PLUS_PNM:
	  printf("noun plus post-nominal phrase\n");
	  break;
	  
	case NOM_PLUS_RELPRO:
	  printf("nominative plus relative pronoun\n");
	  break;
	  
	case PREP_PLUS_NP:
	  printf("preposition plus noun phrase\n");
	  break; 
      
	case START:
	  printf("start\n");
	  break;      
        
	case START_MINUS_RELCL:
	  printf("start minus relative clause\n");
	  break;

	case START_MINUS_SUBCL:
	  printf("start minus subclause\n");
	  break;

	case TOP_MINUS_MAIN:
	  printf("top minus main clause\n");
	  break;

        case WH_PLUS_VERB:
	  printf("wh-phrase plus verb\n");
	  break;

	  default:
	  printf("Scan history contains an invalid rule number: %d.\n", 
		 j);
	  break;
	}
      printf("\nThe complete sentence is '%s', with segments:\n", 
	     sentence->start->surface);
      display_segments(sentence->start->segments);
    }
  
  FREELOCVEC(sentence);
  
} /* end: void  read_it() : finished */


/* 
 *  FUNCTION      int  splice_category(CATEGORY  category,
 *                                     int  position,
 *                                     CATEGORY  segments)
 *   
 *  PURPOSE       to splice some segments into a category at a given position.
 *   
 *  INPUT	  see above
 *  
 *  OUTPUT        returns 1 if successful, 0 otherwise
 * 
 *  SIDE EFFECTS  the category gets segments spliced into it.
 */

extern int
  splice_category(category, position, segments)
CATEGORY  category, segments;
int       position;
{
  int  seg_index, i;
  

  i = category[0];
  
  if (category[0] + segments[0] <= MAX_SEGMENTS)
    category[0] += segments[0];
  else
    return(0);
  
  /* avoid Saber whining */
  for (seg_index = i + 1; seg_index <= i + segments[0];
       seg_index++)
    category[seg_index] = NOSEG;

  for (seg_index = 1; seg_index <= segments[0]; seg_index++)
    {
      /* preserve old segments after position */
      for (i = category[0]; i > position; i--)
	category[i] = category[i - 1];
      category[position++] = segments[seg_index];
    }
  
  return(1);
} /* end: splice_category(category, position, segments) : finished */


/*
  
  FUNCTION      int  undefine_word(char  *word)
  
  PURPOSE       To remove one or more of a word's definitions from the 
  dictionary
  
  INPUT		char  *word
  
  OUTPUT        returns 1 if successful, 0 otherwise
  
  */

int
  undefine_word(word, mode)
char  *word;
int   mode;
{
  int           index = 0, dec = 0, cindex, sindex, stored;
  char          last, next;
  BUFFER        buffer;
  LEXENTRY      *lex;
  datum         bufkey, bufrec;
  static datum  fetched;
  
  lex = NEW(LEXENTRY);
  bufrec.dptr = NEWVEC(char, BUFFER_LEN);
  
  /* verify word from dictionary */
  lex = get_next_word(word);
  if (lex->segments[0][0] == 0)
    return(1);

  if (lex->segments[0][0] > 1) {
    /* indicate current meanings, poll for ones to delete */
    printf("   Here are the current meanings for '%s'. Indicate which ones you would like to remove, by indicating each appropriate number (or zero for all definitions, \"c\" to cancel), separated by spaces.\n", lex->surface);
    display_segments(lex->segments);
    printf("proof: undefine> ");
    gets(buffer);
  }
  else {
    buffer[index] = '1';
    buffer[index + 1] = '\0';
  }
  
  while (buffer[index] != '\0') {
    if (index == 0)
      last = buffer[index] - 48;

    if (buffer[index] == 'c')
      {
	printf("Definition for '%s' unchanged.\n", lex->surface);
	return(-1);
      }
    
    else if (buffer[index] == ' ')
      index++;

    else if (buffer[index] == '0')
      {
	mode = ALL;
	dec++;
	buffer[index] = '\0';
      }
    
    else if ((buffer[index] < 49) || (buffer[index] > 
				      (lex->segments[0][0] + 48)))
      {
	printf("   The lexeme '%s' has no meaning #%c.\n", lex->surface, 
	       buffer[index]);
	printf("proof: undefine> ");
	gets(buffer);
      }
    else
      {
	/* remove categorization buffer[index] */
	if ((buffer[index] - 48) > last)
	  next = buffer[index] - 49;
	else
	  next = buffer[index] - 48;
	
	for (cindex = next;
	     (cindex + 1) <= lex->segments[0][0]; cindex++)  {
	  for (sindex = 0; sindex <= lex->segments[cindex + 1][0]; 
	       sindex++)
	    {
	      lex->segments[cindex][sindex] = 
		lex->segments[cindex + 1][sindex];
	    }
	}
	dec +=1;
	index += 1;
	last = next;
	if (lex->segments[0][0] == 0)
	  buffer[index] = '\0';
      }	  
  }
  
  if (dec > 0)
    {
      /* decrement the number of categorizations */
      lex->segments[0][0] -=dec;
      
      /* put the new definition back in the hashtable and dictionary */
      /* read the defintion into the 'ndbm' buffer */
      bufkey.dptr = lex->surface;
      bufkey.dsize = strlen(lex->surface);
      index = 0;

      /* ...must be some way to just remove the local hashtable entry...? */
      if (mode == ALL)
	{
	  lex->segments[0][index++] = 0;
	  gdbm_delete(lexicon, bufkey);
	}
      else
	{
	  for (cindex = 1; cindex <= lex->segments[0][0]; cindex++)
	    {
	      for (sindex = 1; sindex <= lex->segments[cindex][0]; sindex++)
		bufrec.dptr[index++] = lex->segments[cindex][sindex];
	      bufrec.dptr[index++] = CAT_BUFFER;
	    }
	  
	  bufrec.dptr[index] = '\0';
	  bufrec.dsize = strlen(bufrec.dptr) + 1; 
	  
	  stored = gdbm_store(lexicon, bufkey, bufrec, GDBM_REPLACE);
	  fetched = gdbm_fetch(lexicon, bufkey);
	  
	  if ((stored != 0) || (fetched.dptr == NULL)) {
	    printf("Error in database / hashtable update.");
	    return(0);
	  }
	}
      
      InsertHash(table, lex->surface, bufrec.dptr);
    }
  else
    {
      printf("Definition for '%s' unchanged.\n", lex->surface);
      FREELOCVEC(bufrec.dptr);
      return(0);
    }
  
  FREELOCVEC(bufrec.dptr);
  return(1);
}


/*
  
  FUNCTION      void  unpack(char *record, LEXENTRY *lex)
  
  PURPOSE       To unpack lexical information from the database.
  
  INPUT		char     *record  -- source
  LEXENTRY *lex     -- target
  
  OUTPUT        Returns the unpacked LEXENTRY
  
  */

void
  unpack(record, lex)
char        *record;
LEXENTRY    *lex;
{
  /* 
    rec indexes the characters in 'record', cat_index is the 
    categorization index for the unpacked lexentry, seg_index is the 
    segment index for the unpacked lexentry
    */
  int         rec, cat_index = 1, seg_index = 1; 
  
  for (rec = 0; cat_index < MAX_CATEGORIES && record[rec] != NULL; rec++) {
    if (record[rec] == CAT_BUFFER)
      {
	/* 
	  Terminate the categorization by writing the length to 
	  lex->segments[cat_index][0]; move to the next categorization.
	  */
	if (seg_index - 1)
	  lex->segments[cat_index++][0] = seg_index - 1;
	/* reset the segment index */
	seg_index = 1;
      }
    else
      lex->segments[cat_index][seg_index++] = record[rec];
  }
  
  /*
    terminate the definition by setting the first segment of the
    diagnostic categorization to the number of actual categorizations
    */
  lex->segments[0][0] = --cat_index;
  
} /* end: LEXENTRY *unpack(char *record, LEXENTRY *lex) : finished */


/* 
 *  FUNCTION      void  main(int argc, char *argv[])
 *  
 *  PURPOSE       To read each sentence in an input file or the standard 
 *                input, and generate data for lexical maintainence, any
 *                requested statistics, and/or to perform various actions
 *                as per supplied flags or interactive commands.
 *  
 *  INPUT	  int argc, char *argv[]
 *  
 *  FLAGS         
 *  -d <word>     : define word
 *  -f <file>     : scan from file
 *  -h            : help
 *  -l <word>     : look up word
 *  -r <word>     : redefine word
 *  -s <text>     : scan text
 *  -u <word>     : undefine word
 *  -U <word>     : undefine word completely
 *  
 *  EXIT SIGNALS  0 : success
 *                1 : file not found
 *                2 : error during file read
 *  
 *  OUTPUT        See above.
 * 
 *  TODO          Get system() calls to 'more' to work via "demo"
 */

void
  main(int argc, char *argv[])
{
  int            i, numcomm = 0, text_length, cont = 0;
  BUFFER         buffer1, buffer2, sentence, text, word, filename;
  LEXENTRY       *lex;
  FILE           *file;
  char           *prog = argv[0], c, d = -1;

  filename[0] = '\0';
  
  /* create a new local hash table */
  table = NewHash(START_HASH_LEN);
  
  /* create derivation history */
  history = NEWVEC(HISTENTRY, MAX_DERIVATIONS);
  
  /* 
    define the lexicon to be used. NOTE: the file is not designated (as far as
    the "ndbm" routines are concerned) as "in use" 
    */
  lexicon = gdbm_open(LEX_FILENAME, 512, GDBM_WRCREAT, 0666, 0);

  printf("\n");
  
  if (argc > 1) /* compile request array */ {
    for (i = 1; i < argc; i++)
      {
	if (strcmp(argv[i], "-d") == 0) /* a define was requested */ {
	  strcpy(command[numcomm], "define ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
	if (strcmp(argv[i], "-f") == 0)
	  /* an scan from disk was requested */ {
	    strcpy(command[numcomm], "scan -f ");
	    strcat(command[numcomm++], argv[i + 1]);
	  }
	if (strcmp(argv[i], "-h") == 0)
	  {
	    strcpy(command[numcomm], "help");
	  }
	if (strcmp(argv[i], "-l") == 0) /* a lookup was requested */ {
	  strcpy(command[numcomm], "lookup ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
	if (strcmp(argv[i], "-r") == 0) /* a redefine was requested */ {
	  strcpy(command[numcomm], "redefine ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
	if (strcmp(argv[i], "-R") == 0) /* a reorganize was requested */ {
	  strcpy(command[numcomm], "reorganize ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
	if (strcmp(argv[i], "-s") == 0)
	  /* an interactive scan was requested */ {
	    strcpy(command[numcomm], "scan ");
	    strcat(command[numcomm++], argv[i + 1]);
	  }
	if (strcmp(argv[i], "-u") == 0) /* an undefine was requested */ {
	  strcpy(command[numcomm], "undefine ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
	if (strcmp(argv[i], "-U") == 0) /* a total undefine was requested */ {
	  strcpy(command[numcomm], "Undefine ");
	  strcat(command[numcomm++], argv[i + 1]);
	}
      }
  }
  
  else {
    text[0] = '\0';
    printf("\n\n");

    if (vfork() == 0)
      execl("more", "more", MOTD, (char *)0);
    wait();
    printf("\n\n");
    if (vfork() == 0)
      execl("more", "more", GLOAT, (char *)0);
    wait();
    printf("\n\n");
  }

  /* execute command array, possibly filling it first */
  for (comm = 0; (numcomm == 0) || (comm < numcomm) || (cont); comm++) {
    if ((argc == 1) && (!cont))
      {
	if (d != 2)
	  printf("\n\nproof> ");
	else
	  {
	    d = -1;
	    comm = 0;
	    gets(command[comm]);
	    continue;
	  }
	numcomm = comm + 2;
	gets(command[comm]);
	if (command[comm] == NULL)
	  break;
	else if (strcmp(command[comm], "quit") == 0)
	  {
	    printf("\n\n");
	    exit(0);
	  }
      }

    if (strncmp(command[comm], "help", 4) == 0)
      {
	while (d != 'q')
	  {
	    if (d == -1)
	      {
		printf("Help is available on the following topics:\n\n");
		printf("1) How 'proof' works\n");
		printf("2) How to define words\n");
		printf("3) How to achieve world peace\n");
		printf("\nPlease select a topic by number, or type \"q\" to quit.\n\nproof: help> ");
	      }

	    d = getchar();
	    
	    switch (d)
	      {
	      case '1':
		/* display help on "How 'proof' works" */
		if (vfork() == 0)
		  execl("more", "more", WORKS, (char *)0);
		wait();
		printf("\n\n");
		d = -1;
		break;
		
	      case '2':
		/* display help on "How to define words" */
		if (vfork() == 0)
		  execl("more", "more", DEFINE, (char *)0);
		wait();
		printf("\n\n");
		d = -1;
		break;
		
	      case '3':
		/* display help on "How to achieve world peace" */
		if (vfork() == 0)
		  execl("more", "more", PEACE, (char *)0);
		wait();
		printf("\n\n");
		d = -1;
		break;
		

		default:
		if ((d == '1') || (d == '2'))
		  d = -1;
		break;
	      }
	    if (d == -1)
	      printf("\n\n");
	  }
	d = 2;
      }

    else if ((strncmp(command[comm], "define", 6) == 0) || 
	(strncmp(command[comm], "lookup", 6) == 0))
      {
	if ((argc > 1) && (strncmp(command[comm], "define", 6) == 0)) {
	  strcpy(command[comm], command[comm] + 7);
	  get_first_surface(word, command[comm]);
	  define_word(word, MERGE, NOQUERY);
	  strcpy(command[comm], command[comm] + 1);
	}
	else  {
	  curr_condition = VALID;
	  if (command[comm][6] == NULL)
	    {
	      printf("proof: define: lexeme> ");
	      gets(word);
	    }
	  else  
	    strcpy(word, command[comm] + 7);
	  
	  if (word[0] != NULL)
	    {
	      
	      /* 
	       * if the word currently has no meanings, then the call to 
	       * get_next_word() will take care of that, by calling 
	       * define_word(). When it returns: if 'lex' has no
	       * definitions, 
	       * then the user canceled the 'define' request; if 'lex' has
	       * some definitions, then we display the current
	       * meanings, etc. 
	       */
	      
	      strcpy(buffer1, word);
	      lex = get_next_word(word);
	      if ((curr_condition != VA_NEW) && 
		  (strncmp(command[comm], "define", 6) == 0))
		{
		  printf("   Here are the current meanings for '%s'. Any new meanings you provide, via 'define', will be merged with them. You can undefine meanings with the 'undefine' command.\n", lex->surface);
		  display_segments(lex->segments);
		  define_word(lex->surface, MERGE, QUERY);
		  curr_condition = VA_NEW;
		}
	    }
	  
	  if (curr_condition == VALID)
	    /* got a new word */
	    {
	      /* indicate current meanings */
	      printf("   Here are the current meanings for '%s'. Any new meanings you provide, via 'define', will be merged with them. You can undefine meanings with the 'undefine' command.\n", lex->surface);
	      lex = get_next_word(lex->surface);
	      display_segments(lex->segments);
	    }
	}
      }

    else if (strncmp(command[comm], "redefine", 8) == 0)
      {
	if (command[comm][8] != NULL)
	  strcpy(word, command[comm] + 9);
	else {
	  printf("proof: redefine: lexeme> ");
	  gets(word);
	}
	
	if (word[0] != NULL)
	  {
	    strcpy(buffer1, word);
	    if (undefine_word(word, SOME) == 1)
	      printf("Undefined.\n");
	    define_word(buffer1, MERGE, QUERY);
	  }
      }

    else if (strncmp(command[comm], "reorganize", 10) == 0)
      gdbm_reorganize(lexicon);
    
    else if (strncmp(command[comm], "scan", 4) == 0) 
      {
	strcpy(command[comm], command[comm] + 5);
	if (strncmp(command[comm], "-f", 2) == 0) {
	  strcpy(filename, command[comm] + 3);
	  printf("Opening file '%s'...\n", filename);
	  if ((file = fopen(filename, "r")) == NULL)
	    {
	      printf("Can't open file '%s'.\n", filename);
	      perror(prog);
	      exit(1);
	    }
	  
	  i = 0;
	  
	  while ((c = getc(file)) != EOF)
	    text[i++] = c;
	  text[i] = NULL;
	  strcpy(buffer2, text);
	}
	
	else
	  if (command[comm][0] != NULL)
	    strcpy(text, command[comm]);
	
	  else
	    {
	      /*
	       * query for a test-string (this section will be replaced by code
	       * which reads text in from files, or from user input from a 
	       * front end)
	       */
	      
	      printf("proof: scan: text> ");
	      gets(text);
	    }
	
	/* 
	 * Chop a sentence off from the text and feed it to read_it(), 
	 * unless there isn't any more input, in which case quit.
	 */
	
	text_length = strlen(text);
	
	for (i = 0; (text[i] != '!') && (text[i] != '?') &&
	     (text[i] != '.') && (text[i] != ';') && (text[i] != ':') &&
	     (i < text_length); i++)
	  ;
	
	strncpy(sentence, text, i);

	if ((text[i] == '.') || (text[i] != '!') || (text[i] != '?') ||
	    (text[i] != ';'))
	  sentence[i++] = text[i];
	
	sentence[i] = '\0';
	strcpy(text, text + i);
	if (text[0] == ' ')
	  strcpy(text, text + 1);
	
	strcpy(buffer1, sentence);
	
	/* clear and initialize the history */
	curr_deriv = 1;
	backtracked = 0;
	FREELOCVEC(history);
	history = NEWVEC(HISTENTRY, MAX_DERIVATIONS);
	history[0].rules[0] = 1;
	history[0].rules[1] = START;
	
	read_it(expand(sentence));
	
	printf("\n\n");
	
	if ((curr_condition != VALID) && 
	    (curr_condition != VA_NEW))
	  /* the sentence is neither valid nor new */ {
	    if (curr_condition == INV_LONG) 
	      {
		fprintf(stderr, "   Analysis of the sentence '%s' stopped because of a lexeme which was too long.\n\n", buffer1);
		exit(1);
	      }
	    else
	      {
		fprintf(stderr, "   Analysis of the sentence '%s' stopped.\n   %s\n", buffer1, error);
	      }
	  }
	else {
	  /* sentence was parseable, comment on it */
	  printf("Finished with sentence '%s'.\n", buffer1);
	  
	  /* 		
	   * Load next sentence, if one exists, and continue, else
	   * quit, saving a list of unknown words for lexicon
	   * maintainence, and notifying the user of questionable
	   * sentences.
	   */

	  /* remove leading punctuation and whitespace */
	  while ((text[0] == ':') || (text[0] == ' ') || (text[0] == '\n'))
	    strcpy(text, text + 1);
	  
	  if ((text[0] != '\0') && (text[1] != '\0'))
	    {
	      cont = 1;
	      strcpy(command[comm + 1], "scan ");
	      strcat(command[comm + 1], text);
	    }
	  else
	    {
	      if (cont)
		{
		  if (filename[0] != '\0')
		    /* reading from a file */
		    {
		      printf("Finished with text '%s'.\n", buffer2);
		      if (argc == 1)
			numcomm = 0;
		      else
			exit(0);
		    }
		  else
		    printf("Finished with text '%s'.\n", command[0]);
		  i++;
		  cont = 0;
		}
	    }
	  printf("\n\n\n");
	}
      }
    
    else if (strncmp(command[comm], "undefine", 8) == 0)
      {
	if (command[comm][8] != NULL)
	  strcpy(word, command[comm] + 9);
	else {
	  printf("proof: undefine: lexeme> ");
	  gets(word);
	}
	
	if (word[0] != NULL)
	  if (undefine_word(word, SOME) == 1)
	    printf("Undefined.\n");
      }

    else if (strncmp(command[comm], "Undefine", 8) == 0)
      {
	if (command[comm][8] != NULL)
	  strcpy(word, command[comm] + 9);
	else {
	  printf("proof: Undefine: lexeme> ");
	  gets(word);
	}
	
	if (word[0] != NULL)
	  if (undefine_word(word, ALL) == 1)
	    printf("Undefined.\n");
      }
    else {
      printf("\n");
      printf("Usage: proof [-d|l|u|U lexeme] [-f filename] [-h|t] [-s text]\n\n");
      printf("Interactive commands:\n\n");
      printf("define      ");
      printf("help        ");
      printf("lookup      ");
      printf("quit        ");
      printf("redefine    ");
      printf("reorganize  ");
      printf("scan        ");
      printf("undefine    ");
      printf("Undefine    ");
      printf("\n\n");
      
      numcomm++;
    }
  }
  
  gdbm_close(lexicon);

  if (curr_condition != VALID)
    exit(1);

  exit(0);
  
} /* end: main() : finished */
