/*
File:  qsystemic.pl
Date:  Mon Mar 28 16:45:26 1988
By:    Jo

routines for handling systemic networks for the Protolexicon

*/

eccs_q_expanded_entry(Entry, Graph) :-
    eccs_q_left_most_node(LMN),
    eccs_member(LMN, Entry), !,
    findall(path(Path, N),
    		(eccs_member(Node, Entry), eccs_q_int_node(LMN, Path, N, Node)),
		Nodes),
    eccs_q_make_set(Nodes, N1),
    findall(reentrant_path(P1, P2),
    		(eccs_member(path(P1, Node), N1), eccs_member(path(P2, Node), N1), eccs_not_eq(P2, P1)),
		REPaths),
    eccs_q_sort_by_paths(N1, Nodes1),
    eccs_q_make_graph(LMN, Nodes, REPaths, Graph),
    eccs_q_apply_defaults([], Graph, Graph).

directly_linked(N, N1) :-
    eccs_get_from_database(requires, requires, requires(N, L)),
    eccs_member(E, L),
    eccs_get_from_database(fulfils, fulfils, fulfils(E, N1)).

directly_linked_via_dimension(N, E, N1) :-
    eccs_get_from_database(requires, requires, requires(N, L)),
    eccs_member(E, L),
    eccs_get_from_database(fulfils, fulfils, fulfils(E, N1)).

linked(X, X).
linked(X, Y) :-
    directly_linked(X, Z),
    linked(Z, Y).

linked_via_dimensions(X, [], X).
linked_via_dimensions(X, [D|R], Y) :-
    directly_linked_via_dimension(X, D, Z),
    linked_via_dimensions(Z, R, Y).

eccs_q_intermediate_node(LN, Node, RN) :-
    linked(LN, Node), linked(Node, RN).


eccs_q_int_node(LN, Path, Node, RN) :-
    linked_via_dimensions(LN, Path, Node),
    linked_via_dimensions(Node, _, RN).


eccs_q_sort_by_paths([], []).
eccs_q_sort_by_paths([path(Path, Node)|R], Paths) :-
    eccs_q_delete(path(Path1, Node1), R, Remainder),
    eccs_q_prefix(Path1, Path), !,
    eccs_q_sort_by_paths([path(Path1, Node1), path(Path, Node)|Remainder], Paths).
eccs_q_sort_by_paths([path(Path, Node)|R], [path(Path, Node)|Paths]) :-
    eccs_q_sort_by_paths(R, Paths).

eccs_q_prefix(Path1, Path2) :-
    eccs_append(Path1, _, Path2).

eccs_q_make_graph(Node, Nodes, REPaths, Remainder) :- 
    eccs_q_make_graph0(Node, Nodes, Remainder).

eccs_q_make_graph0(Node, [], _).
eccs_q_make_graph0(MotherNode, [path(Path, Node)|R], G) :-
    eccs_q_value_in_graph(MotherNode, Path, G, Node, error),
    eccs_q_make_graph0(MotherNode, R, G).

eccs_q_value_in_graph(MotherNode, [], G, N, Error) :-
    (G = node(N, _); Error = error, eccs_error([entry, inconsistent, G = N])), !.
eccs_q_value_in_graph(MotherNode, [H|T], node(_, L), N, Error) :-
    eccs_q_has_requirement(MotherNode, L),
    eccs_memberchk(H = Val, L),
    Val = node(Node, _),
    eccs_q_value_in_graph(Node, T, Val, N, Error).

eccs_q_apply_defaults(_, [], _).
eccs_q_apply_defaults(Path, [H|R], Graph) :-
    eccs_q_apply_defaults(Path, H, Graph),
    eccs_q_apply_defaults(Path, R, Graph).
eccs_q_apply_defaults(_, node(N, []), _) :-
    \+ eccs_q_has_requirement(N, _), !.
eccs_q_apply_defaults(Path, node(N, [H|T]), Graph) :-
    eccs_q_has_requirement(N, [H|T]),
    eccs_q_gather_defaults([N|Path], [H|T], Graph, Deeper),
    eccs_q_apply_defaults([N|Path], Deeper, Graph).

eccs_q_gather_defaults(_, [], _, []).
eccs_q_gather_defaults(Path, [Dimension = node(Val, L)|T], Graph, Deeper) :-
    eccs_sys_nonvar(Val), !,
    (eccs_q_has_requirement(Val, L) -> Deeper = [node(Val, L)|DL];
    				  L = nothing, Deeper = DL),
    eccs_q_gather_defaults(Path, T, Graph, DL).
eccs_q_gather_defaults(Path, [Dimension = node(Val, L)|T], Graph, Deeper) :-
    eccs_sys_var(Val),
    (eccs_q_default(Dimension, Df) -> true;
    				 eccs_error([no, default, for, Dimension])),
    eccs_q_handle_default(Path, Dimension, Df, Graph, Val, L, Deeper, DL),
    eccs_q_gather_defaults(Path, T, Graph, DL).

eccs_q_handle_default(_, Dimension, Df, _, Df, L, Deeper, DL) :-
    eccs_not_eq(Df, [_|_]), !,
    (eccs_q_has_requirement(Df, L) -> Deeper = [node(Df, L)|DL];
    				L = nothing, Deeper = DL),
    eccs_debug(3, eccs_message([value, for, dimension, Dimension, defaults, to, Df])).
eccs_q_handle_default(Path, Dimension, Df, Graph, Val, L, Deeper, DL) :-
    Df = [_|_],
    eccs_reverse(Path, RPath),
    findall(N,
        (Tail = [], eccs_member(if(C, N), Df), eccs_q_prefix(P, RPath),
	 eccs_q_value_in_graph(Node, P, Graph, C, no_errors)),
	 PossibleValues),
    (PossibleValues = [Val] -> true;
    			       eccs_error([conditions, on, default, not,
			       		fulfilled, '$nl$', PossibleValues])),
    (eccs_q_has_requirement(Val, L) -> Deeper = [node(Val, L)|DL];
				  L = nothing, Deeper = DL).
/* 

eccs_q_strip_outwith_network(L1, L2)

L1 is a superset of L2 the difference being those properties which are not
connected with any other property
*/

eccs_q_strip_outwith_network([], [], []) :- !.
eccs_q_strip_outwith_network([Term|R], R2, [Term|OutWith]) :-
    eccs_q_outwith_network(Term), !,
    eccs_q_strip_outwith_network(R, R2, OutWith).
eccs_q_strip_outwith_network([Term|R], [Term|R2], OutWith) :-
    eccs_q_strip_outwith_network(R, R2, OutWith).

eccs_q_get_requirements([], []).
eccs_q_get_requirements([H|T], [requires(H, L)|Rest]) :-
	(eccs_q_has_requirement(H, L); L = nothing), !,
	eccs_q_get_requirements(T, Rest).



eccs_q_valid_property(Prop, Props) :-
	eccs_member(requires(E, _L), Props),
	eccs_q_left_most_node(E),
	eccs_q_path_from_to(E, Prop, Props, _, _).


eccs_q_has_requirement(Prop, ReqV) :-
	eccs_q_requires(Prop, Req),
	eccs_q_add_vars(Req, ReqV).

eccs_q_add_vars([], []).
eccs_q_add_vars([Dimension|R], [Dimension = _|T]) :-
	eccs_q_add_vars(R, T).

eccs_q_fulfilled([], _, []).
eccs_q_fulfilled([Dimension = val(E, V1)|T], P, R) :-
	eccs_member(requires(E, Val), P),
	eccs_q_fulfils(Dimension, E),
	V1 = Val, % done like this to prevent cyclic structures
	!,
	(eccs_q_fulfils(Dimension, E1), eccs_member(requires(E1, _V), P), \+ E1 = E ->
		eccs_error([entry, inconsistent, E+E1]);
	 true),
	eccs_q_fulfilled(T, P, R).
eccs_q_fulfilled([Dimension|T], P, [Dimension|R]) :-
	eccs_q_fulfilled(T, P, R).


eccs_q_delete(H, [H|T], T).
eccs_q_delete(H, [F|T], [F|R]) :-
	eccs_q_delete(H, T, R).

eccs_q_precedes(E, E).
eccs_q_precedes(requires(Term, List), Sub) :-
	eccs_member(H, List),
	eccs_q_fulfils(H, E),
	once( (eccs_q_has_requirement(E, L2); L2 = nothing)),
	eccs_q_precedes(requires(E, L2), Sub).

eccs_q_fulfils(E, L) :-
    eccs_get_from_database(fulfils, fulfils, fulfils(E, L)).

eccs_q_requires(E, L) :-
    eccs_get_from_database(requires, requires, requires(E, L)).

eccs_q_left_most_node(N) :-
	eccs_q_requires(N, _), \+  eccs_q_fulfils(_R, N).

eccs_q_left_most_nodes(Ns) :-
	setof(N, eccs_q_left_most_node(N), Ns).

eccs_q_terminals(Ts) :-
	setof(T, eccs_q_terminal(T), Ts).

eccs_q_terminal(N) :-
	eccs_q_fulfils(_, N),
	\+ eccs_q_has_requirement(N, _).

eccs_q_defining_term(T) :-
	eccs_q_requires(T, _); eccs_q_fulfils(_, L), eccs_member(T, L), \+requires(T, _).

eccs_q_default(Dimension, Defaults) :-
    findall(if(C, Df), eccs_get_from_database(default, default, default(Dimension, Df, C)), L),
    (L = [] -> eccs_error([no, default, for, Dimension]);
	       L = [if(no_condition, Default)] -> Defaults = Default;
	       					  Defaults = L).


/*
eccs_q_check_net

run a series of tests on the network
Also change the form of the defaults
*/
eccs_q_check_net :-
	eccs_q_cyclic_graph(_,_), fail;
	eccs_q_conflicting_inheritance, fail;
	eccs_q_inappropriate_default(_,_), fail;
	eccs_q_conditionalized_defaults_inconsistent, fail;
	eccs_message([finished, checking, network, out, for, acyclicity, and,
			defaults, 'etc.']).
/*

sanity checks

No graph can be cyclic.

*/
eccs_q_no_cycles :-
	eccs_q_cyclic_graph(_,_), fail.

eccs_q_cyclic_graph(Super, Sub) :-
	Super = requires(X, [H|T]),
	eccs_q_has_requirement(X, [H|T]),
	eccs_q_precedes(Super, Sub),
	eccs_q_precedes(Sub, Super),
	eccs_not_eq(Sub, Super),
	eccs_message([graph, is, cyclic, between, Super, and, Sub]).

/*

Inheritance must be orthogonal.

*/

eccs_q_no_conflicting_inheritance :-
    eccs_q_conflicting_inheritance, fail.

eccs_q_conflicting_inheritance :-
	eccs_q_fulfils(Req3, D1),
	eccs_q_fulfils(Req3, D2),
	\+ D1 = D2,
	eccs_get_from_database(default, default, default(Req1, D1, C)),
	eccs_get_from_database(default, default, default(Req2, D2, C)),
	eccs_message([defaults, D1, and, D2, conflict, for, dimension, Req3]).

/*

eccs_q_consistent(Entry)

*/

eccs_q_consistent(Entry) :-
	\+ (eccs_member(E, Entry), eccs_member(E1, Entry), \+ E = E1, 
	    eccs_q_fulfils(Dim, E), eccs_q_fulfils(Dim, E1)).

/*

Defaults can only be associated with the appropriate dimension

*/

eccs_q_defaults_appropriate :-
	eccs_q_inappropriate_default(_DT, _Df), fail.

eccs_q_inappropriate_default(DT, Df) :-
	eccs_get_from_database(default, default, default(DT, Df, _)), 
	\+ eccs_q_fulfils(DT, Df),
	eccs_message([default, Df, inappropriate, for, dimension, DT]).

eccs_q_conditionalized_defaults_inconsistent :-
    eccs_get_from_database(default, default, default(Dimension, Default, C)),
    eccs_not_eq(C, no_condition),
    (eccs_get_from_database(default, default, default(Dimension, Df, no_condition)) -> 
    	eccs_error([Dimension, has, both, conditional, and, non-conditional, defaults]); true).
%    (eccs_q_has_requirement(C, L), eccs_q_path_from_to(C, Default, L, _) -> true;
%    eccs_error([default, Default, is, not, dependent, upon, condition, C])).

eccs_q_root_dimensions(Ds) :-
	eccs_q_left_most_nodes(Ds).

	
eccs_q_outwith_network(Term) :-
	\+ eccs_q_fulfils(_, Term), \+eccs_q_has_requirement(Term, _).




