% chart Parser - based on Thompson (1981), Popowich (1989)
%              - looks for all possible parses using a failure driven loop
%              - by Fred Popowich and Carl Vogel
%		 modified by Sandi Kodric
%
% Edge format:  
%   edge(Expects, Sign, Constituents, Constraints, LeftPos, RightPos, Num).

:- dynamic edge/7,		% element of the chart 
           edge/3, 		% reconstructed edge (used in treetool)
           to_process/1, 	% agenda
           parse_found/1,  	% successful parse's edge number
           num/1,        	% edge counter 
           edge_spy/1.   	% debugging flag

/*

Tue Mar 23 11:44:50 1993 JC

Change for abbreviations

*/
parse :-                           % Top-level Goal
  read_sent(Sentence),
  filter_abbreviations(Sentence, UnAbbd),
  parse(UnAbbd).

parse(Sentence) :-
  cputime(Time), 
  clear_edges,                     % clear chart
  set_num_to_one,                  % initialize counter
  init_chart(Sentence),            % initialize chart
  parse(Sentence, Time, _Result).

parse(WordList, Time, Result) :-
  select_next(WordList, Edge),!,          % get next edge
  (parse(Edge, WordList, Time, Result)    % always fails
  ; parse(WordList, Time, Result)).       % any more edges to process?

parse(WordList, Time, _Res) :-    % no more edges on the agenda
  \+ clause(parse_found(_),_),    % if no parse was found so far
  retract(flag(optimize)),!,      % and "optimize" flag was on
  init_chart(empty),              % then add empty phon signs to the chart
  parse(WordList, Time, _Res),    % and proceed...
  assert(flag(optimize)).         % ...finally restore "optimize" flag

parse(_, Time, 'END'):-
  num(X),                         % no edges left, parsing finished
  N is X - 1,
  writemessage('Edges Processed: ', N),  % display summary
  writetime('Done at ', Time), 
  fail.
 

% This "parse" predicate describes a successful parse.  If it succeeds then
% an edge has been constructed accounting for the information in the entire
% sentence.

parse(edge([],ISign,C,[],Wordlist,[],N), Wordlist, Time, _Result) :-
  path(ISign,phon,Wordlist),
  complete(ISign),
  writetime('Parse Found at ', Time),
  ( flag(tracer) -> writemessage('Edge Number: ',N); true ),
  writesem(ISign),                      % Write out the semantics
  makePSign(ISign, C, []),              % Reconstruct the Phrasal Sign
  writePSign(ISign, C, N, Time),        % Write out the Phrasal Sign
  assert(parse_found(N)),
  fail.                                 % look for other parses

% This "parse" predicate encodes the essential behavior of the chart
% parser; failure driven execution.  spying/2 and talker/5 are used
% for debugging.

parse(Edge, _, _, _) :- 
  spying(Edge,'Processing '),
  fundamental_rule(Edge),      % Compare current edge with chart edges
  find_rules(Edge).            % Compare current edge with grammar rules

init_chart(empty):-            % looking for entries with empty phonology
  lookup(+EName,AVM,Cs),
  talker(initializing,[EName],[],[],['EMPTY PHON']),
  make_edge(AVM, [], [], Cs, AnyPos, AnyPos), 
  fail.

init_chart(empty).

init_chart([]):- flag(optimize) -> true; init_chart(empty).

init_chart([Word|Sentence]) :-
  lookup(Word, AVM, Constraints),
  talker(initializing, [Word], [], [], [entry(Word)]),
  make_edge(AVM, [], [], Constraints, [Word|Sentence], Sentence), 
  fail.                                         % look for other entries

/*

Comment added by JC Mon Mar 29 15:00:42 1993

The following two clauses added by someone, to allow ``compound
words'' containing spaces. These are stored in the database as atoms,
and so we have to try putting n words together separated by a space to
see if we get back a word in the lexicon.  n is fixed arbitrarily
between 2 and 3 inclusive.  

*/

init_chart([Word1,Word2|Sentence]) :-
  compound_word(Word1,Word2,Word),
  lookup(Word, AVM, Constraints),
  talker(initializing, [Word], [], [], [entry(Word)]),
  make_edge(AVM, [], [], Constraints, [Word1,Word2|Sentence], Sentence), 
  fail.                                         % look for other entries

init_chart([Word1,Word2,Word3|Sentence]) :-
  concat_atom([Word1,' ',Word2,' ',Word3],Word),
  lookup(Word, AVM, Constraints),
  talker(initializing, [Word], [], [], [entry(Word)]),
  make_edge(AVM, [], [], Constraints, [Word1,Word2,Word3|Sentence], Sentence), 
  fail.                                         % look for other entries

init_chart([_|Sentence]) :- init_chart(Sentence), !.

compound_word(A1,A2,B) :- concat_atom([A1,' ',A2],B).

make_edge(Category, Constituents, Expects, Constraints, LeftPos, RightPos) :-
  numgen(N),
  asserta(
    to_process(
       edge(Expects,Category,Constituents,Constraints,LeftPos,RightPos,N))
    ), !.

% An inactive edge meets an active edge:

fundamental_rule(edge([],PSign,_,[],Lpos,Rpos,Num)) :- 
  edge([PSign|Rest],ActSign,_,Constraints, % 'hidden' unification of PSign and
       ActLpos,ActRpos,N),                 % first expecting sign
  meet(Lpos,Rpos,ActLpos,ActRpos,NewLpos,NewRpos),
  constraint_check(ActSign,Rest,NewLpos,NewRpos,Constraints,NewConstraints),
  talker(completer,NewLpos,NewRpos,Rest,[edge(Num),edge(N)]),
  make_edge(
         ActSign,
         [active(N),inactive(Num)],
         Rest,
         NewConstraints,
         NewLpos,
         NewRpos),
  fail.


% An active edge meets an inactive edge:

fundamental_rule(edge([Exp|Rest],ActSign,_,Constraints,ActLpos,ActRpos,N)):-
  edge([],Exp,_,[],    % 'hidden' unification of first expecting sign and
       Lpos,Rpos,Num), % inactive edge's sign
  meet(Lpos,Rpos,ActLpos,ActRpos,NewLpos,NewRpos),
  constraint_check(ActSign,Rest,NewLpos,NewRpos,Constraints,NewConstraints),
  talker(completer,NewLpos,NewRpos,Rest,[edge(N),edge(Num)]),
  make_edge(
         ActSign,
         [active(N),inactive(Num)],
         Rest,
         NewConstraints,
         NewLpos,
         NewRpos),
  fail.

fundamental_rule(_).

meet(P1, P, P, P2, P1, P2) :- !.
meet(P, P2, P1, P, P1, P2).

constraint_check(PSign,Expect,L,R,Cs,NewCs) :-
  process(Cs, NewCs, Expect),  % first evaluate any ORDER and SEM functions
  path(PSign, phon, P),    % get the phonology of the phrasal sign
  ( var(P) -> true;        % don't check if we have a variable
    phon_ok(P, L, R)), !.

phon_ok([], _, _) :- !.
phon_ok([Word|RestPhon], [Word|Rest], R) :- !, phon_ok(RestPhon, Rest, R).
phon_ok(Phon, [_|Rest], R) :- phon_ok(Phon, Rest, R).


%Triggered on inactive edges.

find_rules(edge([],PSign,_,[],Lpos,Rpos,Num)) :-
  rule(R, Rule, Constraints),
  get_dtr(head_dtr, Rule, HSign),     % get the value at dtr:head_dtr
  sign_unify(HSign,PSign),            % see if values unify 
  make_expects(Rule, Constraints, NConstraints, Expects),
  talker(predicter,Lpos,Rpos,Expects,[edge(Num),rule(R)]),
  newsign(Rule, Sign),
  make_edge(Sign, [rule(R),Num], Expects, NConstraints, Lpos, Rpos),
  fail.                   

make_expects(PSign, Cs, NewerCs, Expects) :-
  get_all_dtrs(PSign, _Head, Comps, Adjs, Fillers),
  process(Cs, NewCs),                  % process any function calls/constraints
  reverse(Comps,Tmp),                  % reverse complements 
  concat(Adjs,Tmp,Tmp1),	       % prepend adjuncts 
  concat(Fillers,Tmp1,Expects),	       % prepend fillers
  process(NewCs, NewerCs, Expects),!.

% newsign(PSign, Sign)
%
%   This predicate effectively gets rid of the DTRS of a phrasal
%   sign thus allowing us to build implicit phrasal signs.
%   Explicit phrasal signs are constructed if "explicit" flag is on.
 
newsign([[dtrs,X]|Sign], [[dtrs,Y]|Sign]) :- flag(explicit) -> X=Y ; true.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% initializers and edge bookkeeping stuff
%

set_num_to_one :- abolish(num,1), assert(num(1)).

numgen(N) :- retract(num(N)), N1 is N + 1, assert(num(N1)), !.

clear_edges :- clear_agenda, abolish(edge,7), abolish(edge,3),
               !,garbage_collect.

clear_agenda :- abolish(to_process,1), retractall(parse_found(_)).

% Agenda management

select_next(S, Edge) :- retract_edge(S, Edge), asserta(Edge).

% Agenda selection strategy - select edges spanning the sentence first

%retract_edge(S,edge([],A,B,C,S,[],F)) :-       % this clause could be omitted
%   retract(to_process(edge([],A,B,C,S,[],F))). % for faster execution

retract_edge(_, Edge) :- retract(to_process(Edge)).


% sign_unify - could be replaced by a alternative routine if
%              you want to see where something goes wrong.

sign_unify(X, X).

