/*

File:	/home/dk2/jcalder/Pleuk/HPSG-PL/hpsg.pl
Date:	Thu Mar  4 13:30:48 1993
By:	Jo Calder

Reverse engineering of Popowich, Vogel and Kodric's HPSG-PL system for
Pleuk purposes.

The intention is that no part of the code from that system is altered,
although some of it is made redundant.  

The files that Pleuk makes use of.  

*/

:- eccs_srcload([hpsgpleukvars, hpsgprint, hpsgdc]).


/*

Files used by the HPSG-PL system

*/

:- eccs_srcload(
	[ main,lex,parser,constraints,utils,io,treetool]).

/*

The list of predicates that get overridden if in SystemX mode.  

*/

sx_abolish_list([writesem/1,
		     order/2,
		     order/5,
		     order_comps/2,
		     combine/4,
		     get_dtr/3,
		     tugtree/2]).
sx_abolish :-
    sx_abolish_list(L),
    eccs_member(Pred, L),
    eccs_sys_abolish(Pred),
    fail.
sx_abolish.

 
/*

The following routines called by Pleuk at start up.  

The former may load a file defining the SystemX version of some
predicates.  The latter checks that HPSG-PL's flags reflect the
current value of Pleuk's variables.

The asserts are necessary so that loading the compiled version of this
file doesn't result in these definitions getting ignored.  

*/ 
:- eccs_sys_assert(spec_init_hook(hpsg_maybe_install_systemX)).

:- eccs_sys_assert(spec_init_hook(hpsg_prepare_flags)).

/*

How we load stuff specific to SystemX.  Note that we assume
nofileerrors here.

*/

hpsg_maybe_install_systemX :-
    FileToLoad = library('HPSG-PL/systemx.pl'),
    eccs_absolute_file_name(FileToLoad, Path),
    (eccs_global_variable('SystemX mode', true) ->
    	(eccs_file_exists(Path) -> 
	    sx_abolish, 
	    eccs_sys_ensure_loaded(Path)
	  ; eccs_message(['SystemX', mode, unavailable]),
	    eccs_set_variable('SystemX mode', false))).


/*

Readers for HPSG

*/

hpsg_read(_, Name, Type, Definition) :-
    eccs_sys_read(X),
    (X == end_of_file -> 
	Definition = end_of_file
      ; (hpsg_match_read(X, Name, Type, Definition) ->
	    true;
	    eccs_error([unable, to, make, sense,of, X]))).


/*

Determine a name and type for any read object.  

This is tricky in the case of load-time directives, the ninth clause
below.  As Pleuk requires Type to be instantiated by the reader, we
have to try and determine a type.  In well-behaved grammars, one can
imagine this not causing any problem, but it certainly make Pleuk
accept a narrower range of possible statements than the standalone
system.

According to the Pleuk documentation, _Phon should not be a variable
in the following case.  At the moment, we can sneak it through.

*/

hpsg_match_read(entry(X), _Phon, entry, X).
hpsg_match_read(entry(X, Y), X, entry, Y).
hpsg_match_read(unknown_word(X), unknown_word, unknown_word, X).

hpsg_match_read(has(A, B), A, has, B).
hpsg_match_read(domain(A, B), A, domain, B).
hpsg_match_read(type(A, B), Name, type, type(A, B)) :-
    hpsg_make_name(A, Name).
hpsg_match_read(rule(N, A), N, rule, A).
hpsg_match_read(ideal(S), ideal, ideal, S).

hpsg_match_read((I -> T; E), _Name, Type, (I -> T; E)) :-
    hpsg_match_read1(T, E, Type).

hpsg_make_name(A, A) :- eccs_sys_atomic(A), !.
hpsg_make_name(T, A/N) :-
    eccs_sys_functor(T, A, N).

hpsg_match_read1(T, E, Type) :-
    eccs_commas_to_list(T, T1),
    eccs_commas_to_list(E, E1),
    ((hpsg_check_all_same_type(T1, Type),
      hpsg_check_all_same_type(E1, Type)) -> true;
	eccs_error([different, kinds, of, statements, appear, to, right, of, (->),
		    in, type, definition])).

hpsg_check_all_same_type([], _).
hpsg_check_all_same_type([T|Ts], Type) :-
    eccs_sys_functor(T, Type, _),
    hpsg_check_all_same_type(Ts, Type).


/*

Compilers 



These are perhaps slightly more general that they need be.  They allow
any sort of definition to appear in any kind of file.  

The following definitions are based on lex.pl.  Each assert/1 has been
removed and the item that it computes added as an argument.

This leads to problems with the last few routines which are to do with
load-time expansion of definitions.

I have tried to keep to a minimum amount of textual changes.  In some
cases, additional brackets are necessary to prevent problems with ->;
syntax.  (i.e. no cuts can appear to the left of ->).

All occurrences of convert/3 changed to hpsg_convert/3.

*/

hpsg_compile(FType, OType, Name, _Args, Def, CDef, _) :-
    hpsg_compile(FType, OType, Name, Def, CDef).

hpsg_compile(_F, has, Name, Structure, sign(Name, [FName, Matrix], Constraints)) :-
  (hpsg_convert(Structure, Matrix, Constraints) ->
  	feature_name(Name, FName)
%    assertz( sign(Name, [FName, Matrix], Constraints) );
      ; error('Has', Name)).

% convert:  (name type definition, conditions) to
%           sign(name,definition, conditions)

% The memberchk statement here is to catch load-time directives, 
% which are handled lower down.

hpsg_compile(_F, type, Name, Def, 
			     sign(NameInFile, Matrix, Constraints)) :-
%    (Name == dateloc -> trace; true),
    \+ eccs_memberchk(Def, [(_,_), (_;_), (_->_)]),
    Def = type(NameInFile, Structure),
    (eccs_sys_var(Name) -> hpsg_make_name(NameInFile, Name); true),
    (hpsg_convert(Structure, Matrix, Constraints) ->
    	true			% assertz( sign(Name, Matrix, Constraints) );
    ; error('Type', Name)).

hpsg_compile(_F, rule, Name, Structure, rule(Name, Matrix, Constraints)) :-
    hpsg_convert(Structure, Matrix, Constraints) ->
	true			% assertz( rule(Name, Matrix, Constraints) );
      ; error('Rule', Name).

hpsg_compile(_F, domain, Name, Structure, List) :-
    hpsg_convert(Structure, List, []) ->
    	true 			% assertz( domain(Name, List) );
      ; error('Domain', Name).

% entry - for improved efficiency, include Word as an argument

% entries with empty phonology

hpsg_compile(_F, entry, Name,  Structure, entry(Name, Matrix, Constraints )) :- 
  eccs_sys_nonvar(Name),
  +(_) = Name, !,
  ((hpsg_convert(Structure, Matrix, Constraints),
   path(Matrix, phon, [])) ->
   	true 		   % assertz( entry(Name, Matrix, Constraints) )
      ; error('Entry', Structure)).

/*

We rely on some looseness in the definition of readers and compilers
for Pleuk.  The doc says that readers should instantiate Name.
However, in the case of this system, the phonology attribute may be
given implicitly, as the value to a parameterized template.  The third
clause below (and the nonvar/1 in the preceding clause) allows this
situation to go through and post hoc instantiates the value for Name.
 
In the first clause, problems were caused by backtracking into compound/2. 

*/

hpsg_compile(_F, entry, Word, Structure, entry(Word, Matrix, Constraints)) :-
  eccs_sys_nonvar(Word), !,
  (hpsg_convert(Structure, Matrix, Constraints) ->
	( flag(lisp) -> concat_atom(['"',Word,'"'], SWord); eccs_once(compound(Word,SWord))),
	  path(Matrix, phon, SWord)
%    assertz( entry(Word, Matrix, Constraints) );
        ; error('Entry', Structure)).

hpsg_compile(_F, entry, Word, Structure, entry(Word, Matrix, Constraints) ) :-
    hpsg_convert(Structure, Matrix, Constraints) ->
	path(Matrix, phon, [Word])
%    assertz( entry(Word, Matrix, Constraints) );
      ; error('Entry', Structure).

hpsg_compile(_F, UK, UK, Structure, unknown_word(Matrix, Constraints)) :-
  UK = unknown_word,
  (hpsg_convert(Structure, Matrix, Constraints) ->
	true 		% assertz( unknown_word(Matrix, Constraints) );
      ; error('Unknown Word', Structure)).

hpsg_compile(_F, ideal, ideal, Structure, ideal_sign(Matrix, Constraints)) :-
    hpsg_convert(Structure, Matrix, Constraints) ->
    	true			% assertz(ideal_sign(Matrix, Constraints));
      ; error('Ideal', Structure).

/*

In the case of load-time directives, things are a bit more tricky.  

There's wierd stuff going on here as well.  In the case of

lisp -> var(X) type var(v(X),_),
        nil type 'NIL',
        value(X) type var(_,X),
        empty type '(0)';  % Lisp

        var(X) type X,
        nil type _,
        value(X) type x(X),
        empty type [].     % Prolog

taken from demo8.typ, one shurely can't mean that the occurrences of X
mean coinstantiation across different type statements.  Below we cash
these out as backtracking.

*/

hpsg_compile(_F, Type, Name, (Test -> Command), Out) :- 
    hpsg_safe_call(Test) -> hpsg_compile(_F, Type, Name, Command, Out).

hpsg_compile(_F, Type, Name, (Command1 ; Command2), Out ) :- 
    ( hpsg_compile(_F, Type, Name, Command1, Out); 
      hpsg_compile(_F, Type, Name, Command2, Out)).

hpsg_compile(_F, Type, Name, (Command1, Command2), Out) :- 
    ( hpsg_compile(_F, Type, Name, Command1, Out); 
      hpsg_compile(_F, Type, Name, Command2, Out)).


hpsg_convert(Structure, Matrix, Constraints) :-
    convert(Structure, Matrix, Constraints).

hpsg_safe_call(Pred) :-
    eccs_sys_nonvar(Pred),
    eccs_sys_functor(Pred, F, N),
    eccs_sys_functor(MGHead, F, N),
    eccs_sys_current_predicate(F, MGHead), !,
    eccs_sys_call(Pred).


/*  

The following procedures replicate the information asserted into the 
database by the routine command/1 in lex.pl

*/



sign(FName, X, Z) :-
    eccs_get_from_database(has, FName, sign(FName, X, Z)).

sign(TName, X, Constraints) :-
    hpsg_make_name(TName, Name),
    (TName = Name -> 
	TName = NameInFile
      ; TName =.. [F|Args], 
        convertlist(Args,NewArgs,Cs), 
	NameInFile =.. [F|NewArgs]),
    eccs_get_from_database(type, Name, sign(NameInFile, X, Z)),
    eccs_append(Cs, Z, Constraints).

domain(X, Y) :-
    eccs_get_from_database(domain, X, Y).

entry(Word, Matrix, Constraints) :-
    E = entry(Word, Matrix, Constraints),
    eccs_get_from_database(entry, Word,  E).

rule(Name, Matrix, Constraints) :-
    R = rule(Name, Matrix, Constraints),
    eccs_get_from_database(rule, Name, R).


:- (eccs_global_variable('SystemX mode', true) -> 
	true
      ; C = (unknown_word(Matrix, Constraints) :-
	  	UK = unknown_word(Matrix, Constraints),
                eccs_get_from_database(unknown_word, unknown_word, 
				        UK), !),
	eccs_sys_assert(C)).

/*

The following is slightly different to the version in lex.pl, which
requires a clause exists for ideal_sign/2, but when called the second
argument (Constraints) has to be [].  This means that complete(_) will
not succeed in case the user-supplied definition for ideal_sign
results in some constraints holding over that sign.

What we do here is mask off any constraints that hold so that
complete/1 can never fail for this reason.

The dynamic declaration is here so that complete/1 won't barf on a
call to clause/2 which references a compiled predicate.  The existence
of the clause below also means that the second clause of complete/1
*never* gets called.

*/

:- dynamic ideal_sign/2.


ideal_sign(Sign,[]) :-
    Term = ideal_sign(Sign, _),
    (eccs_get_from_database(ideal_sign, ideal_sign, Term)
    ; true).


/*

Parser stuff

*/



eccs_user_parser_spec(hpsg_parse,
	[prehook = clear_edges,
	 fs_drawer = hpsg_printfs_from_parse]).

:- eccs_new_variable(parser, hpsg_parse, parsing, run, 
               "The name of the parser to be used for analyzing input").


/*

Pleuk's version of the parser.  Adapted from parser.pl.  

hpsg_prepare_flags/0 checks that HPSG-PL's version of flags
corresponds to the state represented in Pleuk's variables.  See
hpsgpleukvars.pl for details.

*/

hpsg_parse(_Args, String, Results) :-
      hpsg_prepare_flags,
      eccs_sys_retractall(eccs_last_sentence_analysis(_, _, _)),
      eccs_post_last_sentence(String),
      (eccs_global_variable('SystemX mode', true) -> 
	filter_abbreviations(String, UnAbbd)
      ; String = UnAbbd),
      cputime(Time), 
      set_num_to_one,                  % initialize counter
      init_chart(UnAbbd),            % initialize chart
      eccs_once((parse(UnAbbd, Time, _Result); true)),
      findall(ISign+Cs, (parse_found(N), edge(_, ISign, _, Cs, _, _, N)), Is),
      Is = Results.


/*

Stuff for testing

We check that what we have produced is as follows:

there is a one-one relationship between terms stored in the database
and what is produced by compile using the standard set of predicates. 



*/


hpsg_test_all :-
    hpsg_test_all_abolish_pleuk_links,
    consult(lex),
    gram(demo8),
    hpsg_test_all_db_types.

hpsg_test_all_abolish_pleuk_links :-
    hpsg_test_all_abolish_pleuk_links_pred_list(L),
    eccs_member(P, L),
    eccs_sys_abolish(P),
    fail.
hpsg_test_all_abolish_pleuk_links.

hpsg_test_all_abolish_pleuk_links_pred_list(L) :-
    L = [sign/3,
         domain/2,
	 entry/3,
	 rule/3,
	 ideal_sign/2].


hpsg_test_all_db_types :-
    write('checking signs ...'), nl,
    Goal = sign(Name, Matrix, Constraints),
    eccs_sys_call(Goal),
    (eccs_sys_atomic(Name) -> Name = PName; eccs_sys_functor(Name, F, N), PName = F/N),
    ((eccs_member(T, [type, has]),
      eccs_get_from_database(T, PName, S),
      eccs_variant(S, Goal)) -> 
	write(PName), write(' ok'), nl
      ; write('problems with '), write(Name), nl),
    fail.
hpsg_test_all_db_types :-
    write('checking domains ...'), nl,
    Goal = domain(Name, D),
    eccs_sys_call(Goal),
    ((eccs_get_from_database(domain, Name, D1),
      eccs_variant(D, D1)) ->
 	write(D), write(' domain ok'), nl
      ; write(D), write(' domain bad'), nl),
    fail.
hpsg_test_all_db_types :-
    write('checking rules ...'), nl,
    Goal = rule(N, M,C),
    eccs_sys_call(Goal),
    ((eccs_get_from_database(rule, N, R),
      eccs_variant(Goal, R)) -> true;
      write(N), write(' rule failed'), nl),
    fail.
hpsg_test_all_db_types :-
    write('checking lexical entries ...'), nl,
    Goal = entry(N, M, C),
    eccs_sys_call(Goal),
    ((eccs_get_from_database(entry, N, R),
      eccs_variant(Goal, R)) -> true;
      write(N), write(' entry failed'), nl),
    fail.
    
hpsg_test_all_db_types.


    
/*

Mon Mar 15 13:50:25 1993 JC
Additional stuff for abbreviations

*/

hpsg_read_abbrev(_, Name, abbrev_def, Definition) :-
    eccs_sys_read(X),
    (X == end_of_file -> 
	Definition = end_of_file
      ; (X = abbrev(Name, Definition) ->
	    true;
	    eccs_error([unable, to, make, sense,of, X]))).



hpsg_compile_abbrev(abbreviations, abbrev_def, Name, _, Def, CDef, _) :-
    check_abbrev(abbrev(Name, Def), abbrev_def(Name, CDef)).


:- eccs_sys_abolish(abbrev_def/2).

hpsg_abbrev_prehook :-
    tokenize_entries.


abbrev_def(Name, Definition) :-
    eccs_get_from_database(abbrev_def, Name, Definition).


    
/*

A utility predicate for testing formulas

*/

hpsg_test_exp(Exp) :-
    convert(Exp, M, Cs),
    eccs_once((hpsg2spf(M+Cs, SPF), eccs_output(SPF))),
    (eccs_read_line(X), X = ";" -> fail; true).


/*

Replacement for +/1 and -/1.

We only really want to represent state in Pleuk variables, otherwise
we may lose track of the grammar settings.

*/

:- abolish((+)/1).
:- abolish((-)/1).

+(Flag) :- 
    eccs_set_variable(Flag, true).
-(Flag) :-
    eccs_set_variable(Flag, false).

/*

Taken from utils.pl

*/

add_flag(Flag) :-
    flag(Flag) -> true; assert(flag(Flag)).

remove_flag(Flag) :-
    flag(Flag) -> retract(flag(Flag)); true.

/*

Finally, a pointer into the help system.

*/

eccs_spec_info(specializations, 'HPSG-PL').
