/****************************************************************
*
*   File:        treetool.c
*   Written by:  Ian Harder, Simon Fraser University
*                as a research project for Dr. Fred Popowich
*   Date:        Sept, 1989
*
*   Revised and enhanced by: Sue Baker, Simon Fraser University
*                      Date:      October 1989 - April 1990
*
*   Revised and enhanced by: Rob Hamm, Simon Fraser University
*                      Date:      May 1990 - November 1990
*
*   Ported to Xview by     : Ko van der Sloot, Tilburg University
*                   Date   :      august 1991
*
*   This program will take input in the form of "fa_structures"
*   and provide a graphical representation of those which are
*   selected.
*
*   Usage:  treetool [-i input]
*
*   where:  input is the name of the file containing the
*                 structures to be displayed.  If this argument
*                 is not used, input will be taken from stdin.
*
****************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/scrollbar.h>
#include <xview/panel.h>
#include "treetool.h"

#define FR_ROWS 35
#define max( a, b ) (a > b ? a : b )
/* Use the Icon defined by Fred */

static short       icon_image[] = {
#include "tree.icon"
};

/* S.B. - Global variables for use with SunView defined procedures 
*/

FILE               *stream ;
static Frame       base_frame;
static Canvas      canvas;
static Scrollbar   h_scroll, v_scroll;
static Panel       panel;
static Pixwin      *pw;
static FA_PTR      fa[FASZE];
static int         display=0;
static int         next_structures=TRUE ;
static int         num_structs; 
static int         done=FALSE;
Menu_generate      operation ; 
static Menu	   display_menu;
static Menu	   init_display_menu;
static Menu	   testmen_menu;
char               in[STRSZE];
Pixfont 	   *pw_font ;
Pixfont 	   *panel_font ;

Icon jump_icon;

extern Pixfont * xv_pf_open();

/* ----------------------------------------------------------- */
void main(argc, argv)
   int  argc;
   char *argv;

{
   int i;
   char *str[STRSZE];
   Server_image jump_image;
   char font[50] ;    /* length of "/usr/lib/fonts/fixedwidthfonts/screen.b.14" */

   strcpy(font,"/usr/lib/fonts/fixedwidthfonts/") ; /* set font to fonts pathname*/
   get_args(argc, argv, in, font);
   pw_font = xv_pf_open( font );
   panel_font  = pw_font ;

   jump_image = (Server_image)xv_create( NULL, SERVER_IMAGE,
                                         XV_WIDTH, 64,
                                         XV_HEIGHT, 64,
                                         SERVER_IMAGE_BITS, icon_image,
                                         NULL );

   stream = open_input(in);
   eat_white_space(stream);

    /* create and open the window */

   base_frame = xv_create(NULL, FRAME,
                   WIN_COLUMNS, 130,
                   WIN_ROWS,    FR_ROWS,
	 	   XV_LABEL, "                                         TreeTool Graphical Interface (1.0)",
                   WIN_X,	60,
		   WIN_Y,	60,
                   0);
   jump_icon = (Icon)xv_create( base_frame, ICON, ICON_IMAGE, 
                                jump_image, NULL );

   canvas = xv_create(base_frame, CANVAS,
         WIN_COLUMNS,              105,
         CANVAS_AUTO_SHRINK,       FALSE, 
         CANVAS_WIDTH,             C_WIDTH,
         CANVAS_HEIGHT,            C_HEIGHT,
         NULL );

   h_scroll = xv_create( canvas, SCROLLBAR,
                         SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL, NULL );
   v_scroll = xv_create( canvas, SCROLLBAR,
                         SCROLLBAR_DIRECTION,  SCROLLBAR_VERTICAL, NULL );

   panel = xv_create(base_frame, PANEL,
	      XV_FONT,		   panel_font,
              WIN_COLUMNS,         25,
              WIN_ROWS,            FR_ROWS,
              0);

   num_structs=get_structs(stream,fa);

   make_control_panel(panel);

   pw = canvas_pixwin(canvas);

#ifdef XVIEW_COMMENT
     XView CONVERSION - If attempting to fit a frame around multiple subwindows
     where at least one is a text, tty or term subwindow, you should explicitly
     xv_set the size of the windows first, then call window_fit, see window_fit
     in documentation
#endif
window_fit(base_frame);

                 /*  determine how many structures there are and 
                      place them into a fa_structure.          S.B. */
                 /* display the tree */ 
    fa_display(); 

       /*  window_main_loop(pw); */
    window_main_loop(base_frame);
   fclose(stream);
   exit(0);

   return;
}
/* ----------------------------------------------------------- */

void fa_display()
{
/* Displays a tree in the canvas window */
   int    depth, left_lim;

   depth = left_lim = 0;
   
   tree_display(pw, fa[display]->tree, depth, &left_lim,
                fa[display]->aux_list);

   return;
} 
/* ----------------------------------------------------------- */
int tree_display(pw, tp, depth, left_lim, aux_list)
   Pixwin   *pw;
   TREE_PTR tp;
   int      depth, *left_lim;
   char     aux_list[];
{
/* Recursive module used to display a TUG tree in the window
   pointed to by pw.  The value returned is the centre position
   of the node in the x-direction.  The value pointed to by
   left_lim is updated whenever it is surpassed by the length
   of the node text (ie: *left_lim is altered as a side-effect
   of the function).  This function dynamically optimizes the 
   node position.
*/
   int width, old_lim, centre, tree_center[MAX_TREES], 
       x_size, y1, y2, i;
 
   width = node_width(tp->node); 

   /* if this is not a leaf node then recurse */

   if (!( (tp->ntree[0] == (TREE_PTR)NULL)))   /* not a leaf */ 
    {
      old_lim = *left_lim;

      /* make sure variable nodes are centred beneath parent */
      if (tp->ntree[0]->node->phonology[0] == '\0')
         *left_lim = *left_lim + width/2;

            /* this calculates the center position of all
		the trees and prints them and as a side effect
                alters left_lim                             */

	for (i= (tp->num_trees - 1) ; i>=0 ; i--)
	    {
		tree_center[i] = tree_display(pw, tp->ntree[i], depth+1, 
                                          left_lim, aux_list); 
	    }

	centre = max(average(tree_center,tp->num_trees),
            HORIZ_GAP + old_lim + width/2); 

        *left_lim = max(*left_lim, centre + width/2); 

      /* calculate vector end-points */
      y1 = (depth + 1) * ((NODE_SIZE * VERTI_PITCH) + VERTI_GAP);
      y2 = y1 + VERTI_GAP - VERTI_PITCH;

	   /* calculate size of right edge in x-direction */

/*      if (tp->num_trees > 1) */ 
    	   x_size = edge_x_size(3, (tree_center[tp->num_trees-1]-centre),
			      (y2-y1)); 

      /* draw thin edge to left children */ 
      
      for (i= (tp->num_trees - 1) ; i>=1 ; i--)
        pw_vector(pw, centre-x_size/2, y1, tree_center[i],
            y2, PIX_SRC, 1); 

      if (tp->num_trees == 1) 
        pw_vector(pw, centre-x_size/2+1, y1, centre-x_size/2+1,
            y2, PIX_SRC, 1); 


          /* draw wide edge to right child */
      if (tp->num_trees > 1) 
         for (i = 0; i < x_size; i++)
            pw_vector(pw, centre-x_size/2+i, y1, 
                tree_center[0]-x_size/2+i, y2, PIX_SRC, 1);
    }

    else { /* this is a leaf node */
      centre    = HORIZ_GAP + *left_lim + width/2; 
      *left_lim = HORIZ_GAP + *left_lim + width; 
   }
   /* print contents of node along with associated labels */
     node_display(tp, pw, centre, width, depth, aux_list); 

   return (centre);
}   
/* ----------------------------------------------------------- */
int edge_x_size(width, run, rise)
   int width, run, rise;
{
/* calculates the number of pixels an edge must have in order
   for its width to be as prescribed.  It is assumed that 
   the edge is not horizontal (a fair constraint in the 
   context of this program).
*/
   return ((int)(width * sqrt(1.0 + (run/rise)*(run/rise))));
} 
/* ----------------------------------------------------------- */
void node_display(tp, pw, centre, width, depth, aux_list)
   TREE_PTR tp;
   Pixwin  *pw;
   int     centre, width, depth;
   char    aux_list[];
{
/* prints contents of node along with associated labels
*/
   int      i, j, x, y, k=0;
   char     aux_label_str[STRSZE];
   NODE_PTR np;


   /* S.B - print name of structure */
   pw_text(pw, 20, 20, PIX_SRC,pw_font, "Structure Name :");
   pw_text(pw, 240, 20, PIX_SRC, pw_font, fa[display]->name);

   /* S.B. - print component list of structure */
   pw_text(pw, 20, 40, PIX_SRC, pw_font, "Component List :");
   
   pw_text(pw, 240, 40, PIX_SRC, pw_font, fa[display]->total_comp);
 
   np = tp->node;

   /* calculate position of node text */
   x = centre - width/2;
   y = depth * ((NODE_SIZE * VERTI_PITCH) + VERTI_GAP)
         + VERTI_GAP;

   /* print node */
   pw_text(pw, x, y,               PIX_SRC, pw_font , np->phonology);
   pw_text(pw, x, y+VERTI_PITCH,   PIX_SRC, pw_font, np->syntax);
   pw_text(pw, x, y+2*VERTI_PITCH, PIX_SRC, pw_font, np->semantics);
   pw_text(pw, x, y+3*VERTI_PITCH, PIX_SRC, pw_font, np->anaphors);
   pw_text(pw, x, y+4*VERTI_PITCH, PIX_SRC, pw_font, np->pronouns);

   /* print list or (nonempty) label beside node */
   if (depth == 0) {  /* make string from list */
      i = 0; j = 0;
      aux_label_str[j++] = '<';
      while (i < AUXLSTSZE && aux_list[i] != '\0') {
         aux_label_str[j++] = aux_list[i++];
         aux_label_str[j++] = ',';
      }
      j = max(1, j-1);
      aux_label_str[j++] = '>';    
      aux_label_str[j++] = '\0';
      x = x - (j+1)*HORIZ_PITCH;
   } else {  
      if (tp->aux_label != ' ') {
         aux_label_str[0] = tp->aux_label;
         aux_label_str[1] = ':';
         aux_label_str[2] = '\0';
         x = x - 3*HORIZ_PITCH;
      } else {
         aux_label_str[0] = '\0';
      }
   }
   pw_text(pw, x, y, PIX_SRC, pw_font, aux_label_str);

   return;
}      
/* ----------------------------------------------------------- */
int node_width(np)
   NODE_PTR np;
{
/* Returns the node width in pixels.
*/
   int maxlen = 0;

   maxlen = max(maxlen, strlen(np->phonology));
   maxlen = max(maxlen, strlen(np->syntax));
   maxlen = max(maxlen, strlen(np->semantics));
   maxlen = max(maxlen, strlen(np->anaphors));
   maxlen = max(maxlen, strlen(np->pronouns));

   return (maxlen * HORIZ_PITCH);
}
/* ----------------------------------------------------------- */
int get_structs(stream, fa)
   FILE *stream;
   FA_PTR fa[];
{
   char *str[STRSZE];
   int  i     = 0,
	ended = FALSE ;
   

  
   while ( !(ended) && (fgets(str, strlen("fa_structure(")+1, stream))) {
     if (strcmp(str, "end_of_trees.")) 
       {
        if (strcmp(str, "fa_structure(") )  {
	   printf("the incorrect value of str is %s\n",str); 
           ferr("Invalid input (1)"); } 
 	fa[i] = create_fa_struct();
      	get_fa_name(fa[i]->name, stream);
      	read_str(",",stream);
      	fa[i]->tree = get_fa_tree(stream);
      	read_str(",", stream);
      	get_fa_aux_list(fa[i]->aux_list, fa[i]->tree, stream);
        read_str(",",stream);
        get_fa_comp_list(fa[i]->total_comp,fa[i]->comp_list, stream);
        read_str(")", stream); 
        read_str(".", stream);
        eat_white_space(stream);
        i++;
       }
       else ended = TRUE ;
   }
   fa[i] = (FA_PTR)NULL;
    

   return i;
}
/* ----------------------------------------------------------- */
FA_PTR create_fa_struct()
{
/* Allocates memory for an fa_structure, returning a pointer to
   the structure. Failure causes program termination. 
*/
   FA_STRUC *fa;

   if ((fa = (FA_STRUC *)malloc(sizeof(FA_STRUC))) 
         == (FA_STRUC *)NULL)
      ferr("Cannot allocate memory");
   
   fa->tree = (TREE *)NULL;

   return (fa);
}
/* ----------------------------------------------------------- */
TREE_PTR create_tree_node()
{
/* Allocates memory for a single tree node (not to be 
   confused with the type NODE which is contained within this 
   structure). Failure causes program termination.
*/
   TREE *tp;
   NODE *np;
   int i;

   if ((tp = (TREE *)malloc(sizeof(TREE))) == (TREE *)NULL
         || (np = (NODE *)malloc(sizeof(NODE))) == (NODE *)NULL) 
      ferr("Cannot allocate memory");

   tp->aux_label = ' ';
   tp->node      = np; 
   for (i=0 ;i<MAX_TREES ; i++) 
         tp->ntree[i]   = (TREE *)NULL;
   tp->num_trees = 0 ;
   return(tp);
} 
/* ----------------------------------------------------------- */
void eat_white_space(stream)
   FILE *stream;
{
/* Passes by white space in the given stream.
*/
   char ch;
   
   ch = fgetc(stream);
   while (isspace(ch)) ch = fgetc(stream);
   if (ch != EOF) ungetc(ch, stream);

   return;
}
/* ----------------------------------------------------------- */
void get_fa_name(name, stream)
   char name[];
   FILE *stream;
{
/* Reads the name of the fa_structure into the associated C data
   structure.
*/
   char ch;
   int i = 0;
   
   eat_white_space(stream);
   ch = fgetc(stream);
   
   while ((ch != ',') && (ch != EOF)) {
      name[i] = ch;
      if (++i >= STRSZE) ferr("Structure name too long");
      ch = fgetc(stream);
   }
   name[i] = '\0';
   
   /* assure stream is set up for next input */
   if (ch == ',') ungetc(ch, stream); 
   
   return;
}       
/* ----------------------------------------------------------- */
TREE_PTR get_fa_tree(stream)
   FILE *stream;
{
/* Reads in a tree from the data stream into a dynamically 
   created tree structure. Invalid input results in program 
   termination. This is a recursive procedure.
*/
   TREE_PTR tp;
   char ch;
   char var[STRSZE]; /* Modified by Fred, 4 Sep 90 */
   int done=FALSE ;
   eat_white_space(stream);
   ch = fgetc(stream);
   ungetc(ch, stream);

   switch (ch) {
   case 't': /* typical tree */
      read_str("tree(", stream);
      tp = create_tree_node();
      get_tree_node(tp->node, stream);
      read_str(",", stream);

      while (!(done)) 
       {
	 tp->ntree[tp->num_trees] = get_fa_tree(stream) ;

	 tp->num_trees ++ ;
	 if ( next_char(stream) == ')' ) done=TRUE ;
	 else {
            read_str(",", stream);
	  }
       }
      read_str(")", stream); 
      break;

   case '_': /* tree is a variable */
      read_str("_", stream);
      get_variable(var,stream);
      tp = create_tree_node();
      strcpy(tp->node->phonology, "");          
      strcpy(tp->node->syntax, "");
      strcpy(tp->node->semantics, "");
      strcpy(tp->node->anaphors, "");
      strcpy(tp->node->pronouns, "");
      break;
   case 'n': /* tree is empty */
      read_str("nil", stream); 
      tp = (TREE *)NULL;          
      break;
   default : ferr("Invalid input (2)"); 
   } 
   return (tp);
}
/* ----------------------------------------------------------- */
void get_tree_node(tn, stream)
   NODE *tn;
   FILE *stream;
{
/* Obtains the data contents of the tree node tn from the data
   stream.
*/
   get_node_element(tn->phonology, stream);
   read_str(":", stream);
   get_node_element(tn->syntax, stream);
   read_str(":", stream);
   get_node_element(tn->semantics, stream);
   read_str(":", stream);
   get_node_element(tn->anaphors, stream);
   read_str(":", stream);
   get_node_element(tn->pronouns, stream);

   return;
}
/* ----------------------------------------------------------- */
void get_node_element(str, stream)
   char str[];
   FILE *stream;
{
/* Obtains the contents of one element of a tree node from the
   data stream.
*/
   int  i = 0, brack = 0, sbrack = 0;
   char ch;

   eat_white_space(stream);
   ch = fgetc(stream);
   while (TRUE) {
      if (ch == EOF)   ferr("Unexpected end of input"); 
      if (isspace(ch)) ch = ' ';
      
      switch (ch) {
      case '(': brack++;  break;
      case ')': brack--;  break;
      case '[': sbrack++; break;
      case ']': sbrack--; break;
      }
      if ((ch == ':' || ch == ',') && !brack && !sbrack) {
         break;
      } else {
         str[i] = ch;
         ch = fgetc(stream);
         if (++i >= STRSZE) ferr("Node element too long");
      }
   }
   str[i] = '\0';
   ungetc(ch, stream);

   return; 
}
/*-------------------------------------------------------------*/
void get_variable(str, stream)
   char str[];
   FILE *stream;
{
/* S.B. - Obtains the contents of one variable from the data stream
*/
   int i = 0;
   char ch;

   eat_white_space(stream);
   ch = fgetc(stream);
   while (TRUE) {
      if (ch == EOF) ferr("Unexpected end of input");
      if (isspace(ch)) ch = ' ';

      if ((ch >= 65 && ch <= 90) || (ch >= 97 && ch <= 122)
           || (ch >= 48 && ch <= 57))
      {
         str[i] = ch;
         ch = fgetc(stream);
         if (++i >= STRSZE) ferr("Node element too long"); 
      }
      else break;

   }/*end while*/
   str[i] = '\0';
   ungetc(ch,stream);

   return;
}/*end get_variable*/
/* ----------------------------------------------------------- */
void get_fa_aux_list(aux_list, tp, stream)
   char     aux_list[]; 
   TREE_PTR tp;
   FILE     *stream;
{
/* Reads in the elements of the auxiliary list, updating the
   corresponding labels of the tree nodes.
*/ 
   int  i = 0;
   char ch, label = 'A';

   eat_white_space(stream);
   read_str("[", stream);
   eat_white_space(stream);

   while ((ch = fgetc(stream)) != ']') {
      if (ch != ',') ungetc(ch, stream);
      label_tree(label, tp, stream);
      aux_list[i] = label;
      eat_white_space(stream);
      if (ch == EOF) ferr("Unexpected end of input");
      if (++i > AUXLSTSZE) ferr("Auxiliary list too long");
      label++;
   }
	aux_list[i] = '\0';
   return;
}
/* ----------------------------------------------------------- */
void get_fa_comp_list(total_comp,comp_list, stream)
   char 	comp_list[][STRSZE];
   char         total_comp[];
   FILE		*stream;
{
/* S.B.
   Reads in the component list.
*/
   int i = 0,j,k=0,l=0;
   char ch;
   
   eat_white_space(stream);
   read_str("[", stream);
   eat_white_space(stream);

   /* while there are still elements of the component list
      to be retrieved */
   while (!get_fa_comp_element(comp_list[i], stream) )
   {
      eat_white_space(stream);
      read_str(",", stream);
      if (++i >= COMPLSTSZE) ferr("Component list too long");
   }

   /* S.B.
   Concatenate all the elements of the 2 dimensional component
   list into a list to be displayed on the canvas */ 

   for (j=0;j<=i;j++)
   {
      k = 0;
      while (comp_list[j][k] != '\0')
      {
         total_comp[l]=comp_list[j][k]; 
         k++;
         l++;
      }
      if (j < i){
         total_comp[l]= ',';
         l++;
      }
   }
  
   total_comp[l] = '\0';
 
   return;
      
}

/* ----------------------------------------------------------- */
int get_fa_comp_element(comp_element, stream)
   char	        comp_element[];
   FILE		*stream;
{
/* S.B.
   Reads in one element of the component list
   i.e. one row of the 2-d array component list
*/
   int i = 0;
   char ch; 
   int done = FALSE;

   eat_white_space(stream);

   while ( ((ch = fgetc(stream)) != ',') && !done) {
      eat_white_space(stream);
      if (ch == ']')
      {
          done = TRUE;
      }
      else
      {
         comp_element[i] = ch;
      }
      if (ch == EOF) ferr("Unexpected end of input");
      if (++i >= STRSZE) ferr("Component list element too long");
   }
      comp_element[i] = '\0'; 
      ungetc(ch, stream); 

   return (done);
}

/* ----------------------------------------------------------- */

void label_tree(label, tp, stream)
   char     label;
   TREE_PTR tp;
   FILE     *stream;
{
/* Inserts the character label into the aux_label of the 
   appropriate tree node.  This is done by traversing the tree
   according to directions in the aux_list of the input stream.
   Note: for backward compatibility, "f" corresponds to "0", and "a"
   to "1".  The stream will contain digits corresponding to which
   subtree is selected.
*/
   char ch;
   int num ;

   read_str("[", stream);
	eat_white_space(stream);
   ch = fgetc(stream);

   if (ch == ']') tp=tp ;   /* this was added to account for
			       entering [] as a valid aux list
			       RH                             */
   while (ch != ']')
    {
      /* Modified by Fred 1991/07/02 */
      switch (ch) {
         case 'f': 
         case 'F': 
	   num = 0; break;
         case 'a': 
         case 'A': 
	   num = 1; break;
         default : num = (int)ch  - 48; }

      if (ch == EOF) ferr("Unexpected end of input");

      if ((num >= 0) && (num <= 4)) 
       {
 	if (tp->ntree[num] == (TREE *)NULL)
            ferr("Invalid auxiliary list")
	else 
           tp = tp->ntree[num];
       }
      else if (ch != ',') 
           ferr("Invalid auxiliary list element");

      eat_white_space(stream);
      ch = fgetc(stream);
    }
   tp->aux_label = label;

   return;
}
/* ----------------------------------------------------------- */
FILE *open_input(in)
   char *in;
{
/* Opens input file for reading.
*/
   FILE *stream;
   char errmsg[STRSZE];

   if (strcmp(in, "stdin")) {
      if ((stream = fopen(in, "r")) == NULL) {
         sprintf(errmsg, "Cannot open input file %s\n", in);
         ferr(errmsg);
      }
   } else stream = stdin;

   return (stream);
}
/* ----------------------------------------------------------- */   
void read_str(str, stream)
   char *str;
   FILE *stream;
{
/* Reads in a required string from the input stream, ignoring
   leading blanks. If the string is not found, then the program
   terminates.
*/
   char dummy[STRSZE];

   eat_white_space(stream);

   if (!fgets(dummy, strlen(str) + 1, stream) || strcmp(dummy, str)) 
      ferr ("Invalid input (3)");    
   return;
}
/* ----------------------------------------------------------- */   
    /* function to read in the next character in the file 
       and return it.                                     RH   */

char next_char (stream) 
     FILE *stream;
{
        char next ;

      	eat_white_space(stream);

	next = fgetc(stream) ;
        ungetc(next, stream) ;
	return(next) ;
} 

/* ----------------------------------------------------------- */

int average(num_array,size_of_array)
    int *num_array ,
        size_of_array ;

{	
	int temp = 0 ,
	    i ;

	for (i=0;i<size_of_array;i++)
		temp = temp + num_array[i] ;
	return(temp/size_of_array) ;
}
	
/* ----------------------------------------------------------- */
void get_args(argc, argv, in,font)
   int  argc;
   char *argv[], *in;
   char *font;
{
/* Gets the command line arguments for the program. If they are
   entered incorrectly the usage line is displayed.  The single
   argument is calculated as a side-effect rather than returned
   as the function value so that if more arguments are used
   they are all calculated in the same manner (a matter of
   consistency).
*/
   int i, ok = 1;
	char fontin[15];		/* size of typical filename-RH */
   
   strcpy(in, "stdin");
   fontin[0] = '\0';
   for (i = 1; (ok && (i < argc-1) && (*argv[i] == '-')); i++) {
      ++argv[i];
      switch (*argv[i++]) {
      case 'i': in = strcpy(in, argv[i]); break;
      case 'f': strcpy(fontin, argv[i]); break;
      default : ok = 0;
      }
   }
   strcat(font,fontin) ;	       /* concat font and fontin, return font*/
   if (i != argc || !ok) {
      printf("Usage: treetool [-i input]\n");
      exit(-1);
   }
   return;
}
/* ----------------------------------------------------------- */

void menu_action_proc();

make_control_panel(panel)
   Panel  panel;
{
	int i=0;
/* S.B.
   Creates the panel with buttons for manipulating the treetool
   displays.
*/

   Panel_item     heading_item, next_item,
                  previous_item, quit_item,
                  clear_item, display_item, super_item;


   heading_item = xv_create(panel, PANEL_MESSAGE,
      XV_X, xv_col(panel,1),
      XV_Y, xv_row(panel,2), 
      PANEL_LABEL_STRING, "TREE DISPLAY TOOL:", /* New label - Fred */
      0);

   next_item = xv_create(panel, PANEL_BUTTON,
                XV_X, xv_col(panel,5),
                XV_Y, xv_row(panel,6),
                PANEL_NOTIFY_PROC, next_proc,
                PANEL_LABEL_STRING, "Next",
		0);

   previous_item = xv_create(panel, PANEL_BUTTON,
                XV_X, xv_col(panel,5),
                XV_Y, xv_row(panel,8),
                PANEL_NOTIFY_PROC, previous_proc,
                PANEL_LABEL_STRING, "Previous",
		0);

   clear_item = xv_create(panel, PANEL_BUTTON,
                    XV_X, xv_col(panel,5),
                    XV_Y, xv_row(panel,10),
                    PANEL_NOTIFY_PROC, clear_proc,
                    PANEL_LABEL_STRING,"Clear",
                    0);

   super_item = xv_create(panel, PANEL_BUTTON,
                XV_X, xv_col(panel,5),
                XV_Y, xv_row(panel,12),
		PANEL_NOTIFY_PROC, super_proc,
                PANEL_LABEL_STRING, "Next Batch ",
		0);

   quit_item = xv_create(panel, PANEL_BUTTON,
                XV_X, xv_col(panel,5),
                XV_Y, xv_row(panel,16),
		PANEL_NOTIFY_PROC, quit_proc,
                PANEL_LABEL_STRING, "Quit",
		0);

 /* init_display_menu (display_menu, operation) 
   { 
	if (operation == MENU_CREATE) {
   		for (i=0 ; i< num_structs ;i++) 
		    menu_set(m, MENU_SET, fa[i]->name,i+1,0) ;
	}
   }
*/



    display_menu = (Menu)xv_create( NULL, MENU,
                                    MENU_NOTIFY_PROC, menu_action_proc,
                                    NULL ); 
    for (i=0 ; i< num_structs ;i++) 
      menu_set(display_menu, MENU_STRING_ITEM, fa[i]->name,i+1,0);


   display_item = xv_create(panel, PANEL_BUTTON,
                XV_X, xv_col(panel,5),
                XV_Y, xv_row(panel,14),
                PANEL_LABEL_STRING, "Display",
                PANEL_EVENT_PROC, display_proc,
                0);



}
/* -------------------------------------------------------------- */

void next_proc(next_item, event)
   Panel_item next_item;
   Event *event;
{
/* S.B. - Notify procedure for Next button on panel.
          This procedure is executed when Next button is selected
          with the left mouse button.
*/

   display++;  /*move to next treetool display*/
   if (display == num_structs)
      display = 0;

   /* clear canvas for next tree to be displayed */
   pw_writebackground(pw,0,0,CANVAS_WIDTH,XV_HEIGHT,PIX_CLR);
   fa_display();

   return;
}
/* -------------------------------------------------------------- */

void display_proc(display_item, event)
   Panel_item display_item;
   Event *event;
{
/* S.B. - Event procedure for Display button on panel.
          This procedure is executed when Display button is    
          selected with the right mouse button.
*/
   if ( event_id(event) == MS_RIGHT && event_is_down(event))
   {
     menu_show(display_menu, panel, event, 0);
   }
  return;
}

#ifdef XVIEW_COMMENT
     XView CONVERSION - No longer blocks until menu taken down, should specify
     a MENU_ACTON_PROC for each menu item or at least a MENU_DONE_PROC,
     see Sect 3.9 and XVPM Chap 11
#endif

void menu_action_proc( menu, menu_item)
   Menu menu;
   Menu_item menu_item;
{ 
   int value = (int)xv_get(menu_item, MENU_VALUE);
   /* if no choice from the menu is made, then redisplay same tree */ 
   if ( value == 0 )
      display = display;
   else
      display = value - 1;

   /* clear canvas for appropriate tree to be displayed */
   pw_writebackground(pw,0,0,CANVAS_WIDTH,XV_HEIGHT,PIX_CLR);
   fa_display();

   return;
}
/* -------------------------------------------------------------- */

void quit_proc(quit_item, event)
   Panel_item quit_item;
   Event *event;
{
/* S.B. - Notify procedure for Quit button on panel.
          This procedure is executed when Quit button is 
          selected with the left mouse button.
*/

   
   done = TRUE; /*finished displaying trees*/
   xv_set(base_frame,FRAME_NO_CONFIRM,TRUE,0);
   xv_destroy_safe(base_frame);
   exit(0);  
   return;
}
/* -------------------------------------------------------------- */
void super_proc(super_item,display_item, event)
   Panel_item super_item;
   Panel_item *display_item;
   Event *event;
{
/* R.H. - Notify procedure for Next batch button on panel.
          This procedure is executed when next batch button
          is selected with the left mouse button.
*/
   int i;
   display=0;
   window_done(pw);        /* close window in prepartion for new */
   for (i=0;i<=num_structs;i++) free(fa[i]) ;   /*free up fa's */
   eat_white_space(stream);
   num_structs=get_structs(stream,fa);  /* read in new data */
   if (num_structs > 0) {               /* if there is more to do */
   	make_control_panel(panel) ;  /* must rebuild the panel */
   	pw_writebackground(pw,0,0,CANVAS_WIDTH,XV_HEIGHT,PIX_CLR);
   	fa_display();
    }
   else {    /* else quit */
         xv_set(base_frame,FRAME_NO_CONFIRM,TRUE,0);
         xv_destroy_safe(base_frame);  /* closes all windows */
         exit(0);
        }
   return;
}
/* -------------------------------------------------------------- */

void previous_proc(previous_item, event)
   Panel_item previous_item;
   Event *event;
{
/* S.B. - Notify procedure for Previous button on panel.
          This procedure is executed when Previous button
          is selected with the left mouse button.
*/

   display--;  /*move to the previous tree*/
   if (display < 0)
      display = num_structs - 1;
   
   /* clear canvas for previous tree to be displayed */ 
   pw_writebackground(pw,0,0,CANVAS_WIDTH,XV_HEIGHT,PIX_CLR);
   fa_display();

   return;
}
/* -------------------------------------------------------------- */

void clear_proc(clear_item, event)
   Panel_item clear_item;
   Event *event;
{
/* S.B. - Notify procedure for Clear button on panel.
          This procedure is executed when Clear button 
          is selected with the left mouse button.
*/

  
   /* clear the canvas */
   pw_writebackground(pw,0,0,CANVAS_WIDTH,XV_HEIGHT,PIX_CLR);

   return;
}
/* -------------------------------------------------------------- */
