%%%  -*- Mode: PROLOG; Package: PROLOG-USER -*-

%%% DEFINES X-BAR THEORY AND S-STRUCTURE RULES
%%%
%%% (c) 1991, 1992, 1993, 1994  Sandiway Fong, NEC Research Institute, Inc.
%%%

%%% EXPORT
%%% X-BAR CATEGORIES AND RELATIONS
%%%	head(C)			head projection
%%%	bar(C)			intermediate projection
%%%	max(C)			maximal projection
%%%	lexicalCat(C)		lexical categories
%%%	proj(C1,C2)		C2 is the immediate projection of C1
%%%
%%%	spec(C1,SpecC)
%%%	overtSpec(C1,SpecC)
%%%	nullSpec(C1)
%%%	compl(C,ComplementC)
%%%
%%% S-STRUCTURE GRAMMAR RULES
%%%	rule LHS -> RHS
%%%	adjunction rule LHS -> RHS
%%%	Side add_goals [G1,..,Gn]
%%%	Side app_goals [G1,..,Gn]
%%%
%%% IMPORT
%%% PARAMETERS
%%%	specInitial/specFinal				parameters(Language)
%%%	headInitial/headFinal
%%%	agr(strong/weak)
%%%	negationMoves
%%%	(operators)					opDecls

%% CATEGORY LABELS

head(n). head(v). head(a). head(p). head(i). head(c). head(neg). 

bar(n1). bar(v1). bar(a1). bar(p1). bar(i1). bar(c1). bar(neg1).
         bar(v2).

max(np). max(vp). max(ap). max(pp). max(i2). max(c2). max(negp). 

lexicalCat(n). 	lexicalCat(v). 	lexicalCat(a). 	lexicalCat(p). lexicalCat(neg).



%% BASIC RELATIONS

%% for category labels X & Y, proj(X,Y) holds if Y is the
%% immediate projection of X.

proj(n,n1). 	proj(n1,np).
proj(v,v1). 	proj(v1,v2). 	proj(v2,vp).
proj(a,a1). 	proj(a1,ap).
proj(p,p1). 	proj(p1,pp).
proj(c,c1). 	proj(c1,c2).
proj(i,i1). 	proj(i1,i2).
proj(neg,neg1). proj(neg1,negp).

%% head(X,Y) holds if Y is the head of X.
%% NB. Reflexive relation.

head(n,n). 	head(n1,n). 	head(np,n).
head(v,v). 	head(v1,v). 	head(v2,v). 	head(vp,v).
head(a,a). 	head(a1,a). 	head(ap,a).
head(p,p). 	head(p1,p). 	head(pp,p).     
head(c,c). 	head(c1,c). 	head(c2,c).
head(i,i). 	head(i1,i). 	head(i2,i).
head(neg,neg).	head(neg1,neg).	head(negp,neg).

%% spec(X,Y) holds if Y can be the specifier for X (single-bar).
%% NB.	1. Option for no specifier indicated by Y=[].
%%      2. X is the category label for the sister to Y.

:- multifile spec/2.
spec(n1,det).	spec(n1,np). 	spec(n1,[]).
spec(i1,np).
spec(c1,[]).

overtSpec(X1,Y) :- spec(X1,Y), Y \== [].
nullSpec(X1) :- spec(X1,[]) ; \+ spec(X1,_).

%% compl(X,Y) holds if Y can be the complement of X.
%% NB.	1. No complement option not available here.
%%	2. Additional definitions from lexicon (theta-grids,subcat)
%%	3. X is the category label for the sister to Y

compl(i,vp).	compl(i,negp). compl(neg,vp). compl(c,i2).

:- multifile (rule)/1.

%% X-BAR RULES

rule XP -> [XB|spec(XB)] ordered specFinal st max(XP), proj(XB,XP).

rule spec(XB) -> [Y] st overtSpec(XB,Y).
rule spec(XB) -> [] st nullSpec(XB).

rule XB2 -> [XB1|icompl(XB1)] ordered headInitial(X0)
			      st bar(XB2), proj(XB1,XB2), bar(XB1),
			         head(XB2,X0), head(X0).

rule icompl(XB) -> [] st lexicalProperty(XB,subcat(_,_),_).
rule icompl(XB) -> [] st lexicalProperty(XB,grid(_,Roles),no2ndRole(Roles),_).
rule icompl(XB) -> [Y] st lexicalProperty(XB,grid(_,Roles),csr2nd(Roles,Y),Y).

rule XB -> [X|compl(X)] ordered headInitial(X)
			st bar(XB), proj(X,XB), head(X).

rule compl(X) -> [] st lexicalProperty(X,grid(_,Roles),no1stRole(Roles),_).
rule compl(X) -> [Y] st compl(X,Y).
rule compl(X) -> [Y] st lexicalProperty(X,grid(_,Roles),csr1st(Roles,Y),Y).
rule compl(X) -> [Y] st lexicalProperty(X,subcat(Y$_,_),Y).

%% NON-X-BAR RULES

% Empty categories

rule np with Fs -> [] st ecNPFeatures(Fs).
rule c with Fs -> [] st emptyCompFeatures(Fs).	  % language-specific

%% HEAD MOVEMENT
%% NB. requires rule for empty C for adjunction in periphery

rule v(V) moves_to i provided agr(strong), finite(V).
rule v(V) moves_to i provided agr(weak), V has_feature aux.

rule v(V) moves_to i moves_to c provided agr(strong), finite(V).
rule v(V) moves_to i moves_to c provided 
	agr(weak), V has_feature aux, finite(V).

rule i(I) moves_to v(V) provided agr(weak), \+ V has_feature aux,
	post_cond addFeature(noECP(_),I).
rule i(I) moves_to v(V) provided agr(strong), \+ finite(V),
	post_cond addFeature(noECP(_),I).

rule v doesnt_move provided v$vp subcat_by v.
rule v(V) moves_through neg provided_iff negationMoves, V has_feature neg.

rule i(I) derived_from v(V) provided recoverIfromV(I,V).
rule neg(Neg) derived_from v(V) provided negFromV(Neg,V).

% Coping with antecedent government of head traces
rule head_adjoined _ add_feature transparent(ecp).

% Special categories

rule relClNP -> [overtNP] st true.
rule overtNP -> [np(NP)] st \+ ec(NP).
rule cpAdjAP -> [ap(AP)] st AP has_feature allowExt(reason).

:- multifile (adjunction)/1.

%% ADJUNCTION

% Adjunction to head

adjunction rule n -> [a,n] st lexicalProperty(a,grid([_],_)).
adjunction rule a -> [adv,a] st lexicalProperty(adv,degree).
adjunction rule v -> [v,adv] ordered headInitial(v) 
				st lexicalProperty(adv,degree).

adjunction rule i -> [cl,i] st lexicalProperty(cl,morphC(nom)).
adjunction rule i -> [cl,i] st lexicalProperty(cl,polarity(negative)).

% Adjunction to max

adjunction rule vp -> [vp,pp] ordered headInitial(v).
adjunction rule vp -> [vp,adv] st lexicalProperty(adv,adjoin(right)).
adjunction rule vp -> [adv,vp] st lexicalProperty(adv,adjoin(left)).

adjunction rule np -> [relClNP,c2] ordered headInitial(n).

adjunction rule ap -> [cpAdjAP,c2].

% Bracketing (local rules respect bracketing)

:- multifile (bracket)/1.

bracket c2.
bracket i2 idominated_by v1.
bracket i2 idominated_by n1.
bracket i2 idominated_by p1.

%% MODIFICATION RULES

%% Define subjects to have feature apos (A-position)

:- multifile (app_goals)/2.
rhs [np(NP),i1] ordered specInitial app_goals [aPos(NP)].
rhs [np(NP),n1(N1)] ordered specInitial app_goals [npSpecNP(NP,N1)].

:- multifile (add_goals)/2.
rhs anywhere i1(X) add_goals [matrixIPFeature(X)].

:- multifile (left_bracket)/1.
left_bracket c2 add_goals [unboundedLookahead(input(i))].
left_bracket i2 idominated_by v1 add_goals [unboundedLookahead(input(i))].
left_bracket i2 idominated_by n1 add_goals [unboundedLookahead(input(i))].
left_bracket i2 idominated_by p1 add_goals [unboundedLookahead(input(i))].

%%% SUPPORTING DEFINITIONS

%% allow [SPEC,NP] 
%% 1a. N has a non-empty 0-grid, or
%% 1b. SPEC is a nonargument
%% And:
%% 2. Its complement (if one exists) is not a CP.
%% 3. Anti-coindexation holds between [SPEC,NP] and NP

npSpecNP(NP,N1) :-
	( nonEmptyGrid(N1) ; NP has_feature nonarg(+)),
	!,					  % red
	\+ cat(Compl,c2) if Compl complement_of N1,
	index(NP,I),
	index(N1,J),
	addFeature(neq(I,J),N1),
	aPos(NP).

%% Feature [matrix] is added to all matrix IPs
matrixIPFeature(X) :- addFeature(matrix,X) if isMatrixLevel.

%% INFL and V

finite(V) :- \+ V has_feature inf(_).

recoverIfromV(I,V) :-
	inflFeatures(V,Fs),
	mkEC(i,Fs,I).
	
inflFeatures(V,Fs) :- infinitival(V,Fs).
inflFeatures(V,Fs) :- tnsAGRFeatures(V,Fs).
	
%% either
%% 1. inf([]) from an infinitival marker, or
%% 2. language-specific
%% nb. only for non-simple verbs

infinitival(V,Fs) :-
	( V has_feature inf(X) 
	; inf(V,X)),				  % language-specific
	!,
	mkFs([index(_),inf(X)],Fs).

%% only for tensed forms

tnsAGRFeatures(V,Fs) :-
	V has_feature morph(Base,Form),
	\+ V has_feature inf(_),
	vMorphToFs(Base,Form,[TnsAGR|Fs1]),
	mkFs([index(_),TnsAGR|Fs1],Fs).





