%%%  -*- Mode: PROLOG; Package: PROLOG-USER -*-

%%% SAMPLE US ENGLISH LEXICON
%%%
%%% (c) 1991, 1992, 1993  Sandiway Fong, NEC Research Institute, Inc.
%%%

%%% EXPORT
%%%	term(C)				terminals
%%%	lexicon(Word,C,Fs)		Word has category label C
%%%					and feature list Fs
%%%	probeLexicon(Word)		Word is in lexicon
%%%	vMorphToFs(Base,Form,Features)	TNS/AGR features
%%%	inf(Verb,Type)
%%%	relevant(C)			constraints imposed by markers
%%%					apply to C
%%% REFERENCES
%%%	(list processing)		utilities
%%%	ensureWhComp(Ci)		lex
%%%	optWhComp(Ci)			

term(n). term(a). term(v). term(p). term(c). 
term(adv).  term(det). %term(conj).
term(neg). term(mrkr). term('$').

%%% 	Most lexical entries are stored directly as
%%%
%%%		 lex(Word,Category,Features)
%%%
%%%	Non-base forms require inference: 
%%%	1. plural nouns		all features except agr(_) inherited
%%%				from the sg. form
%%%	2. nominalized verbs	inherits verb features except morph(_,_)
%%%	3. non-base verb forms	all features except morph(_,_) inherited
%%%				from the base form

:- dynamic edrEng_lexicon/3.

lexicon(Word,C,Fs) :-
	nonvar(Word)
	-> (probeLexicon(Word)
	   -> builtin(Word,C,Fs)
	   ;  edrEng_lexicon(Word,C,Fs))
	;  builtin(Word,C,Fs).

probeLexicon(Word) :- lex(Word,_,_) ; lex(Word,_,_,_).

builtin(Word,C,Fs) :- lex(Word,C,Fs).		% directly available
builtin(Form,v,Fs) :-				% non-base verb forms
	lex(Form,v,Base,F1),
	verbFeatures(Base,F2),
	append1(F1,F2,Fs).

% lexicon(Word,C,Fs) :- lex(Word,C,Fs).		% directly available
% lexicon(Form,v,Fs) :-				% non-base verb forms
%	lex(Form,v,Base,F1),
%	verbFeatures(Base,F2),
%	append1(F1,F2,Fs).

%%% NEGATION

lex(not,neg,[polarity(-)]).

%%% PREPOSITIONS

lex(about,p,[grid([],[theme]),predicate(theme),
	     adjR([goal(roleNotPresent(X,theme),X)])]).
lex(after,p,[grid([],[reducedProp]),selR([not(feature(inf([])))]),
	     predicate(manner)]). % use+ing
lex(because,p,[grid([],[reducedProp]),predicate(reason)]).
lex(before,p,[grid([],[reducedProp]),selR([not(feature(inf([])))]),
	      predicate(manner)]). % use+ing
lex(by,p,[grid([],[agent]),adjR([goal(unsaturatedExtRole(X,agent),X)]),
	  predicate(agent)]).
lex(in,p,[grid([],[location]),predicate(location)]).
lex(under,p,[grid([],[location]),predicate(location)]).

lex(on,p,[grid([],[location]),predicate(location)]).
lex(to,p,[grid([],[destination]),predicate(destination),
	  adjR([goal(vpAllowExt(destination,X),X)])]).
lex(toward,p,[grid([],[destination]),predicate(destination),
	  adjR([goal(vpAllowExt(destination,X),X)])]).
lex(at,p,[grid([],[patient]),predicate(patient),  % .. at something/body
	  adjR([goal(vpAllowExt(patient,X),X)])]).

lex(with,p,[grid([],[patient]),predicate(patient), % with s.o.
	  adjR([goal(vpAllowExt(patient,X),X)])]).

lex(without,p,[grid([],[reducedProp]),selR([not(feature(inf([])))]),
	       predicate(manner)]). % use+ing 

%% Adverbs

lex(how,adv,[adjoin(right),wh,predicate(manner)]).
lex(why,adv,[adjoin(right),wh,predicate(reason)]).
lex(completely,adv,[adjoin(left),predicate(manner)]).
lex(often,adv,[adjoin(left),predicate(manner)]).
lex(sincerely,adv,[adjoin(left),predicate(manner)]).
lex(here,adv,[adjoin(right),predicate(location)]).
lex(late,adv,[adjoin(right),predicate(time)]).
lex(yesterday,adv,[adjoin(right),predicate(time)]).
lex(hard,adv,[degree,adjR([feature(allow(adv,degree))])]). % ... special case
lex(well,adv,[adjoin(right),predicate(manner)]).
lex(badly,adv,[adjoin(right),predicate(manner)]).

%% Degree Adverbs (modify adjectives)

% permit the resulting AP to take an optional clausal adjunct
lex(too,adv,[degree,adjR([addFeatures([allowExt(reason),adjR([goal(nonfiniteClause(X),X)])])])]).

%% Determiners
%%	Combine only with nouns that have the count(+/-/_) feature (common nouns) 
%%	Six classes of determiners:
%%	Class Examples                     Features
%%	1     the,my,our,whose,which,what  count(_) agr([3,[],[]])
%%	2     some,any,enough              count(_) agr([3,[pl,m],[]])
%%	3     this,that                    count(_) agr([3,[sg,m],[]])
%%	4     few,these,those              count(+) agr([3,pl,[]])
%%	5     a,every,each,either,neither  count(+) agr(([3,sg,[]])
%%	6     much                         count(-) agr(([3,m,[]])
%%
%%	op(+) elements that are moved by QR and 
%%	form operator-variable structures

lex(a,det,[count(+),agr([3,sg,[]]),vow(-),def(-)]).
lex(an,det,[count(+),agr([3,sg,[]]),vow(+),def(-)]).
lex(a_certain,det,[count(+),agr([3,sg,[]])]).
lex(any,det,[count(_),agr([3,[pl,m],[]])]).
lex(each,det,[count(+),agr([3,sg,[]]),op(+)]).
lex(either,det,[count(+),agr([3,sg,[]])]).
lex(enough,det,[count(_),agr([3,[pl,m],[]]),op(+)]).
lex(every,det,[count(+),agr([3,sg,[]]),op(+)]).
lex(few,det,[count(+),agr([3,pl,[]]),op(+)]).
lex(many,det,[count(+),agr([3,pl,[]]),op(+)]).
lex(much,det,[count(-),agr([3,m,[]]),op(+)]).
% lex(my,det,[count(_),agr([3,[],[]])]).
% lex(our,det,[count(_),agr([3,[],[]])]).
lex(neither,det,[count(+),agr([3,sg,[]])]).
lex(some,det,[count(_),agr([3,[],[]]),op(+)]).
lex(that,det,[count(_),agr([3,[sg,m],[]])]).
lex(the,det,[count(_),agr([3,[],[]])]).
lex(these,det,[count(+),agr([3,pl,[]])]).
lex(this,det,[count(_),agr([3,[sg,m],[]])]).
lex(those,det,[count(+),agr([3,pl,[]])]).
% lex(what,det,[count(_),agr([3,[],[]]),wh]).
lex(which,det,[count(_),agr([3,[],[]]),wh]).
lex(whose,det,[count(_),agr([3,[],[]]),wh]).

%% Nouns
%% 	Obligatory features
%% 	1. Anaphoric: a(+/-), Pronominal: p(+/-) for Binding Theory
%% 	2. Agreement: agr([Per,Num,Gen]) 
%%	3. O-grid: grid(Ext,Int) 
%% 	Optional features
%% 	1. Morphologically realized Case: morphC(C)
%% 	2. Common nouns: count(+/-/_) (not present for proper nouns)
%%	3. Quantifiers: op(+/-) 
%%	4. Wh-nouns: wh

% Numbers and Ordinals

lex(one,n,[grid([],[]),morphC(gen),count(_),agr([3,sg,[]]),nonarg(+)]).
lex(three,n,[grid([],[]),morphC(gen),count(_),agr([3,pl,[]]),nonarg(+)]).

lex(first,n,[grid([],[]),morphC(gen),count(+),agr([3,sg,[]]),nonarg(+)]).
lex(third,n,[grid([],[]),morphC(gen),count(+),agr([3,sg,[]]),nonarg(+)]).

% Pleonastic elements

lex(there,n,[grid([],[]),agr([3,[],[]]),nonarg(+),linkTo(np)]).
lex(it,n,[grid([],[]),agr([3,sg,n]),
    goal(nonarg(A),setFromPlusMinus(A,[noCoindex,linkTo(c2)],[a(-),p(+)]))]).

% not sure?
lex(now,n,[grid([],[]),agr([3,[],[]])]).

% Proper nouns
lex(bill,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(john,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(mary,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).
lex(rome,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(susan,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).

% Anaphors and pronominals
lex(each_other,n,[grid([],[]),a(+),p(-),agr([3,pl,[m,f]])]).

lex(he,n,[grid([],[]),morphC(nom),a(-),p(+),agr([3,sg,m])]).
lex(him,n,[grid([],[]),morphC(acc),a(-),p(+),agr([3,sg,m])]).
lex(himself,n,[grid([],[]),a(+),p(-),agr([3,sg,m])]).

% 'their', 'his' ambiguous wrt Binding features
lex(his,n,[grid([],[]),morphC(gen),agr([3,sg,m]),
	   goal(a(A),inv_plus_minus(A,P)),p(P)]).
lex(their,n,[grid([],[]),morphC(gen),agr([3,pl,[m,f]]),
	   goal(a(A),inv_plus_minus(A,P)),p(P)]).

lex(them,n,[grid([],[]),morphC(acc),a(-),p(+),agr([3,pl,[m,f]])]).

lex(themselves,n,[grid([],[]),a(+),p(-),agr([3,pl,[m,f]]),
		  oneof(_,[[morphC(acc)],[morphC(gen)]])]).

lex(they,n,[grid([],[]),morphC(nom),a(-),p(+),agr([3,pl,[m,f]])]).

lex(she,n,[grid([],[]),morphC(nom),a(-),p(+),agr([3,sg,f])]).
lex(her,n,[grid([],[]),morphC(acc),a(-),p(+),agr([3,sg,f])]).
lex(her,n,[grid([],[]),morphC(gen),agr([3,sg,f]),
	   goal(a(A),inv_plus_minus(A,P)),p(P)]).
lex(herself,n,[grid([],[]),a(+),p(-),agr([3,sg,f])]).

lex(i,n,[grid([],[]),morphC(nom),a(-),p(+),agr([1,sg,[m,f]])]).
lex(me,n,[grid([],[]),morphC(acc),a(-),p(+),agr([1,sg,[m,f]])]).
lex(my,n,[grid([],[]),morphC(gen),a(-),p(+),agr([1,sg,[m,f]])]).
lex(myself,n,[grid([],[]),a(+),p(-),agr([1,sg,[m,f]])]).

lex(you,n,[grid([],[]),a(-),p(-),agr([2,[],[m,f]])]).

lex(we,n,[grid([],[]),morphC(nom),a(-),p(+),agr([1,pl,[m,f]])]).
lex(us,n,[grid([],[]),morphC(acc),a(-),p(+),agr([1,pl,[m,f]])]).

lex(itself,n,[grid([],[]),a(+),p(-),agr([3,sg,n])]).

% Quantifier nouns
lex(everyone,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),op(+)]).
lex(someone,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),op(+)]).

% Wh-nouns
lex(who,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),wh]).
lex(whom,n,[grid([],[]),morphC(acc),a(-),p(-),agr([3,sg,[m,f]]),wh]).
lex(what,n,[grid([],[]),a(-),p(-),agr([3,[],n]),wh]).
lex(which,n,[grid([],[]),a(-),p(-),agr([3,[],n]),wh]).

% Common nouns
lex(actress,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),vow]).
lex(actor,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vow]).
lex(answer,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n]),vow]).
lex(article,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n]),vow]).
lex(bastard,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]]),epithet]).
lex(boy,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(building,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(bus,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(cake,n,[grid([],[]),a(-),p(-),count(_),agr([3,[],n])]).
lex(cat,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(cats,n,[grid([],[]),a(-),p(-),count(+),agr([3,pl,[m,f]])]).
lex(cheese,n,[grid([],[]),a(-),p(-),count(+),agr([3,[],n])]).
lex(chicken,n,[grid([],[]),a(-),p(-),count(+),agr([3,[],n])]).
lex(cleaner,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(city,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(demonstration,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(dog,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(explosion,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n]),vow]).
lex(fire,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(floor,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(friend,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(garden,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(ice_cream,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],n]),vow]).
lex(lawyer,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(letter,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(liar,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(man,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(medal,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(play,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(paper,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(playground,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(police,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,n])]).
lex(policeman,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(problem,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(race,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(railroad_station,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(rat,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(report,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(road,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(room,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(stain,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(student,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),count(_)]).
lex(symphony,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(syntactic_structures,n,[grid([],[]),a(-),p(-),count(-),agr([3,sg,n])]).
lex(time,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,n])]).
lex(this,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,n])]).
lex(unicorn,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(woman,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(worker,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,n])]).

% Nouns with O-grids
lex(book,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(brother,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(father,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(furniture,n,[grid([possessor],[]),a(-),p(-),count(-),agr([3,m,n])]).
lex(mother,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(picture,n,[grid([possessor],[theme]),a(-),p(-),count(+),agr([3,sg,n])]).
lex(teacher,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).

% Non-base noun forms

lex(actresses,n,[agr([3,pl,f])|F]) :- nounFeatures(actress,F).
lex(actors,n,[agr([3,pl,m])|F]) :- nounFeatures(actor,F).
lex(answers,n,[agr([3,pl,n])|F]) :- nounFeatures(answer,F).
lex(articles,n,[agr([3,pl,n])|F]) :- nounFeatures(article,F).
lex(bastards,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(bastard,F).
lex(beliefs,n,[agr([3,pl,n])|F]) :- nounFeatures(belief,F).
lex(books,n,[agr([3,pl,n])|F]) :- nounFeatures(book,F).
lex(boys,n,[agr([3,pl,m])|F]) :- nounFeatures(boy,F).
lex(buildings,n,[agr([3,pl,n])|F]) :- nounFeatures(building,F).
lex(buses,n,[agr([3,pl,n])|F]) :- nounFeatures(bus,F).
lex(cities,n,[agr([3,pl,n])|F]) :- nounFeatures(city,F).
lex(cleaners,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(cleaner,F).
lex(demonstrations,n,[agr([3,pl,n])|F]) :- nounFeatures(demonstration,F).
lex(departures,n,[agr([3,pl,n])|F]) :- nounFeatures(departure,F).
lex(dogs,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(dog,F).
lex(explosions,n,[agr([3,pl,n])|F]) :- nounFeatures(explosion,F).
lex(fires,n,[agr([3,pl,n])|F]) :- nounFeatures(fire,F).
lex(floors,n,[agr([3,pl,n])|F]) :- nounFeatures(floor,F).
lex(friends,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(friend,F).
lex(gardens,n,[agr([3,pl,n])|F]) :- nounFeatures(garden,F).
lex(ice_creams,n,[agr([3,pl,n])|F]) :- 
	nounFeatures(ice_cream,F), 
	in1(count(+),F).
lex(lawyers,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(lawyer,F).
lex(letters,n,[agr([3,pl,n])|F]):- nounFeatures(letter,F).
lex(liars,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(liar,F).
lex(medals,n,[agr([3,pl,n])|F]):- nounFeatures(medal,F).
lex(men,n,[agr([3,pl,m])|F]):- nounFeatures(man,F).
lex(mothers,n,[agr([3,pl,f])|F]) :- nounFeatures(mother,F).
lex(papers,n,[agr([3,pl,n])|F]) :- nounFeatures(paper,F).
lex(plays,n,[agr([3,pl,n])|F]) :- nounFeatures(play,F).
lex(playgrounds,n,[agr([3,pl,n])|F]) :- nounFeatures(playground,F).
lex(policemen,n,[agr([3,pl,m])|F]) :- nounFeatures(policeman,F).
lex(pictures,n,[agr([3,pl,n])|F]) :- nounFeatures(picture,F).
lex(problems,n,[agr([3,pl,n])|F]) :- nounFeatures(problem,F).
lex(races,n,[agr([3,pl,n])|F]) :- nounFeatures(race,F).
lex(rats,n,[agr([3,pl,n])|F]) :- nounFeatures(rat,F).
lex(reports,n,[agr([3,pl,n])|F]) :- nounFeatures(report,F).
lex(roads,n,[agr([3,pl,n])|F]) :- nounFeatures(road,F).
lex(rooms,n,[agr([3,pl,n])|F]) :- nounFeatures(room,F).
lex(stains,n,[agr([3,pl,n])|F]) :- nounFeatures(stain,F).
lex(students,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(student,F).
lex(symphonies,n,[agr([3,pl,n])|F]) :- nounFeatures(symphony,F).
lex(teachers,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(teacher,F).
lex(unicorns,n,[agr([3,pl,n])|F]):- nounFeatures(unicorn,F).
lex(women,n,[agr([3,pl,f])|F]) :- nounFeatures(woman,F).
lex(workers,n,[agr([3,pl,n])|F]) :- nounFeatures(worker,F).

% Nominalized verbs
lex(belief,n,[count(+),agr([3,sg,n]),a(-),p(-)|F]) :-
	verbToNounFeatures(believe,F).
lex(claim,n,[count(+),agr([3,sg,n]),a(-),p(-)|F]) :-
	verbToNounFeatures(claim,F).
lex(departure,n,[count(+),agr([3,sg,n]),a(-),p(-)|F]) :-
	verbToNounFeatures(depart,F).
lex(destruction,n,[count(-),agr([3,m,n]),a(-),p(-)|F]) :-
	verbToNounFeatures(destroy,F).
lex(fight,n,[count(+),agr([3,sg,[]]),a(-),p(-)|F]) :-
	verbToNounFeatures(fight,F).

%%% VERBS - BASE FORMS
%
% Obligatory Elements
% 	1. grid(Ext,Int) denotes the O-grid
% 	2. morph(Base,[]/ing/s/ed(1)/ed(2)) 
% Optional Features
% 	1. ecm(opt/oblig) for ECM verbs
% 	2. noCasemark(+) for technically intransitive verbs
% 	3. subcat(C$Fs,Fs1) subcategorizes for an element of category C 
%    	   with feature restriction list Fs.
%    	   Feature list Fs1 will be appended to the feature list of C
% 	4. selR(Rs), selR(Type,R) restriction(s) Rs or R on direct object.

lex(appear,v,[morph(appear,[]),grid([theme],[])]).
lex(appreciate,v,[morph(appreciate,[]),grid([agent],[theme])]).
lex(arrest,v,[morph(arrest,[]),grid([agent],[patient])]).
lex(arrive,v,[morph(arrive,[]),grid([theme],[])]).
lex(ask,v,[morph(ask,[]),grid([agent],[proposition]),
	   selR(head(c2),goal(ensureWhComp(X),X))]).
lex(behave,v,[morph(behave,[]),grid([theme],[]),
	      allowExt(patient),requiresExt(manner)]).
lex(believe,v,[morph(believe,[]),grid([agent],[proposition]),ecm(oblig)]).
lex(bite,v,[morph(bite,[]),grid([agent],[patient])]).
lex(buy,v,[morph(buy,[]),grid([agent],[theme])]).
lex(call,v,[morph(call,[]),grid([agent],[patient])]).
lex(care,v,[morph(care,[]),grid([agent],[proposition]),noCasemark(+),
	   selR(head(c2),goal(ensureWhComp(X),X))]).
lex(claim,v,[morph(claim,[]),grid([agent],[proposition]),ecm(oblig)]).
lex(depart,v,[morph(depart,[]),grid([theme],[])]).
lex(destroy,v,[morph(destroy,[]),grid([agent],[patient])]).
lex(die,v,[morph(die,[]),grid([theme],[])]).
lex(disappear,v,[morph(disappear,[]),grid([theme],[])]).
lex(eat,v,[morph(eat,[]),grid([agent],[[patient]])]).
lex(fall,v,[morph(fall,[]),grid([theme],[])]).
lex(fight,v,[morph(fight,[]),grid([agent],[[patient]])]).
lex(file,v,[morph(file,[]),grid([agent],[patient])]).
lex(fire,v,[morph(fire,[]),grid([agent],[patient])]).
lex(give,v,[morph(give,[]),grid([agent],[theme]),allowExt(destination)]).
lex(give,v,[morph(give,[]),grid([agent],[goal,theme]),idoCase(acc)]).
lex(hear,v,[morph(hear,[]),grid([agent],[theme])]).
lex(hire,v,[morph(hire,[]),grid([agent],[patient])]).
lex(hit,v,[morph(hit,[]),grid([agent],[patient])]).
lex(instruct,v,[morph(instruct,[]),grid([agent],[patient])]).
lex(insult,v,[morph(insult,[]),grid([agent],[patient])]).
lex(keep,v,[morph(keep,[]),grid([agent],[patient])]).
lex(kill,v,[morph(kill,[]),grid([agent],[patient])]).
lex(know,v,[morph(know,[]),grid([agent],[proposition]),
	    selR(head(c2),goal(optWhComp(X),X))]).
lex(laugh,v,[morph(laugh,[]),grid([theme],[]),allowExt(patient)]).
lex(leave,v,[morph(leave,[]),grid([theme],[])]).
lex(lead,v,[morph(lead,[]),grid([agent],[patient])]).
lex(like,v,[morph(like,[]),grid([agent],[patient])]).
lex(love,v,[morph(love,[]),grid([agent],[patient])]).
lex(meet,v,[morph(meet,[]),grid([agent],[patient])]).
lex(mention,v,[morph(mention,[]),grid([agent],[proposition])]).
lex(park,v,[morph(park,[]),grid([agent],[])]).
lex(persuade,v,[morph(persuade,[]),grid([agent],[patient,proposition])]).
lex(please,v,[morph(please,[]),grid([agent],[patient])]).
lex(promise,v,[morph(promise,[]),grid([agent],[patient,proposition]),
	       selR([addFeature(noControl)])]).
lex(put,v,[morph(put,[]),grid([agent],[patient]),requiresExt(location)]).
lex(rain,v,[morph(rain,[]),grid([],[])]).
lex(read,v,[morph(read,[]),grid([agent],[theme])]).
lex(remove,v,[morph(remove,[]),grid([agent],[patient])]).
lex(resign,v,[morph(resign,[]),grid([theme],[])]).
lex(say,v,[morph(say,[]),grid([agent],[proposition]),
	   selR([goal(agrConstraint(X),X)])]).
lex(see,v,[morph(see,[]),grid([agent],[patient])]).
lex(seem,v,[morph(seem,[]),grid([],[proposition]),ecm(oblig),noCasemark(+)]).
lex(send,v,[morph(send,[]),grid([agent],[recipient,theme]),idoCase(acc)]).
lex(show,v,[morph(show,[]),grid([agent],[recipient,theme]),idoCase(acc)]).
lex(sleep,v,[morph(sleep,[]),grid([agent],[])]).
lex(snow,v,[morph(snow,[]),grid([],[])]).
lex(solve,v,[morph(solve,[]),grid([agent],[patient])]).
lex(speak,v,[morph(speak,[]),grid([agent],[]),allowExt(destination)]).
lex(stand,v,[morph(stand,[]),grid([agent],[proposition])]).
lex(start,v,[morph(start,[]),grid([agent],[patient])]).
lex(surprise,v,[morph(surprise,[]),grid([agent],[patient,proposition])]).
lex(talk,v,[morph(talk,[]),grid([agent],[]),allowExt(destination)]).
lex(tell,v,[morph(tell,[]),grid([agent],[patient,proposition])]).
lex(think,v,[morph(think,[]),grid([agent],[proposition]),
	doCase(gen),selR([goal(agrConstraint(X),X)])]).
lex(try,v,[morph(try,[]),grid([agent],[proposition])]).
lex(understand,v,[morph(understand,[]),grid([agent],[patient])]).
lex(visit,v,[morph(visit,[]),grid([agent],[patient])]).
lex(walk,v,[morph(walk,[]),grid([agent],[]),allowExt(destination)]).
lex(want,v,[morph(want,[]),grid([agent],[proposition]),ecm(opt)]).
lex(win,v,[morph(win,[]),grid([agent],[])]).
lex(win,v,[morph(win,[]),grid([agent],[patient])]).
lex(wonder,v,[morph(wonder,[]),grid([agent],[proposition]),noCasemark(+),
	      selR(head(c2),goal(ensureWhComp(X),X))]).
lex(work,v,[morph(work,[]),grid([theme],[]),allow(adv,degree)]).
lex(write,v,[morph(write,[]),grid([agent],[theme])]).

%% Auxiliaries


lex(be,v,[morph(be,[]),aux,
	  oneof(_,[		   [grid([eq],[eq])],
[grid([],[theme]),noCasemark(+),	% unicorn
		    selR([goal(overt(X),X)])],
		   [grid([pHolder(P)],subcat),subcat(adv$[predicate(P)],[])],
		   [subcat(ap$[grid(_,_)],[])],
		   [subcat(vp$[morph(_,ed(2)),grid([_],_)],
			   [noCasemark(+)]),blockTheta],
		   [subcat(vp$[morph(_,ing)],[prog])]])]).

% Stage 2
% lex(be,v,[morph(be,[]),aux,grid([eq],[eq])]).	  % john is the man
% lex(be,v,[morph(be,[]),aux,grid([pHolder(P)],subcat),	% john is here
%	  subcat(adv$[predicate(P)],[])]).
% Stage 1
% lex(be,v,[morph(be,[]),aux,subcat(ap$[grid(_,_)],[])]).	% john is dead
% lex(be,v,[morph(be,[]),aux,blockTheta,	  % john was arrested
%	  subcat(vp$[morph(_,ed(2))],[noCasemark(+)])]).
% lex(be,v,[morph(be,[]),aux,			  % john is 
%	  subcat(vp$[morph(_,ing)],[prog])]).

lex(have,v,[morph(have,[]),aux,perf,subcat(vp$[morph(_,ed(2))],[perf])]).
lex(do,v,[morph(do,[]),aux,subcat(vp$[morph(_,[])],[])]).

lex(can,v,[morph(can,def(past(-))),aux,modal,
	   subcat(vp$[morph(_,[])],[])]).
lex(could,v,[morph(could,def(past(+))),aux,modal,
	     subcat(vp$[morph(_,[])],[])]).
lex(would,v,[morph(would,def(past(+))),aux,modal,	
	     subcat(vp$[morph(_,[])],[])]).
lex(should,v,[morph(should,def(past(+))),aux,modal,
	      subcat(vp$[morph(_,[])],[])]).
lex(will,v,[morph(will,def(past(-))),aux,modal,
	    subcat(vp$[morph(_,[])],[])]).

%% Adjectives
%%

lex(aaaaa,a,[grid([],[patient])]).		% too complicated to explain

lex(certain,a,[grid([],[proposition]),ecm(oblig)]).
lex(clever,a,[grid([theme],[])]).
lex(crazy,a,[grid([theme],[])]).
lex(crucial,a,[grid([],[proposition])]).
lex(dumb,a,[grid([theme],[])]).
lex(eager,a,[grid([theme],[proposition])]).
lex(easy,a,[grid([],[proposition])]).
lex(gold,a,[grid([theme],[])]).
lex(hard,a,[grid([],[proposition])]).
lex(happy,a,[grid([theme],[])]).
lex(illegal,a,[grid([],[proposition])]).
lex(ill_founded,a,[grid([theme],[])]).
lex(illogical,a,[grid([theme],[])]).
lex(important,a,[grid([],[proposition])]).
lex(intelligent,a,[grid([theme],[])]).
lex(likely,a,[grid([],[proposition]),ecm(oblig)]).
lex(nice,a,[grid([theme],[])]).
lex(on_sale,a,[grid([theme],[])]).
lex(pathological,a,[grid([theme],[])]).
lex(proud,a,[grid([theme],[proposition])]).
lex(ready,a,[grid([theme],[]),allowExt(reason)]). % actually 'purpose'
lex(rotten,a,[grid([theme],[])]).
lex(sad,a,[grid([theme],[])]).
lex(stubborn,a,[grid([theme],[])]).
lex(tough,a,[grid([weakTheme$optional],[]),allowExt(reason)]).

% Complementizers
lex(for,c,[ecm,selR([feature(inf([]))])]).
lex(that,c,[selR([not(feature(inf(_)))]),specR(addFeature(noECP(ss)))]).
lex(whether,c,[wh]).
lex(if,c,[wh]).

% Markers: morphological realizations of particular features
lex(neg,mrkr,[left(v,aux,neg)]).		  % negation marker
lex(to,mrkr,[right(v,morph(_,[]),inf([]))]).	  % infinitival marker
lex(of,mrkr,[right(np,case(gen),[])]).		  % object genitive Case

% subject of NP genitive Case
lex('''s',mrkr,[left(n,[not(morphC(_)),case(gen)],[])]). 
lex('''',mrkr,[left(n,[not(morphC(_)),case(gen)],[])]).

:- initialization((no lexFeature(_,_))).

% Coordinating conjunctions
lex(and,conj,[overt(+)]).

% relevant for marker constraints
relevant(n).
relevant(v).

irregular(be).

% Non-base verb forms:
%
% Form x Category x Base x Features

lex(appearing,v,appear,[morph(appear,ing)]).
lex(appears,v,appear,[morph(appear,s)]).
lex(appeared,v,appear,[morph(appear,ed(_))]).

lex(appreciating,v,appreciate,[morph(appreciate,ing)]).
lex(appreciates,v,appreciate,[morph(appreciate,s)]).
lex(appreciated,v,appreciate,[morph(appreciate,ed(_))]).

lex(arresting,v,arrest,[morph(arrest,ing)]).
lex(arrests,v,arrest,[morph(arrest,s)]).
lex(arrested,v,arrest,[morph(arrest,ed(_))]).

lex(arriving,v,arrive,[morph(arrive,ing)]).
lex(arrives,v,arrive,[morph(arrive,s)]).
lex(arrived,v,arrive,[morph(arrive,ed(1))]).
lex(arriven,v,arrive,[morph(arrive,ed(2))]).

lex(asking,v,ask,[morph(ask,ing)]).
lex(asks,v,ask,[morph(ask,s)]).
lex(asked,v,ask,[morph(ask,ed(_))]).

lex(am,_,be,[morph(be,am)]).  % Cat=mrkr, aux
lex(are,_,be,[morph(be,are)]).
lex(been,_,be,[morph(be,ed(2))]).
lex(being,_,be,[morph(be,ing)]).
lex(is,_,be,[morph(be,s)]).
lex(was,_,be,[morph(be,was)]).
lex(were,_,be,[morph(be,were)]).

lex(behaving,v,behave,[morph(behave,ing)]).
lex(behaves,v,behave,[morph(behave,s)]).
lex(behaved,v,behave,[morph(behave,ed(_))]).

lex(believing,v,believe,[morph(believe,ing)]).
lex(believes,v,believe,[morph(believe,s)]).
lex(believed,v,believe,[morph(believe,ed(_))]).

lex(biting,v,bite,[morph(bite,ing)]).
lex(bites,v,bite,[morph(bite,s)]).
lex(bit,v,bite,[morph(bite,ed(1))]).
lex(bitten,v,bite,[morph(bite,ed(1))]).

lex(buying,v,buy,[morph(buy,ing)]).
lex(buys,v,buy,[morph(buy,s)]).
lex(bought,v,buy,[morph(buy,ed(_))]).

lex(calling,v,call,[morph(call,ing)]).
lex(calls,v,call,[morph(call,s)]).
lex(called,v,call,[morph(call,ed(_))]).

lex(caring,v,care,[morph(care,ing)]).
lex(cares,v,care,[morph(care,s)]).
lex(cared,v,care,[morph(care,ed(_))]).

lex(claiming,v,claim,[morph(claim,ing)]).
lex(claims,v,claim,[morph(claim,s)]).
lex(claimed,v,claim,[morph(claim,ed(_))]).

lex(departing,v,depart,[morph(depart,ing)]).
lex(departs,v,depart,[morph(depart,s)]).
lex(departed,v,depart,[morph(depart,ed(_))]).

lex(destroying,v,destroy,[morph(destroy,ing)]).
lex(destroys,v,destroy,[morph(destroy,s)]).
lex(destroyed,v,destroy,[morph(destroy,ed(_))]).

lex(dying,v,die,[morph(die,ing)]).
lex(dies,v,die,[morph(die,s)]).
lex(died,v,die,[morph(die,ed(_))]).

lex(disappearing,v,disappear,[morph(disappear,ing)]).
lex(disappears,v,disappear,[morph(disappear,s)]).
lex(disappeared,v,disappear,[morph(disappear,ed(_))]).

lex(doing,_,do,[morph(do,ing)]).
lex(does,_,do,[morph(do,s)]).
lex(did,_,do,[morph(do,ed(1))]).
lex(done,_,do,[morph(do,ed(2))]).

lex(eating,v,eat,[morph(eat,ing)]).
lex(eats,v,eat,[morph(eat,s)]).
lex(ate,v,eat,[morph(eat,ed(1))]).
lex(eaten,v,eat,[morph(eat,ed(2))]).

lex(falling,v,fall,[morph(fall,ing)]).
lex(falls,v,fall,[morph(fall,s)]).
lex(fell,v,fall,[morph(fall,ed(1))]).
lex(fallen,v,fall,[morph(fall,ed(2))]).

lex(fighting,v,fight,[morph(fight,ing)]).
lex(fights,v,fight,[morph(fight,s)]).
lex(fought,v,fight,[morph(fight,ed(_))]).

lex(filing,v,file,[morph(file,ing)]).
lex(files,v,file,[morph(file,s)]).
lex(filed,v,file,[morph(file,ed(_))]).

lex(firing,v,fire,[morph(fire,ing)]).
lex(fires,v,fire,[morph(fire,s)]).
lex(fired,v,fire,[morph(fire,ed(_))]).

lex(giving,v,give,[morph(give,ing)]).
lex(gives,v,give,[morph(give,s)]).
lex(gave,v,give,[morph(give,ed(1))]).
lex(given,v,give,[morph(give,ed(1))]).

lex(having,_,have,[morph(have,ing)]).
lex(has,_,have,[morph(have,s)]).
lex(had,_,have,[morph(have,ed(_))]).

lex(hearing,v,hear,[morph(hear,ing)]).
lex(hears,v,hear,[morph(hear,s)]).
lex(heard,v,hear,[morph(hear,ed(_))]).

lex(hiring,v,hire,[morph(hire,ing)]).
lex(hires,v,hire,[morph(hire,s)]).
lex(hired,v,hire,[morph(hire,ed(_))]).

lex(hitting,v,hit,[morph(hit,ing)]).
lex(hits,v,hit,[morph(hit,s)]).
lex(hit,v,hit,[morph(hit,ed(_))]).

lex(insulting,v,insult,[morph(insult,ing)]).
lex(insults,v,insult,[morph(insult,s)]).
lex(insulted,v,insult,[morph(insult,ed(_))]).

lex(instructing,v,instruct,[morph(instruct,ing)]).
lex(instructs,v,instruct,[morph(instruct,s)]).
lex(instructed,v,instruct,[morph(instruct,ed(_))]).

lex(keeping,v,keep,[morph(keep,ing)]).
lex(keeps,v,keep,[morph(keep,s)]).
lex(kept,v,keep,[morph(keep,ed(_))]).

lex(killing,v,kill,[morph(kill,ing)]).
lex(kills,v,kill,[morph(kill,s)]).
lex(killed,v,kill,[morph(kill,ed(_))]).

lex(knowing,v,know,[morph(know,ing)]).
lex(knows,v,know,[morph(know,s)]).
lex(knew,v,know,[morph(know,ed(1))]).
lex(known,v,know,[morph(know,ed(2))]).

lex(laughing,v,laugh,[morph(laugh,ing)]).
lex(laughs,v,laugh,[morph(laugh,s)]).
lex(laughed,v,laugh,[morph(laugh,ed(_))]).

lex(leaving,v,leave,[morph(leave,ing)]).
lex(leaves,v,leave,[morph(leave,s)]).
lex(left,v,leave,[morph(leave,ed(_))]).

lex(leading,v,lead,[morph(lead,ing)]).
lex(leads,v,lead,[morph(lead,s)]).
lex(led,v,lead,[morph(lead,ed(_))]).

lex(liking,v,like,[morph(like,ing)]).
lex(likes,v,like,[morph(like,s)]).
lex(liked,v,like,[morph(like,ed(_))]).

lex(loving,v,love,[morph(love,ing)]).
lex(loves,v,love,[morph(love,s)]).
lex(loved,v,love,[morph(love,ed(_))]).

lex(meeting,v,meet,[morph(meet,ing)]).
lex(meets,v,meet,[morph(meet,s)]).
lex(met,v,meet,[morph(meet,ed(_))]).

lex(mentioning,v,mention,[morph(mention,ing)]).
lex(mentions,v,mention,[morph(mention,s)]).
lex(mentioned,v,mention,[morph(mention,ed(_))]).

lex(parking,v,park,[morph(park,ing)]).
lex(parks,v,park,[morph(park,s)]).
lex(parked,v,park,[morph(park,ed(_))]).

lex(persuading,v,persuade,[morph(persuade,ing)]).
lex(persuades,v,persuade,[morph(persuade,s)]).
lex(persuaded,v,persuade,[morph(persuade,ed(_))]).

lex(pleasing,v,please,[morph(please,ing)]).
lex(pleases,v,please,[morph(please,s)]).
lex(pleased,v,please,[morph(please,ed(_))]).

lex(promising,v,promise,[morph(promise,ing)]).
lex(promises,v,promise,[morph(promise,s)]).
lex(promised,v,promise,[morph(promise,ed(_))]).

lex(putting,v,put,[morph(put,ing)]).
lex(puts,v,put,[morph(put,s)]).
lex(put,v,put,[morph(put,ed(_))]).

lex(raining,v,rain,[morph(rain,ing)]).
lex(rains,v,rain,[morph(rain,s)]).
lex(rained,v,rain,[morph(rain,ed(_))]).

lex(reading,v,read,[morph(read,ing)]).
lex(reads,v,read,[morph(read,s)]).
lex(read,v,read,[morph(read,ed(_))]).

lex(removing,v,remove,[morph(remove,ing)]).
lex(removes,v,remove,[morph(remove,s)]).
lex(removed,v,remove,[morph(remove,ed(_))]).

lex(resigning,v,resign,[morph(resign,ing)]).
lex(resigns,v,resign,[morph(resign,s)]).
lex(resigned,v,resign,[morph(resign,ed(_))]).

lex(saying,v,say,[morph(say,ing)]).
lex(says,v,say,[morph(say,s)]).
lex(said,v,say,[morph(say,ed(_))]).

lex(seeing,v,see,[morph(see,ing)]).
lex(sees,v,see,[morph(see,s)]).
lex(saw,v,see,[morph(see,ed(1))]).
lex(seen,v,see,[morph(see,ed(2))]).

lex(seeming,v,seem,[morph(seem,ing)]).
lex(seems,v,seem,[morph(seem,s)]).
lex(seemed,v,seem,[morph(seem,ed(_))]).

lex(sending,v,send,[morph(send,ing)]).
lex(sends,v,send,[morph(send,s)]).
lex(sent,v,send,[morph(send,ed(_))]).

lex(showing,v,show,[morph(show,ing)]).
lex(shows,v,show,[morph(show,s)]).
lex(showed,v,show,[morph(show,ed(1))]).
lex(shown,v,show,[morph(show,ed(2))]).

lex(sleeping,v,sleep,[morph(sleep,ing)]).
lex(sleeps,v,sleep,[morph(sleep,s)]).
lex(slept,v,sleep,[morph(sleep,ed(_))]).

lex(snowing,v,snow,[morph(snow,ing)]).
lex(snows,v,snow,[morph(snow,s)]).
lex(snowed,v,snow,[morph(snow,ed(_))]).

lex(solving,v,solve,[morph(solve,ing)]).
lex(solves,v,solve,[morph(solve,s)]).
lex(solved,v,solve,[morph(solve,ed(_))]).

lex(speaking,v,speak,[morph(speak,ing)]).
lex(speaks,v,speak,[morph(speak,s)]).
lex(spoke,v,speak,[morph(speak,ed(1))]).
lex(spoken,v,speak,[morph(speak,ed(2))]).

lex(standing,v,stand,[morph(stand,ing)]).
lex(stands,v,stand,[morph(stand,s)]).
lex(stood,v,stand,[morph(stand,ed(_))]).

lex(starting,v,start,[morph(start,ing)]).
lex(starts,v,start,[morph(start,s)]).
lex(started,v,start,[morph(start,ed(_))]).

lex(surprising,v,surprise,[morph(surprise,ing)]).
lex(surprises,v,surprise,[morph(surprise,s)]).
lex(surprised,v,surprise,[morph(surprise,ed(_))]).

lex(talking,v,talk,[morph(talk,ing)]).
lex(talks,v,talk,[morph(talk,s)]).
lex(talked,v,talk,[morph(talk,ed(_))]).

lex(telling,v,tell,[morph(tell,ing)]).
lex(tells,v,tell,[morph(tell,s)]).
lex(told,v,tell,[morph(tell,ed(_))]).

lex(trying,v,try,[morph(try,ing)]).
lex(tries,v,try,[morph(try,s)]).
lex(tried,v,try,[morph(try,ed(_))]).

lex(thinking,v,think,[morph(think,ing)]).
lex(thinks,v,think,[morph(think,s)]).
lex(thought,v,think,[morph(think,ed(_))]).

lex(understanding,v,understand,[morph(understand,ing)]).
lex(understands,v,understand,[morph(understand,s)]).
lex(understood,v,understand,[morph(understand,ed(_))]).

lex(visiting,v,visit,[morph(visit,ing)]).
lex(visits,v,visit,[morph(visit,s)]).
lex(visited,v,visit,[morph(visit,ed(_))]).

lex(walking,v,walk,[morph(walk,ing)]).
lex(walks,v,walk,[morph(walk,s)]).
lex(walked,v,walk,[morph(walk,ed(_))]).

lex(wanting,v,want,[morph(want,ing)]).
lex(wants,v,want,[morph(want,s)]).
lex(wanted,v,want,[morph(want,ed(_))]).

lex(winning,v,win,[morph(win,ing)]).
lex(wins,v,win,[morph(win,s)]).
lex(won,v,win,[morph(win,ed(_))]).

lex(wondering,v,wonder,[morph(wonder,ing)]).
lex(wonders,v,wonder,[morph(wonder,s)]).
lex(wondered,v,wonder,[morph(wonder,ed(_))]).

lex(working,v,work,[morph(work,ing)]).
lex(works,v,work,[morph(work,s)]).
lex(worked,v,work,[morph(work,ed(_))]).

lex(writing,v,write,[morph(write,ing)]).
lex(writes,v,write,[morph(write,s)]).
lex(wrote,v,write,[morph(write,ed(1))]).
lex(written,v,write,[morph(write,ed(2))]).

% CONTRACTIONS

contraction('''d',[would]).
contraction('''ve',[have]).
contraction(cannot,[can,not]).
contraction(can,'''t',[can,neg]).
contraction(X+n,'''t',[X=morph(_,_),neg]).

% MISCELLANEOUS

nounFeatures(Base,F) :-	lex(Base,n,F1),	pick1(agr(_),F1,F).

verbFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).

verbToNounFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).
%	pickNF1(ecm(_),F2,F).	e.g. belief still needs to properly govern ...

%% MAPS MORPHOLOGY INTO SYNTACTIC FEATURES
%%	Verb morphology and Agreement
%% 
%%	Form	Tense		AGR
%%	base 	infinitival
%%	base 	past(-) 	agr(not 3,sg)
%%	-s	past(-) 	agr(3,sg)
%%	-ed(1) 	past(+)		agr(_)
%% 	-ing	prog, infinitival
%%	-ed(2)	perf, passive
%%	am	past(-) 	agr(1,sg)
%%	are	past(-) 	agr(not [1,3],sg)
%%	was	past(+)		agr([1,3],sg)
%%	were	past(+) 	agr(not 1,3,sg)

% non-deterministic
vMorphToFs(_,def(F),[F,agr(_)]) :-		% defective: modal auxiliaries
	!.
vMorphToFs(_,[],[]).				% infinitival form
vMorphToFs(Base,Form,Features) :-
	\+ irregular(Base,Form),
	tns(Form,F1),
	agr(Form,F2),
	append(F1,F2,Features).

% base form is irregular: (regular [] -> past(-) not 3sg) 
irregular(Base,[]) :- irregular(Base).

% verb form <-> AGR feature
agr([],[agr(not([3,[sg,m],[m,f,n]]))]).		
agr(s,[agr([3,[sg,m],[m,f,n]])]).
agr(ed(1),[agr(_)]).				
agr(ing,[]).
agr(ed(2),[]).					
agr(am,[agr([1,sg,[m,f,n]])]).
agr(are,[agr(not([[1,3],[sg,m],[m,f,n]]))]). 	
agr(was,[agr([[1,3],[sg,m],[m,f,n]])]).
agr(were,[agr(not([[1,3],[sg,m],[m,f,n]]))]).

% verb form <-> Tense feature
tns([],[past(-)]).		
tns(s,[past(-)]).		
tns(ed(1),[past(+)]).	
tns(ing,[]).			
tns(ed(2),[]).			
tns(am,[past(-)]).
tns(are,[past(-)]).		
tns(was,[past(+)]).		
tns(were,[past(+)]).

%%% PREDICATES FOR INDIVIDUAL LEXICAL ITEMS

agrConstraint(X) :- intersectAGR([[],[],[n]],X) if cat(X,np).

inf(V,ing) :- V has_feature morph(_,ing).
	
% Head movement and negation

negFromV(Neg,V) :-
	V has_feature neg,
	lexicon(not,neg,L),
	mkFs([index(_)|L],Fs),
	cat(Neg,neg),
	Neg has_features Fs,
	containsWord(Neg,'n''t').


noMax(X) :- cat(X,C), \+ max(C).

overt(X) :- \+ ec(X).
