%%%  -*- Mode: PROLOG; Package: PROLOG-USER -*-

%%% SAMPLE SPANISH LEXICON
%%%
%%% (c) 1991, 1992, 1993  Sandiway Fong, NEC Research Institute, Inc.
%%%     1994  Martine Smets

%%% EXPORT
%%%	term(C)				terminals
%%%	lexicon(Word,C,Fs)		Word has category label C
%%%					and feature list Fs
%%%	vMorphToFs(Base,Form,Features)	TNS/AGR features
%%%	inf(Verb,Type)
%%%	relevant(C)			constraints imposed by markers
%%%					apply to C
%%% REFERENCES
%%%	(list processing)		utilities
%%%	ensureWhComp(Ci)		lex
%%%	optWhComp(Ci)			

term(n). term(a). term(v). term(p). term(c). 
term(adv).  term(det). %term(conj).
term(neg). term(mrkr). term('$'). term(cl).

%%% 	Most lexical entrees are stored directly as
%%%
%%%		 lex(Word,Category,Features)
%%%
%%%	Non-base forms require inference: 
%%%	1. plural nouns		all features except agr(_) inherited
%%%				from the sg. form
%%%	2. nominalized verbs	inherits verb features except morph(_,_)
%%%	3. non-base verb forms	all features except morph(_,_) inherited
%%%				from the base form

lexicon(Word,C,Fs) :-				% directly available
	lex(Word,C,Fs1),
	verbInf(C,Fs1,Fs).
lexicon(Form,v,Fs) :-				% non-base verb forms
	lex(Form,v,Base,F1),
	verbFeatures(Base,F2),
	append1(F1,F2,Fs).

probeLexicon(Word) :- lex(Word,_,_) ; lex(Word,_,_,_).

%%% NEGATION

lex(no,neg,[polarity(negative)]).  


%%% PREPOSITIONS
lex(acerca_de,p,[grid([],[theme]),predicate(theme),   % about
	     adjR([goal(roleNotPresent(X,theme),X)])]).
lex(de,p,[grid([],[theme]),predicate(theme),         % about, of
	     adjR([goal(roleNotPresent(X,theme),X)])]).
lex(de,p,[grid([],[source]),predicate(source),      % from
	  adjR([goal(vpAllowExt(source,X),X)])]).
lex(despus_de,p,[grid([],[reducedProp]),predicate(manner)]). % after
lex(porque,p,[grid([],[reducedProp]),predicate(reason)]). % because
lex(antes_de,p,[grid([],[reducedProp]),predicate(manner)]). % before
lex(por,p,[grid([],[agent]),adjR([goal(unsaturatedExtRole(X,agent),X)]), % by
	  predicate(agent)]).
lex(en,p,[grid([],[location]),predicate(location)]).        % in
lex(debajo_de,p,[grid([],[location]),predicate(location)]). % under
lex(sin,p,[grid([],[reducedProp]),predicate(manner)]).      % without
lex(con,p,[grid([],[patient]),predicate(patient)]).
lex(sobre,p,[grid([],[location]),predicate(location)]).       % on
lex(a,p,[grid([],[destination]),predicate(destination),      % to
	  adjR([goal(vpAllowExt(destination,X),X)])]).
lex(a,p,[grid([],[patient]),predicate(patient),      % .. at something/body
	  adjR([goal(vpAllowExt(patient,X),X)])]).

%% Adverbs

lex(como,adv,[adjoin(right),wh,predicate(manner)]).   % how
lex(porque,adv,[adjoin(right),wh,predicate(reason)]). % why
lex(completamente,adv,[adjoin(right),predicate(manner)]). % completely
lex(muchas_veces,adv,[adjoin(right),predicate(manner)]).      % often
lex(sinceramente,adv,[adjoin(right),predicate(manner)]). % sincerely
lex(aqu,adv,[adjoin(right),predicate(location)]).  % here
lex(tarde,adv,[adjoin(right),predicate(time)]).    % late
lex(ayer,adv,[adjoin(right),predicate(time)]).    % yesterday
lex(bien,adv,[adjoin(right),predicate(manner)]).  % well
lex(muy,adv,[degree]).                            % very
lex(mucho,adv,[degree,adjR([feature(allow(adv,degree))])]). % ... special case
lex(mal,adv,[adjoin(right),predicate(manner)]).   % badly

%% Degree Adverbs (modify adjectives)

% permit the resulting AP to take an optional clausal adjunct
lex(demasiado,adv,[degree,adjR([addFeatures([allowExt(reason),adjR([goal(nonfiniteClause(X),X)])])])]).

%% Determiners
%%	Combine only with nouns that have the count(+/-/_) feature (common nouns) 
%%	Six classes of determiners:
%%	Class Examples                     Features
%%	1     le,mon,quel                  count(_) agr([3,[sg,m],m])
%%	2     un_peu de, assez de          count(_) agr([3,m,[]])
%%	3     ce                           count(_) agr([3,[sg,m],m])
%%	4     peu_de,ces                   count(+) agr([3,[pl,m],[]])
%%	5     chaque                       count(+) agr(([3,sg,[]])
%%	6     beaucoup_de                  count(-) agr(([3,[pl,m],[]])
%%
%%	op(+) elements that are moved by QR and 
%%	form operator-variable structures

lex(un,det,[count(+),agr([3,sg,m])]).   % undefinite determiner
lex(una,det,[count(+),agr([3,sg,f])]).
lex(unos,det,[count(+),agr([3,pl,m])]).  % mean "some"
lex(unas,det,[count(+),agr([3,pl,f])]).
lex(cada,det,[count(+),agr([3,sg,[m,f]]),op(+)]). % each
lex(bastante,det,[count(_),agr([3,[pl,m],[]]),op(+)]). % enough
lex(poco,det,[count(-),agr([3,m,m]),op(+)]).  % little
lex(poca,det,[count(-),agr([3,m,f]),op(+)]).
lex(pocos,det,[count(+),agr([3,pl,m]),op(+)]). % few
lex(pocas,det,[count(+),agr([3,pl,f]),op(+)]).
lex(mucho,det,[count(-),agr([3,m,m]),op(+)]).  % much
lex(mucha,det,[count(-),agr([3,m,f]),op(+)]).
lex(muchos,det,[count(+),agr([3,pl,m]),op(+)]). % many
lex(muchas,det,[count(+),agr([3,pl,f]),op(+)]).
lex(este,det,[count(_),agr([3,[sg,m],m])]).  % this
lex(esta,det,[count(_),agr([3,[sg,m],f])]).
lex(el,det,[count(_),agr([3,[sg,m],m])]).    % the 
lex(la,det,[count(_),agr([3,[sg,m],f])]).
lex(los,det,[count(+),agr([3,pl,[]])]).
lex(las,det,[count(+),agr([3,pl,[]])]).
lex(estos,det,[count(+),agr([3,pl,m])]).
lex(estas,det,[count(+),agr([3,pl,f])]).
lex(ese,det,[count(_),agr([3,[sg,m],m])]).  % that
lex(esa,det,[count(_),agr([3,[sg,m],f])]).
lex(esos,det,[count(+),agr([3,pl,m])]).
lex(esas,det,[count(+),agr([3,pl,f])]).
% %lex(whose,det,[count(_),agr([3,[],[]]),wh]).%not in French
lex(ningun,det,[count(+),agr([3,sg,m])]). % no
lex(ninguna,det,[count(+),agr([3,sg,f])]).
%lex(neither,det,[count(+),agr([3,sg,[]])]).
lex(nuestro,det,[count(_),agr([3,[sg,m],m])]). % our
lex(nuestra,det,[count(_),agr([3,[sg,m],f])]).
lex(nuestros,det,[count(_),agr([3,pl,m])]).
lex(nuestras,det,[count(_),agr([3,pl,f])]).

lex(vuestro,det,[count(_),agr([3,[sg,m],m])]). % your
lex(vuestra,det,[count(_),agr([3,[sg,m],f])]).
lex(vuestros,det,[count(_),agr([3,pl,m])]).
lex(vuestras,det,[count(_),agr([3,pl,f])]).

lex(mi,det,[count(_),agr([3,[sg,m],[m,f]])]). % my
lex(mis,det,[count(_),agr([3,pl,[]])]).
lex(tu,det,[count(_),agr([3,[sg,m],[m,f]])]). % your
lex(tus,det,[count(_),agr([3,pl,[]])]).
lex(cuyo,det,[count(_),agr([3,[sg,m],m])]).  % whose
lex(cuya,det,[count(_),agr([3,[sg,m],f])]).
lex(cuyos,det,[count(_),agr([3,pl,m])]).
lex(cuyas,det,[count(_),agr([3,pl,f])]).

%% Nouns
%% 	Obligatory features
%% 	1. Anaphoric: a(+/-), Pronominal: p(+/-) for Binding Theory
%% 	2. Agreement: agr([Per,Num,Gen]) 
%%	3. O-grid: grid(Ext,Int) 
%% 	Optional features
%% 	1. Morphologically realized Case: morphC(C)
%% 	2. Common nouns: count(+/-/_) (not present for proper nouns)
%%	3. Quantifiers: op(+/-) 
%%	4. Wh-nouns: wh

% Pleonastic elements

%% No pleonastic elements: pro drop language.

% not sure?
%?lex(now,n,[grid([],[]),agr([3,[],[]])]).

% Proper nouns
lex(pedro,n,[grid([],[]),a(-),p(-),agr([3,sg,m]),animate]).
lex(juan,n,[grid([],[]),a(-),p(-),agr([3,sg,m]),animate]).
lex(mara,n,[grid([],[]),a(-),p(-),agr([3,sg,f]),animate]).
lex(roma,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).
lex(susana,n,[grid([],[]),a(-),p(-),agr([3,sg,f]),animate]).
lex(estructuras_sintcticas,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).

% Anaphors and pronominals

lex(lo,pf(cl),[a(-),p(+),agr([3,sg,m])|Fs]) :- objCliticFeatures(Fs).
lex(le,pf(cl),[a(-),p(+),agr([[2,3],sg,m])|Fs]) :- objCliticFeatures(Fs).
lex(le,pf(cl),[a(-),p(+),agr([[2,3],sg,[f,m]])|Fs]) :-
	iobjCliticFeatures(Fs).
lex(se,pf(cl),[a(+),p(-),agr([[2,3],[sg,pl],[f,m]])|Fs]) :- 
	objCliticFeatures(Fs).
lex(se,pf(cl),[a(+),p(-),agr([[2,3],[sg,pl],[f,m]])|Fs]) :- 
	iobjCliticFeatures(Fs).
lex(l,n,[grid([],[]),a(-),p(+),agr([3,sg,m]),animate]).
lex(ello,n,[grid([],[]),a(-),p(+),agr([3,sg,n]),animate]).
lex(s,n,[grid([],[]),morphC(obl),a(+),p(-),agr([[2,3],[pl,sg],[m,f]])]).
lex(les,pf(cl),[a(-),p(+),agr([[2,3],pl,m])|Fs]) :-
	objCliticFeatures(Fs).
lex(las,pf(cl),[a(-),p(+),agr([[2,3],pl,f])|Fs]) :-
	objCliticFeatures(Fs).
lex(les,pf(cl),[a(-),p(+),agr([[2,3],pl,[m,f]])|Fs]) :-
	iobjCliticFeatures(Fs).
lex(ellos,n,[grid([],[]),a(-),p(+),agr([3,pl,m]),animate]).

lex(ella,n,[grid([],[]),a(-),p(+),agr([3,sg,f]),animate]).
lex(la,pf(cl),[a(-),p(+),agr([[2,3],sg,f])|Fs]) :-
	objCliticFeatures(Fs).
lex(ellas,n,[grid([],[]),a(-),p(+),agr([3,pl,f]),animate]).

lex(yo,n,[grid([],[]),a(-),p(+),agr([1,sg,[m,f]]),morphC(nom),animate]).
lex(me,pf(cl),[a(_),p(_),agr([1,sg,[m,f]])|Fs]) :-
	objCliticFeatures(Fs).
lex(me,pf(cl),[a(_),p(_),agr([1,sg,[m,f]])|Fs]) :-
	iobjCliticFeatures(Fs).
lex(mi,n,[grid([],[]),a(-),p(+),agr([1,sg,[m,f]]),morphC(nonNom),animate]).

lex(tu,n,[grid([],[]),a(-),p(+),agr([2,[],[m,f]]),animate]).
lex(te,pf(cl),[a(_),p(_),agr([2,[],[m,f]])|Fs]) :-
	objCliticFeatures(Fs).
lex(te,pf(cl),[a(_),p(_),agr([2,[],[m,f]])|Fs]) :-
	iobjCliticFeatures(Fs).

lex(nosotros,n,[grid([],[]),a(-),p(+),agr([1,pl,m]),animate]).
lex(nosotras,n,[grid([],[]),a(-),p(+),agr([1,pl,f]),animate]).

lex(nos,pf(cl),[a(_),p(_),agr([1,pl,[m,f]])|Fs]) :-
	objCliticFeatures(Fs).
lex(nos,pf(cl),[a(_),p(_),agr([1,pl,[m,f]])|Fs]) :-
	iobjCliticFeatures(Fs).

lex(vosotros,n,[grid([],[]),a(-),p(+),agr([2,pl,m]),animate]).
lex(vosotras,n,[grid([],[]),a(-),p(+),agr([2,pl,f]),animate]).
lex(os,pf(cl),[a(_),p(_),agr([2,pl,[m,f]])|Fs]) :-
	objCliticFeatures(Fs).
lex(os,pf(cl),[a(_),p(_),agr([2,pl,[m,f]])|Fs]) :-
	iobjCliticFeatures(Fs).

lex(su,n,[morphC(gen),grid([],[]),a(-),p(+),agr([3,[sg,m],[m,f]]),
	agr1([3,[],[]])]).
lex(su,n,[morphC(gen),grid([],[]),a(+),p(-),agr([3,[sg,m],[m,f]]),
	agr1([3,[],[]])]).
lex(sus,n,[morphC(gen),grid([],[]),a(-),p(+),agr([3,pl,[m,f]]),
	agr1([3,[],[]])]).
lex(sus,n,[morphC(gen),grid([],[]),a(+),p(-),agr([3,pl,[m,f]]),
	agr1([3,[],[]])]).


% Quantifier nouns
lex(cada_uno,n,[grid([],[]),a(-),p(-),agr([3,sg,m]),op(+)]).
lex(cada_una,n,[grid([],[]),a(-),p(-),agr([3,sg,f]),op(+)]).
lex(alguien,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),op(+),animate]).

% Wh-nouns
lex(quin,n,[grid([],[]),a(-),p(-),agr([3,[sg,m],[m,f]]),wh,animate]).
lex(quines,n,[grid([],[]),a(-),p(-),agr([3,pl,[m,f]]),wh,animate]).
lex(a_quin,n,[grid([],[]),morphC(dat),a(-),p(-),agr([3,[sg,pl,m],[m,f]]),wh,
	animate]).
lex(de_quin,n,[grid([],[]),morphC(gen),a(-),p(-),agr([3,[sg,pl,m],[m,f]]),wh,
	animate]).
lex(qu,n,[grid([],[]),morphC(acc),a(-),p(-),agr([3,[sg,pl,m],[m,f]]),wh]).
lex(cul,n,[grid([],[]),a(-),p(-),agr([3,[sg,m],[m,f]])]).
lex(cules,n,[grid([],[]),a(-),p(-),agr([3,pl,[m,f]])]).

% Common nouns
% actress, actor
lex(actriz,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
lex(actor,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% article
lex(artculo,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% bastard
lex(cabrn,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),epithet,animate]).
% boy
lex(muchacho,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% building
lex(edificio,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% bus
lex(autobs,n,[grid([],[]),a(-),p(-),count(+),agr([3,[sg,pl],m])]).
% cake
lex(tarta,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
% cat (masculine, feminine)
lex(gato,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(gata,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% cheese
lex(queso,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
%lex(cleaner,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
% city
lex(ciudad,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% demonstration
lex(manifestacin,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% explosion
lex(explosin,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% fire
lex(fuego,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
% floor
lex(piso,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% ice-cream
lex(helado,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,m])]).
% lawyer(masc;fem)
lex(abogado,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
lex(abogada,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
% lyer (masc;fem)
lex(mentiroso,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
lex(mentirosa,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
% man
lex(hombre,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% room
lex(pieza,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% playground
lex(patio_de_recreo,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% police
lex(polica,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,f])]).
% policeman
lex(polica,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% problem
lex(problema,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% race
lex(carrera,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% rat
lex(rata,n,[grid([],[]),a(-),p(-),count(+),agr([3,[],f]),animate]).
% report
lex(informe,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% stain
lex(mancha,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% student
lex(estudiante,n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),count(+),animate]).
% symphony
lex(sinfona,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
% time
lex(tiempo,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,m]) ]).
% this
lex(esto,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,m]) ]).
% woman
lex(mujer,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
% worker
lex(trabajador,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% worker(factory; masc, fem)
lex(obrero,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
lex(obrera,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
% face
lex(cara,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).

% Nouns with O-grids
% book
lex(libro,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% brother
lex(hermano,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% father
lex(padre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m]),animate]).
% furniture
lex(mueble,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
% mother
lex(madre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f]),animate]).
% picture
lex(foto,n,[grid([possessor],[theme]),a(-),p(-),count(+),agr([3,sg,f])]).
% teacher(masc;fem)
lex(profesor,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m]),
	animate]).
lex(profesora,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f]),
	animate]).
	                                            

% Non-base noun forms

lex(actrizes,n,[agr([3,pl,f]) |F]) :- nounFeatures(actriz,F).
lex(actores,n,[agr([3,pl,m])|F]) :- nounFeatures(actor,F).
lex(artculos,n,[agr([3,pl,m])|F]) :- nounFeatures(artculo,F).
lex(cabrnes,n,[agr([3,pl,m])|F]) :- nounFeatures(cabrn,F).
lex(autobses,n,[agr([3,pl,m])|F]) :- nounFeatures(autobs,F).
lex(manifestacines,n,[agr([3,pl,f])|F]) :- nounFeatures(manifestacin,F).
lex(creencias,n,[agr([3,pl,f])|F]) :- nounFeatures(creencia,F).
lex(libros,n,[agr([3,pl,m])|F]) :- nounFeatures(libro,F).
lex(edificios,n,[agr([3,pl,m])|F]) :- nounFeatures(edificio,F).
lex(ciudades,n,[agr([3,pl,f])|F]) :- nounFeatures(ciudad,F).
lex(explosines,n,[agr([3,pl,f])|F]) :- nounFeatures(explosin,F).
lex(fuegos,n,[agr([3,pl,m])|F]) :- nounFeatures(fuego,F).
lex(salidas,n,[agr([3,pl,f])|F]) :- nounFeatures(salida,F).
lex(pisos,n,[agr([3,pl,m])|F]) :- nounFeatures(piso,F).
lex(jardnes,n,[agr([3,pl,m])|F]) :- nounFeatures(jardn,F).
lex(abogados,n,[agr([3,pl,m])|F]) :- nounFeatures(abogado,F).
lex(mentirosos,n,[agr([3,pl,m])|F]) :- nounFeatures(mentiroso,F).
lex(hombres,n,[agr([3,pl,m])|F]) :- nounFeatures(hombre,F).
lex(piezas,n,[agr([3,pl,f])|F]) :- nounFeatures(pieza,F).
lex(patios_de_recreo,n,[agr([3,pl,m])|F]) :- nounFeatures(patio_de_recreo,F).
lex(policas,n,[agr([3,pl,m])|F]) :- nounFeatures(polica,F).
lex(madres,n,[agr([3,pl,f])|F]) :- nounFeatures(madre,F).
lex(fotos,n,[agr([3,pl,f])|F]) :- nounFeatures(foto,F).
lex(problemas,n,[agr([3,pl,m])|F]) :- nounFeatures(problema,F).
lex(carreras,n,[agr([3,pl,f])|F]) :- nounFeatures(carrera,F).
lex(ratas,n,[agr([3,pl,f])|F]) :- nounFeatures(rata,F).
lex(profesores,n,[agr([3,pl,[f,m]])|F]) :- nounFeatures(profesor,F).
lex(cuartos,n,[agr([3,pl,m])|F]) :- nounFeatures(cuarto,F).
lex(manchas,n,[agr([3,pl,f])|F]) :- nounFeatures(mancha,F).
lex(estudiantes,n,[agr([3,pl,[f,m]])|F]) :- nounFeatures(estudiante,F).
lex(sinfonas,n,[agr([3,pl,f])|F]) :- nounFeatures(sinfona,F).
lex(mujeres,n,[agr([3,pl,f])|F]) :- nounFeatures(mujer,F).
lex(trabajadores,n,[agr([3,pl,[f,m]])|F]) :- nounFeatures(trabajador,F).
lex(obreros,n,[agr([3,pl,m])|F]) :- nounFeatures(obrero,F).
lex(obreras,n,[agr([3,pl,f])|F]) :- nounFeatures(obrera,F).

% Nominalized verbs
lex(creencia,n,[count(+),agr([3,sg,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(creer,F).
% lex(afirmacion,n,[count(+),agr([3,sg,f]),a(-),p(-)|F]) :-
%	verbToNounFeatures(afirmar,F).
lex(salida,n,[count(+),agr([3,sg,m]),a(-),p(-)|F]) :-
	verbToNounFeatures(salir,F).
lex(destruccin,n,[count(+),agr([3,m,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(destruir,F).
lex(lucha,n,[count(+),agr([3,sg,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(luchar,F).

%%% VERBS - BASE FORMS
%
% Obligatory Elements
% 	1. grid(Ext,Int) denotes the O-grid
% 	2. morph(Base,[]/ing/s/ed(1)/ed(2)) 
% Optional Features
% 	1. ecm(opt/oblig) for ECM verbs
% 	2. noCasemark(+) for technically intransitive verbs
% 	3. subcat(C$Fs,Fs1) subcategorizes for an element of category C 
%    	   with feature restriction list Fs.
%    	   Feature list Fs1 will be appended to the feature list of C
% 	4. selR(Rs), selR(Type,R) restriction(s) Rs or R on direct object.

lex(aparecer,v,[morph(aparecer,er),grid([theme],[])]).
lex(apreciar,v,[morph(apreciar,ar),grid([agent],[theme])]).
lex(arrestar,v,[morph(arrestar,ar),grid([agent],[[patient]])]).
lex(llegar,v,[morph(llegar,ar),grid([theme],[]),allowExt(source)]).
lex(preguntar,v,[morph(preguntar,ar),grid([agent],[proposition,[destination]])]).
	
lex(creer,v,[morph(creer,er),grid([agent],[proposition])]).
lex(comprar,v,[morph(comprar,ar),grid([agent],[theme,[destination]]),
	                                                     idocase(dat)]).
lex(llamar,v,[morph(llamar,ar),grid([agent],[patient])]).
% lex(care,v,[morph(care,[]),grid([agent],[proposition]),noCasemark(+),
	%   selR(head(c2),goal(ensureWhComp(X),X))]).
lex(afirmar,v,[morph(afirmar,ar),grid([agent],[proposition,[destination]]),
	                                                      idoCase(dat)]).
lex(salir,v,[morph(salir,ir),grid([theme]), allowExt(source)]).
lex(destruir,v,[morph(destruir,ir),grid([agent],[patient])]).
lex(morir,v,[morph(morir,ir),grid([theme],[])]).
lex(desaparecer,v,[morph(desaparecer,er),grid([theme],[])]).
lex(comer,v,[morph(comer,er),grid([agent],[[patient]])]).
lex(caer,v,[morph(caer,er),grid([theme],[])]).
lex(luchar,v,[morph(luchar,ar),grid([agent]),allowExt(patient)]).
lex(archivar,v,[morph(archivar,ar),grid([agent],[patient])]).
lex(despedir,v,[morph(despedir,ir),grid([agent],[patient]),allowExt(source)]).
lex(dar,v,[morph(dar,ar),grid([agent],[theme,[destination]]),
	                                                     idoCase(dat)]).
lex(or,v,[morph(o,ir),grid([agent],[theme])]).
lex(contratar,v,[morph(contratar,ar),grid([agent],[patient])]).
lex(golpear,v,[morph(golpear,ar),grid([agent],[patient])]).
lex(ensear,v,[morph(ensear,ar),grid([agent],[theme,destination]),
	idoCase(dat)]).
lex(insultar,v,[morph(insultar,ar),grid([agent],[patient])]).
lex(guardar,v,[morph(guardar,ar),grid([agent],[patient])]).
lex(matar,v,[morph(matar,ar),grid([agent],[patient])]).
lex(saber,v,[morph(saber,er),grid([agent],[proposition]), % with c2 or n2
	    selR(head(c2),goal(optWhComp(X),X))]).
lex(conocer,v,[morph(conocer,er),grid([agent],[theme])]). % with np
lex(rer,v,[morph(rer,ir),grid([theme],[]),allowExt(patient)]).
lex(dejar,v,[morph(dejar,ar),grid([agent],[patient])]).
% let
lex(dejar,v,[morph(dejar,ar),grid([agent],[proposition,patient]),idoCase(dat)]).
lex(acaudillar,v,[morph(acaudillar,ar),grid([agent],[patient]),allowExt(destination)]).
lex(gustar,v,[morph(gustar,ar),grid([theme],[experencer])]).
lex(encontrarse,v,[morph(encontrar,ar),grid([agent],[]),requiresExt(patient),
	                                          allowExt(location)]).
lex(mencionar,v,[morph(mencionar,ar),grid([agent],[proposition])]).
lex(estacionar,v,[morph(estacionar,ar),grid([agent],[[patient]]),allowExt(location)]).
lex(persuadir,v,[morph(persuadir,ir),grid([agent],[patient,proposition])]).
lex(prometer,v,[morph(prometer,er),grid([agent],[proposition,patient]),
	 idoCase(dat), iselR([addFeature(noControl)])]).
lex(poner,v,[morph(poner,er),grid([agent],[patient]),
	                                            requiresExt(location)]).
lex(llover,v,[morph(llover,er),grid([],[])]).
lex(leer,v,[morph(leer,er),grid([agent],[theme,[destination]]),idoCase(dat)]).
lex(quitar,v,[morph(quitar,ar),grid([agent],[patient]),allowExt(source)]).
lex(dimitir,v,[morph(dimitir,ir),grid([theme],[])]).
lex(decir,v,[morph(decir,ir),grid([agent],[proposition, [destination]]),
	                                           idoCase(dat)]).
lex(ver,v,[morph(ver,er),grid([agent],[patient])]).
lex(parecer,v,[morph(parecer,er),grid([],[proposition]),noCasemark(+)]).
lex(dormir,v,[morph(dormir,ir),grid([agent],[])]).
lex(nevar,v,[morph(nevar,ar),grid([],[])]).
lex(resolver,v,[morph(resolver,er),grid([agent],[patient])]).
lex(soportar,v,[morph(soportar,ar),grid([agent],[proposition])]).
lex(empezar,v,[morph(empezar,ar),grid([agent],[patient])]).
lex(sorprender,v,[morph(sorprender,er),grid([agent],[patient,proposition])]).
lex(hablar,v,[morph(hablar,ar),grid([agent],[[theme]]),
                                   doCase(gen),allowExt(patient)]).
lex(contar,v,[morph(contar,ar),grid([agent],[proposition,[destination]]),
	                                                       idoCase(dat)]).
lex(lavar,v,[morph(lavar,ar),grid([agent],[patient,[destination]]),
	                                       idoCase(dat)]).
lex(pensar,v,[morph(pensar,ar),grid([agent],[[proposition]])]).
lex(probar,v,[morph(probar,ar),grid([agent],[proposition])]).
lex(entender,v,[morph(entender,er),grid([agent],[patient])]).
lex(querer,v,[morph(querer,er),grid([agent],[proposition])]).
lex(ganar,v,[morph(ganar,ar),grid([agent],[])]).
lex(ganar,v,[morph(ganar,ar),grid([agent],[patient])]).
lex(trabajar,v,[morph(trabajar,ar),grid([agent],[]),
	                                allow(adv,degree)]).
lex(escribir,v,[morph(escribir,ir),grid([agent],[[theme],[destination]]),
	                         idoCase(dat)]).
lex(visitar,v,[morph(visitar,ar),grid([agent],[patient])]).
lex(venir,v,[morph(venir,ir),grid([agent],[]),allowExt(source)]).
lex(tener,v,[morph(tener,er),grid([agent],[patient])]).
lex(ir,v,[morph(ir,ir),grid([agent],[]),requiresExt(destination)]).

%% Auxiliaries

lex(ser,v,[morph(ser,er),aux,transmitAgr,
	  oneof(_,[[grid([],[eq])],
                   [grid([eq],[eq])],
		   [subcat(ap$[grid(_,_)],[])],
		   [subcat(vp$[morph(_,ed(2))],[noCasemark(+)]),blockTheta]])]).

lex(estar,v,[morph(estar,ar),aux,transmitAgr,
	oneof(_,[[grid([pHolder(P)],subcat),subcat(adv$[predicate(P)],[])],
                 [subcat(ap$[grid(_,_)],[])],	         
                 [subcat(vp$[morph(_,ando)],[prog])]])]).
                  
% Stage 2
% lex(be,v,[morph(be,[]),aux,grid([eq],[eq])]).	  % john is the man
% lex(be,v,[morph(be,[]),aux,grid([pHolder(P)],subcat),	% john is here
%	  subcat(adv$[predicate(P)],[])]).
% Stage 1
% lex(be,v,[morph(be,[]),aux,subcat(ap$[grid(_,_)],[])]).	% john is dead
% lex(be,v,[morph(be,[]),aux,blockTheta,	  % john was arrested
%	  subcat(vp$[morph(_,ed(2))],[noCasemark(+)])]).
% lex(be,v,[morph(be,[]),aux,			  % john is 
%	  subcat(vp$[morph(_,ing)],[prog])]).

lex(haber,v,[morph(haber,er),perf,subcat(vp$[morph(_,ed(2))],[perf])]).
lex(ir,v,[morph(ir,ir),modal,subcat(vp$[morph(_,[ar,ir,er])],[])]).
lex(poder,v,[morph(poder,er),modal,subcat(vp$[morph(_,[ar,ir,er])],[])]).


%% Adjectives
%%
%% Optional features:
%%	ecm(oblig) really means S-bar Deleter in this context.

lex(aaaaa,a,[grid([],[patient])]).		% too complicated to explain

lex(seguro,a,[grid([],[proposition]),agr([[],[sg,m],m])]).
lex(segura,a,[grid([],[proposition]),agr([[],[sg,m],f])]).
lex(seguros,a,[grid([],[proposition]),agr([[],pl,m])]).
lex(seguras,a,[grid([],[proposition]),agr([[],pl,f])]).
lex(inteligente,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(inteligentes,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(loco,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(loca,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(locos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(locas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(crucial,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(cruciales,a,[grid([],[proposition]),agr([[],pl,[f,m]])]).
lex(estpido,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(estpida,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(estpidos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(estpidas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(ansioso,a,[grid([theme],[proposition]),agr([[],[sg,m],m])]).
lex(ansiosa,a,[grid([theme],[proposition]),agr([[],[sg,m],f])]).
lex(ansiosos,a,[grid([theme],[proposition]),agr([[],pl,m])]).
lex(ansiosas,a,[grid([theme],[proposition]),agr([[],pl,f])]).
lex(fcil,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(fciles,a,[grid([],[proposition]),agr([[],pl,[m,f]])]).
lex(difcil,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(difciles,a,[grid([],[proposition]),agr([[],pl,[m,f]])]).
lex(feliz,a,[grid([theme],[]),agr([[],[sg,m],[f,m]])]).
lex(felices,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(ilegal,a,[grid([],[proposition]),agr([[],[sg,m],[f,m]])]).
lex(ilegales,a,[grid([],[proposition]),agr([[],pl,[m,f]])]).
%lex(ill_founded,a,[grid([theme],[])]).
lex(ilgico,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(ilgica,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(ilgicos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(ilgicas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(importante,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(importantes,a,[grid([],[proposition]),agr([[],pl,[f,m]])]).
lex(probable,a,[grid([],[proposition]),agr([3,[sg,m],[m,f]])]).
lex(probables,a,[grid([],[proposition]),agr([3,pl,[m,f]])]).
lex(simptico,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(simptica,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(simpticos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(simpticas,a,[grid([theme],[]),agr([[],pl,f])]).
%lex(en_solde,a,[grid([theme],[]),agr([[],[],[]])]).
lex(orgulloso,a,[grid([theme],[proposition]),agr([[],[sg,m],m])]).
lex(orgullosa,a,[grid([theme],[proposition]),agr([[],[sg,m],f])]).
lex(orgullosos,a,[grid([theme],[proposition]),agr([[],pl,m])]).
lex(orgullosas,a,[grid([theme],[proposition]),agr([[],pl,f])]).
lex(testarudo,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(testaruda,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(testarudos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(testarudas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(patolgico,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(patolgica,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(patolgicos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(patolgicas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(podrido,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(podrida,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(podridos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(podridas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(contento,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(contenta,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(contentos,a,[grid([theme],[]),agr([[],pl,m])]).
lex(contentas,a,[grid([theme],[]),agr([[],pl,f])]).
lex(triste,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(tristes,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
%lex(tough,a,[grid([weakTheme$optional],[]),allowExt(reason)]).

% Complementizers

lex(que,c,[selR([not(feature(inf(_)))])]).
lex(si,c,[wh]).
lex(para,c,[selR([feature(inf(_))])]).

% Markers: morphological realizations of particular features

lex(de,mrkr,[right(np,case(gen),[])]).		  % object genitive Case
lex(a,mrkr,[right(np,case(dat),[])]).		  % dative case
lex(a,mrkr,[right(np,animate,[case(acc)])]).       % animate direct object
					
                                        
:- initialization((no lexFeature(_,_))).

% Coordinating conjunctions
lex(y,conj,[overt(+)]).

% relevant for marker constraints
relevant(n).


irregular(ser).
irregular(haber).


% Non-base verb forms:
%
% Form x Category x Base x Features

lex(apareciendo,v,aparecer,[morph(aparecer,ando)]).
lex(aparezco,v,aparecer,[morph(aparecer,o)]).
lex(apareces,v,aparecer,[morph(aparecer,s)]).
lex(aparece,v,aparecer,[morph(aparecer,e)]).
lex(aparecemos,v,aparecer,[morph(aparecer,emos)]).
lex(aparecis,v,aparecer,[morph(aparecer,is)]).
lex(aparecen,v,aparecer,[morph(aparecer,en)]).
lex(aparecida,v,aparecer,[morph(aparecer,ed(2)),agr([[],[sg,m],f])]).
lex(aparecido,v,aparecer,[morph(aparecer,ed(2)),agr([[],[sg,m],m])]).
lex(aparecidos,v,aparecer,[morph(aparecer,ed(2)),agr([[],pl,m])]).
lex(aparecidas,v,aparecer,[morph(aparecer,ed(2)),agr([[],pl,f])]).

lex(apreciando,v,apreciar,[morph(apreciar,ando)]).
lex(aprecio,v,apreciar,[morph(apreciar,o)]).
lex(aprecias,v,apreciar,[morph(apreciar,s)]).
lex(aprecia,v,apreciar,[morph(apreciar,a)]).
lex(apreciamos,v,apreciar,[morph(apreciar,amos)]).
lex(apreciis,v,apreciar,[morph(apreciar,is)]).
lex(aprecian,v,apreciar,[morph(apreciar,an)]).
lex(apreciado,v,apreciar,[morph(apreciar,ed(2)),agr([[],[sg,m],m])]).
lex(apreciada,v,apreciar,[morph(apreciar,ed(2)),agr([[],[sg,m],f])]).
lex(apreciadas,v,apreciar,[morph(apreciar,ed(2)),agr([[],pl,f])]).
lex(apreciados,v,apreciar,[morph(apreciar,ed(2)),agr([[],pl,m])]).

lex(arrestando,v,arrestar,[morph(arrestar,ando)]).
lex(arresto,v,arrestar,[morph(arrestar,o)]).
lex(arrestas,v,arrestar,[morph(arrestar,s)]).
lex(arresta,v,arrestar,[morph(arrestar,a)]).
lex(arrestamos,v,arrestar,[morph(arrestar,amos)]).
lex(arrestis,v,arrestar,[morph(arrestar,is)]).
lex(arrestan,v,arrestar,[morph(arrestar,an)]).
lex(arrestado,v,arrestar,[morph(arrestar,ed(2)),agr([[],[sg,m],m])]).
lex(arrestada,v,arrestar,[morph(arrestar,ed(2)),agr([[],[sg,m],f])]).
lex(arrestadas,v,arrestar,[morph(arrestar,ed(2)),agr([[],pl,f])]).
lex(arrestados,v,arrestar,[morph(arrestar,ed(2)),agr([[],pl,m])]).

lex(llegando,v,llegar,[morph(llegar,ando)]).
lex(llego,v,llegar,[morph(llegar,o)]).
lex(llegas,v,llegar,[morph(llegar,s)]).
lex(llega,v,llegar,[morph(llegar,a)]).
lex(llegamos,v,llegar,[morph(llegar,amos)]).
lex(llegis,v,llegar,[morph(llegar,is)]).
lex(llegan,v,llegar,[morph(llegar,an)]).
lex(llegado,v,llegar,[morph(llegar,ed(2)),agr([[],[sg,m],m])]).
lex(llegada,v,llegar,[morph(llegar,ed(2)),agr([[],[sg,m],f])]).
lex(llegadas,v,llegar,[morph(llegar,ed(2)),agr([[],pl,f])]).
lex(llegados,v,llegar,[morph(llegar,ed(2)),agr([[],pl,m])]).

lex(preguntando,v,preguntar,[morph(preguntar,ando)]).
lex(pregunto,v,preguntar,[morph(preguntar,o)]).
lex(pregunta,v,preguntar,[morph(preguntar,a)]).
lex(preguntas,v,preguntar,[morph(preguntar,s)]).
lex(preguntamos,v,preguntar,[morph(preguntar,amos)]).
lex(preguntis,v,preguntar,[morph(preguntar,is)]).
lex(preguntan,v,preguntar,[morph(preguntar,an)]).
lex(preguntado,v,preguntar,[morph(preguntar,ed(2)),agr([[],[sg,m],m])]).
lex(preguntada,v,preguntar,[morph(preguntar,ed(2)),agr([[],[sg,m],f])]).
lex(preguntados,v,preguntar,[morph(preguntar,ed(2)),agr([[],pl,m])]).
lex(preguntadas,v,preguntar,[morph(preguntar,ed(2)),agr([[],pl,f])]).

lex(soy,v,ser,[morph(ser,soy)]).  % Cat= v
lex(eres,v,ser,[morph(ser,eres)]).
lex(es,v,ser,[morph(ser,es)]).
lex(somos,v,ser,[morph(ser,somos)]).
lex(sos,v,ser,[morph(ser,sos)]).
lex(son,v,ser,[morph(ser,son)]).
lex(sido,v,ser,[morph(ser,ed(2))]).
lex(siendo,v,ser,[morph(ser,ando)]).

lex(creyendo,v,creer,[morph(cre,ando)]).
lex(creo,v,creer,[morph(creer,o)]).
lex(crees,v,creer,[morph(creer,es)]).
lex(cree,v,creer,[morph(creer,e)]).
lex(creemos,v,creer,[morph(creer,emos)]).
lex(creis,v,creer,[morph(creer,is)]).
lex(creen,v,creer,[morph(creer,en)]).
lex(credo,v,creer,[morph(creer,ed(2)),agr([[],[sg,m],m])]).
lex(creda,v,creer,[morph(creer,ed(2)),agr([[],[sg,m],f])]).
lex(credos,v,creer,[morph(creer,ed(2)),agr([[],pl,m])]).
lex(credas,v,creer,[morph(creer,ed(2)),agr([[],pl,f])]).

lex(comprando,v,comprar,[morph(comprar,ando)]).
lex(compro,v,comprar,[morph(comprar,o)]).
lex(compras,v,comprar,[morph(comprar,s)]).
lex(compra,v,comprar,[morph(comprar,a)]).
lex(compramos,v,comprar,[morph(comprar,amos)]).
lex(compris,v,comprar,[morph(comprar,is)]).
lex(compran,v,comprar,[morph(comprar,an)]).
lex(comprado,v,comprar,[morph(comprar,ed(2)),agr([[],[sg,m],m])]).
lex(comprada,v,comprar,[morph(comprar,ed(2)),agr([[],[sg,m],f])]).
lex(comprados,v,comprar,[morph(comprar,ed(2)),agr([[],pl,m])]).
lex(compradas,v,comprar,[morph(comprar,ed(2)),agr([[],pl,f])]).

lex(llamando,v,llamar,[morph(llamar,ando)]).
lex(llamo,v,llamar,[morph(llamar,o)]).
lex(llamas,v,llamar,[morph(llamar,s)]).
lex(llama,v,llamar,[morph(llamar,a)]).
lex(llamamos,v,llamar,[morph(llamar,amos)]).
lex(llamis,v,llamar,[morph(llamar,is)]).
lex(llaman,v,llamar,[morph(llamar,an)]).
lex(llamado,v,llamar,[morph(llamar,ed(2)),agr([[],[sg,m],m])]).
lex(llamada,v,llamar,[morph(llamar,ed(2)),agr([[],[sg,m],f])]).
lex(llamados,v,llamar,[morph(llamar,ed(2)),agr([[],pl,m])]).
lex(llamadas,v,llamar,[morph(llamar,ed(2)),agr([[],pl,f])]).

lex(afirmando,v,afirmar,[morph(afirmar,ando)]).
lex(afirmo,v,afirmar,[morph(afirmar,o)]).
lex(afirmas,v,afirmar,[morph(afirmar,s)]).
lex(afirma,v,afirmar,[morph(afirmar,a)]).
lex(afirmamos,v,afirmar,[morph(afirmar,amos)]).
lex(afirmis,v,afirmar,[morph(afirmar,is)]).
lex(afirman,v,afirmar,[morph(afirmar,an)]).
lex(afirmado,v,afirmar,[morph(afirmar,ed(2)),agr([[],[sg,m],m])]).
lex(afirmada,v,afirmar,[morph(afirmar,ed(2)),agr([[],[sg,m],f])]).
lex(afirmados,v,afirmar,[morph(afirmar,ed(2)),agr([[],pl,m])]).
lex(afirmadas,v,afirmar,[morph(afirmar,ed(2)),agr([[],pl,f])]).

lex(saliendo,v,salir,[morph(salir,iendo)]).
lex(salgo,v,salir,[morph(salir,o)]).
lex(sales,v,salir,[morph(salir,s)]).
lex(sale,v,salir,[morph(salir,e)]).
lex(salimos,v,salir,[morph(salir,imos)]).
lex(sals,v,salir,[morph(salir,s)]).
lex(salen,v,salir,[morph(salir,en)]).
lex(salido,v,salir,[morph(salir,ed(2)),agr([[],sg,m])]).
lex(salida,v,salir,[morph(salir,ed(2)),agr([[],sg,f])]).
lex(salidos,v,salir,[morph(salir,ed(2)),agr([[],pl,m])]).
lex(salidas,v,salir,[morph(salir,ed(2)),agr([[],pl,f])]).

lex(destruyendo,v,destruir,[morph(destruir,ando)]).
lex(destruyes,v,destruir,[morph(destruir,s)]).
lex(destruye,v,destruir,[morph(destruir,e)]).
lex(destruimos,v,destruir,[morph(destruir,imos)]).
lex(destrus,v,destruir,[morph(destruir,s)]).
lex(destruyen,v,destruir,[morph(destruir,en)]).
lex(destruido,v,desstruir,[morph(destruir,ed(2)),agr([[],[sg,m],m])]).
lex(destruida,v,destruir,[morph(destruir,ed(2)),agr([[],[sg,m],f])]).
lex(destruidos,v,destruir,[morph(destruir,ed(2)),agr([[],pl,m])]).
lex(destruidas,v,destruir,[morph(destruir,ed(2)),agr([[],pl,f])]).

lex(murendo,v,morir,[morph(morir,ando)]).
lex(muero,v,morir,[morph(morir,o)]).
lex(mueres,v,morir,[morph(morir,s)]).
lex(muere,v,morir,[morph(morir,e)]).
lex(morimos,v,morir,[morph(morir,imos)]).
lex(mors,v,morir,[morph(morir,s)]).
lex(mueren,v,morir,[morph(morir,en)]).
lex(muerto,v,morir,[morph(morir,ed(2)),agr([[],[sg,m],m])]).
lex(muerta,v,morir,[morph(morir,ed(2)),agr([[],[sg,m],f])]).
lex(muertas,v,morir,[morph(morir,ed(2)),agr([[],pl,f])]).
lex(muertos,v,morir,[morph(morir,ed(2)),agr([[],pl,m])]).

lex(desapareciendo,v,desaparecer,[morph(desaparecer,ando)]).
lex(desaparezco,v,desaparecer,[morph(desaparecer,o)]).
lex(desapareces,v,desaparecer,[morph(desaparecer,s)]).
lex(desaparece,v,desaparecer,[morph(desaparecer,e)]).
lex(desaparecemos,v,desaparecer,[morph(desaparecer,emos)]).
lex(desaparecis,v,desaparecer,[morph(desaparecer,is)]).
lex(desaparecen,v,desaparecer,[morph(desaparecer,en)]).
lex(desaparecido,v,desaparecer,[morph(desaparecer,ed(2))]).

lex(comiendo,v,comer,[morph(comer,ando)]).
lex(como,v,comer,[morph(comer,o)]).
lex(come,v,comer,[morph(comer,e)]).
lex(comes,v,comer,[morph(comer,s)]).
lex(comemos,v,comer,[morph(comer,emos)]).
lex(comis,v,comer,[morph(comer,is)]).
lex(comen,v,comer,[morph(comer,en)]).
lex(comido,v,comer,[morph(comer,ed(2)),agr([[],[sg,m],m])]).

lex(cayendo,v,caer,[morph(caer,ando)]).
lex(caigo,v,caer,[morph(caer,o)]).
lex(cae,v,caer,[morph(caer,e)]).
lex(caes,v,caer,[morph(caer,s)]).
lex(caemos,v,caer,[morph(caer,emos)]).
lex(cais,v,caer,[morph(caer,is)]).
lex(caen,v,caer,[morph(caer,en)]).
lex(cado,v,caer,[morph(caer,ed(2)),agr([[],[sg,m],m])]).
lex(cada,v,caer,[morph(caer,ed(2)),agr([[],[sg,m],f])]).
lex(cados,v,caer,[morph(caer,ed(2)),agr([[],pl,m])]).
lex(cadas,v,caer,[morph(caer,ed(2)),agr([[],pl,f])]).

lex(luchando,v,luchar,[morph(luchar,ando)]).
lex(lucho,v,luchar,[morph(luchar,o)]).
lex(luchas,v,luchar,[morph(luchar,s)]).
lex(lucha,v,luchar,[morph(luchar,a)]).
lex(luchamos,v,luchar,[morph(luchar,amos)]).
lex(luchis,v,luchar,[morph(luchar,is)]).
lex(luchan,v,luchar,[morph(luchar,an)]).
lex(luchado,v,luchar,[morph(luchar,ed(2))]).

lex(archivando,v,archivar,[morph(archivar,ando)]).
lex(archivo,v,archivar,[morph(archivar,o)]).
lex(archivas,v,archivar,[morph(archivar,s)]).
lex(archiva,v,archivar,[morph(archivar,a)]).
lex(archivamos,v,archivar,[morph(archivar,amos)]).
lex(archivis,v,archivar,[morph(archivar,is)]).
lex(archivan,v,archivar,[morph(archivar,an)]).
lex(archivado,v,archivar,[morph(archivar,ed(2)),agr([[],[sg,m],m])]).

lex(despidiendo,v,despedir,[morph(despedir,ando)]).
lex(despido,v,despedir,[morph(despedir,o)]).
lex(despides,v,despedir,[morph(despedir,s)]).
lex(despide,v,despedir,[morph(despedir,e)]).
lex(despedimos,v,despedir,[morph(despedir,imos)]).
lex(despeds,v,despedir,[morph(despedir,s)]).
lex(despiden,v,despedir,[morph(despedir,en)]).
lex(despidiendo,v,despedir,[morph(despedir,ed(2)),agr([[],[sg,m],m])]).
lex(despidienda,v,despedir,[morph(despedir,ed(2)),agr([[],[sg,m],f])]).
lex(despidiendos,v,despedir,[morph(despedir,ed(2)),agr([[],pl,m])]).
lex(despidiendas,v,despedir,[morph(despedir,ed(2)),agr([[],pl,f])]).

lex(dando,v,dar,[morph(dar,ando)]).
lex(doy,v,dar,[morph(dar,o)]).
lex(das,v,dar,[morph(dar,s)]).
lex(da,v,dar,[morph(dar,a)]).
lex(damos,v,dar,[morph(dar,amos)]).
lex(dis,v,dar,[morph(dar,is)]).
lex(dan,v,dar,[morph(dar,an)]).
lex(dado,v,dar,[morph(dar,ed(2)),agr([[],[sg,m],m])]).
lex(dada,v,dar,[morph(dar,ed(2)),agr([[],[sg,m],f])]).
lex(dados,v,dar,[morph(dar,ed(2)),agr([[],pl,m])]).
lex(dadas,v,dar,[morph(dar,ed(2)),agr([[],pl,f])]).

lex(habiendo,v,haber,[morph(haber,ando)]).
lex(he,v,haber,[morph(haber,he)]).
lex(has,v,haber,[morph(haber,s)]).
lex(ha,v,haber,[morph(haber,a)]).
lex(hemos,v,haber,[morph(haber,hemos)]).
lex(habis,v,haber,[morph(haber,is)]).
lex(han,v,haber,[morph(haber,han)]).
lex(habido,v,haber,[morph(haber,ed(2)),agr([[],[sg,m],m])]).
lex(habida,v,haber,[morph(haber,ed(2)),agr([[],[sg,m],f])]).
lex(habidos,v,haber,[morph(haber,ed(2)),agr([[],pl,m])]).
lex(habidas,v,haber,[morph(haber,ed(2)),agr([[],pl,f])]).

lex(oyendo,v,or,[morph(or,ando)]).
lex(oigo,v,or,[morph(or,o)]).
lex(oyes,v,or,[morph(or,s)]).
lex(oye,v,or,[morph(or,e)]).
lex(omos,v,or,[morph(or,imos)]).
lex(os,v,or,[morph(or,s)]).
lex(oyen,v,or,[morph(or,en)]).
lex(odo,v,or,[morph(or,ed(2)),agr([[],[sg,m],m])]).
lex(oda,v,or,[morph(or,ed(2)),agr([[],[sg,m],f])]).
lex(odos,v,or,[morph(or,ed(2)),agr([[],pl,m])]).
lex(odas,v,or,[morph(or,ed(2)),agr([[],pl,f])]).


lex(contratando,v,contratar,[morph(contratar,ando)]).
lex(contrato,v,contratar,[morph(contratar,o)]).
lex(contratas,v,contratar,[morph(contratar,s)]).
lex(contrata,v,contratar,[morph(contratar,a)]).
lex(contratamos,v,contratar,[morph(contratar,amos)]).
lex(contratis,v,contratar,[morph(contratar,is)]).
lex(contratan,v,contratar,[morph(contratar,an)]).
lex(contratado,v,contratar,[morph(contratar,ed(2)),agr([[],[sg,m],m])]).
lex(contratada,v,contratar,[morph(contratar,ed(2)),agr([[],[sg,m],f])]).
lex(contratados,v,contratar,[morph(contratar,ed(2)),agr([[],pl,m])]).
lex(contratadas,v,contratar,[morph(contratar,ed(2)),agr([[],pl,f])]).

lex(golpando,v,golpear,[morph(golpear,ando)]).
lex(golpeo,v,golpear,[morph(golpear,o)]).
lex(golpeas,v,golpear,[morph(golpear,s)]).
lex(golpea,v,golpear,[morph(golpear,a)]).
lex(golpeamos,v,golpear,[morph(golpear,amos)]).
lex(golpeis,v,golpear,[morph(golpear,is)]).
lex(golpean,v,golpear,[morph(golpear,an)]).
lex(golpeado,v,golpear,[morph(golpear,ed(2)),agr([[],[sg,m],m])]).
lex(golpeada,v,golpear,[morph(golpear,ed(2)),agr([[],[sg,m],f])]).
lex(golpeados,v,golpear,[morph(golpear,ed(2)),agr([[],pl,m])]).
lex(golpeadas,v,golpear,[morph(golpear,ed(2)),agr([[],pl,f])]).

lex(insultando,v,insultar,[morph(insultar,ando)]).
lex(insulto,v,insultar,[morph(insultar,o)]).
lex(insultas,v,insultar,[morph(insultar,s)]).
lex(insulta,v,insultar,[morph(insultar,a)]).
lex(insultamos,v,insultar,[morph(insultar,amos)]).
lex(insultis,v,insultar,[morph(insultar,is)]).
lex(insultan,v,insultar,[morph(insultar,an)]).
lex(insultado,v,insultar,[morph(insultar,ed(2)),agr([[],[sg,m],m])]).
lex(insultada,v,insultar,[morph(insultar,ed(2)),agr([[],[sg,m],f])]).
lex(insultados,v,insultar,[morph(insultar,ed(2)),agr([[],pl,m])]).
lex(insultadas,v,insultar,[morph(insultar,ed(2)),agr([[],pl,f])]).

lex(enseando,v,ensear,[morph(ensear,ando)]).
lex(enseo,v,ensear,[morph(ensear,o)]).
lex(enseas,v,ensear,[morph(ensear,s)]).
lex(ensea,v,ensear,[morph(ensear,a)]).
lex(enseamos,v,ensear,[morph(ensear,amos)]).
lex(enseis,v,ensear,[morph(ensear,is)]).
lex(ensean,v,ensear,[morph(ensear,an)]).
lex(enseado,v,ensear,[morph(ensear,ed(2)),agr([[],[sg,m],m])]).
lex(enseada,v,ensear,[morph(ensear,ed(2)),agr([[],[sg,m],f])]).
lex(enseados,v,ensear,[morph(ensear,ed(2)),agr([[],pl,m])]).
lex(enseadas,v,ensear,[morph(ensear,ed(2)),agr([[],pl,f])]).

lex(guardando,v,guardar,[morph(guardar,ando)]).
lex(guardo,v,guardar,[morph(guardar,o)]).
lex(guardas,v,guardar,[morph(guardar,s)]).
lex(guarda,v,guardar,[morph(guardar,a)]).
lex(guardamos,v,guardar,[morph(guardar,amos)]).
lex(guardis,v,guardar,[morph(guardar,is)]).
lex(guardan,v,guardar,[morph(guardar,an)]).
lex(guardado,v,guardar,[morph(guardar,ed(2)),agr([[],[sg,m],m])]).
lex(guardada,v,guardar,[morph(guardar,ed(2)),agr([[],[sg,m],f])]).
lex(guardados,v,guardar,[morph(guardar,ed(2)),agr([[],pl,m])]).
lex(guardadas,v,guardar,[morph(guardar,ed(2)),agr([[],pl,f])]).

lex(matando,v,matar,[morph(matar,ando)]).
lex(mato,v,matar,[morph(matar,o)]).
lex(matas,v,matar,[morph(matar,s)]).
lex(mata,v,matar,[morph(matar,a)]).
lex(matamos,v,matar,[morph(matar,amos)]).
lex(matis,v,matar,[morph(matar,is)]).
lex(matan,v,matar,[morph(matar,an)]).
lex(matado,v,matar,[morph(matar,ed(2)),agr([[],[sg,m],m])]).
lex(matada,v,matar,[morph(matar,ed(2)),agr([[],[sg,m],f])]).
lex(matados,v,matar,[morph(matar,ed(2)),agr([[],pl,m])]).
lex(matadas,v,matar,[morph(matar,ed(2)),agr([[],pl,f])]).

lex(conociendo,v,conocer,[morph(conocer,ando)]).
lex(conozco,v,conocer,[morph(conocer,o)]).
lex(conoces,v,conocer,[morph(conocer,s)]).
lex(conoce,v,conocer,[morph(conocer,e)]).
lex(conocemos,v,conocer,[morph(conocer,emos)]).
lex(conocis,v,conocer,[morph(conocer,is)]).
lex(conocen,v,conocer,[morph(conocer,en)]).
lex(conocido,v,conocer,[morph(conocer,ed(2)),agr([[],[sg,m],m])]).
lex(conocida,v,conocer,[morph(conocer,ed(2)),agr([[],[sg,m],f])]).
lex(conocidos,v,conocer,[morph(conocer,ed(2)),agr([[],pl,m])]).
lex(conocidas,v,conocer,[morph(conocer,ed(2)),agr([[],pl,f])]).

lex(sabiendo,v,saber,[morph(saber,ando)]).
lex(s,v,saber,[morph(saber,se)]).
lex(sabes,v,saber,[morph(saber,s)]).
lex(sabe,v,saber,[morph(saber,e)]).
lex(sabemos,v,saber,[morph(saber,emos)]).
lex(sabis,v,saber,[morph(saber,is)]).
lex(saben,v,saber,[morph(saber,en)]).
lex(sabido,v,saber,[morph(saber,ed(2)),agr([3,[sg,m],m])]).
lex(sabida,v,saber,[morph(saber,ed(2)),agr([3,[sg,m],f])]).
lex(sabidos,v,saber,[morph(saber,ed(2)),agr([3,pl,m])]).
lex(sabidas,v,saber,[morph(saber,ed(2)),agr([3,pl,f])]).

lex(rendo,v,rer,[morph(rer,ando)]).
lex(ro,v,rer,[morph(rer,o)]).
lex(res,v,rer,[morph(rer,s)]).
lex(re,v,rer,[morph(rer,e)]).
lex(remos,v,rer,[morph(rer,mos)]).
lex(res,v,rer,[morph(rer,s)]).
lex(ren,v,rer,[morph(rer,en)]).
lex(redo,v,rer,[morph(rer,ed(2))]).

lex(dejando,v,dejar,[morph(dejar,ando)]).
lex(dejo,v,dejar,[morph(dejar,o)]).
lex(dejas,v,dejar,[morph(dejar,s)]).
lex(deja,v,dejar,[morph(dejar,a)]).
lex(dejamos,v,dejar,[morph(dejar,amos)]).
lex(dejis,v,dejar,[morph(dejar,is)]).
lex(dejan,v,dejar,[morph(dejar,an)]).
lex(dejado,v,dejar,[morph(dejar,ed(2)),agr([[],[sg,m],m])]).
lex(dejada,v,dejar,[morph(dejar,ed(2)),agr([[],[sg,m],f])]).
lex(dejados,v,dejar,[morph(dejar,ed(2)),agr([[],pl,m])]).
lex(dejadas,v,dejar,[morph(dejar,ed(2)),agr([[],pl,f])]).

% to lead
lex(acaudillando,v,acaudillar,[morph(acaudillar,ando)]).
lex(acaudillas,v,acaudillar,[morph(acaudillar,s)]).
lex(acaudilla,v,acaudillar,[morph(acaudillar,a)]).
lex(acaudillamos,v,acaudillar,[morph(acaudillar,amos)]).
lex(acaudillis,v,acaudillar,[morph(acaudillar,is)]).
lex(acaudillan,v,acaudillar,[morph(acaudillar,an)]).
lex(acaudillado,v,acaudillar,[morph(acaudillar,ed(2)),agr([[],[sg,m],m])]).
lex(acaudillada,v,acaudillar,[morph(acaudillar,ed(2)),agr([[],[sg,m],f])]).
lex(acaudillados,v,acaudillar,[morph(acaudillar,ed(2)),agr([[],pl,m])]).
lex(acaudilladas,v,acaudillar,[morph(acaudillar,ed(2)),agr([[],pl,f])]).

% to please: me gusta el libro : I like the book
lex(gustando,v,gustar,[morph(gustar,ando)]).
lex(gusto,v,gustar,[morph(gustar,o)]).
lex(gustas,v,gustar,[morph(gustar,s)]).
lex(gusta,v,gustar,[morph(gustar,a)]).
lex(gustamos,v,gustar,[morph(gustar,amos)]).
lex(gustis,v,gustar,[morph(gustar,is)]).
lex(gustan,v,gustar,[morph(gustar,an)]).
lex(gustado,v,gustar,[morph(gustar,ed(2)),agr([[],[sg,m],m])]).
lex(gustada,v,gustar,[morph(gustar,ed(2)),agr([[],[sg,m],f])]).
lex(gustados,v,gustar,[morph(gustar,ed(2)),agr([[],pl,m])]).
lex(gustadas,v,gustar,[morph(gustar,ed(2)),agr([[],pl,f])]).

lex(encontrando,v,encontrarse,[morph(encontrarse,ando)]).
lex(me_encuentro,v,encontrarse,[morph(encontrarse,o)]).
lex(te_encuentras,v,encontrarse,[morph(encontrarse,s)]).
lex(se_encuentra,v,encontrarse,[morph(encontrarse,a)]).
lex(nos_encontramos,v,encontrarse,[morph(encontrarse,amos)]).
lex(os_encontris,v,encontrarse,[morph(encontrarse,is)]).
lex(se_encuentran,v,encontrarse,[morph(encontrarse,an)]).
lex(encontrado,v,encontrarse,[morph(encontrarse,ed(2)),arg([[],[sg,m],m])]).
lex(encontrada,v,encontrarse,[morph(encontrarse,ed(2)),arg([[],[sg,m],f])]).
lex(encontrados,v,encontrarse,[morph(encontrarse,ed(2)),arg([[],pl,m])]).
lex(encontradas,v,encontrarse,[morph(encontrarse,ed(2)),arg([[],pl,f])]).

lex(mencionando,v,mencionar,[morph(mencionar,ando)]).
lex(menciono,v,mencionar,[morph(mencionar,o)]).
lex(mencionas,v,mencionar,[morph(mencionar,s)]).
lex(menciona,v,mencionar,[morph(mencionar,a)]).
lex(mencionamos,v,mencionar,[morph(mencionar,amos)]).
lex(mencionis,v,mencionar,[morph(mencionar,is)]).
lex(mencionan,v,mencionar,[morph(mencionar,an)]).
lex(mencionado,v,mencionar,[morph(mencionar,ed(2)),agr([[],[sg,m],m])]).
lex(mencionada,v,mencionar,[morph(mencionar,ed(2)),agr([[],[sg,m],f])]).
lex(mencionados,v,mencionar,[morph(mencionar,ed(2)),agr([[],pl,m])]).
lex(mencionadas,v,mencionar,[morph(mencionar,ed(2)),agr([[],pl,f])]).

lex(estacionando,v,estacionar,[morph(estacionar,ando)]).
lex(estaciono,v,estacionar,[morph(estacionar,o)]).
lex(estacionas,v,estacionar,[morph(estacionar,s)]).
lex(estaciona,v,estacionar,[morph(estacionar,a)]).
lex(estacionamos,v,estacionar,[morph(estacionar,amos)]).
lex(estacionis,v,estacionar,[morph(estacionar,is)]).
lex(estacionan,v,estacionar,[morph(estacionar,an)]).
lex(estacionado,v,estacionar,[morph(estacionar,ed(2)),agr([[],[sg,m],m])]).
lex(estacionada,v,estacionar,[morph(estacionar,ed(2)),agr([[],[sg,m],f])]).
lex(estacionados,v,estacionar,[morph(estacionar,ed(2)),agr([[],pl,m])]).
lex(estacionadas,v,estacionar,[morph(estacionar,ed(2)),agr([[],pl,f])]).

lex(persuadiendo,v,persuadir,[morph(persuadir,ando)]).
lex(persuado,v,persuadir,[morph(persuadir,o)]).
lex(persuades,v,persuadir,[morph(persuadir,s)]).
lex(persuade,v,persuadir,[morph(persuadir,e)]).
lex(persuadimos,v,persuadir,[morph(persuadir,imos)]).
lex(persuads,v,persuadir,[morph(persuadir,s)]).
lex(persuaden,v,persuadir,[morph(persuadir,en)]).
lex(persuadido,v,persuadir,[morph(persuadir,ed(2)),agr([[],[sg,m],m])]).
lex(persuadida,v,persuadir,[morph(persuadir,ed(2)),agr([[],[sg,m],f])]).
lex(persuadidos,v,persuadir,[morph(persuadir,ed(2)),agr([[],pl,m])]).
lex(persuadidas,v,persuadir,[morph(persuadir,ed(2)),agr([[],pl,f])]).

lex(prometiendo,v,prometer,[morph(prometer,ando)]).
lex(prometo,v,prometer,[morph(prometer,o)]).
lex(prometes,v,prometer,[morph(prometer,s)]).
lex(promete,v,prometer,[morph(prometer,e)]).
lex(prometemos,v,prometer,[morph(prometer,emos)]).
lex(prometis,v,prometer,[morph(prometer,is)]).
lex(prometen,v,prometer,[morph(prometer,en)]).
lex(prometido,v,prometer,[morph(prometer,ed(2)),agr([[],[sg,m],m])]).
lex(prometida,v,prometer,[morph(prometer,ed(2)),agr([[],[sg,m],f])]).
lex(prometidas,v,prometer,[morph(prometer,ed(2)),agr([[],pl,f])]).
lex(prometidos,v,prometer,[morph(prometer,ed(2)),agr([[],pl,m])]).

lex(poniendo,v,poner,[morph(poner,ando)]).
lex(pongo,v,poner,[morph(poner,o)]).
lex(pones,v,poner,[morph(poner,s)]).
lex(pone,v,poner,[morph(poner,e)]).
lex(ponemos,v,poner,[morph(poner,emos)]).
lex(ponis,v,poner,[morph(poner,is)]).
lex(ponen,v,poner,[morph(poner,en)]).
lex(puesto,v,poner,[morph(poner,ed(2)),agr([[],[pl,sg,m],m])]).
lex(puesta,v,poner,[morph(poner,ed(2)),agr([[],[sg,m],f])]).
lex(puestas,v,poner,[morph(poner,ed(2)),agr([[],pl,f])]).
lex(puestos,v,poner,[morph(poner,ed(2)),agr([[],pl,m])]).


lex(lloviendo,v,llover,[morph(llover,ando)]).
lex(llueve,v,llover,[morph(llover,e)]).
lex(llovido,v,llover,[morph(llover,ed(2))]).
lex(llova,v,llover,[morph(llover,ed(1))]).

lex(leyendo,v,leer,[morph(leer,ando)]).
lex(leo,v,leer,[morph(leer,o)]).
lex(lees,v,leer,[morph(leer,s)]).
lex(lee,v,leer,[morph(leer,e)]).
lex(leemos,v,leer,[morph(leer,emos)]).
lex(leis,v,leer,[morph(leer,is)]).
lex(leen,v,leer,[morph(leer,en)]).
lex(ledo,v,leer,[morph(leer,ed(2)),agr([[],[sg,m],m])]).
lex(leda,v,leer,[morph(leer,ed(2)),agr([[],[sg,m],f])]).
lex(ledos,v,leer,[morph(leer,ed(2)),agr([[],pl,m])]).
lex(ledas,v,leer,[morph(leer,ed(2)),agr([[],pl,f])]).

lex(quitando,v,quitar,[morph(quitar,ando)]).
lex(quito,v,quitar,[morph(quitar,o)]).
lex(quitas,v,quitar,[morph(quitar,s)]).
lex(quita,v,quitar,[morph(quitar,a)]).
lex(quitamos,v,quitar,[morph(quitar,amos)]).
lex(quitis,v,quitar,[morph(quitar,is)]).
lex(quitan,v,quitar,[morph(quitar,an)]).
lex(quitado,v,quitar,[morph(quitar,ed(2)),agr([[],[sg,m],m])]).
lex(quitada,v,quitar,[morph(quitar,ed(2)),agr([[],[sg,m],f])]).
lex(quitados,v,quitar,[morph(quitar,ed(2)),agr([[],pl,m])]).
lex(quitadas,v,quitar,[morph(quitar,ed(2)),agr([[],pl,f])]).

lex(dimitiendo,v,dimitir,[morph(dimitir,ando)]).
lex(dimites,v,dimitir,[morph(dimitir,s)]).
lex(dimite,v,dimitir,[morph(dimitir,e)]).
lex(dimitimos,v,dimitir,[morph(dimitir,imos)]).
lex(dimits,v,dimitir,[morph(dimitir,s)]).
lex(dimiten,v,dimitir,[morph(dimitir,en)]).
lex(dimitido,v,dimitir,[morph(dimitir,ed(2))]).

lex(diciendo,v,decir,[morph(decir,ando)]).
lex(digo,v,decir,[morph(decir,o)]).
lex(dices,v,decir,[morph(decir,s)]).
lex(dice,v,decir,[morph(decir,e)]).
lex(decimos,v,decir,[morph(decir,imos)]).
lex(decs,v,decir,[morph(decir,is)]).
lex(dicen,v,decir,[morph(decir,en)]).
lex(dicho,v,decir,[morph(decir,ed(2)),agr([[],[sg,m],m])]).
lex(dichos,v,decir,[morph(decir,ed(2)),agr([[],pl,m])]).
lex(dicha,v,decir,[morph(decir,ed(2)),agr([[],[sg,m],f])]).
lex(dichas,v,decir,[morph(decir,ed(2)),agr([[],pl,f])]).

lex(viendo,v,ver,[morph(ver,ando)]).
lex(veo,v,ver,[morph(ver,o)]).
lex(ves,v,ver,[morph(ver,s)]).
lex(ve,v,ver,[morph(ver,e)]).
lex(vemos,v,ver,[morph(ver,emos)]).
lex(veis,v,ver,[morph(ver,is)]).
lex(ven,v,ver,[morph(ver,en)]).
lex(visto,v,ver,[morph(ver,ed(2)),agr([[],[sg,m],m])]).
lex(vista,v,ver,[morph(ver,ed(2)),agr([[],[sg,m],f])]).
lex(vistos,v,ver,[morph(ver,ed(2)),agr([[],pl,m])]).
lex(vistas,v,ver,[morph(ver,ed(2)),agr([[],pl,f])]).

lex(pareciendo,v,parecer,[morph(parecer,ando)]).
lex(parezco,v,parecer,[morph(parecer,o)]).
lex(pareces,v,parecer,[morph(parecer,s)]).
lex(parece,v,parecer,[morph(parecer,e)]).
lex(parecemos,v,parecer,[morph(parecer,emos)]).
lex(parecis,v,parecer,[morph(parecer,is)]).
lex(parecen,v,parecer,[morph(parecer,en)]).
lex(parecido,v,parecer,[morph(parecer,ed(2))]).

lex(durmiendo,v,dormir,[morph(dormir,ando)]).
lex(duermo,v,dormir,[morph(dormir,o)]).
lex(duermes,v,dormir,[morph(dormir,s)]).
lex(duerme,v,dormir,[morph(dormir,e)]).
lex(dormimos,v,dormir,[morph(dormir,imos)]).
lex(dorms,v,dormir,[morph(dormir,is)]).
lex(duermen,v,dormir,[morph(dormir,ent)]).
lex(dormido,v,dormir,[morph(dormir,ed(2))]).

% to snow
lex(nevando,v,nevar,[morph(nevar,ando)]).
lex(nieva,v,nevar,[morph(nevar,a)]).
lex(nevado,v,nevar,[morph(nevar,ed(2))]).
lex(nevaba,v,nevar,[morph(nevar,ed(1))]).

% to solve
lex(resolviendo,v,resolver,[morph(resolver,ando)]).
lex(resuelvo,v,resolver,[morph(resolver,o)]).
lex(resuelves,v,resolver,[morph(resolver,s)]).
lex(resuelve,v,resolver,[morph(resolver,e)]).
lex(resolvemos,v,resolver,[morph(resolver,emos)]).
lex(resolvis,v,resolver,[morph(resolver,is)]).
lex(resuelven,v,resolver,[morph(resolver,en)]).
lex(resuelto,v,resolver,[morph(resolver,ed(2)),agr([[],[sg,m],m])]).
lex(resuelta,v,resolver,[morph(resolver,ed(2)),agr([[],[sg,m],f])]).
lex(resueltos,v,resolver,[morph(resolver,ed(2)),agr([[],pl,m])]).
lex(resueltas,v,resolver,[morph(resolver,ed(2)),agr([[],pl,f])]).

% to stand
lex(soportando,v,soportar,[morph(soportar,ando)]).
lex(soporto,v,soportar,[morph(soportar,o)]).
lex(soportas,v,soportar,[morph(soportar,s)]).
lex(soporta,v,soportar,[morph(soportar,a)]).
lex(soportamos,v,soportar,[morph(soportar,amos)]).
lex(soportis,v,soportar,[morph(soportar,is)]).
lex(soportan,v,soportar,[morph(soportar,an)]).
lex(soportado,v,soportar,[morph(soportar,ed(2)),agr([[],[sg,m],m])]).
lex(soportada,v,soportar,[morph(soportar,ed(2)),agr([[],[sg,m],f])]).
lex(soportados,v,soportar,[morph(soportar,ed(2)),agr([[],pl,m])]).
lex(soportadas,v,soportar,[morph(soportar,ed(2)),agr([[],pl,f])]).

% to start
lex(empezando,v,empezar,[morph(empezar,ando)]).
lex(empiezo,v,empezar,[morph(empezar,o)]).
lex(empiezas,v,empezar,[morph(empezar,s)]).
lex(empieza,v,empezar,[morph(empezar,a)]).
lex(empezamos,v,empezar,[morph(empezar,amos)]).
lex(empezis,v,empezar,[morph(empezar,is)]).
lex(empiezan,v,empezar,[morph(empezar,an)]).
lex(empezado,v,empezar,[morph(empezar,ed(2)),agr([[],[sg,m],m])]).
lex(empezada,v,empezar,[morph(empezar,ed(2)),agr([[],[sg,m],f])]).
lex(empezados,v,empezar,[morph(empezar,ed(2)),agr([[],pl,m])]).
lex(empezadas,v,empezar,[morph(empezar,ed(2)),agr([[],pl,f])]).

% to surprise
lex(sorprendiendo,v,sorprender,[morph(sorprender,ando)]).
lex(sorprendo,v,sorprender,[morph(sorprender,o)]).
lex(sorprendes,v,sorprender,[morph(sorprender,s)]).
lex(sorprende,v,sorprender,[morph(sorprender,e)]).
lex(sorprendemos,v,sorprender,[morph(sorprender,emos)]).
lex(sorprendis,v,sorprender,[morph(sorprender,is)]).
lex(sorprenden,v,sorprender,[morph(sorprender,en)]).
lex(sorprendido,v,sorprender,[morph(sorprender,ed(2)),agr([[],[sg,m],m])]).
lex(sorprendida,v,sorprender,[morph(sorprender,ed(2)),agr([[],[sg,m],f])]).
lex(sorprendidos,v,sorprender,[morph(sorprender,ed(2)),agr([[],pl,m])]).
lex(sorprendidas,v,sorprender,[morph(sorprender,ed(2)),agr([[],pl,f])]).

% to talk
lex(hablando,v,hablar,[morph(hablar,ando)]).
lex(hablo,v,hablar,[morph(hablar,o)]).
lex(hablas,v,hablar,[morph(hablar,s)]).
lex(habla,v,hablar,[morph(hablar,a)]).
lex(hablamos,v,hablar,[morph(hablar,amos)]).
lex(hablis,v,hablar,[morph(hablar,is)]).
lex(hablan,v,hablar,[morph(hablar,an)]).
lex(hablado,v,hablar,[morph(hablar,ed(2))]).

% to wash
lex(lavando,v,lavar,[morph(lavar,ando)]).
lex(lavo,v,lavar,[morph(lavar,o)]).
lex(lavas,v,lavar,[morph(lavar,s)]).
lex(lava,v,lavar,[morph(lavar,a)]).
lex(lavamos,v,lavar,[morph(lavar,amos)]).
lex(lavis,v,lavar,[morph(lavar,is)]).
lex(lavan,v,lavar,[morph(lavar,an)]).
lex(lavado,v,lavar,[morph(lavar,ed(2))]).

% to tell
lex(contando,v,contar,[morph(contar,ando)]).
lex(cuento,v,contar,[morph(contar,o)]).
lex(cuentas,v,contar,[morph(contar,s)]).
lex(cuenta,v,contar,[morph(contar,a)]).
lex(contamos,v,contar,[morph(contar,amos)]).
lex(contis,v,contar,[morph(contar,is)]).
lex(cuentan,v,contar,[morph(contar,an)]).
lex(contado,v,contar,[morph(contar,ed(2)),agr([[],[sg,m],m])]).
lex(contada,v,contar,[morph(contar,ed(2)),agr([[],[sg,m],f])]).
lex(contados,v,contar,[morph(contar,ed(2)),agr([[],pl,m])]).
lex(contadas,v,contar,[morph(contar,ed(2)),agr([[],pl,f])]).

% to try
lex(probando,v,probar,[morph(probar,ando)]).
lex(pruebo,v,probar,[morph(probar,o)]).
lex(pruebas,v,probar,[morph(probar,s)]).
lex(prueba,v,probar,[morph(probar,a)]).
lex(probamos,v,probar,[morph(probar,amos)]).
lex(probis,v,probar,[morph(probar,is)]).
lex(prueban,v,probar,[morph(probar,an)]).
lex(probado,v,probar,[morph(probar,ed(2)),agr([[],[sg,m],m])]).
lex(probada,v,probar,[morph(probar,ed(2)),agr([[],[sg,m],f])]).
lex(probados,v,probar,[morph(probar,ed(2)),agr([[],pl,m])]).
lex(probadas,v,probar,[morph(probar,ed(2)),agr([[],pl,f])]).

% to think
lex(pensando,v,pensar,[morph(pensar,ando)]).
lex(pienso,v,pensar,[morph(pensar,o)]).
lex(piensas,v,pensar,[morph(pensar,s)]).
lex(piensa,v,pensar,[morph(pensar,a)]).
lex(pensamos,v,pensar,[morph(pensar,amos)]).
lex(pensis,v,pensar,[morph(pensar,is)]).
lex(piensan,v,pensar,[morph(pensar,an)]).
lex(pensado,v,pensar,[morph(pensar,ed(2)),agr([[],[sg,m],m])]).
lex(pensada,v,pensar,[morph(pensar,ed(2)),agr([[],[sg,m],f])]).
lex(pensados,v,pensar,[morph(pensar,ed(2)),agr([[],pl,m])]).
lex(pensadas,v,pensar,[morph(pensar,ed(2)),agr([[],pl,f])]).

% to understand
lex(entendiendo,v,entender,[morph(entender,ando)]).
lex(entiendo,v,entender,[morph(entender,o)]).
lex(entiendes,v,entender,[morph(entender,s)]).
lex(entiende,v,entender,[morph(entender,e)]).
lex(entendemos,v,entender,[morph(entender,imos)]).
lex(entendis,v,entender,[morph(entender,is)]).
lex(entienden,v,entender,[morph(entender,en)]).
lex(entendido,v,entender,[morph(entender,ed(2)),agr([[],[sg,m],m])]).
lex(entendidos,v,entender,[morph(entender,ed(2)),agr([[],pl,m])]).
lex(entendida,v,entender,[morph(entender,ed(2)),agr([[],[sg,m],f])]).
lex(entendidas,v,entender,[morph(entender,ed(2)),agr([[],pl,f])]).

% to visit
lex(visitando,v,visitar,[morph(visitar,ando)]).
lex(visito,v,visitar,[morph(visitar,o)]).
lex(visitas,v,visitar,[morph(visitar,s)]).
lex(visita,v,visitar,[morph(visitar,a)]).
lex(visitamos,v,visitar,[morph(visitar,amos)]).
lex(visitis,v,visitar,[morph(visitar,is)]).
lex(visitan,v,visitar,[morph(visitar,an)]).
lex(visitado,v,visitar,[morph(visitar,ed(2)),agr([[],[sg,m],m])]).
lex(visitada,v,visitar,[morph(visitar,ed(2)),agr([[],[sg,m],f])]).
lex(visitados,v,visitar,[morph(visit,ed(2)),agr([[],pl,m])]).
lex(visitadas,v,visitar,[morph(visit,ed(2)),agr([[],pl,f])]).

% to want; to love
lex(queriendo,v,querer,[morph(querer,ando)]).
lex(quiero,v,querer,[morph(querer,o)]).
lex(quieres,v,querer,[morph(querer,s)]).
lex(quiere,v,querer,[morph(querer,e)]).
lex(queremos,v,querer,[morph(querer,emos)]).
lex(queris,v,querer,[morph(querer,is)]).
lex(quieren,v,querer,[morph(querer,en)]).
lex(querido,v,querer,[morph(querer,ed(2)),agr([[],[sg,m],m])]).
lex(querida,v,querer,[morph(querer,ed(2)),agr([[],[sg,m],f])]).
lex(queridos,v,querer,[morph(querer,ed(2)),agr([[],pl,m])]).
lex(queridas,v,querer,[morph(querer,ed(2)),agr([[],pl,f])]).

% win
lex(ganando,v,ganar,[morph(ganar,ando)]).
lex(gano,v,ganar,[morph(ganar,o)]).
lex(ganas,v,ganar,[morph(ganar,s)]).
lex(gana,v,ganar,[morph(ganar,a)]).
lex(ganamos,v,ganar,[morph(ganar,amos)]).
lex(ganis,v,ganar,[morph(ganar,is)]).
lex(ganan,v,ganar,[morph(ganar,an)]).
lex(ganado,v,ganar,[morph(ganar,ed(2)),agr([[],[sg,m],m])]).
lex(ganada,v,ganar,[morph(ganar,ed(2)),agr([[],[sg,m],f])]).
lex(ganados,v,ganar,[morph(ganar,ed(2)),agr([[],pl,m])]).
lex(ganadas,v,ganar,[morph(ganar,ed(2)),agr([[],pl,f])]).

% to work
lex(trabajando,v,trabajar,[morph(trabajar,ando)]).
lex(trabajo,v,trabajar,[morph(trabajar,o)]).
lex(trabajas,v,trabajar,[morph(trabajar,s)]).
lex(trabaja,v,trabajar,[morph(trabajar,a)]).
lex(trabajamos,v,trabajar,[morph(trabajar,amos)]).
lex(trabajis,v,trabajar,[morph(trabajar,is)]).
lex(trabajan,v,trabajar,[morph(trabajar,an)]).
lex(trabajado,v,trabajar,[morph(trabajar,ed(2)),agr([[],[sg,m],m])]).
lex(trabajada,v,trabajar,[morph(trabajar,ed(2)),agr([[],[sg,m],f])]).
lex(trabajados,v,trabajar,[morph(trabajar,ed(2)),agr([[],pl,m])]).
lex(trabajadas,v,trabajar,[morph(trabajar,ed(2)),agr([[],pl,f])]).

% to write
lex(escribiendo,v,escribir,[morph(escribir,ando)]).
lex(escribo,v,escribir,[morph(escribir,o)]).
lex(escribes,v,escribir,[morph(escribir,s)]).
lex(escribe,v,escribir,[morph(escribir,e)]).
lex(escribimos,v,escribir,[morph(escribir,imos)]).
lex(escribs,v,escribir,[morph(escribir,s)]).
lex(escriben,v,escribir,[morph(escribir,en)]).
lex(escrito,v,escribir,[morph(escribir,ed(2)),agr([[],[sg,m],m])]).
lex(escrita,v,escribir,[morph(escribir,ed(2)),agr([[],[sg,m],f])]).
lex(escritos,v,escribir,[morph(escribir,ed(2)),agr([[],pl,m])]).
lex(escritas,v,escribir,[morph(escribir,ed(2)),agr([[],pl,f])]).

% to come
lex(viniendo,v,venir,[morph(venir,ando)]).
lex(vengo,v,venir,[morph(venir,o)]).
lex(vienes,v,venir,[morph(venir,s)]).
lex(viene,v,venir,[morph(venir,e)]).
lex(venimos,v,venir,[morph(venir,imos)]).
lex(vens,v,venir,[morph(venir,s)]).
lex(vienen,v,venir,[morph(venir,en)]).
lex(venido,v,venir,[morph(venir,ed(2)),agr([[],[sg,m],m])]).
lex(venida,v,venir,[morph(venir,ed(2)),agr([[],[sg,m],f])]).
lex(venidos,v,venir,[morph(venir,ed(2)),agr([[],pl,m])]).
lex(venidas,v,venir,[morph(venir,ed(2)),agr([[],pl,f])]).

% to go
lex(yendo,v,ir,[morph(ir,ando)]).
lex(voy,v,ir,[morph(ir,o)]).
lex(vas,v,ir,[morph(ir,s)]).
lex(va,v,ir,[morph(ir,a)]).
lex(vamos,v,ir,[morph(ir,amos)]).
lex(vis,v,ir,[morph(ir,is)]).
lex(van,v,ir,[morph(ir,an)]).
lex(ido,v,ir,[morph(ir,ed(2)),agr([[],[sg,m],m])]).

% to be
lex(estando,v,estar,[morph(estar,ando)]).
lex(estoy,v,estar,[morph(estar,o)]).
lex(estas,v,estar,[morph(estar,s)]).
lex(esta,v,estar,[morph(estar,a)]).
lex(estamos,v,estar,[morph(estar,amos)]).
lex(estis,v,estar,[morph(estar,is)]).
lex(estan,v,estar,[morph(estar,an)]).
lex(estado,v,estar,[morph(estar,ed(2)),agr([[],[sg,m],m])]).

% to have (not the auxiliary)
lex(teniendo,v,tener,[morph(tener,ando)]).
lex(tengo,v,tener,[morph(tener,o)]).
lex(tienes,v,tener,[morph(tener,s)]).
lex(tiene,v,tener,[morph(tener,e)]).
lex(tenemos,v,tener,[morph(tener,emos)]).
lex(tenis,v,tener,[morph(tener,is)]).
lex(tienen,v,tener,[morph(tener,en)]).
lex(tenido,v,tener,[morph(tener,ed(2)),agr([[],[sg,m],m])]).
lex(tenida,v,tener,[morph(tener,ed(2)),agr([[],[sg,m],f])]).
lex(tenidos,v,tener,[morph(tener,ed(2)),agr([[],pl,m])]).
lex(tenidas,v,tener,[morph(tener,ed(2)),agr([[],pl,f])]).


% CONTRACTIONS

:- initialization(no contraction(_,_)).

:- initialization(no contraction(_,_,_)).

% MISCELLANEOUS

nounFeatures(Base,F) :-	lex(Base,n,F1),	pick1(agr(_),F1,F).

verbFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).

verbToNounFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).
%	pickNF1(ecm(_),F2,F).	e.g. belief still needs to properly govern ...

%% MAPS MORPHOLOGY INTO SYNTACTIC FEATURES
%%	Verb morphology and Agreement
%% 
%%	Form	Tense		AGR
%%	base 	infinitival
%%      -o      past(-)         agr(1,sg)
%%	-es	past(-) 	agr(2,sg)
%%      -as     past(-)         agr(2,sg)
%%      -e      past(-)         agr(3,sg)
%%      -a      past(-)         agr(3,sg)
%%      -emos, -imos, amos      past(-)         agr(1,pl)
%%      -is, is, s  past(-)         agr(2,pl)
%%      -en, an                 past(-)         agr(3,pl)
%%	-ed(1) 	past(+)		agr(_)
%% 	-ando	infinitival
%%	-ed(2)	perf, passive
%%	soy	past(-) 	agr(1,sg)
%%	eres	past(-) 	agr(2,sg)
%%      es      past(-)         agr(3,sg)
%%      somos   past(-)         agr(1,pl)
%%      sos    past(-)         agr(2,pl)
%%      son     past(-)         agr(3,pl)
%%	was	past(+)		agr([1,3],sg)
%%	were	past(+) 	agr(not 1,3,sg)
%%      he      past(-)         agr(1,sg)
%%      has      past(-)         agr(2,sg)
%%      ha       past(-)         agr(3,sg)
%%      han     past(-)         agr(3,pl)





% non-deterministic
vMorphToFs(_,def(F),[F,agr(_)]) :-		% defective: modal auxiliares
	!.
vMorphToFs(_,[],[]).				% infinitival form
vMorphToFs(Base,Form,Features) :-
	\+ irregular(Base,Form),
	tns(Form,F1),
	agr(Form,F2),
	append(F1,F2,Features).

% base form is irregular: (regular [] -> past(-) not 3sg) 
irregular(Base,[]) :- irregular(Base).

% verb form <-> AGR feature
agr([],[]).
agr(o,[agr([1,sg,[]])]).
agr(s,[agr([2,sg,[]])]).
agr(e,[agr([3,[sg,m],[]])]).
agr(a,[agr([3,[sg,m],[]])]).
agr(amos,[agr([1,pl,[]])]).
agr(emos,[agr([1,pl,[]])]).
agr(imos,[agr([1,pl,[]])]).
agr(is,[agr([2,pl,[]])]).
agr(is,[agr([2,pl,[]])]).
agr(is,[agr([2,pl,[]])]).
agr(en,[agr([3,pl,[]])]).
agr(an,[agr([3,pl,[]])]).
%agr(ed(1),[agr(_)]).				
agr(ando,[]).
agr(ed(2),[]).					
agr(soy,[agr([1,sg,[]])]).
agr(eres,[agr([2,sg,[]])]).
agr(es,[agr([3,[sg,m],[]])]).
agr(somos,[agr([1,pl,[]])]).
agr(sos,[agr([2,pl,[]])]).
agr(son,[agr([3,pl,[]])]).
agr(he,[agr([1,sg,[]])]).
agr(has,[agr([2,sg,[]])]).
agr(ha,[agr([3,[sg,m],[]])]).
agr(han,[agr([3,pl,[]])]).
%agr(was,[agr([[1,3],[sg,m],[m,f,n]])]).
%agr(were,[agr(not([[1,3],[sg,m],[m,f,n]]))]).

% verb form <-> Tense feature
tns([],[]).
tns(o,[past(-)]).
tns(s,[past(-)]).
tns(e,[past(-)]).
tns(a,[past(-)]).
tns(amos,[past(-)]).
tns(emos,[past(-)]).
tns(imos,[past(-)]).
tns(is,[past(-)]).
tns(is,[past(-)]).
tns(s,[past(-)]).
tns(en,[past(-)]).		
tns(an,[past(-)]).		
tns(ed(1),[past(+)]).	
tns(ando,[]).			
tns(ed(2),[]).			
tns(soy,[past(-)]).
tns(eres,[past(-)]).		
tns(es,[past(-)]).
tns(somos,[past(-)]).
tns(sos,[past(-)]).
tns(son,[past(-)]).							
tns(he,[past(-)]).
tns(has,[past(-)]).
tns(ha,[past(-)]).
tns(han,[past(-)]).
tns(was,[past(+)]).		
tns(were,[past(+)]).

%%% PREDICATES FOR INDIVIDUAL LEXICAL ITEMS

agrConstraint(X) :- intersectAGR([[],[],[n]],X) if cat(X,np).

inf(V,er) :- V has_feature morph(_,er).
inf(V,ar) :- V has_feature morph(_,ar).
inf(V,ir) :- V has_feature morph(_,ir).

inf(V,ando) :- V has_feature morph(_,ando).

% Infinitival verbs <=> morph(_,[])
	
verbInf(C,Fs,Fsp) :-
	C == v
    ->  (in1(morph(_,er),Fs)
     ;  in1(morph(_,ir),Fs)
     ;  in1(morph(_,ar),Fs)),
	Fsp = [inf([ar,er,ir])|Fs]
    ;   Fsp = Fs.

% All object clitics have these lexical properties
%
% NB. 
% 1. Must ensure that the noCasemark(+) feature gets to the real predicate
% 2. Must disallow "juan ha lo visto" i.e. prevent clitic from 
%    attaching to a perfective verb.

objCliticFeatures([adjoin(v,left), argument,
	adjR([not(feature(morph(_,ed(2)))),not(feature(inf(_))),
	addFeature(transparent(ecp)),
	goal(propagateNoCase(noCasemark(+),X),X)])]).

objCliticFeatures([adjoin(v,right), argument,
	adjR([feature(inf(_)),
	addFeature(transparent(ecp)),
	goal(propagateNoCase(noCasemark(+),X),X)])]).

iobjCliticFeatures([adjoin(v,left),indirectObject, argument,
	adjR([not(feature(morph(_,ed(2)))),not(feature(inf(_))),
	addFeatures([transparent(ecp),indCl])])]).

iobjCliticFeatures([adjoin(v,right),indirectObject, argument,
	adjR([feature(inf(_)),
	addFeatures([transparent(ecp),indCl])])]).

propagateNoCase(NoCaseFeature,X) :-
	X has_feature subcat(YP$Fs,AFs)
       ->  AFsp = [NoCaseFeature|AFs],
	   addFeature(subcat(YP$Fs,AFsp),X)
       ; addFeature(NoCaseFeature,X).		
   
   

	
	
	













