%%%  -*- Mode: PROLOG; Package: PROLOG-USER -*-

%%% SAMPLE JAPANESE LEXICON
%%%
%%% (c) 1991, 1992, 1993  Sandiway Fong, NEC Research Institute, Inc.
%%%

%%% EXPORT
%%%	term(C)				terminals
%%%	lexicon(Word,C,Fs)		Word has category label C 
%%%					and feature list Fs
%%%	probeLexicon(Word)		Word is in lexicon
%%%	vMorphToFs(Base,Form,Features)	TNS/AGR features
%%%	inf(Verb,Type)
%%%	relevant(C)			constraints imposed by markers 
%%%					apply to C
%%% REFERENCES
%%%	optWhComplement(X)		xbar
%%%	(list processing)		utilities

term(n). term(v). term(a). term(p). term(c). term(adv). term(det). term($).
term(nq).

:- dynamic edrJap:jlookup/4.

lexicon(Word,C,Fs) :-
	nonvar(Word)
	-> (probeLexicon(Word)
	   -> builtin(Word,C,Fs)
	   ;  edrJap:jlookup(Word,_,C,Fs))
	;  builtin(Word,C,Fs).


/**** ORIGINAL CODE

% lexicon(Word,C,Fs) :- lex(Word,C,Fs).		% directly available
% lexicon(Form,v,Fs) :-				% non-base verb forms
%	lex(Form,v,Base,F1),
%	verbFeatures(Base,F2),
%	append1(F1,F2,Fs).

**** END ORIGINAL */

probeLexicon(Word) :- lex(Word,_,_) ; lex(Word,_,_,_).

builtin(Word,C,Fs) :- lex(Word,C,Fs).		% directly available
builtin(Form,v,Fs) :-				% non-base verb forms
	lex(Form,v,Base,F1),
	verbFeatures(Base,F2),
	append1(F1,F2,Fs).
%% nouns

% Wh-nouns

lex(nani,n,[a(-),p(-),agr([[],[],n]),wh,grid([],[]),k(b2bf)]). % what
lex(nan,n,[a(-),p(-),agr([[],[],n]),wh,grid([],[]),k(b2bf)]). % what
lex(dare,n,[a(-),p(-),agr([[],[],[m,f]]),wh,grid([],[]),k(c3af)]). % who
lex(dareka,n,[a(-),p(-),agr([[],[],[m,f]]),op(+),grid([],[]),k(c3afa4ab)]). % someone
lex(daremo,n,[a(-),p(-),agr([[],[],[m,f]]),op(+),grid([],[]),k(c3afa4e2)]). % everyone
lex(doko,n,[a(-),p(-),agr([[],[],n]),wh,grid([],[]),k(a4c9a4b3)]). % where

% People
% chomsky
lex(chomsky,n,[a(-),p(-),agr([3,sg,m]),grid([],[]),class(person),k(a5c1a5e7a5e0a5b9a5ada1bc)]).
lex(hanako,n,[a(-),p(-),agr([3,sg,f]),grid([],[]),class(person),k(b2d6bbd2)]). % mary
lex(biru,n,[a(-),p(-),agr([3,sg,m]),grid([],[]),class(person),k(a5d3a5eb)]). % bill
lex(bill,n,[a(-),p(-),agr([3,sg,m]),grid([],[]),class(person)]).
lex(masao,n,[a(-),p(-),agr([3,sg,f]),grid([],[]),class(person),k(c0b5c3cb)]). % male name (v. ambiguous kanji)
lex(mearii,n,[a(-),p(-),agr([3,sg,f]),grid([],[]),class(person),k(a5e1a5a2a5eaa1bc)]). % mary
lex(mary,n,[a(-),p(-),agr([3,sg,f]),grid([],[]),class(person)]). 
lex(taroo,n,[a(-),p(-),agr([3,sg,m]),grid([],[]),class(person),k(c2c0cfba)]). % john
lex(john,n,[a(-),p(-),agr([3,sg,m]),grid([],[]),class(person)]).

% Pronouns and Anaphors

lex(kare,n,[agr([3,sg,m]),grid([],[]),notVar,k(c8e0),goal(a(A),inv_plus_minus(A,P)),p(P)]).

lex(kimi,n,[a(-),p(+),agr([2,sg,[m,f]]),grid([],[]),k(b7af)]). % you
lex(karera,n,[a(-),p(+),agr([3,pl,[m,f]]),grid([],[]),k(c8e0a4e9)]). % they

lex(soitsu,n,[a(-),p(+),agr([[],[],m]),grid([],[]),k(a4bda4a4a4c4)]). % (that) guy 
lex(soitsu,n,[a(+),p(-),agr([[],[],m]),grid([],[]),k(a4bda4a4a4c4)]). 
lex(watashi,n,[a(-),p(+),agr([1,sg,[m,f]]),grid([],[]),k(bbe4)]). % I

lex(otagai,n,[a(+),p(-),agr([3,pl,[m,f]]),grid([],[]),k(a4aab8dfa4a4)]). 
% each other
lex(zibunzisin,n,[a(+),p(-),agr([3,sg,[m,f]]),grid([],[]),k(bcabcaacbcabbfc8)]).
% him/herself

% Nouns taking objects
% time (about)
lex(koro,n,[a(-),p(-),agr([[],[],n]),grid([],[reducedProp]),k(baa2)]). 
% fact (hiragana: a4b3a4c8)
lex(koto,n,[a(-),p(-),agr([[],[],n]),grid([],[reducedProp]),k(bbf6)]).

% Common nouns
lex(kore,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a4b3a4ec)]). % this one
lex(sore,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a4bda4ec)]). % that one (1)
lex(are,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a4a2a4ec)]). % that one (2)
lex(aisukuriimu,n,[a(-),p(-),agr([[],[],n]),grid([],[]),
	k(a5a2a5a4a5b9a5afa5eaa1bca5e0)]). % ice-cream
lex(barriers,n,[a(-),p(-),agr([[],[],n]),grid([possessor],[]),
		k(a1d6a5d0a5eaa5e4a1bca5baa1d7)]).  % barriers
lex(gengogaku,n,[a(-),p(-),agr([[],[],n]),grid([],[]),
		 k(b8c0b8ecb3d8)]). % linguistics
lex(gakusei,n,[a(-),p(-),agr([[],[],[m,f]]),grid([],[]),
	       class(person),k(b3d8c0b8)]). % student
lex(gakkoo,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(b3d8b9bb)]). % school
lex(gohan,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(b8e6c8d3)]). % boiled rice, meal
lex(haha,n,[a(-),p(-),agr([3,sg,f]),grid([possessor],[]),
	    class(person),k(caec)]). % mother
lex(hahaoya,n,[a(-),p(-),agr([3,sg,f]),grid([possessor],[]),
 	       class(person),k(caecbfc6)]). % mother
lex(hito,n,[a(-),p(-),agr([[],[],[m,f]]),grid([],[]),
	    class(person),k(bfcd)]). % man
lex(hon,n,[a(-),p(-),agr([[],[],n]),grid([],[]),
	   class(volume),k(cbdc)]). % book
lex(kami,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(c8b1)]). % hair
lex(kubi,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(bcf3)]). % neck/throat
lex(menyuu,n,[a(-),p(-),agr([[],[],n]),grid([],[]),
	      k(a5e1a5cba5e5a1bc)]).  % menu
lex(mura,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(c2bc)]).  % village
lex(neko,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(c7ad)]). % cat
lex(okaasan,n,[a(-),p(-),agr([[],[],f]),grid([possessor],[]),
	       class(person),k(a4aacaeca4b5a4f3)]). % mother
lex(okane,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a4aab6e2)]). % money
lex(nezumi,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(c1cd)]). % rat
lex(piza,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a5d4a5b6)]). % pizza
lex(sensei,n,[a(-),p(-),agr([[],[],[m,f]]),grid([possessor],[]),
	      class(person),k(c0e8c0b8)]). % teacher
lex(roka,n,[a(-),p(-),agr([]),grid([],[])]).	  % Roka
lex(sake,n,[a(-),p(-),agr([[],[],n]),class(hon),
	    grid([],[]),k(bcf2)]).	  % sake/alcohol
lex(table,n,[a(-),p(-),agr([[],[],n]),grid([],[]),
	     k(a5c6a1bca5d6a5eb)]). % table
lex(te,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(bcea)]). % hand
lex(tegami,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(bceabbe6)]). % letter

lex(tiizu,n,[a(-),p(-),agr([[],[],n]),grid([],[]),k(a5c1a1bca5ba)]). % cheese

lex(ue,n,[a(-),p(-),agr([[],[],n]),grid([theme],[]),k(bee5)]). % top

% determiners
lex(kono,det,[k(a4b3a4ce)]).	% this
lex(sono,det,[k(a4bda4ce)]).	% that (1)
lex(ano,det,[k(a4a2a4ce)]).	% that (2)
lex(dono,det,[wh,k(a4c9a4ce)]).	% which

%% verbs
%% stem forms - inconsistent

lex(itta,v,[allowExt(goal),
	    oneof(_,[[grid([agent],[]),morph(ik,past(+)),k(b9d4a4c3a4bf)],
		     [grid([agent],[proposition1]),morph(i,past(+)),k(b8c0a4c3a4bf)]])]).

% give s.t. (to s.o.)
lex(ager,v,[morph(ager,[]),grid([agent],[theme]),allowExt(goal)]).
lex(aisur,v,[morph(aisur,[]),grid([agent],[patient])]).
lex(a,v,[morph(a,[]),grid([agent],[]),allowExt(goal)]). % meet (with)
lex(hihans,v,[morph(hihans,[]),grid([agent],[patient])]). % criticize
lex(d,v,[morph(d,[]),aux,subcat(ap$[grid(_,_)],[])]). % copula
lex(ir,v,[morph(ir,[]),aux,prog,subcat(vp$[morph(_,te)],[])]). % be (prog.)
% get s.t. (using s.t.)
lex(irer,v,[morph(irer,[]),grid([agent],[themeNA]),allowExt(goal)]).
% say s.t.(to s.o.)
lex(i,v,[morph(i,[]),grid([agent],[proposition1]),allowExt(goal)]). 
lex(ik,v,[morph(ik,[]),grid([agent],[]),allowExt(goal)]). % go
lex(kak,v,[morph(kak,[]),grid([agent],[themeNA])]). % write
lex(kar,v,[morph(kar,[]),grid([agent],[theme])]). % keep
lex(kau,v,[morph(kau,[]),grid([agent],[theme])]). % buy
lex(kirawa,v,[morph(kirau,neg),grid([agent],[theme]),neg,k(a4ada4e9a4ef)]). % dislike
lex(kur,v,[morph(kur,[]),subcat(vp$[morph(_,te)],[])]). % come (conj).
lex(koru,v,[morph(koru,[]),grid([agent],[theme])]). % kill
lex(kusar,v,[morph(kusar,[]),grid([theme],[])]). % rotten
lex(mir,v,[morph(mir,[]),grid([agent],[theme])]). % see
lex(mora,v,[morph(mora,[]),grid([recipient],[theme]),allowExt(source)]). % receive
lex(mots,v,[morph(mots,[]),grid([agent],[theme])]). % have
lex(nagur,v,[morph(nagur,[]),grid([agent],[patient])]). % beat
lex(nar,v,[morph(nar,[]),grid([experiencer],[goal]),doCase(dat)]). % become
lex(ner,v,[morph(ner,[]),grid([theme],[])]).	  % sleep
lex(nemur,v,[morph(nemur,[]),grid([theme],[])]).  % to go to bed
lex(nom,v,[morph(nom,[]),grid([agent],[themeNA])]). % drink s.t.
lex(oi,v,[morph(oi,[]),grid([agent],[theme]),allowExt(goal)]). % put
% be angry about s.t.
lex(okor,v,[morph(okor,[]),grid([experiencer],[proposition])]). 
%lex(okur,v,[morph(okur,[]),grid([agent],[patient]),allowExt(goal)]). % send
lex(okur,v,[morph(okur,[]),grid([agent],[patient,goal]),idoCase(dat)]). % send
lex(omo,v,[morph(omo,[]),grid([agent],[proposition])]).	% think
lex(rare,v,[morph(rareru,[]),subcat(vp$[morph(_,[])],[noCasemark(+)]),
	    passive]). % passive 
lex(re,v,[morph(rareru,[]),subcat(vp$[morph(_,neg)],[noCasemark(+)]),
	  passive]).	% passive 
lex(sagas,v,[morph(sagas,[]),grid([agent],[theme])]). % search for s.t.
lex(shir,v,[morph(shir,[]),grid([theme],[proposition])]).  % know
lex(suisen,v,[morph(suisen,[]),grid([agent],[theme])]).    % recommend
lex(sukinina,v,[morph(sukinina,[]),grid([agent],[patient])]). % fall in love with
%lex(su,v,[morph(su,[]),grid([agent],[]),allowExt(goal)]). % reside s.w.
lex(su,v,[morph(su,[]),grid([agent],[goal]),doCase(dat)]). % reside s.w.
%lex(shookai,v,[morph(shookai,[]),grid([agent],[patient]),allowExt(goal),
%	       k(bed2b2f0)]). % introduce s.o
lex(shookai,v,[morph(shookai,[]),grid([agent],[patient,goal]),idoCase(dat),k(bed2b2f0)]). % introduce s.o
lex(tabe,v,[morph(taberu,[]),grid([agent],[theme]),k(bfa9)]). % eat
%lex(watas,v,[morph(watas,[]),grid([agent],[theme]),allowExt(goal)]). % give st.
lex(watas,v,[morph(watas,[]),grid([agent],[theme,goal]),idoCase(dat)]). % give st.
lex(yom,v,[morph(yom,[]),grid([agent],[themeNA])]). % read (s.t.)

% complementizers
lex(ka,c,[wh,yn,k(a4ab)]).			  % Q
lex(ka_doo_ka,c,[wh,merge(yn),k(a4aba4c9a4a6a4ab)]).
						  % Q - yes/no (whether or not)
lex(no,c,[wh,yn,k(a4ce),left(v,[],[])]).			  % only in matrix clause
lex(to,c,[selR([not(feature(inf(_)))]),k(a4c8)]).	  % that
lex(tte,c,[k(a4c3a4c6)]).			  % that

% adverbs
lex(doo,adv,[adjoin(left),predicate(manner),wh,k(a4c1a4a6)]).	% how
lex(itsu,adv,[adjoin(left),predicate(time),wh]).                % when
lex(naze,adv,[adjoin(left),predicate(reason),wh,k(a4caa4bc)]). 	
						  % why (kanji b2bfb8ce)
lex(kinoo,adv,[adjoin(left),predicate(time),k(baf2c6fc)]).	% yesterday

% V-adj
lex(hitome,adv,[degree,k(b0ecccdc)]).		  % one glance
lex(sonnani,adv,[degree,k(a4bda4f3a4caa4cb)]).	  % so much
lex(mada,adv,[degree,k(a4dea4c0)]).		  % +/- still/yet

% adjectives
lex(nagai,a,[grid([theme],[]),k(a4caa4aca4a4)]).  % long (kanji c4b9a4a4)
lex(shizuka,a,[grid([theme],[]),k(c0c5a4ab)]).	  % quiet (na)
lex(genki,a,[grid([theme],[]),k(c0c5a4ab)]).	  % well
lex(tensai,a,[grid([theme],[]),k(c5b7bacd)]).	  % genius

% postpositions

lex(de,p,[grid([],[location]),transparent,predicate(location),k(a4c7)]).  % at

lex(kara,p,[grid([],[source]),transparent,predicate(source),k(a4aba4e9),	% from
	  adjR([goal(vpAllowExt(source,X),X)])]).

% NB. Goal (.=. Location) (or Time - not added yet)
lex(ni,p,[grid([],[role(Role)]),transparent,predicate(Role),k(a4cb),
	  adjR([goal(externalRolesForNi(X,Role),X)])]).

lex(maeni,p,[grid([],[reducedProp]),predicate(time),k(c1b0a4cb)]). % before
lex(to,p,[grid([],role(Role)),conj,k(a4c8),adjR([feature(theta(Role))])]). % and

% markers
lex(ta,mrkr,[left(v,morph(_,past(+)),[]),k(a4bf)]).
lex(wa,mrkr,[left(n,[],morphC(topic)),k(a4cf)]).
lex(ga,mrkr,[left(n,[],morphC(nom)),k(a4ac)]).
lex(o,mrkr,[left(n,[],morphC(acc)),k(a4f2)]).
lex(wo,mrkr,[left(n,[],morphC(acc)),k(a4f2)]).	  % orthography
lex(no,mrkr,[left(n,[],morphC(gen)),k(a4ce)]).
lex(ni,mrkr,[left(n,[],morphC(dat)),k(a4f2)]).


% NQs

% default
lex(hitotsu,nq,[number(1),classifier(default),k(a3b1a4c4)]).
lex(futatsu,nq,[number(2),classifier(default),k(a3b2a4c4)]).
lex(mittsu,nq,[number(3),classifier(default),k(a3b3a4c4)]).
lex(yottsu,nq,[number(4),classifier(default),k(a3b4a4c4)]).
lex(itsutsu,nq,[number(5),classifier(default),k(a3b5a4c4)]).
lex(muttsu,nq,[number(6),classifier(default),k(a3b6a4c4)]).
lex(nanatsu,nq,[number(7),classifier(default),k(a3b7a4c4)]).
lex(yattsu,nq,[number(8),classifier(default),k(a3b8a4c4)]).
lex(kokonotsu,nq,[number(9),classifier(default),k(a3b9a4c4)]).
lex(too,nq,[number(10),classifier(default),k(a3ba)]).

% person
lex(hitori,nq,[number(1),classifier(person),k(a3b1bfcd)]).
lex(futari,nq,[number(2),classifier(person),k(a3b2bfcd)]).
lex(sannin,nq,[number(3),classifier(person),k(a3b3bfcd)]).
lex(yonin,nq,[number(4),classifier(person),k(a3b4bfcd)]).
lex(gonin,nq,[number(5),classifier(person),k(a3b5bfcd)]).
lex(rokunin,nq,[number(6),classifier(person),k(a3b6bfcd)]).
lex(shichinin,nq,[number(7),classifier(person),k(a3b6bfcd)]).
lex(nananin,nq,[number(7),classifier(person),k(a3b7bfcd)]).
lex(hachinin,nq,[number(8),classifier(person),k(a3b8bfcd)]).
lex(kyuunin,nq,[number(9),classifier(person),k(a3b9bfcd)]).
lex(kunin,nq,[number(9),classifier(person),k(a3b9bfcd)]).
lex(juunin,nq,[number(10),classifier(person),k(a3babfcd)]).

% volume
lex(issatsu,nq,[number(1),classifier(volume),k(a3b1bafd)]).
lex(nisatsu,nq,[number(2),classifier(volume),k(a3b2bafd)]).
lex(sansatsu,nq,[number(3),classifier(volume),k(a3b3bafd)]).
lex(yonsatsu,nq,[number(4),classifier(volume),k(a3b4bafd)]).
lex(gosatsu,nq,[number(5),classifier(volume),k(a3b5bafd)]).
lex(rokusatsu,nq,[number(6),classifier(volume),k(a3b6bafd)]).
lex(shichisatsu,nq,[number(7),classifier(volume),k(a3b6bafd)]).
lex(nanasatsu,nq,[number(7),classifier(volume),k(a3b7bafd)]).
lex(hassatsu,nq,[number(8),classifier(volume),k(a3b8bafd)]).
lex(kyuusatsu,nq,[number(9),classifier(volume),k(a3b9bafd)]).
lex(jussatsu,nq,[number(10),classifier(volume),k(a3babafd)]).

% long object
lex(ippon,nq,[number(1),classifier(hon),k(a3b1cbdc)]).
lex(nihon,nq,[number(2),classifier(hon),k(a3b2cbdc)]).
lex(sanbon,nq,[number(3),classifier(hon),k(a3b3cbdc)]).
lex(yonhon,nq,[number(4),classifier(hon),k(a3b4cbdc)]).
lex(gohon,nq,[number(5),classifier(hon),k(a3b5cbdc)]).
lex(roppon,nq,[number(6),classifier(hon),k(a3b6cbdc)]).
lex(shichihon,nq,[number(7),classifier(hon),k(a3b6cbdc)]).
lex(nanahon,nq,[number(7),classifier(hon),k(a3b7cbdc)]).
lex(hachihon,nq,[number(8),classifier(hon),k(a3b8cbdc)]).
lex(happon,nq,[number(8),classifier(hon),k(a3b8cbdc)]).
lex(kyuhon,nq,[number(9),classifier(hon),k(a3b9cbdc)]).
lex(juppon,nq,[number(10),classifier(hon),k(a3bacbdc)]).


lexFeature(morphC(_),n).

% relevant for marker constraints
relevant(n).  relevant(v). relevant(p).

% non-base forms

lex(ageta,v,ager,[morph(ager,past(+)),k(a4a2a4b2a4bf)]).
lex(aishite,v,aisur,[morph(aisur,te),k(b0a6a4b7a4c6)]).

lex(au,v,a,[morph(a,past(-)),k(b2f1a4a6)]).
lex(atta,v,a,[morph(a,past(+)),k(b2f1a4c3a4bf)]).
lex(aitagatte,v,a,[morph(a,te),aux(signs(want)),k(b2f1a4a4a4bfa4aca4c3a4c6)]).

lex(da,v,d,[morph(d,past(-)),k(a4c0)]).
lex(desu,v,d,[morph(d,past(-)),formal,k(a4c7a4c4)]).
lex(datta,v,d,[morph(d,past(+)),k(a4c0a4c3a4bf)]).
lex(deshita,v,d,[morph(d,past(+)),formal,k(a4c7a4b7a4bf)]).

lex(hihanshita,v,hihans,[morph(hihans,past(+)),k(c8e3c8bda4b7a4bf)]).
lex(ireta,v,irer,[morph(irer,past(+)),k(c6fea4eca4bf)]).

lex(iru,v,ir,[morph(ir,past(-)),k(a4a4a4eb)]).
lex(ita,v,ir,[morph(ir,past(+)),k(a4a4a4bf)]).
lex(inai,v,ir,[morph(ir,npast(-)),neg,k(a4a4a4caa4a4)]).

lex(iku,v,ik,[morph(ik,past(+)),k(b9d4a4af)]).
% lex(itta,v,ik,[morph(ik,past(+)),k(b9d4a4c3a4bf)]).

lex(iu,v,i,[morph(i,past(-)),k(b8c0a4a6)]).
% lex(itta,v,i,[morph(i,past(+)),k(b8c0a4c3a4bf)]).

lex(kaita,v,kak,[morph(kak,past(+)),k(bdf1a4a4a4bf)]).

lex(katta,v,kau,[morph(kau,past(+)),k(c7e3a4c3a4bf)]).

lex(katte,v,kar,[morph(kar,te),k(bbf4a4c3a4c6)]).

lex(kite,v,kur,[morph(kur,te),k(cde8a4c6)]).

lex(kuru,v,kur,[morph(kur,past(-)),k(cde8a4eb)]).

lex(kita,v,kur,[morph(kur,past(+)),k(cde8a4bf)]).

lex(koroshita,v,koru,[morph(koru,past(+)),k(bba6a4b7a4bf)]).
lex(kusatte,v,kusar,[morph(kusar,te),k(c9e5a4c3a4c6)]).

lex(mita,v,mir,[morph(mir,past(+)),k(b8aba4bf)]).

lex(morau,v,mora,[morph(mora,past(-)),k(cce3a4a6)]).
lex(moratta,v,mora,[morph(mora,past(+)),k(cce3a4c3a4bf)]).

lex(motsu,v,mots,[morph(mots,past(-)),k(bbfda4c4)]).
lex(motte,v,mots,[morph(mots,te),k(bbfda4c3a4c6)]).
lex(motta,v,mots,[morph(mots,past(+)),k(bbfda4c3a4bf)]).

lex(naguru,v,nagur,[morph(nagur,past(-)),k(b2a5a4eb)]).
lex(nagutta,v,nagur,[morph(nagur,past(+)),k(b2a5a4c3a4bf)]).

lex(natta,v,nar,[morph(nar,past(+)),k(a4caa4c3a4bf)]).

lex(nemuru,v,ner,[morph(nemur,past(-)),k(ccb2a4eb)]).

lex(neru,v,ner,[morph(ner,past(-)),k(bfb2a4eb)]).

lex(nonda,v,nom,[morph(nom,past(+)),k(c6c9a4f3a4c0)]).
lex(nonde,v,nom,[morph(nom,te),k(c6c9a4f3a4c7)]).
lex(nomimashita,v,nom,[morph(nom,past(+)),k(c6c9a4dfa4dea4b7a4bf)]).

lex(oita,v,oi,[morph(oi,past(+)),k(c3d6a4a4a4bf)]).

lex(okotte,v,okor,[morph(okor,te),k(c5dca4c3a4c6)]).

lex(okuru,v,okur,[morph(okur,past(-)),k(c1f7a4eb)]).
lex(okutta,v,okur,[morph(okur,past(-)),k(c1f7a4c3a4bf)]).

lex(omou,v,omo,[morph(omou,past(-)),k(bbd7a4a6)]).
lex(omotta,v,omo,[morph(omou,past(+)),k(bbd7a4c3a4bf)]).
lex(omotte,v,omo,[morph(omou,te),k(bbd7a4c3a4c6)]).
lex(omoikonde,v,omo,[morph(omoikomu,te),k(bbd7a4a4b9fea4f3a4c7)]). 

lex(rareru,v,rare,[morph(rareru,past(-)),k(a4b1a4eca4eb)]).
lex(rareta,v,rare,[morph(rareru,past(+)),k(a4b1a4eca4bf)]).
lex(reru,v,re,[morph(rareru,past(-)),k(a4eca4eb)]).
lex(reta,v,re,[morph(rareru,past(+)),k(a4eca4bf)]).
lex(rete,v,re,[morph(rareru,te),k(a4eca4c6)]).

lex(sagashite,v,sagas,[morph(sagas,te),k(c3b5a4b7a4c6)]).

lex(shitte,v,shir,[morph(shir,te),k(c3cea4c3a4c6)]).
lex(shiranai,v,shir,[morph(shir,npast(-)),neg,k(c3cea4e9a4caa4a4)]).

lex(shookaisuru,v,shookai,[morph(shookai,past(-)),k(bed2b2f0a4b9a4bf)]).
lex(shookaishita,v,shookai,[morph(shookai,past(+)),k(bed2b2f0a4b7a4bf)]).

lex(suisenshita,v,suisen,[morph(suisen,past(+)),k(bfe4c1a6a4b7a4bf)]).

lex(sukininaru,v,sukinina,[morph(sukinina,past(-)),k(b9a5a4adaa4cba4ca4eb)]).
lex(sukininatta,v,sukinina,[morph(sukinina,past(+)),k(b9a5a4ada4cba4caa4c3a4bf)]).

lex(sunde,v,su,[morph(su,te),k(bdbba4c7)]).
lex(sumu,v,su,[morph(su,past(-)),k(bdbba4e0)]).

lex(taberu,v,tabe,[morph(taberu,past(-)),k(bfa9a4d9a4eb)]).
lex(tabeta,v,tabe,[morph(taberu,past(+)),k(bfa9a4d9a4bf)]).

lex(watashita,v,watas,[morph(watas,past(+)),k(c5cfa4b7a4bf)]).

lex(yomu,v,yom,[morph(yom,past(-)),k(c6c9a4e0)]).
lex(yonda,v,yom,[morph(yom,past(+)),k(c6c9a4f3a4c0)]).
lex(yonde,v,yom,[morph(yom,te),k(c6c9a4f3a4c7)]).
lex(yomimashita,v,yom,[morph(yom,past(+)),k(c6c9a4dfa4dea4b7a4bf)]).

% CONTRACTIONS

contraction(X+iru,[X=morph(_,te),iru]).
contraction(X+rareta,[X=morph(_,[]),rareta]).
contraction(X+rareru,[X=morph(_,[]),rareru]).
contraction(X+reta,[X=morph(_,neg),reta]).
contraction(X+rete,[X=morph(_,neg),rete]).

:- initialization(no contraction(_,_,_)).

% MISCELLANEOUS

verbalize([Y|s]) :- lex(Y,n,_).

nounFeatures(Base,F) :-	lex(Base,n,F1),	pick1(agr(_),F1,F).
verbFeatures(Base,F) :- 
	lex(Base,v,F1), 
	pick1(morph(_,_),F1,F2),
	pickNF1(k(_),F2,F).

%% MAPS MORPHOLOGY INTO SYNTACTIC FEATURES
%%	Verb morphology and Agreement
%% 
%%	Form	Tense		AGR
%%	base 	infinitival
%%	te	
%%	past(+) past(+)		agr(_)
%%	past(-) past(-)		agr(_)

vMorphToFs(_,Form,Features) :-
	formToFeatures(Form,Features).

formToFeatures([],[]).
formToFeatures(neg,[]).
formToFeatures(te,[]).
formToFeatures(past(X),[past(X),agr(_)]).
formToFeatures(npast(X),[past(X),agr(_)]).
formToFeatures(prog,[prog,past(-),agr(_)]).

% verb morphology
% iru1 (exists) -> ite
% iru2 (prog) -> ite
% iu (say) -> itte
% iru3 (need) -> itte

:- dynamic inf/2.

% Head movement and negation

negFromV(Neg,V) :-
	V has_feature neg,
	mkFs([index(_),polarity(-)],Fs),
	mkEC(neg,Fs,Neg).

