%%%  -*- Mode: PROLOG; Package: PROLOG-USER -*-

%%% SAMPLE FRENCH LEXICON
%%%
%%% (c) 1991, 1992, 1993  Sandiway Fong, NEC Research Institute, Inc.
%%%     1994  Martine Smets

%%% EXPORT
%%%	term(C)				terminals
%%%	lexicon(Word,C,Fs)		Word has category label C
%%%					and feature list Fs
%%%	probeLexicon(Word)		Word is in lexicon
%%%	vMorphToFs(Base,Form,Features)	TNS/AGR features
%%%	inf(Verb,Type)
%%%	relevant(C)			constraints imposed by markers
%%%					apply to C
%%% REFERENCES
%%%	(list processing)		utilities
%%%	ensureWhComp(Ci)		lex
%%%	optWhComp(Ci)			

term(n). term(a). term(v). term(p). term(c). 
term(adv).  term(det). %term(conj).
term(neg). term(mrkr). term('$'). term(cl).

%%% 	Most lexical entries are stored directly as
%%%
%%%		 lex(Word,Category,Features)
%%%
%%%	Non-base forms require inference: 
%%%	1. plural nouns		all features except agr(_) inherited
%%%				from the sg. form
%%%	2. nominalized verbs	inherits verb features except morph(_,_)
%%%	3. non-base verb forms	all features except morph(_,_) inherited
%%%				from the base form

lexicon(Word,C,Fs) :-				% directly available
	lex(Word,C,Fs1),
	verbInf(C,Fs1,Fs).
lexicon(Form,v,Fs) :-				% non-base verb forms
	lex(Form,v,Base,F1),
	verbFeatures(Base,F2),
	append1(F1,F2,Fs).

probeLexicon(Word) :- lex(Word,_,_) ; lex(Word,_,_,_).

%%% NEGATION

lex(pas,neg,[polarity(negative),transparent(ecp)]).
% lex(personne,neg,[polarity(negative),transparent(ecp)]).
lex(ne,cl,[polarity(negative)]).
lex(plus,neg,[polarity(negative)]).

%%% PREPOSITIONS
lex(au_sujet_de,p,[grid([],[theme]),predicate(theme),  % about
	     adjR([goal(roleNotPresent(X,theme),X)])]).
lex(de,p,[grid([],[source]),predicate(source),          % from
	  adjR([goal(vpAllowExt(source,X),X)])]).
lex(aprs,p,[grid([],[reducedProp]),predicate(manner)]). % after
lex(_cause_de,p,[grid([],[reason]),predicate(reason)]). % because of
lex(parce,p,[grid([],[proposition1]),predicate(reason)]). % because
lex(contre,p,[grid([],[patient]),predicate(patient)]).      % against
lex(avant,p,[grid([],[reducedProp]),predicate(manner)]).  % before
lex(par,p,[grid([],[agent]),adjR([goal(unsaturatedExtRole(X,agent),X)]), % by
	  predicate(agent)]).
lex(dans,p,[grid([],[location]),predicate(location)]).   % in
lex(en,p,[grid([],[location]),predicate(location)]).     % in
lex(sous,p,[grid([],[location]),predicate(location)]).   % under
lex(sans,p,[grid([],[reducedProp]),predicate(manner)]).  % without
lex(avec,p,[grid([],[patient]),predicate(patient)]).     % with
lex(sur,p,[grid([],[location]),predicate(location)]).    % on
lex(,p,[grid([],[destination]),predicate(destination), % to
	  adjR([goal(vpAllowExt(destination,X),X)])]).
lex(,p,[grid([],[patient]),predicate(patient),  % .. at something/body
	  adjR([goal(vpAllowExt(patient,X),X)])]).


%% Adverbs

lex(comment,adv,[adjoin(right),wh,predicate(manner)]).
lex(quand,adv,[adjoin(right),wh,predicate(time)]). % when
lex(o,adv,[adjoin(right),wh,predicate(location)]). % when
lex(pourquoi,adv,[adjoin(right),wh,predicate(reason)]).
lex(compltement,adv,[adjoin(right),predicate(manner)]).
lex(souvent,adv,[adjoin(right),predicate(manner)]).
lex(sincrement,adv,[adjoin(right),predicate(manner)]).
lex(ici,adv,[adjoin(right),predicate(location)]).
lex(tard,adv,[adjoin(right),predicate(time)]).
lex(hier,adv,[adjoin(right),predicate(time)]).
lex(y,pf(cl),[predicate(location) | Fs]) :- 
	                        advCliticFeatures(Fs).
lex(en,pf(cl),[predicate(source),adjR([feature(allowExt(source))]) | Fs]) :- 
	                                       advCliticFeatures(Fs).
lex(bien,adv,[adjoin(right),predicate(manner)]).
lex(trs,adv,[degree]).
lex(dur,adv,[degree,adjR([feature(allow(adv,degree))])]). % ... special case
lex(mal,adv,[adjoin(right),predicate(manner)]).

%% Degree Adverbs (modify adjectives)

% permit the resulting AP to take an optional clausal adjunct
lex(trop,adv,[degree,adjR([addFeatures([allowExt(reason),adjR([goal(nonfiniteClause(X),X)])])])]).

%% Determiners
%%	Combine only with nouns that have the count(+/-/_) feature (common nouns) 
%%	Six classes of determiners:
%%	Class Examples                     Features
%%	1     le,mon,quel                  count(_) agr([3,[sg,m],m])
%%	2     un_peu de, assez de          count(_) agr([3,m,[]])
%%	3     ce                           count(_) agr([3,[sg,m],m])
%%	4     peu_de,ces                   count(+) agr([3,[pl,m],[]])
%%	5     chaque                       count(+) agr(([3,sg,[]])
%%	6     beaucoup_de                  count(-) agr(([3,[pl,m],[]])
%%
%%	op(+) elements that are moved by QR and 
%%	form operator-variable structures

lex(un,det,[count(+),agr([3,sg,m])]).
lex(une,det,[count(+),agr([3,sg,f])]).
lex(des,det,[count(+),agr([3,pl,[]])]).
lex(un_certain,det,[count(+),agr([3,sg,m])]).
lex(une_certaine,det,[count(+),agr([3,sg,f])]).
lex(chaque,det,[count(+),agr([3,sg,[m,f]]),op(+)]).
lex(assez_de,det,[count(_),agr([3,[pl,m],[]]),op(+)]).
lex(peu_de,det,[count(_),agr([3,[pl,m],[]]),op(+)]).
lex(de,det,[count(_),agr([3,[],[]]),polarity(-)]).
lex(beaucoup_de,det,[count(_),agr([3,[pl,m],[]]),op(+)]).
lex(quelques,det,[count(+),agr([3,pl,[]])]).
lex(du,det,[count(-),agr([3,m,m])]).
lex(de_la,det,[count(-),agr([3,m,f])]).
%lex(un_peu_de,det,[count(-),agr([3,m,[]])]).
lex(un_peu,det,[count(-),agr([3,m,[]])]).
lex(ce,det,[count(_),agr([3,[sg,m],m])]).
lex(cette,det,[count(_),agr([3,[sg,m],f])]).
lex(le,det,[count(_),agr([3,[sg,m],m])]).
lex(la,det,[count(_),agr([3,[sg,m],f])]).
lex(les,det,[count(+),agr([3,pl,[]])]).
lex(ces,det,[count(+),agr([3,pl,[]])]).
lex(quel,det,[count(_),agr([3,[sg,m],m]),wh]).
lex(quels,det,[count(+),agr([3,pl,m]),wh]).
lex(quelle,det,[count(_),agr([3,[sg,m],f]),wh]).
lex(quelles,det,[count(+),agr([3,pl,f]),wh]).
% %lex(whose,det,[count(_),agr([3,[],[]]),wh]).%not in French
lex(mon,det,[count(_),a(-),p(+),agr([3,[sg,m],m])]).
lex(ma,det,[count(_),a(-),p(+),agr([3,[sg,m],f])]).
lex(mes,det,[count(+),a(-),p(+),agr([3,pl,[]])]).
lex(ton,det,[count(_),a(-),p(+),agr([3,[sg,m],m])]).
lex(ta,det,[count(_),a(-),p(+),agr([3,[sg,m],f])]).
lex(tes,det,[count(+),a(-),p(+),agr([3,pl,[]])]).
% lex(son,det,[count(_),a(-),p(+),agr([3,[sg,m],m])]).
% lex(son,det,[count(_),a(+),p(-),agr([3,[sg,m],m])]).
% lex(sa,det,[count(_),a(-),p(+),agr([3,[sg,m],f])]).
% lex(sa,det,[count(_),a(+),p(-),agr([3,[sg,m],f])]).
% lex(ses,det,[count(+),a(-),p(+),agr([3,pl,[]])]).
% lex(ses,det,[count(+),a(+),p(-),agr([3,pl,[]])]).
% lex(notre,det,[count(_),a(-),p(+),agr([1,[sg,m],[]])]).
% lex(nos,det,[count(+),a(-),p(+),agr([1,pl,[]])]).
% lex(votre,det,[count(_),a(-),p(+),agr([2,[sg,m],[]])]).
% lex(vos,det,[count(+),a(-),p(+),agr([2,pl,[]])]).
% lex(leur,det,[count(_),a(-),p(+),agr([3,[sg,m],[]])]).
% lex(leur,det,[count(_),a(+),p(-),agr([3,[sg,m],[]])]).
% lex(leurs,det,[count(+),a(-),p(+),agr([3,pl,[]])]).
% lex(leurs,det,[count(+),a(+),p(-),agr([3,pl,[]])]).
lex(aucun,det,[count(+),agr([3,sg,m])]).
lex(aucune,det,[count(+),agr([3,sg,f])]).
%lex(neither,det,[count(+),agr([3,sg,[]])]).

%% Nouns
%% 	Obligatory features
%% 	1. Anaphoric: a(+/-), Pronominal: p(+/-) for Binding Theory
%% 	2. Agrement: agr([Per,Num,Gen]) 
%%	3. O-grid: grid(Ext,Int) 
%% 	Optional features
%% 	1. Morphologically realized Case: morphC(C)
%% 	2. Common nouns: count(+/-/_) (not present for proper nouns)
%%	3. Quantifiers: op(+/-) 
%%	4. Wh-nouns: wh

% Pleonastic elements

%lex(there,n,[grid([],[]),agr([3,[],[]]),nonarg(+)]).
lex(il,cl,[agr([3,sg,m]),nonarg(+)|Fs]) :- subjCliticFeatures(nom,Fs).

% not sure?
%?lex(now,n,[grid([],[]),agr([3,[],[]])

% Proper nouns
lex(pierre,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(jean,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(eric,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(ric,n,[grid([],[]),a(-),p(-),agr([3,sg,m])]).
lex(marie,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).
lex(rome,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).
lex(sylvie,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).
lex(structures_syntaxiques,n,[grid([],[]),a(-),p(-),agr([3,sg,f])]).

% Anaphors and pronominals
lex(l_un_l_autre,n,[grid([],[]),a(+),p(-),agr([3,pl,m])]).
lex(l_une_l_autre,n,[grid([],[]),a(+),p(-),agr([3,pl,f])]).
%?lex(les_uns_les_autres,n,[grid([],[]),a(+),p(-),agr([3,pl,m])]).
lex(il,cl,[a(-),p(+),agr([3,sg,m])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(il,pf(cl),[a(-),p(+),agr([3,sg,m])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(le,pf(cl),[a(-),p(+),agr([3,sg,m])|Fs]) :- objCliticFeatures1(Fs).
lex(se,pf(cl),[a(+),p(-),agr([3,[sg,pl],[f,m]])|Fs]) :- 
	objCliticFeaturesR(Fs).
lex(se,pf(cl),[a(+),p(-),agr([3,[sg,pl],[f,m]])|Fs]) :- 
	iobjCliticFeaturesR(Fs).
lex(lui,pf(cl),[a(-),p(+),agr([3,sg,[f,m]]),animate|Fs]) :-
	iobjCliticFeatures1(Fs).
lex(lui,n,[grid([],[]),a(-),p(+),agr([3,sg,m]),animate,morphC(nonNom)]).
lex(lui,n,[grid([],[]),a(+),p(-),agr([3,sg,m]),animate,morphC(nonNom)]).
lex(les,pf(cl),[a(-),p(+),agr([3,pl,[m,f]])|Fs]) :-
	objCliticFeatures1(Fs).
lex(ils,cl,[a(-),p(+),agr([3,pl,m])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(ils,pf(cl),[a(-),p(+),agr([3,pl,m])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(leur,pf(cl),[a(-),p(+),agr([3,pl,[m,f]]),animate|Fs]) :-
	iobjCliticFeatures1(Fs).
lex(eux,n,[grid([],[]),a(-),p(+),agr([3,pl,m]),animate,morphC(nonNom)]).
lex(eux,n,[grid([],[]),a(+),p(-),agr([3,pl,m]),animate,morphC(nonNom)]).

lex(elle,cl,[a(-),p(+),agr([3,sg,f])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(elle,pf(cl),[a(-),p(+),agr([3,sg,f])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(elle,n,[grid([],[]),a(-),p(+),agr([3,sg,f]),animate,morphC(nonNom)]).
lex(elle,n,[grid([],[]),a(+),p(-),agr([3,sg,f]),animate,morphC(nonNom)]).
lex(la,pf(cl),[a(-),p(+),agr([3,sg,f])|Fs]) :-
	objCliticFeatures1(Fs).
lex(elles,cl,[a(-),p(+),agr([3,pl,f])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(elles,pf(cl),[a(-),p(+),agr([3,pl,f])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(elles,n,[grid([],[]),morphC(nonNom),a(-),p(+),agr([3,pl,f]),animate]).
lex(elles,n,[grid([],[]),morphC(nonNom),a(+),p(-),agr([3,pl,f]),animate]).

lex(en,pf(cl),[a(-),p(+),agr([3,[sg,pl,m],[m,f]]),vowel,nAnimate|Fs]) :-
	objCliticFeatures(Fs).
lex(y,pf(cl),[a(-),p(+),agr([3,[sg,m,pl],[m,f]]),vowel,nAnimate|Fs]) :-
	iobjCliticFeatures1(Fs).

lex(je,cl,[a(-),p(+),agr([1,sg,[m,f]])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(je,pf(cl),[a(-),p(+),agr([1,sg,[m,f]])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(me,pf(cl),[a(-),p(+),agr([1,sg,[m,f]])|Fs]) :-
	objCliticFeatures2(Fs).
lex(me,pf(cl),[a(-),p(+),agr([1,sg,[m,f]])|Fs]) :-
	iobjCliticFeatures2(Fs).
lex(me,pf(cl),[a(+),p(-),agr([1,sg,[m,f]])|Fs]) :-
	objCliticFeaturesR(Fs).
lex(me,pf(cl),[a(+),p(-),agr([1,sg,[m,f]])|Fs]) :-
	iobjCliticFeaturesR(Fs).
lex(moi,n,[grid([],[]),a(-),p(+),agr([1,sg,[]]),morphC(nonNom)]).

lex(tu,cl,[a(-),p(+),agr([2,[],[m,f]])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(tu,pf(cl),[a(-),p(+),agr([2,sg,[m,f]])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(te,pf(cl),[a(-),p(+),agr([2,[],[m,f]])|Fs]) :-
	objCliticFeatures2(Fs).
lex(te,pf(cl),[a(-),p(+),agr([2,[],[m,f]])|Fs]) :-
	iobjCliticFeatures2(Fs).
lex(te,pf(cl),[a(+),p(-),agr([2,[],[m,f]])|Fs]) :-
	objCliticFeaturesR(Fs).
lex(te,pf(cl),[a(+),p(-),agr([2,[],[m,f]])|Fs]) :-
	iobjCliticFeaturesR(Fs).
lex(toi,n,[grid([],[]),a(-),p(+),agr([2,sg,[]]),morphC(nonNom)]).

lex(nous,cl,[a(-),p(+),agr([1,pl,[m,f]])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(nous,pf(cl),[a(-),p(+),agr([1,pl,[m,f]])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(nous,pf(cl),[a(-),p(+),agr([1,pl,[m,f]])|Fs]) :-
	objCliticFeatures2(Fs).
lex(nous,pf(cl),[a(-),p(+),agr([1,pl,[m,f]])|Fs]) :-
	iobjCliticFeatures2(Fs).
lex(nous,pf(cl),[a(+),p(-),agr([1,pl,[m,f]])|Fs]) :-
	objCliticFeaturesR(Fs).
lex(nous,pf(cl),[a(+),p(-),agr([1,pl,[m,f]])|Fs]) :-
	iobjCliticFeaturesR(Fs).

lex(nous,n,[grid([],[]),a(-),p(+),agr([1,pl,[m,f]]),morphC(nonNom)]).

lex(vous,cl,[a(-),p(+),agr([2,[],[m,f]])|Fs]) :- subjCliticFeatures(nom,Fs).
lex(vous,pf(cl),[a(-),p(+),agr([2,[],[m,f]])|Fs]) :- subjClFeaturesInv(nom,Fs).
lex(vous,pf(cl),[a(-),p(+),agr([2,[],[m,f]])|Fs]) :-
	objCliticFeatures2(Fs).
lex(vous,pf(cl),[a(-),p(+),agr([2,[],[m,f]])|Fs]) :-
	iobjCliticFeatures2(Fs).
lex(vous,pf(cl),[a(+),p(-),agr([2,[],[m,f]])|Fs]) :-
	objCliticFeaturesR(Fs).
lex(vous,pf(cl),[a(+),p(-),agr([2,[],[m,f]])|Fs]) :-
	iobjCliticFeaturesR(Fs).

lex(vous,n,[grid([],[]),a(-),p(+),agr([2,[],[m,f]]),morphC(nonNom)]).
lex(mon,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],m]),
	                                     agr([1,sg,[]])]).
lex(ma,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],f]),
	                                     agr([1,sg,[]])]).
lex(mes,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,pl,[]]),
	                                     agr([1,sg,[]])]).
lex(ton,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],m]),
	                                     agr([2,sg,[]])]).
lex(ta,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([2,[sg,m],f]),
                                              agr([2,sg,[]])]).
lex(tes,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([2,pl,[]]),
                                              agr([2,sg,[]])]).
lex(son,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],m]),
	                                  agr([3,[sg,m],[]])]).
lex(son,n,[morphC(gen),grid([],[]),a(+),p(-),agr1([3,[sg,m],m]),
	                                  agr([3,[sg,m],[]])]).
lex(sa,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],f]),
	                                  agr([3,[sg,m],[]])]).
lex(sa,n,[morphC(gen),grid([],[]),a(+),p(-),agr1([3,[sg,m],f]),
	                                  agr([3,[sg,m],[]])]).
lex(ses,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,pl,[]]),
	                                  agr([3,[sg,m],[]])]).
lex(ses,n,[morphC(gen),grid([],[]),a(+),p(-),agr1([3,pl,[]]),
	                                  agr([3,[sg,m],[]])]).
lex(notre,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],[]]),
	                                  agr([1,pl,[]])]).
lex(nos,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([1,pl,[]]),
	                                  agr([1,pl,[]])]). 
lex(votre,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],[]]),
	                                  agr([2,pl,[]])]).
lex(vos,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([2,pl,[]]),
	                                   agr([2,pl,[]])]).
lex(leur,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,[sg,m],[]]),
	                                     agr([3,pl,[]])]).
lex(leur,n,[morphC(gen),grid([],[]),a(+),p(-),agr1([3,[sg,m],[]]),
	                                     agr([3,pl,[]])]).
lex(leurs,n,[morphC(gen),grid([],[]),a(-),p(+),agr1([3,pl,[]]),
	                                      agr([3,pl,[]])]).
lex(leurs,n,[morphC(gen),grid([],[]),a(+),p(-),agr1([3,pl,[]]),
	                                       agr([3,pl,[]])]).

% Quantifier nouns
lex(chacun,n,[grid([],[]),a(-),p(-),agr([3,sg,m]),op(+)]).
lex(chacune,n,[grid([],[]),a(-),p(-),agr([3,sg,f]),op(+)]).
lex('quelqu''un',n,[grid([],[]),a(-),p(-),agr([3,sg,[m,f]]),op(+)]).

% Wh-nouns
lex(qui,n,[grid([],[]),a(-),p(-),agr([3,[sg,pl,m],[m,f]]),wh,morphC(nonAcc)]).
lex(_qui,n,[grid([],[]),morphC(dat),a(-),p(-),agr([3,[sg,pl,m],[m,f]]),wh]).
lex(dont,n,[grid([],[]),morphC(gen),a(-),p(-),agr([3,[],[m,f]]),wh]).
lex(que,n,[grid([],[]),morphC(acc),a(-),p(-),agr([3,[],[m,f]]),wh,
	lexInGoal(isMatrixLevel)]).




% Common nouns
lex(actrice,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),vowel]).
lex(acteur,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(article,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(salaud,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),epithet]).
lex(garon,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(btiment,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(bus,n,[grid([],[]),a(-),p(-),count(+),agr([3,[sg,pl],m])]).
lex(gteau,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
lex(chat,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(chatte,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(fille,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(fromage,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
%lex(cleaner,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(ville,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(manifestation,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(explosion,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),vowel]).
lex(feu,n,[grid([],[]),a(-),p(-),count(_),agr([3,[sg,m],m])]).
lex(tage,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(jardin,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(crme_glace,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,f])]).
lex(avocat,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(avocate,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),vowel]).
lex(menteur,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(menteuse,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(homme,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(pice,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(plaine_de_jeux,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(police,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,f])]).
lex(policier,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(problme,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(course,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(rat,n,[grid([],[]),a(-),p(-),count(+),agr([3,[],m])]).
lex(rapport,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(tache,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(tudiant,n,[grid([],[]),a(-),p(-),agr([3,sg,m]),count(+),vowel]).
lex(tudiante,n,[grid([],[]),a(-),p(-),agr([3,sg,f]),count(+),vowel]).
lex(symphonie,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(temps,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,m])]).
lex(cela,n,[grid([],[]),a(-),p(-),count(-),agr([3,m,m])]).
lex(femme,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(travailleur,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(ouvrier,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m]),vowel]).
lex(ouvrire,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f]),vowel]).
lex(vacances,n,[grid([],[]),a(-),p(-),count(+),agr([3,pl,f])]).
lex(titre,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(personne,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(personne,n,[grid([],[]),a(-),p(-),agr([3,sg,[]])]).
lex(voleur,n,[grid([],[]),a(-),p(-),count(+),agr([3,sg,m])]).


% Nouns with O-grids
lex(livre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(frre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(pre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(meuble,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,m])]).
lex(mre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(photo,n,[grid([possessor],[theme]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(professeur,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,[m,f]])]).
lex(chaussure,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f])]).
lex(lettre,n,[grid([possessor],[]),a(-),p(-),count(+),agr([3,sg,f])]).

% Non-base noun forms
lex(actrices,n,[agr([3,pl,f])|F]) :- nounFeatures(actrice,F).
lex(acteurs,n,[agr([3,pl,m])|F]) :- nounFeatures(acteur,F).
lex(salauds,n,[agr([3,pl,m])|F]) :- nounFeatures(salaud,F).
lex(croyances,n,[agr([3,pl,f])|F]) :- nounFeatures(croyance,F).
lex(livres,n,[agr([3,pl,m])|F]) :- nounFeatures(livre,F).
lex(btiments,n,[agr([3,pl,m])|F]) :- nounFeatures(btiment,F).
lex(villes,n,[agr([3,pl,f])|F]) :- nounFeatures(ville,F).
lex(dparts,n,[agr([3,pl,m])|F]) :- nounFeatures(dpart,F).
lex(manifestations,n,[agr([3,pl,f])|F]) :- nounFeatures(manifestation,F).
lex(explosions,n,[agr([3,pl,f])|F]) :- nounFeatures(explosion,F).
lex(feux,n,[agr([3,pl,m])|F]) :- nounFeatures(feu,F).
lex(tages,n,[agr([3,pl,m])|F]) :- nounFeatures(tage,F).
lex(jardins,n,[agr([3,pl,m])|F]) :- nounFeatures(jardin,F).
lex(avocats,n,[agr([3,pl,m])|F]) :- nounFeatures(avocat,F).
lex(menteurs,n,[agr([3,pl,m])|F]) :- nounFeatures(menteur,F).
lex(pices,n,[agr([3,pl,f])|F]) :- nounFeatures(pice,F).
lex(plaines_de_jeux,n,[agr([3,pl,f])|F]) :- nounFeatures(plaine_de_jeux,F).
lex(hommes,n,[agr([3,pl,m])|F]):- nounFeatures(homme,F).
lex(mres,n,[agr([3,pl,f])|F]) :- nounFeatures(mere,F).
lex(photos,n,[agr([3,pl,f])|F]) :- nounFeatures(photo,F).
lex(professeurs,n,[agr([3,pl,[m,f]])|F]) :- nounFeatures(professeur,F).
lex(policiers,n,[agr([3,pl,m])|F]) :- nounFeatures(policier,F).
lex(problmes,n,[agr([3,pl,m])|F]) :- nounFeatures(problme,F).
lex(courses,n,[agr([3,pl,f])|F]) :- nounFeatures(course,F).
lex(rats,n,[agr([3,pl,m])|F]) :- nounFeatures(rat,F).
lex(rapports,n,[agr([3,pl,m])|F]) :- nounFeatures(rapport,F).
lex(taches,n,[agr([3,pl,f])|F]) :- nounFeatures(tache,F).
lex(tudiants,n,[agr([3,pl,m])|F]) :- nounFeatures(tudiant,F).
lex(tudiantes,n,[agr([3,pl,f])|F]) :- nounFeatures(tudiante,F).
lex(symphonies,n,[agr([3,pl,f])|F]) :- nounFeatures(symphonie,F).
lex(femmes,n,[agr([3,pl,f])|F]) :- nounFeatures(femme,F).
lex(travailleurs,n,[agr([3,pl,m])|F]) :- nounFeatures(travailleur,F).
lex(ouvriers,n,[agr([3,pl,m])|F]) :- nounFeatures(ouvrier,F).
lex(ouvrires,n,[agr([3,pl,m])|F]) :- nounFeatures(ouvrire,F).
lex(chaussures,n,[agr([3,pl,f])|F]) :- nounFeatures(chaussure,F).
lex(lettres,n,[agr([3,pl,f])|F]) :- nounFeatures(lettre,F).
lex(personnes,n,[agr([3,pl,f])|F]) :- nounFeatures(personne,F).

% Nominalized verbs
lex(croyance,n,[count(+),agr([3,sg,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(croire,F).
% lex(affirmation,n,[count(+),agr([3,sg,f]),a(-),p(-),vowel|F]) :-
%	verbToNounFeatures(affirmer,F).
lex(dpart,n,[count(+),agr([3,sg,m]),a(-),p(-)|F]) :-
	verbToNounFeatures(partir,F).
lex(destruction,n,[count(+),agr([3,m,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(dtruire,F).
lex(lutte,n,[count(+),agr([3,sg,f]),a(-),p(-)|F]) :-
	verbToNounFeatures(lutter,F).

%%% VERBS - BASE FORMS
%
% Obligatory Elements
% 	1. grid(Ext,Int) denotes the O-grid
% 	2. morph(Base,[]/ing/s/ed(1)/ed(2)) 
% Optional Features
% 	1. ecm(opt/oblig) for ECM verbs
% 	2. noCasemark(+) for technically intransitive verbs
% 	3. subcat(C$Fs,Fs1) subcategorizes for an element of category C 
%    	   with feature restriction list Fs.
%    	   Feature list Fs1 will be appended to the feature list of C
% 	4. selR(Rs), selR(Type,R) restriction(s) Rs or R on direct object.

lex(apparatre,v,[morph(apparatre,[]),grid([theme],[]),vowel]).
lex(apprcier,v,[morph(apprcier,[]),grid([agent],[theme]),vowel]).
lex(arrter,v,[morph(arrter,[]),grid([agent],[[patient]]),vowel]).
lex(arriver,v,[morph(arriver,[]),grid([theme],[]),allowExt(source),vowel]).
% lex(demander,v,[morph(demander,[]),grid([agent],[proposition,[patient]]),
%	 selR(head(c2),goal(optWhComp(X),X)),idoCase(dat)]).
lex(demander,v,[morph(demander,[]),grid([agent],[proposition,patient]),
    control,selR(goal(takesDe(X),X)),idoCase(dat)]).
lex(croire,v,[morph(croire,[]),grid([agent],[proposition])]).
lex(acheter,v,[morph(acheter,[]),grid([agent],[theme,[destination]]),
	                                                  idocase(dat),vowel]).
lex(appeler,v,[morph(appeler,[]),grid([agent],[patient]),vowel]).
% lex(care,v,[morph(care,[]),grid([agent],[proposition]),noCasemark(+),
	%   selR(head(c2),goal(ensureWhComp(X),X))]).
lex(affirmer,v,[morph(affirmer,[]),grid([agent],[proposition,[destination]]),
	                                              idoCase(dat),vowel]).
lex(partir,v,[morph(partir,[]),grid([theme],[]), allowExt(source)]).
lex(dtruire,v,[morph(dtruire,[]),grid([agent],[patient])]).
lex(mourir,v,[morph(mourir,[]),grid([theme],[])]).
lex(disparatre,v,[morph(disparatre,[]),grid([theme],[])]).
lex(manger,v,[morph(manger,[]),grid([agent],[[patient]])]).
lex(laver,v,[morph(laver,[]),grid([agent],[[patient]])]).
lex(tomber,v,[morph(tomber,[]),grid([theme],[])]).
lex(lutter,v,[morph(lutter,[]),grid([agent]),allowExt(patient)]).
lex(classer,v,[morph(classer,[]),grid([agent],[patient])]).
lex(renvoyer,v,[morph(renvoyer,[]),grid([agent],[patient]),allowExt(source)]).
lex(donner,v,[morph(donner,[]),grid([agent],[theme,[destination]]),
	      idoCase(dat)]).
lex(entendre,v,[morph(entendre,[]),grid([agent],[theme]),vowel]).
lex(engager,v,[morph(engager,[]),grid([agent],[patient]),vowel]).
lex(frapper,v,[morph(frapper,[]),grid([agent],[patient])]).
lex(instruire,v,[morph(instruire,[]),grid([agent],[patient]),vowel]).
lex(insulter,v,[morph(insulter,[]),grid([agent],[patient]),vowel]).
lex(garder,v,[morph(garder,[]),grid([agent],[patient])]).
lex(tuer,v,[morph(tuer,[]),grid([agent],[patient])]).
lex(savoir,v,[morph(savoir,[]),grid([agent],[proposition]), % with c2 or n2
	    selR(head(c2),goal(optWhComp(X),X))]).
lex(connatre,v,[morph(connatre,[]),grid([agent],[theme])]). % with np
lex(rire,v,[morph(rire,[]),grid([theme],[[patient]])]).
lex(quitter,v,[morph(quitter,[]),grid([agent],[patient])]).
lex(mener,v,[morph(mener,[]),grid([agent],[patient]),allowExt(destination)]).
lex(aimer,v,[morph(aimer,[]),grid([agent],[patient]),vowel]).
lex(rencontrer,v,[morph(rencontrer,[]),grid([agent],[patient]),
	                                          allowExt(location)]).
lex(mentionner,v,[morph(mentionner,[]),grid([agent],[proposition])]).
lex(garer,v,[morph(garer,[]),grid([agent],[[patient]]),allowExt(location)]).
lex(persuader,v,[morph(persuader,[]),grid([agent],[patient,proposition]),
		 control,iselR(goal(takesDe(X),X))]).
lex(plaire,v,[morph(plaire,[]),grid([agent],[patient])]).
lex(promettre,v,[morph(promettre,[]),grid([agent],[proposition,patient]),
control,selR(goal(takesDe(X),X)),idoCase(dat),iselR([addFeature(noControl)])]).
lex(mettre,v,[morph(mettre,[]),grid([agent],[patient]),
	                                            requiresExt(location)]).
lex(pleuvoir,v,[morph(pleuvoir,[]),grid([],[]),intrans1]).
lex(lire,v,[morph(lire,[]),grid([agent],[theme,[destination]]),idoCase(dat)]).
lex(enlever,v,[morph(enlever,[]),grid([agent],[patient]),allowExt(source),
	                                                       vowel]).
lex(dmissionner,v,[morph(dmissionner,[]),grid([theme],[])]).
lex(dire,v,[morph(dire,[]),grid([agent],[proposition, [destination]]),
	    control,selR(goal(takesDe(X),X)),idoCase(dat)]).
lex(voir,v,[morph(voir,[]),grid([agent],[patient])]).
lex(sembler,v,[morph(sembler,[]),grid([],[proposition]),ecm(oblig),
	noCasemark(+), selR(goal(ensureSubj(X),X))]).
lex(dormir,v,[morph(dormir,[]),grid([experiencer],[]),intrans1]).
lex(neiger,v,[morph(neiger,[]),grid([],[]),intrans1]).
lex(rsoudre,v,[morph(rsoudre,[]),grid([agent],[patient])]).
lex(supporter,v,[morph(supporter,[]),grid([agent],[proposition])]).
lex(commencer,v,[morph(commencer,[]),grid([agent],[patient])]).
lex(tonner,v,[morph(tonner,[]),grid([agent],[patient,proposition]),
	                                                      vowel]).
lex(parler,v,[morph(parler,[]),grid([agent],[[theme],[destination]]),doCase(gen), idoCase(dat)]).
lex(raconter,v,[morph(raconter,[]),grid([agent],[proposition,[destination]]),
	                                                       idoCase(dat)]).
% lex(penser,v,[morph(penser,[]),grid([agent],[]),allowExt(theme)]).
lex(penser,v,[morph(penser,[]),grid([agent],[proposition])]).
lex(essayer,v,[morph(essayer,[]),grid([agent],[proposition]),vowel,
		 control,iselR(goal(takesDe(X),X))]).
lex(comprendre,v,[morph(comprendre,[]),grid([agent],[patient])]).
lex(vouloir,v,[morph(vouloir,[]),grid([agent],[proposition]),
	selR(goal(ensureSubj(X),X))]).
lex(gagner,v,[morph(gagner,[]),grid([agent],[])]).
lex(gagner,v,[morph(gagner,[]),grid([agent],[patient])]).
lex(travailler,v,[morph(travailler,[]),grid([theme],[]),
	                                allow(adv,degree)]).
lex(crire,v,[morph(crire,[]),grid([agent],[[theme],[destination]]),
	                         idoCase(dat),vowel]).
lex(visiter,v,[morph(visiter,[]),grid([agent],[patient])]).
lex(venir,v,[morph(venir,[]),grid([agent],[]),allowExt(source)]).
lex(aller,v,[morph(aller,[]),grid([agent],[]),allowExt(destination),vowel]).
lex(aller,v,[morph(aller,[]),grid([agent],[proposition]),vowel,
	selR(goal(ensureInf(X),X))]).
lex(pouvoir,v,[morph(pouvoir,[]),grid([agent],[proposition]),
	selR(goal(ensureInf(X),X))]).

%% Auxiliaries

lex(tre,v,[morph(tre,[]),aux,intrans1,vowel,transmitAgr,
 oneof(_,[[grid([eq],[eq])],
	  [grid([pHolder(P)],subcat),subcat(adv$[predicate(P)],[])],
	  [subcat(ap$[grid(_,_)],[])],
	  [subcat(vp$[morph(_,ed(2)),grid(_,[])],[perf])],
	  [subcat(vp$[morph(_,ed(2)),grid(_,[_X|_])],
	  [noCasemark(+)]),blockTheta]])]).


                  
% Stage 2
% lex(be,v,[morph(be,[]),aux,grid([eq],[eq])]).	  % john is the man
% lex(be,v,[morph(be,[]),aux,grid([pHolder(P)],subcat),	% john is here
%	  subcat(adv$[predicate(P)],[])]).
% Stage 1
% lex(be,v,[morph(be,[]),aux,subcat(ap$[grid(_,_)],[])]).	% john is dead
% lex(be,v,[morph(be,[]),aux,blockTheta,	  % john was arrested
%	  subcat(vp$[morph(_,ed(2))],[noCasemark(+)])]).
% lex(be,v,[morph(be,[]),aux,			  % john is 
%	  subcat(vp$[morph(_,ing)],[prog])]).

lex(avoir,v,[morph(avoir,[]),aux,perf,vowel,subcat(vp$[morph(_,ed(2)), 
	grid(_,[_|_])],[perf])]).

lex(avoir,v,[morph(avoir,[]),aux,perf,subcat(vp$[morph(_,ed(2)),
      	intrans1],[perf])]).


%% Adjectives
%%
%% Optional features:
%%	ecm(oblig) really means S-bar Deleter in this context.

lex(aaaaa,a,[grid([],[patient])]).		% too complicated to explain

lex(certain,a,[grid([],[proposition]),agr([[],[sg,m],m])]).
lex(certaine,a,[grid([],[proposition]),agr([[],[sg,m],f])]).
lex(certains,a,[grid([],[proposition]),agr([[],pl,m])]).
lex(certaines,a,[grid([],[proposition]),agr([[],pl,f])]).
lex(intelligent,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(intelligente,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(intelligents,a,[grid([theme],[]),agr([[],pl,m])]).
lex(intelligentes,a,[grid([theme],[]),agr([[],pl,f])]).
lex(fou,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(folle,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(fous,a,[grid([theme],[]),agr([[],pl,m])]).
lex(folles,a,[grid([theme],[]),agr([[],pl,f])]).
lex(crucial,a,[grid([],[proposition]),agr([[],[sg,m],m])]).
lex(cruciale,a,[grid([],[proposition]),agr([[],[sg,m],f])]).
lex(cruciaux,a,[grid([],[proposition]),agr([[],pl,m])]).
lex(cruciales,a,[grid([],[proposition]),agr([[],pl,f])]).
lex(bte,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(btes,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(anxieux,a,[grid([theme],[proposition]),agr([[],[pl,sg,m],m])]).
lex(anxieuse,a,[grid([theme],[proposition]),agr([[],[sg,m],f])]).
lex(anxieuses,a,[grid([theme],[proposition]),agr([[],pl,f])]).
lex(facile,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(faciles,a,[grid([],[proposition]),agr([[],pl,[m,f]])]).
lex(difficile,a,[grid([],[proposition]),agr([[],[sg,m],[m,f]])]).
lex(difficiles,a,[grid([],[proposition]),agr([[],pl,[m,f]])]).
lex(heureux,a,[grid([theme],[]),agr([[],[pl,sg,m],m])]).
lex(heureuse,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(heureuses,a,[grid([theme],[]),agr([[],pl,f])]).
lex(illgal,a,[grid([],[proposition]),agr([[],[sg,m],m])]).
lex(illgale,a,[grid([],[proposition]),agr([[],[sg,m],f])]).
lex(illgaux,a,[grid([],[proposition]),agr([[],pl,m])]).
lex(illgales,a,[grid([],[proposition]),agr([[],pl,f])]).
%lex(ill_founded,a,[grid([theme],[])]).
lex(illogique,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(illogiques,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(important,a,[grid([],[proposition]),agr([[],[sg,m],m])]).
lex(importante,a,[grid([],[proposition]),agr([[],[sg,m],f])]).
lex(importants,a,[grid([],[proposition]),agr([[],pl,m])]).
lex(importantes,a,[grid([],[proposition]),agr([[],pl,f])]).
lex(probable,a,[grid([],[proposition]),agr([3,[sg,m],[f,m]])]).
lex(probables,a,[grid([],[proposition]),agr([3,pl,[f,m]])]).
lex(gentil,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(gentille,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(gentils,a,[grid([theme],[]),agr([[],pl,m])]).
lex(gentilles,a,[grid([theme],[]),agr([[],pl,f])]).
lex(en_solde,a,[grid([theme],[]),agr([[],[],[]])]).
lex(fier,a,[grid([theme],[proposition]),agr([[],[sg,m],m])]).
lex(fire,a,[grid([theme],[proposition]),agr([[],[sg,m],f])]).
lex(fiers,a,[grid([theme],[proposition]),agr([[],pl,m])]).
lex(fires,a,[grid([theme],[proposition]),agr([[],pl,f])]).
lex(ttu,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(ttue,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(ttus,a,[grid([theme],[]),agr([[],pl,m])]).
lex(ttues,a,[grid([theme],[]),agr([[],pl,f])]).
lex(pathologique,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(pathologiques,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(pourri,a,[grid([theme],[]),agr([[],[sg,m],m])]).
lex(pourrie,a,[grid([theme],[]),agr([[],[sg,m],f])]).
lex(pourris,a,[grid([theme],[]),agr([[],pl,m])]).
lex(pourries,a,[grid([theme],[]),agr([[],pl,f])]).
lex(triste,a,[grid([theme],[]),agr([[],[sg,m],[m,f]])]).
lex(tristes,a,[grid([theme],[]),agr([[],pl,[m,f]])]).
lex(dur,a,[grid([weakTheme$optional],[]),allowExt(reason)]).

% Complementizers

lex(que,c,[selR([not(feature(inf(_)))]),specR(goal(quirkyAcc(X),X))]).
lex(qui,c,[morphC(nom),selR([not(feature(inf(_)))]),
	specR(goal(quirkyNom(X),X))]).
lex(dont,c,[morphC(gen),selR([not(feature(inf(_)))]),
	    specR(addFeature(noECP(_)))]).
lex(si,c,[wh]).
lex(pour,c,[selR([feature(inf([]))])]).
lex(est_ce_que,c,[wh,selR([not(feature(inversion)),goal(isMatrixLevel,_)])]).
lex(est_ce_qui,c,[morphC(nom),wh,selR([not(feature(inversion)),
goal(isMatrixLevel,_)]),specR(addFeature(case(nom)))]).
lex(de,c,[selR([feature(inf([]))]),pAttachR(feature(control))]).

% Markers: morphological realizations of particular features

lex(de,mrkr,[right(np,case(gen),[])]). % object genitive Case
lex(,mrkr,[right(np,case(dat),[])]). % dative case

:- initialization((no lexFeature(_,_))).

% Coordinating conjunctions
lex(et,conj,[overt(+)]).

% relevant for marker constraints
relevant(n).
relevant(v).


irregular(tre).
irregular(avoir).
irregular(aller).

% Non-base verb forms:
%
% Form x Category x Base x Features

lex(apparaissant,v,apparatre,[morph(apparatre,ant)]).
lex(apparais,v,apparatre,[morph(apparatre,s)]).
lex(apparat,v,apparatre,[morph(apparatre,t)]).
lex(apparaissons,v,apparatre,[morph(apparatre,ons)]).
lex(apparaissez,v,apparatre,[morph(apparatre,ez)]).
lex(apparaissent,v,apparatre,[morph(apparatre,ent)]).
lex(apparaisse,v,apparatre,[morph(apparatre,e1)]).
lex(apparaisses,v,apparatre,[morph(apparatre,s2)]).
lex(apparaissions,v,apparatre,[morph(apparatre,ions1)]).
lex(apparaissiez,v,apparatre,[morph(apparatre,iez1)]).
lex(apparue,v,apparatre,[morph(apparatre,ed(2)),agr([[],[sg,m],f])]).
lex(apparu,v,apparatre,[morph(apparatre,ed(2)),agr([[],[sg,m],m])]).
lex(apparus,v,apparatre,[morph(apparatre,ed(2)),agr([[],pl,m])]).
lex(apparues,v,apparatre,[morph(apparatre,ed(2)),agr([[],pl,f])]).

lex(apprciant,v,apprcier,[morph(apprcier,ant)]).
lex(apprcie,v,apprcier,[morph(apprcier,e),endvowel]).
lex(apprcies,v,apprcier,[morph(apprcier,s1)]).
lex(apprcions1,v,apprcier,[morph(apprcier,ons)]).
lex(apprciez,v,apprcier,[morph(apprcier,ez)]).
lex(apprcient,v,apprcier,[morph(apprcier,ent)]).
lex(apprciions,v,apprcier,[morph(apprcier,ions1)]).
lex(apprciiez,v,apprcier,[morph(apprcier,iez1)]).
lex(apprci,v,apprcier,[morph(apprcier,ed(2)),agr([[],[sg,m],m])]).
lex(apprcie,v,apprcier,[morph(apprcier,ed(2)),agr([[],[sg,m],f])]).
lex(apprcies,v,apprcier,[morph(apprcier,ed(2)),agr([[],pl,f])]).
lex(apprcies,v,apprcier,[morph(apprcier,ed(2)),agr([[],pl,m])]).

lex(arrtant,v,arrter,[morph(arrter,ant)]).
lex(arrtes,v,arrter,[morph(arrter,s1)]).
lex(arrte,v,arrter,[morph(arrter,e),endvowel]).
lex(arrtons,v,arrter,[morph(arrter,ons)]).
lex(arrtez,v,arrter,[morph(arrter,ez)]).
lex(arrtent,v,arrter,[morph(arrter,ent)]).
lex(arrtions,v,arrter,[morph(arrter,ions1)]).
lex(arrtiez,v,arrter,[morph(arrter,iez1)]).
lex(arrt,v,arrter,[morph(arrter,ed(2)),agr([[],[sg,m],m])]).
lex(arrte,v,arrter,[morph(arrter,ed(2)),agr([[],[sg,m],f])]).
lex(arrtes,v,arrter,[morph(arrter,ed(2)),agr([[],pl,f])]).
lex(arrtes,v,arrter,[morph(arrter,ed(2)),agr([[],pl,m])]).

lex(arrivant,v,arriver,[morph(arriver,ant)]).
lex(arrives,v,arriver,[morph(arriver,s1)]).
lex(arrive,v,arriver,[morph(arriver,e),endvowel]).
lex(arrivons,v,arriver,[morph(arriver,ons)]).
lex(arrivez,v,arriver,[morph(arriver,ez)]).
lex(arrivent,v,arriver,[morph(arriver,ent)]).
lex(arrivions,v,arriver,[morph(arriver,ions1)]).
lex(arriviez,v,arriver,[morph(arriver,iez1)]).
lex(arriv,v,arriver,[morph(arriver,ed(2)),agr([[],[sg,m],m])]).
lex(arrive,v,arriver,[morph(arriver,ed(2)),agr([[],[sg,m],f])]).
lex(arrives,v,arriver,[morph(arriver,ed(2)),agr([[],pl,f])]).
lex(arrives,v,arriver,[morph(arriver,ed(2)),agr([[],pl,m])]).

lex(demandant,v,demander,[morph(demander,ant)]).
lex(demandes,v,demander,[morph(demander,s1)]).
lex(demande,v,demander,[morph(demander,e),endvowel]).
lex(demandons,v,demander,[morph(demander,ons)]).
lex(demandez,v,demander,[morph(demander,ez)]).
lex(demandent,v,demander,[morph(demander,ent)]).
lex(demandiez,v,demander,[morph(demander,iez1)]).
lex(demandions,v,demander,[morph(demander,ions1)]).
lex(demand,v,demander,[morph(demander,ed(2)),agr([[],[sg,m],m])]).
lex(demande,v,demander,[morph(demander,ed(2)),agr([[],[sg,m],f])]).
lex(demands,v,demander,[morph(demander,ed(2)),agr([[],pl,m])]).
lex(demandes,v,demander,[morph(demander,ed(2)),agr([[],pl,f])]).

lex(suis,v,tre,[morph(tre,suis)]).  % Cat= v
lex(es,v,tre,[morph(tre,es)]).
lex(est,v,tre,[morph(tre,est)]).
lex(sommes,v,tre,[morph(tre,sommes)]).
lex(tes,v,tre,[morph(tre,etes)]).
lex(sont,v,tre,[morph(tre,sont)]).
lex(sois,v,tre,[morph(tre,sois)]).  % Cat= v
lex(soit,v,tre,[morph(tre,soit)]).
lex(soyons,v,tre,[morph(tre,soyons)]).
lex(soyez,v,tre,[morph(tre,soyez)]).
lex(soient,v,tre,[morph(tre,soient)]).
lex(t,v,tre,[morph(tre,ed(2))]).
lex(tant,v,tre,[morph(tre,ant)]).

lex(croyant,v,croire,[morph(croire,ant)]).
lex(crois,v,croire,[morph(croire,s)]).
lex(croit,v,croire,[morph(croire,t)]).
lex(croyons,v,croire,[morph(croire,ons)]).
lex(croyez,v,croire,[morph(croire,ez)]).
lex(croient,v,croire,[morph(croire,ent)]).
lex(croie,v,croire,[morph(croire,e1)]).
lex(croies,v,croire,[morph(croire,s2)]).
lex(croyions,v,croire,[morph(croire,ions1)]).
lex(croyiez,v,croire,[morph(croire,iez1)]).
lex(cru,v,croire,[morph(croire,ed(2)),agr([[],[sg,m],m])]).
lex(crue,v,croire,[morph(croire,ed(2)),agr([[],[sg,m],f])]).
lex(crus,v,croire,[morph(croire,ed(2)),agr([[],pl,m])]).
lex(crues,v,croire,[morph(croire,ed(2)),agr([[],pl,f])]).

lex(achetant,v,acheter,[morph(acheter,ant)]).
lex(achte,v,acheter,[morph(acheter,e),endvowel]).
lex(achtes,v,acheter,[morph(acheter,s1)]).
lex(achetons,v,acheter,[morph(acheter,ons)]).
lex(achetez,v,acheter,[morph(acheter,ez)]).
lex(achtent,v,acheter,[morph(acheter,ent)]).
lex(achtions,v,acheter,[morph(acheter,ions1)]).
lex(achtiez,v,acheter,[morph(acheter,iez1)]).
lex(achet,v,acheter,[morph(acheter,ed(2)),agr([[],[sg,m],m])]).
lex(achete,v,acheter,[morph(acheter,ed(2)),agr([[],[sg,m],f])]).
lex(achets,v,acheter,[morph(acheter,ed(2)),agr([[],pl,m])]).
lex(achetes,v,acheter,[morph(acheter,ed(2)),agr([[],pl,f])]).

lex(appelant,v,appeler,[morph(appeler,ant)]).
lex(appelles,v,appeler,[morph(appeler,s1)]).
lex(appelle,v,appeler,[morph(appeler,e),endvowel]).
lex(appelons,v,appeler,[morph(appeler,ons)]).
lex(appelez,v,appeler,[morph(appeler,ez)]).
lex(appellent,v,appeler,[morph(appeler,ent)]).
lex(appelions,v,appeler,[morph(appeler,ions1)]).
lex(appeliez,v,appeler,[morph(appeler,iez1)]).
lex(appel,v,appeler,[morph(appeler,ed(2)),agr([[],[sg,m],m])]).
lex(appele,v,appeler,[morph(appeler,ed(2)),agr([[],[sg,m],f])]).
lex(appels,v,appeler,[morph(appeler,ed(2)),agr([[],pl,m])]).
lex(appeles,v,appeler,[morph(appeler,ed(2)),agr([[],pl,f])]).

lex(affirmant,v,affirmer,[morph(affirmer,ant)]).
lex(affirmes,v,affirmer,[morph(affirmer,s1)]).
lex(affirme,v,affirmer,[morph(affirmer,e),endvowel]).
lex(affirmons,v,affirmer,[morph(affirmer,ons)]).
lex(affirmez,v,affirmer,[morph(affirmer,ez)]).
lex(affirment,v,affirmer,[morph(affirmer,ent)]).
lex(affirmions,v,affirmer,[morph(affirmer,ions1)]).
lex(affirmiez,v,affirmer,[morph(affirmer,iez1)]).
lex(affirm,v,affirmer,[morph(affirmer,ed(2)),agr([[],[sg,m],m])]).
lex(affirme,v,affirmer,[morph(affirmer,ed(2)),agr([[],[sg,m],f])]).
lex(affirms,v,affirmer,[morph(affirmer,ed(2)),agr([[],pl,m])]).
lex(affirmes,v,affirmer,[morph(affirmer,ed(2)),agr([[],pl,f])]).

lex(partant,v,partir,[morph(partir,ant)]).
lex(pars,v,partir,[morph(partir,s)]).
lex(part,v,partir,[morph(partir,t)]).
lex(partons,v,partir,[morph(partir,ons)]).
lex(partez,v,partir,[morph(partir,ez)]).
lex(partent,v,partir,[morph(partir,ent)]).
lex(partes,v,partir,[morph(partir,s2)]).
lex(parte,v,partir,[morph(partir,e1)]).
lex(partions,v,partir,[morph(partir,ions1)]).
lex(partiez,v,partir,[morph(partir,iez1)]).
lex(parti,v,partir,[morph(partir,ed(2)),agr([[],sg,m])]).
lex(partie,v,partir,[morph(partir,ed(2)),agr([[],sg,f])]).
lex(partis,v,partir,[morph(partir,ed(2)),agr([[],pl,m])]).
lex(parties,v,partir,[morph(partir,ed(2)),agr([[],pl,f])]).

lex(dtruisant,v,dtruire,[morph(dtruire,ant)]).
lex(dtruis,v,dtruire,[morph(dtruire,s)]).
lex(dtruit,v,dtruire,[morph(dtruire,t)]).
lex(dtruisons,v,dtruire,[morph(dtruire,ons)]).
lex(dtruisez,v,dtruire,[morph(dtruire,ez)]).
lex(dtruisent,v,dtruire,[morph(dtruire,ent)]).
lex(dtruises,v,dtruire,[morph(dtruire,s2)]).
lex(dtruise,v,dtruire,[morph(dtruire,e1)]).
lex(dtruisions,v,dtruire,[morph(dtruire,ions1)]).
lex(dtruisiez,v,dtruire,[morph(dtruire,iez1)]).
lex(dtruit,v,dtruire,[morph(dtruire,ed(2)),agr([[],[sg,m],m])]).
lex(dtruite,v,dtruire,[morph(dtruire,ed(2)),agr([[],[sg,m],f])]).
lex(dtruits,v,dtruire,[morph(dtruire,ed(2)),agr([[],pl,m])]).
lex(dtruites,v,dtruire,[morph(dtruire,ed(2)),agr([[],pl,f])]).

lex(mourant,v,mourir,[morph(mourir,ant)]).
lex(meurs,v,mourir,[morph(mourir,s)]).
lex(meurt,v,mourir,[morph(mourir,t)]).
lex(mourons,v,mourir,[morph(mourir,ons)]).
lex(mourez,v,mourir,[morph(mourir,ez)]).
lex(meurent,v,mourir,[morph(mourir,ent)]).
lex(meures,v,mourir,[morph(mourir,s2)]).
lex(meure,v,mourir,[morph(mourir,e1)]).
lex(mourions,v,mourir,[morph(mourir,ions1)]).
lex(mouriez,v,mourir,[morph(mourir,iez1)]).
lex(mort,v,mourir,[morph(mourir,ed(2)),agr([[],[sg,m],m])]).
lex(morte,v,mourir,[morph(mourir,ed(2)),agr([[],[sg,m],f])]).
lex(mortes,v,mourir,[morph(mourir,ed(2)),agr([[],pl,f])]).
lex(morts,v,mourir,[morph(mourir,ed(2)),agr([[],pl,m])]).

lex(disparaissant,v,disparatre,[morph(disparatre,ant)]).
lex(disparais,v,disparatre,[morph(disparatre,s)]).
lex(disparat,v,disparatre,[morph(disparatre,t)]).
lex(disparaissons,v,disparatre,[morph(disparatre,ons)]).
lex(disparaissez,v,disparatre,[morph(disparatre,ez)]).
lex(disparaissent,v,disparatre,[morph(disparatre,ent)]).
lex(disparaisse,v,disparatre,[morph(disparatre,e1)]).
lex(disparaisses,v,disparatre,[morph(disparatre,s2)]).
lex(disparaissions,v,disparatre,[morph(disparatre,ions1)]).
lex(disparaissiez,v,disparatre,[morph(disparatre,iez1)]).
lex(disparu,v,disparatre,[morph(disparatre,ed(2))]).

lex(mangeant,v,manger,[morph(manger,ant)]).
lex(mange,v,manger,[morph(manger,e),endvowel]).
lex(manges,v,manger,[morph(manger,s1)]).
lex(mangeons,v,manger,[morph(manger,ons)]).
lex(mangez,v,manger,[morph(manger,ez)]).
lex(mangent,v,manger,[morph(manger,ent)]).
lex(mangions,v,manger,[morph(manger,ions1)]).
lex(mangiez,v,manger,[morph(manger,iez1)]).
lex(mang,v,manger,[morph(manger,ed(2)),agr([[],[sg,m],m])]).
lex(mange,v,manger,[morph(manger,ed(2)),agr([[],[sg,m],f])]).
lex(manges,v,manger,[morph(manger,ed(2)),agr([[],pl,f])]).
lex(mangs,v,manger,[morph(manger,ed(2)),agr([[],pl,m])]).

lex(lavant,v,laver,[morph(laver,ant)]).
lex(lave,v,laver,[morph(laver,e),endvowel]).
lex(laves,v,laver,[morph(laver,s1)]).
lex(lavons,v,laver,[morph(laver,ons)]).
lex(lavez,v,laver,[morph(laver,ez)]).
lex(lavent,v,laver,[morph(laver,ent)]).
lex(lavions,v,laver,[morph(laver,ions1)]).
lex(laviez,v,laver,[morph(laver,iez1)]).
lex(lav,v,laver,[morph(laver,ed(2)),agr([[],[sg,m],m])]).
lex(lave,v,laver,[morph(laver,ed(2)),agr([[],[sg,m],f])]).
lex(laves,v,laver,[morph(laver,ed(2)),agr([[],pl,f])]).
lex(lavs,v,laver,[morph(laver,ed(2)),agr([[],pl,m])]).

lex(tombant,v,tomber,[morph(tomber,ant)]).
lex(tombe,v,tomber,[morph(tomber,e),endvowel]).
lex(tombes,v,tomber,[morph(tomber,s1)]).
lex(tombons,v,tomber,[morph(tomber,ons)]).
lex(tombez,v,tomber,[morph(tomber,ez)]).
lex(tombent,v,tomber,[morph(tomber,ent)]).
lex(tombions,v,tomber,[morph(tomber,ions1)]).
lex(tombiez,v,tomber,[morph(tomber,iez1)]).
lex(tomb,v,tomber,[morph(tomber,ed(2)),agr([[],[sg,m],m])]).
lex(tombe,v,tomber,[morph(tomber,ed(2)),agr([[],[sg,m],f])]).
lex(tombs,v,tomber,[morph(tomber,ed(2)),agr([[],pl,m])]).
lex(tombes,v,tomber,[morph(tomber,ed(2)),agr([[],pl,f])]).

lex(luttant,v,lutter,[morph(lutter,ant)]).
lex(luttes,v,lutter,[morph(lutter,s1)]).
lex(lutte,v,lutter,[morph(lutter,e),endvowel]).
lex(luttons,v,lutter,[morph(lutter,ons)]).
lex(luttez,v,lutter,[morph(lutter,ez)]).
lex(luttent,v,lutter,[morph(lutter,ent)]).
lex(luttions,v,lutter,[morph(lutter,ions1)]).
lex(luttiez,v,lutter,[morph(lutter,iez1)]).
lex(lutt,v,lutter,[morph(lutter,ed(2))]).

lex(classant,v,classer,[morph(classer,ant)]).
lex(classes,v,classer,[morph(classer,s1)]).
lex(classe,v,classer,[morph(classer,e),endvowel]).
lex(classons,v,classer,[morph(classer,ons)]).
lex(classez,v,classer,[morph(classer,ez)]).
lex(classent,v,classer,[morph(classer,ent)]).
lex(classions,v,classer,[morph(classer,ions1)]).
lex(classiez,v,classer,[morph(classer,iez1)]).
lex(class,v,classer,[morph(classer,ed(2)),agr([[],[sg,m],m])]).
lex(classe,v,classer,[morph(classer,ed(2)),agr([[],[sg,m],f])]).
lex(classs,v,classer,[morph(classer,ed(2)),agr([[],pl,m])]).
lex(classes,v,classer,[morph(classer,ed(2)),agr([[],pl,f])]).

lex(renvoyant,v,renvoyer,[morph(renvoyer,ant)]).
lex(renvoies,v,renvoyer,[morph(renvoyer,s1)]).
lex(renvoie,v,renvoyer,[morph(renvoyer,e),endvowel]).
lex(renvoyons,v,renvoyer,[morph(renvoyer,ons)]).
lex(renvoyez,v,renvoyer,[morph(renvoyer,ez)]).
lex(renvoient,v,renvoyer,[morph(renvoyer,ent)]).
lex(renvoyions,v,renvoyer,[morph(renvoyer,ions1)]).
lex(renvoyiez,v,renvoyer,[morph(renvoyer,iez1)]).
lex(renvoy,v,renvoyer,[morph(renvoyer,ed(2)),agr([[],[sg,m],m])]).
lex(renvoye,v,renvoyer,[morph(renvoyer,ed(2)),agr([[],[sg,m],f])]).
lex(renvoys,v,renvoyer,[morph(renvoyer,ed(2)),agr([[],pl,m])]).
lex(renvoyes,v,renvoyer,[morph(renvoyer,ed(2)),agr([[],p,f])]).

lex(donnant,v,donner,[morph(donner,ant)]).
lex(donnes,v,donner,[morph(donner,s1)]).
lex(donne,v,donner,[morph(donner,e),endvowel]).
lex(donnons,v,donner,[morph(donner,ons)]).
lex(donnez,v,donner,[morph(donner,ez)]).
lex(donnent,v,donner,[morph(donner,ent)]).
lex(donnions,v,donner,[morph(donner,ions1)]).
lex(donniez,v,donner,[morph(donner,iez1)]).
lex(donn,v,donner,[morph(donner,ed(2)),agr([[],[sg,m],m])]).
lex(donne,v,donner,[morph(donner,ed(2)),agr([[],[sg,m],f])]).
lex(donns,v,donner,[morph(donner,ed(2)),agr([[],pl,m])]).
lex(donnes,v,donner,[morph(donner,ed(2)),agr([[],pl,f])]).

lex(ayant,v,avoir,[morph(avoir,ant)]).
lex(ai,v,avoir,[morph(avoir,ai)]).
lex(as,v,avoir,[morph(avoir,as)]).
lex(a,v,avoir,[morph(avoir,a),endvowel]).
lex(avons,v,avoir,[morph(avoir,ons)]).
lex(avez,v,avoir,[morph(avoir,ez)]).
lex(aie,v,avoir,[morph(avoir,aie)]).
lex(aies,v,avoir,[morph(avoir,aies)]).
lex(ait,v,avoir,[morph(avoir,ait),endvowel]).
lex(ayons,v,avoir,[morph(avoir,ayons)]).
lex(ayez,v,avoir,[morph(avoir,ayez)]).
lex(aient,v,avoir,[morph(avoir,aient)]).
lex(eu,v,avoir,[morph(avoir,ed(2)),agr([[],[sg,m],m])]).
lex(eue,v,avoir,[morph(avoir,ed(2)),agr([[],[sg,m],f])]).
lex(eus,v,avoir,[morph(avoir,ed(2)),agr([[],pl,m])]).
lex(eues,v,avoir,[morph(avoir,ed(2)),agr([[],pl,f])]).

lex(entendant,v,entendre,[morph(entendre,ant)]).
lex(entends,v,entendre,[morph(entendre,s)]).
lex(entend,v,entendre,[morph(entendre,t)]).
lex(entendons,v,entendre,[morph(entendre,ons)]).
lex(entendez,v,entendre,[morph(entendre,ez)]).
lex(entendent,v,entendre,[morph(entendre,ent)]).
lex(entendes,v,entendre,[morph(entendre,s2)]).
lex(entende,v,entendre,[morph(entendre,e1)]).
lex(entendions,v,entendre,[morph(entendre,ions1)]).
lex(entendiez,v,entendre,[morph(entendre,iez1)]).
lex(entendu,v,entendre,[morph(entendre,ed(2)),agr([[],[sg,m],m])]).
lex(entendue,v,entendre,[morph(entendre,ed(2)),agr([[],[sg,m],f])]).
lex(entendus,v,entendre,[morph(entendre,ed(2)),agr([[],pl,m])]).
lex(entendues,v,entendre,[morph(entendre,ed(2)),agr([[],pl,f])]).


lex(engageant,v,engager,[morph(engager,ant)]).
lex(engages,v,engager,[morph(engager,s1)]).
lex(engage,v,engager,[morph(engager,e),endvowel]).
lex(engageons,v,engager,[morph(engager,ons)]).
lex(engagez,v,engager,[morph(engager,ez)]).
lex(engagent,v,engager,[morph(engager,ent)]).
lex(engagions,v,engager,[morph(engager,ions1)]).
lex(engagiez,v,engager,[morph(engager,iez1)]).
lex(engag,v,engager,[morph(engager,ed(2)),agr([[],[sg,m],m])]).
lex(engage,v,engager,[morph(engager,ed(2)),agr([[],[sg,m],f])]).
lex(engags,v,engager,[morph(engager,ed(2)),agr([[],pl,m])]).
lex(engages,v,engager,[morph(engager,ed(2)),agr([[],pl,f])]).

lex(frappant,v,frapper,[morph(frapper,ant)]).
lex(frappes,v,frapper,[morph(frapper,s1)]).
lex(frappe,v,frapper,[morph(frapper,e),endvowel]).
lex(frappons,v,frapper,[morph(frapper,ons)]).
lex(frappez,v,frapper,[morph(frapper,ez)]).
lex(frappent,v,frapper,[morph(frapper,ent)]).
lex(frappions,v,frapper,[morph(frapper,ions1)]).
lex(frappiez,v,frapper,[morph(frapper,iez1)]).
lex(frapp,v,frapper,[morph(frapper,ed(2)),agr([[],[sg,m],m])]).
lex(frappe,v,frapper,[morph(frapper,ed(2)),agr([[],[sg,m],f])]).
lex(frapps,v,frapper,[morph(frapper,ed(2)),agr([[],pl,m])]).
lex(frappes,v,frapper,[morph(frapper,ed(2)),agr([[],pl,f])]).

lex(insultant,v,insulter,[morph(insulter,ant)]).
lex(insultes,v,insulter,[morph(insulter,s1)]).
lex(insulte,v,insulter,[morph(insulter,e),endvowel]).
lex(insultons,v,insulter,[morph(insulter,ons)]).
lex(insultez,v,insulter,[morph(insulter,ez)]).
lex(insultent,v,insulter,[morph(insulter,ent)]).
lex(insultions,v,insulter,[morph(insulter,ions1)]).
lex(insultiez,v,insulter,[morph(insulter,iez1)]).
lex(insult,v,insulter,[morph(insulter,ed(2)),agr([[],[sg,m],m])]).
lex(insulte,v,insulter,[morph(insulter,ed(2)),agr([[],[sg,m],f])]).
lex(insults,v,insulter,[morph(insulter,ed(2)),agr([[],pl,m])]).
lex(insultes,v,insulter,[morph(insulter,ed(2)),agr([[],pl,f])]).

lex(instruisant,v,instruire,[morph(instruire,ant)]).
lex(instruis,v,instruire,[morph(instruire,s)]).
lex(instruit,v,instruire,[morph(instruire,t)]).
lex(instruisons,v,instruire,[morph(instruire,ons)]).
lex(instruisez,v,instruire,[morph(instruire,ez)]).
lex(instruisent,v,instruire,[morph(instruire,ent)]).
lex(instruises,v,instruire,[morph(instruire,s2)]).
lex(instruise,v,instruire,[morph(instruire,e1)]).
lex(instruisions,v,instruire,[morph(instruire,ions1)]).
lex(instruisiez,v,instruire,[morph(instruire,iez1)]).
lex(instruit,v,instruire,[morph(instruire,ed(2)),agr([[],[sg,m],m])]).
lex(instruite,v,instruire,[morph(instruire,ed(2)),agr([[],[sg,m],f])]).
lex(instruits,v,instruire,[morph(instruire,ed(2)),agr([[],pl,m])]).
lex(instruites,v,instruire,[morph(instruire,ed(2)),agr([[],pl,f])]).

lex(gardant,v,garder,[morph(garder,ant)]).
lex(gardes,v,garder,[morph(garder,s1)]).
lex(garde,v,garder,[morph(garder,e),endvowel]).
lex(gardons,v,garder,[morph(garder,ons)]).
lex(gardez,v,garder,[morph(garder,ez)]).
lex(gardent,v,garder,[morph(garder,ent)]).
lex(gardions,v,garder,[morph(garder,ions1)]).
lex(gardiez,v,garder,[morph(garder,iez1)]).
lex(gard,v,garder,[morph(garder,ed(2)),agr([[],[sg,m],m])]).
lex(garde,v,garder,[morph(garder,ed(2)),agr([[],[sg,m],f])]).
lex(gards,v,garder,[morph(garder,ed(2)),agr([[],pl,m])]).
lex(gardes,v,garder,[morph(garder,ed(2)),agr([[],pl,f])]).

lex(tuant,v,tuer,[morph(tuer,ant)]).
lex(tues,v,tuer,[morph(tuer,s1)]).
lex(tue,v,tuer,[morph(tuer,e),endvowel]).
lex(tuons,v,tuer,[morph(tuer,ons)]).
lex(tuez,v,tuer,[morph(tuer,ez)]).
lex(tuent,v,tuer,[morph(tuer,ent)]).
lex(tuions,v,tuer,[morph(tuer,ions1)]).
lex(tuiez,v,tuer,[morph(tuer,iez1)]).
lex(tu,v,tuer,[morph(tuer,ed(2)),agr([[],[sg,m],m])]).
lex(tue,v,tuer,[morph(tuer,ed(2)),agr([[],[sg,m],f])]).
lex(tus,v,tuer,[morph(tuer,ed(2)),agr([[],pl,m])]).
lex(tues,v,tuer,[morph(tuer,ed(2)),agr([[],pl,f])]).

lex(connaissant,v,connatre,[morph(connatre,ant)]).
lex(connais,v,connatre,[morph(connatre,s)]).
lex(connat,v,connatre,[morph(connatre,t)]).
lex(connaissons,v,connatre,[morph(connatre,ons)]).
lex(connaissez,v,connatre,[morph(connatre,ez)]).
lex(connaissent,v,connatre,[morph(connatre,ent)]).
lex(connaisses,v,connatre,[morph(connatre,s2)]).
lex(connaisse,v,connatre,[morph(connatre,e1)]).
lex(connaissions,v,connatre,[morph(connatre,ions1)]).
lex(connaissiez,v,connatre,[morph(connatre,iez1)]).
lex(connu,v,connatre,[morph(connatre,ed(2)),agr([[],[sg,m],m])]).
lex(connue,v,connatre,[morph(connatre,ed(2)),agr([[],[sg,m],f])]).
lex(connus,v,connatre,[morph(connatre,ed(2)),agr([[],pl,m])]).
lex(connues,v,connatre,[morph(connatre,ed(2)),agr([[],pl,f])]).

lex(sachant,v,savoir,[morph(savoir,ant)]).
lex(sais,v,savoir,[morph(savoir,s)]).
lex(sait,v,savoir,[morph(savoir,t)]).
lex(savons,v,savoir,[morph(savoir,ons)]).
lex(savez,v,savoir,[morph(savoir,ez)]).
lex(savent,v,savoir,[morph(savoir,ent)]).
lex(sache,v,savoir,[morph(savoir,sache)]).
lex(saches,v,savoir,[morph(savoir,saches)]).
lex(sachions,v,savoir,[morph(savoir,sachions)]).
lex(sachiez,v,savoir,[morph(savoir,sachiez)]).
lex(sachent,v,savoir,[morph(savoir,sachent)]).
lex(su,v,savoir,[morph(savoir,ed(2)),agr([3,[sg,m],m])]).
lex(sue,v,savoir,[morph(savoir,ed(2)),agr([3,[sg,m],f])]).
lex(sus,v,savoir,[morph(savoir,ed(2)),agr([3,pl,m])]).
lex(sues,v,savoir,[morph(savoir,ed(2)),agr([3,pl,f])]).

lex(riant,v,rire,[morph(rire,ant)]).
lex(ris,v,rire,[morph(rire,s)]).
lex(rit,v,rire,[morph(rire,t)]).
lex(rions,v,rire,[morph(rire,ons)]).
lex(riez,v,rire,[morph(rire,ez)]).
lex(rient,v,rire,[morph(rire,ent)]).
lex(rie,v,rire,[morph(rire,e1)]).
lex(ries,v,rire,[morph(rire,s2)]).
lex(riions,v,rire,[morph(rire,ions1)]).
lex(riiez,v,rire,[morph(rire,iez1)]).
lex(ri,v,rire,[morph(rire,ed(2))]).

lex(quittant,v,quitter,[morph(quitter,ant)]).
lex(quittes,v,quitter,[morph(quitter,s1)]).
lex(quitte,v,quitter,[morph(quitter,e),endvowel]).
lex(quittons,v,quitter,[morph(quitter,ons)]).
lex(quittez,v,quitter,[morph(quitter,ez)]).
lex(quittent,v,quitter,[morph(quitter,ent)]).
lex(quittions,v,quitter,[morph(quitter,ions1)]).
lex(quittiez,v,quitter,[morph(quitter,iez1)]).
lex(quitt,v,quitter,[morph(quitter,ed(2)),agr([[],[sg,m],m])]).
lex(quitte,v,quitter,[morph(quitter,ed(2)),agr([[],[sg,m],f])]).
lex(quitts,v,quitter,[morph(quitter,ed(2)),agr([[],pl,m])]).
lex(quittes,v,quitter,[morph(quitter,ed(2)),agr([[],pl,f])]).

lex(menant,v,mener,[morph(mener,ant)]).
lex(mnes,v,mener,[morph(mener,s1)]).
lex(mne,v,mener,[morph(mener,e),endvowel]).
lex(menons,v,mener,[morph(mener,ons)]).
lex(menez,v,mener,[morph(mener,ez)]).
lex(mnent,v,mener,[morph(mener,ent)]).
lex(menions,v,mener,[morph(mener,ions1)]).
lex(meniez,v,mener,[morph(mener,iez1)]).
lex(men,v,mener,[morph(mener,ed(2)),agr([[],[sg,m],m])]).
lex(mene,v,mener,[morph(mener,ed(2)),agr([[],[sg,m],f])]).
lex(mens,v,mener,[morph(mener,ed(2)),agr([[],pl,m])]).
lex(menes,v,mener,[morph(mener,ed(2)),agr([[],pl,f])]).

lex(aimant,v,aimer,[morph(aimer,ant)]).
lex(aimes,v,aimer,[morph(aimer,s1)]).
lex(aime,v,aimer,[morph(aimer,e),endvowel]).
lex(aimons,v,aimer,[morph(aimer,ons)]).
lex(aimez,v,aimer,[morph(aimer,ez)]).
lex(aiment,v,aimer,[morph(aimer,ent)]).
lex(aimions,v,aimer,[morph(aimer,ions1)]).
lex(aimiez,v,aimer,[morph(aimer,iez1)]).
lex(aim,v,aimer,[morph(aimer,ed(2)),agr([[],[sg,m],m])]).
lex(aime,v,aimer,[morph(aimer,ed(2)),agr([[],[sg,m],f])]).
lex(aims,v,aimer,[morph(aimer,ed(2)),agr([[],pl,m])]).
lex(aimes,v,aimer,[morph(aimer,ed(2)),agr([[],pl,f])]).

lex(rencontrant,v,rencontrer,[morph(rencontrer,ant)]).
lex(rencontres,v,rencontrer,[morph(rencontrer,s1)]).
lex(rencontre,v,rencontrer,[morph(rencontrer,e),endvowel]).
lex(rencontrons,v,rencontrer,[morph(rencontrer,ons)]).
lex(rencontrez,v,rencontrer,[morph(rencontrer,ez)]).
lex(rencontrent,v,rencontrer,[morph(rencontrer,ent)]).
lex(rencontrions,v,rencontrer,[morph(rencontrer,ions1)]).
lex(rencontriez,v,rencontrer,[morph(rencontrer,iez1)]).
lex(rencontr,v,rencontrer,[morph(rencontrer,ed(2)),arg([[],[sg,m],m])]).
lex(rencontre,v,rencontrer,[morph(rencontrer,ed(2)),arg([[],[sg,m],f])]).
lex(rencontrs,v,rencontrer,[morph(rencontrer,ed(2)),arg([[],pl,m])]).
lex(rencontres,v,rencontrer,[morph(rencontrer,ed(2)),arg([[],pl,f])]).

lex(mentionnant,v,mentionner,[morph(mentionner,ant)]).
lex(mentionnes,v,mentionner,[morph(mentionner,s1)]).
lex(mentionne,v,mentionner,[morph(mentionner,e),endvowel]).
lex(mentionnons,v,mentionner,[morph(mentionner,ons)]).
lex(mentionnez,v,mentionner,[morph(mentionner,ez)]).
lex(mentionnent,v,mentionner,[morph(mentionner,ent)]).
lex(mentionnions,v,mentionner,[morph(mentionner,ions1)]).
lex(mentionniez,v,mentionner,[morph(mentionner,iez1)]).
lex(mentionn,v,mentionner,[morph(mentionner,ed(2)),agr([[],[sg,m],m])]).
lex(mentionne,v,mentionner,[morph(mentionner,ed(2)),agr([[],[sg,m],f])]).
lex(mentionns,v,mentionner,[morph(mentionner,ed(2)),agr([[],pl,m])]).
lex(mentionnes,v,mentionner,[morph(mentionner,ed(2)),agr([[],pl,f])]).

lex(garant,v,garer,[morph(garer,ant)]).
lex(gares,v,garer,[morph(garer,s1)]).
lex(gare,v,garer,[morph(garer,e),endvowel]).
lex(garons,v,garer,[morph(garer,ons)]).
lex(garez,v,garer,[morph(garer,ez)]).
lex(garent,v,garer,[morph(garer,ent)]).
lex(garions,v,garer,[morph(garer,ions1)]).
lex(gariez,v,garer,[morph(garer,iez1)]).
lex(gar,v,garer,[morph(garer,ed(2)),agr([[],[sg,m],m])]).
lex(gare,v,garer,[morph(garer,ed(2)),agr([[],[sg,m],f])]).
lex(gars,v,garer,[morph(garer,ed(2)),agr([[],pl,m])]).
lex(gares,v,garer,[morph(garer,ed(2)),agr([[],pl,f])]).

lex(persuadant,v,persuader,[morph(persuader,ant)]).
lex(persuades,v,persuader,[morph(persuader,s1)]).
lex(persuade,v,persuader,[morph(persuader,e),endvowel]).
lex(persuadons,v,persuader,[morph(persuader,ons)]).
lex(persuadez,v,persuader,[morph(persuader,ez)]).
lex(persuadent,v,persuader,[morph(persuader,ent)]).
lex(persuadions,v,persuader,[morph(persuader,ions1)]).
lex(persuadiez,v,persuader,[morph(persuader,iez1)]).
lex(persuad,v,persuader,[morph(persuader,ed(2)),agr([[],[sg,m],m])]).
lex(persuade,v,persuader,[morph(persuader,ed(2)),agr([[],[sg,m],f])]).
lex(persuads,v,persuader,[morph(persuader,ed(2)),agr([[],pl,m])]).
lex(persuades,v,persuader,[morph(persuader,ed(2)),agr([[],pl,f])]).

lex(plaisant,v,plaire,[morph(plaire,ant)]).
lex(plais,v,plaire,[morph(plaire,s)]).
lex(plat,v,plaire,[morph(plaire,t)]).
lex(plaisons,v,plaire,[morph(plaire,ons)]).
lex(plaisez,v,plaire,[morph(plaire,ez)]).
lex(plaisent,v,plaire,[morph(plaire,ent)]).
lex(plaises,v,plaire,[morph(plaire,s2)]).
lex(plaise,v,plaire,[morph(plaire,e1)]).
lex(plaisions,v,plaire,[morph(plaire,ions1)]).
lex(plaisiez,v,plaire,[morph(plaire,iez1)]).
lex(plu,v,plaire,[morph(plaire,ed(2))]).

lex(promettant,v,promettre,[morph(promettre,ant)]).
lex(promets,v,promettre,[morph(promettre,s)]).
lex(promet,v,promettre,[morph(promettre,t)]).
lex(promettons,v,promettre,[morph(promettre,ons)]).
lex(promettez,v,promettre,[morph(promettre,ez)]).
lex(promettent,v,promettre,[morph(promettre,ent)]).
lex(promettes,v,promettre,[morph(promettre,s2)]).
lex(promette,v,promettre,[morph(promettre,e1)]).
lex(promettions,v,promettre,[morph(promettre,ions1)]).
lex(promettiez,v,promettre,[morph(promettre,iez1)]).
lex(promis,v,promettre,[morph(promettre,ed(2)),agr([[],[pl,sg,m],m])]).
lex(promise,v,promettre,[morph(promettre,ed(2)),agr([[],[sg,m],f])]).
lex(promises,v,promettre,[morph(promettre,ed(2)),agr([[],pl,f])]).

lex(mettant,v,mettre,[morph(mettre,ant)]).
lex(mets,v,mettre,[morph(mettre,s)]).
lex(met,v,mettre,[morph(mettre,t)]).
lex(mettons,v,mettre,[morph(mettre,ons)]).
lex(mettez,v,mettre,[morph(mettre,ez)]).
lex(mettent,v,mettre,[morph(mettre,ent)]).
lex(mettes,v,mettre,[morph(mettre,s2)]).
lex(mette,v,mettre,[morph(mettre,e1)]).
lex(mettions,v,mettre,[morph(mettre,ions1)]).
lex(mettiez,v,mettre,[morph(mettre,iez1)]).
lex(mis,v,mettre,[morph(mettre,ed(2)),agr([[],[pl,sg,m],m])]).
lex(mise,v,mettre,[morph(mettre,ed(2)),agr([[],[sg,m],f])]).
lex(mises,v,mettre,[morph(mettre,ed(2)),agr([[],pl,f])]).

lex(pleut,v,pleuvoir,[morph(pleuvoir,t)]).
lex(plu,v,pleuvoir,[morph(pleuvoir,ed(2))]).
lex(pleuvait,v,pleuvoir,[morph(pleuvoir,ed(1))]).
lex(pleuve,v,pleuvoir,[morph(pleuvoir,e1)]).

lex(lisant,v,lire,[morph(lire,ant)]).
lex(lis,v,lire,[morph(lire,s)]).
lex(lit,v,lire,[morph(lire,t)]).
lex(lisons,v,lire,[morph(lire,ons)]).
lex(lisez,v,lire,[morph(lire,ez)]).
lex(lises,v,lire,[morph(lire,s2)]).
lex(lise,v,lire,[morph(lire,e1)]).
lex(lisions,v,lire,[morph(lire,ions1)]).
lex(lisiez,v,lire,[morph(lire,iez1)]).
lex(lu,v,lire,[morph(lire,ed(2)),agr([[],[sg,m],m])]).
lex(lue,v,lire,[morph(lire,ed(2)),agr([[],[sg,m],f])]).
lex(lus,v,lire,[morph(lire,ed(2)),agr([[],pl,m])]).
lex(lues,v,lire,[morph(lire,ed(2)),agr([[],pl,f])]).

lex(enlevant,v,enlever,[morph(enlever,ant)]).
lex(enlves,v,enlever,[morph(enlever,s1)]).
lex(enlve,v,enlever,[morph(enlever,e),endvowel]).
lex(enlevons,v,enlever,[morph(enlever,ons)]).
lex(enlevez,v,enlever,[morph(enlever,ez)]).
lex(enlvent,v,enlever,[morph(enlever,ent)]).
lex(enlvions,v,enlever,[morph(enlever,ions1)]).
lex(enlviez,v,enlever,[morph(enlever,iez1)]).
lex(enlev,v,enlever,[morph(enlever,ed(2)),agr([[],[sg,m],m])]).
lex(enleve,v,enlever,[morph(enlever,ed(2)),agr([[],[sg,m],f])]).
lex(enlevs,v,enlever,[morph(enlever,ed(2)),agr([[],pl,m])]).
lex(enleves,v,enlever,[morph(enlever,ed(2)),agr([[],pl,f])]).

lex(dmissionnant,v,dmissionner,[morph(dmissionner,ant)]).
lex(dmissionnes,v,dmissionner,[morph(dmissionner,s1)]).
lex(dmissionne,v,dmissionner,[morph(dmissionner,e),endvowel]).
lex(dmissionnons,v,dmissionner,[morph(dmissionner,ons)]).
lex(dmissionnez,v,dmissionner,[morph(dmissionner,ez)]).
lex(dmissionnent,v,dmissionner,[morph(dmissionner,ent)]).
lex(dmissionnions,v,dmissionner,[morph(dmissionner,ions1)]).
lex(dmissionniez,v,dmissionner,[morph(dmissionner,iez1)]).
lex(dmissionn,v,dmissionner,[morph(dmissionner,ed(2))]).

lex(disant,v,dire,[morph(dire,ant)]).
lex(dis,v,dire,[morph(dire,s)]).
lex(dit,v,dire,[morph(dire,t)]).
lex(disons,v,dire,[morph(dire,ons)]).
lex(dites,v,dire,[morph(dire,dites)]).
lex(disent,v,dire,[morph(dire,ent)]).
lex(dise,v,dire,[morph(dire,e1)]).
lex(dises,v,dire,[morph(dire,s2)]).
lex(disions,v,dire,[morph(dire,ions1)]).
lex(disiez,v,dire,[morph(dire,iez1)]).
lex(dit,v,dire,[morph(dire,ed(2)),agr([[],[sg,m],m])]).
lex(dits,v,dire,[morph(dire,ed(2)),agr([[],pl,m])]).
lex(dite,v,dire,[morph(dire,ed(2)),agr([[],[sg,m],f])]).
lex(dites,v,dire,[morph(dire,ed(2)),agr([[],pl,f])]).

lex(voyant,v,voir,[morph(voir,ant)]).
lex(vois,v,voir,[morph(voir,s)]).
lex(voit,v,voir,[morph(voir,t)]).
lex(voyons,v,voir,[morph(voir,ons)]).
lex(voyez,v,voir,[morph(voir,ez)]).
lex(voient,v,voir,[morph(voir,ent)]).
lex(voie,v,voir,[morph(voir,e1)]).
lex(voies,v,voir,[morph(voir,s2)]).
lex(voyions,v,voir,[morph(voir,ions1)]).
lex(voyiez,v,voir,[morph(voir,iez1)]).
lex(vu,v,voir,[morph(voir,ed(2)),agr([[],[sg,m],m])]).
lex(vue,v,voir,[morph(voir,ed(2)),agr([[],[sg,m],f])]).
lex(vus,v,voir,[morph(voir,ed(2)),agr([[],pl,m])]).
lex(vues,v,voir,[morph(voir,ed(2)),agr([[],pl,f])]).

lex(semblant,v,sembler,[morph(sembler,ant)]).
lex(sembles,v,sembler,[morph(sembler,s1)]).
lex(semble,v,sembler,[morph(sembler,e),endvowel]).
lex(semblons,v,sembler,[morph(sembler,ons)]).
lex(semblez,v,sembler,[morph(sembler,ez)]).
lex(semblions,v,sembler,[morph(sembler,ions1)]).
lex(sembliez,v,sembler,[morph(sembler,iez1)]).
lex(semblent,v,sembler,[morph(sembler,ent)]).
lex(sembl,v,sembler,[morph(sembler,ed(2))]).

lex(dormant,v,dormir,[morph(dormir,ant)]).
lex(dors,v,dormir,[morph(dormir,s)]).
lex(dort,v,dormir,[morph(dormir,t)]).
lex(dormons,v,dormir,[morph(dormir,ons)]).
lex(dormez,v,dormir,[morph(dormir,ez)]).
lex(dorment,v,dormir,[morph(dormir,ent)]).
lex(dorme,v,dormir,[morph(dormir,e1)]).
lex(dormes,v,dormir,[morph(dormir,s2)]).
lex(dormions,v,dormir,[morph(dormir,ions1)]).
lex(dormiez,v,dormir,[morph(dormir,iez1)]).
lex(dormi,v,dormir,[morph(dormir,ed(2))]).

lex(neige,v,neiger,[morph(neiger,e),endvowel]).
lex(neig,v,neiger,[morph(neiger,ed(2))]).
lex(neigeait,v,neiger,[morph(neiger,ed(1))]).

lex(resolvant,v,rsoudre,[morph(rsoudre,ant)]).
lex(resouds,v,rsoudre,[morph(rsoudre,s)]).
lex(resoud,v,rsoudre,[morph(rsoudre,t)]).
lex(resolvons,v,rsoudre,[morph(rsoudre,ons)]).
lex(resolvez,v,rsoudre,[morph(rsoudre,ez)]).
lex(resolvent,v,rsoudre,[morph(rsoudre,ent)]).
lex(resolves,v,rsoudre,[morph(rsoudre,s2)]).
lex(resolve,v,rsoudre,[morph(rsoudre,e1)]).
lex(resolvions,v,rsoudre,[morph(rsoudre,ions1)]).
lex(resolviez,v,rsoudre,[morph(rsoudre,iez1)]).
lex(resolu,v,rsoudre,[morph(rsoudre,ed(2)),agr([[],[sg,m],m])]).
lex(resolue,v,rsoudre,[morph(rsoudre,ed(2)),agr([[],[sg,m],f])]).
lex(resolus,v,rsoudre,[morph(rsoudre,ed(2)),agr([[],pl,m])]).
lex(resolues,v,rsoudre,[morph(rsoudre,ed(2)),agr([[],pl,f])]).

lex(supportant,v,supporter,[morph(supporter,ant)]).
lex(supportes,v,supporter,[morph(supporter,s1)]).
lex(supporte,v,supporter,[morph(supporter,e),endvowel]).
lex(supportons,v,supporter,[morph(supporter,ons)]).
lex(supportez,v,supporter,[morph(supporter,ez)]).
lex(supportent,v,supporter,[morph(supporter,ent)]).
lex(supportions,v,supporter,[morph(supporter,ions1)]).
lex(supportiez,v,supporter,[morph(supporter,iez1)]).
lex(support,v,supporter,[morph(supporter,ed(2)),agr([[],[sg,m],m])]).
lex(supporte,v,supporter,[morph(supporter,ed(2)),agr([[],[sg,m],f])]).
lex(supports,v,supporter,[morph(supporter,ed(2)),agr([[],pl,m])]).
lex(supportes,v,supporter,[morph(supporter,ed(2)),agr([[],pl,f])]).

lex(commencant,v,commencer,[morph(commencer,ant)]).
lex(commences,v,commencer,[morph(commencer,s1)]).
lex(commence,v,commencer,[morph(commencer,e),endvowel]).
lex(commenons,v,commencer,[morph(commencer,ons)]).
lex(commencez,v,commencer,[morph(commencer,ez)]).
lex(commencions,v,commencer,[morph(commencer,ions1)]).
lex(commenciez,v,commencer,[morph(commencer,iez1)]).
lex(commencent,v,commencer,[morph(commencer,ent)]).
lex(commenc,v,commencer,[morph(commencer,ed(2)),agr([[],[sg,m],m])]).
lex(commence,v,commencer,[morph(commencer,ed(2)),agr([[],[sg,m],f])]).
lex(commencs,v,commencer,[morph(commencer,ed(2)),agr([[],pl,m])]).
lex(commences,v,commencer,[morph(commencer,ed(2)),agr([[],pl,f])]).

lex(tonnant,v,tonner,[morph(tonner,ant)]).
lex(tonnes,v,tonner,[morph(tonner,s1)]).
lex(tonne,v,tonner,[morph(tonner,e),endvowel]).
lex(tonnons,v,tonner,[morph(tonner,ons)]).
lex(tonnez,v,tonner,[morph(tonner,ez)]).
lex(tonnent,v,tonner,[morph(tonner,ent)]).
lex(tonnions,v,tonner,[morph(tonner,ions1)]).
lex(tonniez,v,tonner,[morph(tonner,iez1)]).
lex(tonn,v,tonner,[morph(tonner,ed(2)),agr([[],[sg,m],m])]).
lex(tonne,v,tonner,[morph(tonner,ed(2)),agr([[],[sg,m],f])]).
lex(tonns,v,tonner,[morph(tonner,ed(2)),agr([[],pl,m])]).
lex(tonnes,v,tonner,[morph(tonner,ed(2)),agr([[],pl,f])]).

lex(parlant,v,parler,[morph(parler,ant)]).
lex(parles,v,parler,[morph(parler,s1)]).
lex(parle,v,parler,[morph(parler,e),endvowel]).
lex(parlons,v,parler,[morph(parler,ons)]).
lex(parlez,v,parler,[morph(parler,ez)]).
lex(parlions,v,parler,[morph(parler,ions1)]).
lex(parliez,v,parler,[morph(parler,iez1)]).
lex(parlent,v,parler,[morph(parler,ent)]).
lex(parl,v,parler,[morph(parler,ed(2))]).

lex(racontant,v,raconter,[morph(raconter,ant)]).
lex(racontes,v,raconter,[morph(raconter,s1)]).
lex(raconte,v,raconter,[morph(raconter,e),endvowel]).
lex(racontons,v,raconter,[morph(raconter,ons)]).
lex(racontez,v,raconter,[morph(raconter,ez)]).
lex(racontent,v,raconter,[morph(raconter,ent)]).
lex(racontions,v,raconter,[morph(raconter,ions1)]).
lex(racontiez,v,raconter,[morph(raconter,iez1)]).
lex(racont,v,raconter,[morph(raconter,ed(2)),agr([[],[sg,m],m])]).
lex(raconte,v,raconter,[morph(raconter,ed(2)),agr([[],[sg,m],f])]).
lex(raconts,v,raconter,[morph(raconter,ed(2)),agr([[],pl,m])]).
lex(racontes,v,raconter,[morph(raconter,ed(2)),agr([[],pl,f])]).

lex(essayant,v,essayer,[morph(essayer,ant)]).
lex(essaies,v,essayer,[morph(essayer,s1)]).
lex(essaie,v,essayer,[morph(essayer,e),endvowel]).
lex(essayons,v,essayer,[morph(essayer,ons)]).
lex(essayez,v,essayer,[morph(essayer,ez)]).
lex(essaient,v,essayer,[morph(essayer,ent)]).
lex(essayions,v,essayer,[morph(essayer,ions1)]).
lex(essayiez,v,essayer,[morph(essayer,iez1)]).
lex(essay,v,essayer,[morph(essayer,ed(2)),agr([[],[sg,m],m])]).
lex(essaye,v,essayer,[morph(essayer,ed(2)),agr([[],[sg,m],f])]).
lex(essays,v,essayer,[morph(essayer,ed(2)),agr([[],pl,m])]).
lex(essayes,v,essayer,[morph(essayer,ed(2)),agr([[],pl,f])]).

lex(pensant,v,penser,[morph(penser,ant)]).
lex(penses,v,penser,[morph(penser,s1)]).
lex(pense,v,penser,[morph(penser,e),endvowel]).
lex(pensons,v,penser,[morph(penser,ons)]).
lex(pensez,v,penser,[morph(penser,ez)]).
lex(pensent,v,penser,[morph(penser,ent)]).
lex(pensions,v,penser,[morph(penser,ions1)]).
lex(pensiez,v,penser,[morph(penser,iez1)]).
lex(pens,v,penser,[morph(penser,ed(2)),agr([[],[sg,m],m])]).
lex(pense,v,penser,[morph(penser,ed(2)),agr([[],[sg,m],f])]).
lex(penss,v,penser,[morph(penser,ed(2)),agr([[],pl,m])]).
lex(penses,v,penser,[morph(penser,ed(2)),agr([[],pl,f])]).

lex(comprenant,v,comprendre,[morph(comprendre,ant)]).
lex(comprends,v,comprendre,[morph(comprendre,s)]).
lex(comprend,v,comprendre,[morph(comprendre,t)]).
lex(comprenons,v,comprendre,[morph(comprendre,ons)]).
lex(comprenez,v,comprendre,[morph(comprendre,ez)]).
lex(comprennent,v,comprendre,[morph(comprendre,ent)]).
lex(comprenne,v,comprendre,[morph(comprendre,e1)]).
lex(comprennes,v,comprendre,[morph(comprendre,s2)]).
lex(comprenions,v,comprendre,[morph(comprendre,ions1)]).
lex(compreniez,v,comprendre,[morph(comprendre,iez1)]).
lex(compris,v,comprendre,[morph(comprendre,ed(2)),agr([[],[pl,sg,m],m])]).
lex(comprise,v,comprendre,[morph(comprendre,ed(2)),agr([[],[sg,m],f])]).
lex(comprises,v,comprendre,[morph(comprendre,ed(2)),agr([[],pl,f])]).

lex(visitant,v,visiter,[morph(visiter,ant)]).
lex(visites,v,visiter,[morph(visiter,s1)]).
lex(visite,v,visiter,[morph(visiter,e),endvowel]).
lex(visitons,v,visiter,[morph(visiter,ons)]).
lex(visitez,v,visiter,[morph(visiter,ez)]).
lex(visitent,v,visiter,[morph(visiter,ent)]).
lex(visitions,v,visiter,[morph(visiter,ions1)]).
lex(visitiez,v,visiter,[morph(visiter,iez1)]).
lex(visit,v,visiter,[morph(visiter,ed(2)),agr([[],[sg,m],m])]).
lex(visite,v,visiter,[morph(visiter,ed(2)),agr([[],[sg,m],f])]).
lex(visits,v,visiter,[morph(visiter,ed(2)),agr([[],pl,m])]).
lex(visites,v,visiter,[morph(visiter,ed(2)),agr([[],pl,f])]).

lex(voulant,v,vouloir,[morph(vouloir,ant)]).
lex(veux,v,vouloir,[morph(vouloir,s)]).
lex(veut,v,vouloir,[morph(vouloir,t)]).
lex(voulons,v,vouloir,[morph(vouloir,ons)]).
lex(voulez,v,vouloir,[morph(vouloir,ez)]).
lex(veulent,v,vouloir,[morph(vouloir,ent)]).
lex(veuille,v,vouloir,[morph(vouloir,e1)]).
lex(veuilles,v,vouloir,[morph(vouloir,s2)]).
lex(veuillent,v,vouloir,[morph(vouloir,ent1)]).
lex(voulions,v,vouloir,[morph(vouloir,ions1)]).
lex(vouliez,v,vouloir,[morph(vouloir,iez1)]).
lex(voulu,v,vouloir,[morph(vouloir,ed(2)),agr([[],[sg,m],m])]).
lex(voulue,v,vouloir,[morph(vouloir,ed(2)),agr([[],[sg,m],f])]).
lex(voulus,v,vouloir,[morph(vouloir,ed(2)),agr([[],pl,m])]).
lex(voulues,v,vouloir,[morph(vouloir,ed(2)),agr([[],pl,f])]).

lex(gagnant,v,gagner,[morph(gagner,ant)]).
lex(gagnes,v,gagner,[morph(gagner,s1)]).
lex(gagne,v,gagner,[morph(gagner,e),endvowel]).
lex(gagnons,v,gagner,[morph(gagner,ons)]).
lex(gagnez,v,gagner,[morph(gagner,ez)]).
lex(gagnent,v,gagner,[morph(gagner,ent)]).
lex(gagnions,v,gagner,[morph(gagner,ions1)]).
lex(gagniez,v,gagner,[morph(gagner,iez1)]).
lex(gagn,v,gagner,[morph(gagner,ed(2)),agr([[],[sg,m],m])]).
lex(gagne,v,gagner,[morph(gagner,ed(2)),agr([[],[sg,m],f])]).
lex(gagns,v,gagner,[morph(gagner,ed(2)),agr([[],pl,m])]).
lex(gagnes,v,gagner,[morph(gagner,ed(2)),agr([[],pl,f])]).

lex(travaillant,v,travailler,[morph(travailler,ant)]).
lex(travailles,v,travailler,[morph(travailler,s1)]).
lex(travaille,v,travailler,[morph(travailler,e),endvowel]).
lex(travaillons,v,travailler,[morph(travailler,ons)]).
lex(travaillez,v,travailler,[morph(travailler,ez)]).
lex(travaillent,v,travailler,[morph(travailler,ent)]).
lex(travaillions,v,travailler,[morph(travailler,ions1)]).
lex(travailliez,v,travailler,[morph(travailler,iez1)]).
lex(travaill,v,travailler,[morph(travailler,ed(2)),agr([[],[sg,m],m])]).
lex(travaille,v,travailler,[morph(travailler,ed(2)),agr([[],[sg,m],f])]).
lex(travaills,v,travailler,[morph(travailler,ed(2)),agr([[],pl,m])]).
lex(travailles,v,travailler,[morph(travailler,ed(2)),agr([[],pl,f])]).

lex(crivant,v,crire,[morph(crire,ant)]).
lex(cris,v,crire,[morph(crire,s)]).
lex(crit,v,crire,[morph(crire,t)]).
lex(crivons,v,crire,[morph(crire,ons)]).
lex(crivez,v,crire,[morph(crire,ez)]).
lex(crivent,v,crire,[morph(crire,ent)]).
lex(crive,v,crire,[morph(crire,e1)]).
lex(crives,v,crire,[morph(crire,s2)]).
lex(crivions,v,crire,[morph(crire,ions1)]).
lex(criviez,v,crire,[morph(crire,iez1)]).
lex(crit,v,crire,[morph(crire,ed(2)),agr([[],[sg,m],m])]).
lex(crite,v,crire,[morph(crire,ed(2)),agr([[],[sg,m],f])]).
lex(crits,v,crire,[morph(crire,ed(2)),agr([[],pl,m])]).
lex(crites,v,crire,[morph(crire,ed(2)),agr([[],pl,f])]).

lex(venant,v,venir,[morph(venir,ant)]).
lex(viens,v,venir,[morph(venir,s)]).
lex(vient,v,venir,[morph(venir,t)]).
lex(venons,v,venir,[morph(venir,ons)]).
lex(venez,v,venir,[morph(venir,ez)]).
lex(viennent,v,venir,[morph(venir,ent)]).
lex(viennes,v,venir,[morph(venir,s2)]).
lex(vienne,v,venir,[morph(venir,e1)]).
lex(venions,v,venir,[morph(venir,ions1)]).
lex(veniez,v,venir,[morph(venir,iez1)]).
lex(venu,v,venir,[morph(venir,ed(2)),agr([[],[sg,m],m])]).
lex(venue,v,venir,[morph(venir,ed(2)),agr([[],[sg,m],f])]).
lex(venus,v,venir,[morph(venir,ed(2)),agr([[],pl,m])]).
lex(venues,v,venir,[morph(venir,ed(2)),agr([[],pl,f])]).

lex(allant,v,aller,[morph(aller,ant)]).
lex(vais,v,aller,[morph(aller,vais)]).
lex(vas,v,aller,[morph(aller,vas)]).
lex(va,v,aller,[morph(aller,va),endvowel]).
lex(allons,v,aller,[morph(aller,ons)]).
lex(allez,v,aller,[morph(aller,ez)]).
lex(vont,v,aller,[morph(aller,vont)]).
lex(aille,v,aller,[morph(aller,e1)]).
lex(aillent,v,aller,[morph(aller,ent1)]).
lex(ailles,v,aller,[morph(aller,s2)]).
lex(allions,v,aller,[morph(aller,ions1)]).
lex(alliez,v,aller,[morph(aller,iez1)]).
lex(all,v,aller,[morph(aller,ed(2)),agr([[],[sg,m],m])]).
lex(alle,v,aller,[morph(aller,ed(2)),agr([[],[sg,m],f])]).
lex(alls,v,aller,[morph(aller,ed(2)),agr([[],pl,m])]).
lex(alles,v,aller,[morph(aller,ed(2)),agr([[],pl,f])]).

lex(pouvant,v,pouvoir,[morph(pouvoir,ant)]).
lex(peux,v,pouvoir,[morph(pouvoir,s)]).
lex(peut,v,pouvoir,[morph(pouvoir,t)]).
lex(pouvons,v,pouvoir,[morph(pouvoir,ons)]).
lex(pouvez,v,pouvoir,[morph(pouvoir,ez)]).
lex(peuvent,v,pouvoir,[morph(pouvoir,ent)]).
lex(puisses,v,pouvoir,[morph(pouvoir,s2)]).
lex(puisse,v,pouvoir,[morph(pouvoir,e1)]).
lex(puissions,v,pouvoir,[morph(pouvoir,puissions)]).
lex(puissiez,v,pouvoir,[morph(pouvoir,puissiez)]).
lex(puissent,v,pouvoir,[morph(pouvoir,puissent)]).
lex(pu,v,pouvoir,[morph(pouvoir,ed(2)),agr([[],[sg,m],m])]).


% CONTRACTIONS

:- initialization(no contraction(_,_)).

contraction(X,''''+Y,[X+[e,a]=word,Y=word]).
contraction(X,t,[X=morph(_,_),tx]).
contraction(tx,X,[X=adjoin(_,_)]).


% MISCELLANEOUS

nounFeatures(Base,F) :-	lex(Base,n,F1),	pick1(agr(_),F1,F).

verbFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).

verbToNounFeatures(Base,F) :-
	lex(Base,v,F1),
	pick1(morph(Base,_),F1,F).
%	pickNF1(ecm(_),F2,F).	e.g. belief still needs to properly govern ...

%% MAPS MORPHOLOGY INTO SYNTACTIC FEATURES
%%	Verb morphology and Agreement
%% 
%%	Form	Tense		AGR
%%	base 	infinitival
%%	-s	past(-) 	agr(1,2,sg)
%%      -es     past(-)         agr(2,sg)
%%      -e      past(-)         agr(1,3,sg)
%%      -t      past(-)         agr(3,sg)
%%      -ons    past(-)         agr(1,pl)
%%      -ez     past(-)         agr(2,pl)
%%      -ent    past(-)         agr(3,pl)
%%	-ed(1) 	past(+)		agr(_)
%% 	-ant	infinitival
%%	-ed(2)	perf, passive
%%	suis	past(-) 	agr(1,sg)
%%	es	past(-) 	agr(2,sg)
%%      est     past(-)         agr(3,sg)
%%      sommes  past(-)         agr(1,pl)
%%      tes    past(-)         agr(2,pl)
%%      sont    past(-)         agr(3,pl)
%%	was	past(+)		agr([1,3],sg)
%%	were	past(+) 	agr(not 1,3,sg)
%%      ai      past(-)         agr(1,sg)
%%      as      past(-)         agr(2,sg)
%%      a       past(-)         agr(3,sg)
%%      ont     past(-)         agr(3,pl)
%%      dites   past(-)         agr(2,pl)




% non-deterministic
vMorphToFs(_,def(F),[F,agr(_)]) :-		% defective: modal auxiliaries
	!.
vMorphToFs(_,[],[]).				% infinitival form
vMorphToFs(Base,Form,Features) :-
	\+ irregular(Base,Form),
	tns(Form,F1),
	agr(Form,F2),
	asp(Form,F3),
	append(F1,F2,F3,Features).


% base form is irregular: (regular [] -> past(-) not 3sg) 
irregular(Base,[]) :- irregular(Base).

% verb form <-> AGR feature
agr([],[]).
agr(s,[agr([[1,2],[sg,m],[]])]).
agr(s1,[agr([2,sg,[]])]).
agr(s2,[agr([2,sg,[]])]).
agr(e,[agr([[1,3],[sg,m],[]])]).
agr(e1,[agr([[1,3],[sg,m],[]])]).
agr(t,[agr([3,[sg,m],[]])]).
agr(ons,[agr([1,pl,[]])]).
agr(ez,[agr([2,pl,[]])]).
agr(ent,[agr([3,pl,[]])]).
agr(ent1,[agr([3,pl,[]])]).
%agr(ed(1),[agr(_)]).				
agr(ant,[]).
agr(ed(2),[]).					
agr(suis,[agr([1,sg,[]])]).
agr(es,[agr([2,sg,[]])]).
agr(est,[agr([3,[sg,m],[]])]).
agr(sommes,[agr([1,pl,[]])]).
agr(tes,[agr([2,pl,[]])]).
agr(sont,[agr([3,pl,[]])]).
agr(ai,[agr([1,sg,[]])]).
agr(as,[agr([2,sg,[]])]).
agr(a,[agr([3,[sg,m],[]])]).
agr(ont,[agr([3,pl,[]])]).
%agr(was,[agr([[1,3],[sg,m],[m,f,n]])]).
%agr(were,[agr(not([[1,3],[sg,m],[m,f,n]]))]).
agr(vais,[agr([1,sg,[]])]).
agr(vas,[agr([2,sg,[]])]).
agr(va,[agr([3,[sg,m],[]])]).
agr(vont,[agr([3,pl,[]])]).
agr(ions1,[agr,([1,pl,[]])]).
agr(iez1,[agr([2,pl,[]])]).	
agr(sois,[agr([[1,2],sg,[]])]).
agr(soit,[agr([3,sg,[]])]).		
agr(soyons,[agr([1,pl,[]])]).
agr(soyez,[agr([2,pl,[]])]).
agr(soient,[agr([3,pl,[]])]).
agr(aie,[agr([1,sg,[]])]).
agr(aies,[agr([2,sg,[]])]).
agr(ait,[agr([3,sg,[]])]).
agr(ayons,[agr([1,pl,[]])]).
agr(ayez,[agr([2,pl,[]])]).
agr(aient,[agr([3,pl,[]])]).
agr(sache,[agr([[1,3],sg,[]])]).
agr(saches,[agr([2,sg,[]])]).
agr(sachions,[agr([1,pl,[]])]).
agr(sachiez,[agr([2,pl,[]])]).
agr(sachent,[agr([3,pl,[]])]).
agr(puisse,[agr([[1,3],sg,[]])]).
agr(puisses,[agr([2,sg,[]])]).
agr(puissions,[agr([1,pl,[]])]).
agr(puissiez,[agr([2,pl,[]])]).
agr(puissent,[agr([3,pl,[]])]).

% verb form <-> Tense feature
tns([],[]).
tns(s,[past(-)]).
tns(s1,[past(-)]).
tns(s2,[past(-)]).
tns(e,[past(-)]).
tns(e1,[past(-)]).
tns(t,[past(-)]).
tns(ons,[past(-)]).
tns(ez,[past(-)]).
tns(ent,[past(-)]).		
tns(ent1,[past(-)]).		
tns(ed(1),[past(+)]).	
tns(ant,[]).			
tns(ed(2),[]).			
tns(suis,[past(-)]).
tns(es,[past(-)]).		
tns(est,[past(-)]).
tns(sommes,[past(-)]).
tns(tes,[past(-)]).
tns(sont,[past(-)]).							
tns(ai,[past(-)]).
tns(as,[past(-)]).
tns(a,[past(-)]).
tns(ont,[past(-)]).
tns(was,[past(+)]).		
tns(were,[past(+)]).
tns(vais,[past(-)]).
tns(vas,[past(-)]).
tns(va,[past(-)]).
tns(vont,[past(-)]).
tns(ions1,[past(-),past(+)]).
tns(iez1,[past(-),past(+)]).	
tns(sois,[past(-)]).
tns(soit,[past(-)]).		
tns(soyons,[past(-)]).
tns(soyez,[past(-)]).
tns(soient,[past(-)]).
tns(aie,[past(-)]).
tns(aies,[past(-)]).
tns(ait,[past(-)]).
tns(ayons,[past(-)]).
tns(ayez,[past(-)]).
tns(aient,[past(-)]).
tns(sache,[past(-)]).
tns(saches,[past(-)]).
tns(sachions,[past(-)]).
tns(sachiez,[past(-)]).
tns(sachent,[past(-)]).
tns(puisse,[past(-)]).
tns(puisses,[past(-)]).
tns(puissions,[past(-)]).
tns(puissiez,[past(-)]).
tns(puissent,[past(-)]).

% verb form <-> aspect feature
asp([],[]).
asp(s1,[subj(+),subj(-)]).
asp(s2,[subj(+)]).
asp(s,[subj(-)]).
asp(t,[subj(-)]).
asp(ons,[subj(-)]).
asp(ez,[subj(-)]).
asp(e,[subj(+),subj(-)]).
asp(e1,[subj(+)]).
asp(ent,[subj(+),subj(-)]).		
asp(ent1,[subj(+)]).		
asp(ions1,[subj(+)]).
asp(iez1,[subj(+)]).	
asp(ant,[]).			
asp(ed(2),[]).			
asp(sois,[subj(+)]).
asp(soit,[subj(+)]).		
asp(soyons,[subj(+)]).
asp(soyez,[subj(+)]).
asp(soient,[subj(+)]).
asp(aie,[subj(+)]).
asp(aies,[subj(+)]).
asp(ait,[subj(+)]).
asp(ayons,[subj(+)]).
asp(ayez,[subj(+)]).
asp(aient,[subj(+)]).
asp(vais,[subj(-)]).
asp(vas,[subj(-)]).
asp(va,[subj(-)]).
asp(vont,[subj(-)]).
asp(sache,[subj(+)]).
asp(saches,[subj(+)]).
asp(sachions,[subj(+)]).
asp(sachiez,[subj(+)]).
asp(sachent,[subj(+)]).
asp(puisse,[subj(+)]).
asp(puisses,[subj(+)]).
asp(puissions,[subj(+)]).
asp(puissiez,[subj(+)]).
asp(puissent,[subj(+)]).
asp(suis,[subj(-)]).
asp(es,[subj(-)]).		
asp(est,[subj(-)]).
asp(sommes,[subj(-)]).
asp(tes,[subj(-)]).
asp(sont,[subj(-)]).							
asp(ai,[subj(-)]).
asp(as,[subj(-)]).
asp(a,[subj(-)]).
asp(ont,[subj(-)]).

%%% PREDICATES FOR INDIVIDUAL LEXICAL ITEMS

agrConstraint(X) :- intersectAGR([[],[],[n]],X) if cat(X,np).

inf(V,[]) :- V has_feature morph(_,[]).

inf(V,ant) :- V has_feature morph(_,ant).

% Infinitival verbs <=> morph(_,[])
	
verbInf(C,Fs,Fsp) :-
	C == v
    ->  in1(morph(_,[]),Fs),
	Fsp = [inf([])|Fs]
    ;   Fsp = Fs.

% All object clitics have these lexical properties
%
% NB. 
% 1. Must ensure that the noCasemark(+) feature gets to the real predicate
% 2. Must disallow "Pierre a le vu" i.e. prevent clitic from 
%    attaching to a perfective verb.


objCliticFeatures1([adjoin(v,left),argument,
	adjR([not(feature(morph(_,ed(2)))), not(feature(noAcc)), 
	addFeature(transparent(ecp)), 
	goal(propagateNoCase(noCasemark(+),X),X)])]).

objCliticFeatures2([adjoin(v,left),argument,
	adjR([not(feature(morph(_,ed(2)))),not(feature(noAcc)), 
	addFeature(transparent(ecp)), addFeature(noDat),
	goal(propagateNoCase(noCasemark(+),X),X)])]).

objCliticFeatures([adjoin(v,left),argument,
	adjR([not(feature(morph(_,ed(2)))), 
	addFeatures(transparent(ecp)),
	goal(propagateNoCase(noCasemark(+),X),X)])]).

objCliticFeaturesR([adjoin(v,left),argument,
	adjR([not(feature(morph(_,ed(2)))), not(feature(noAcc)), 
	addFeature(transparent(ecp)), 
	goal(propagateReflexive(X),X),
	goal(propagateNoCase(noCasemark(+),Y),Y)])]).


iobjCliticFeatures1([adjoin(v,left),indirectObject,argument,
	adjR([not(feature(morph(_,ed(2)))),not(feature(noCasemark(+))),
	addFeature(transparent(ecp)),
	goal(propagateNoCase(noInherentCasemark,X),X)])]).

iobjCliticFeatures2([adjoin(v,left),indirectObject,argument,
	adjR([not(feature(morph(_,ed(2)))), not(feature(noDat)),
	addFeatures([transparent(ecp),noAcc]),
	goal(propagateNoCase(noInherentCasemark,X),X)])]).

iobjCliticFeaturesR([adjoin(v,left),indirectObject,argument,
	adjR([not(feature(morph(_,ed(2)))),
	not(feature(noDat)),
	addFeatures([transparent(ecp),noAcc]),
	goal(propagateReflexive(X),X),
	goal(propagateNoCase(noInherentCasemark,Y),Y)])]).

propagateNoCase(NoCaseFeature,X) :-
	X has_feature subcat(YP$Fs,AFs)
    ->  AFsp = [NoCaseFeature|AFs],
	addFeature(subcat(YP$Fs,AFsp),X)
    ;   addFeature(NoCaseFeature,X).


% Two cases:
% 1. Aux verb: must have feature reflexive
% 2. Regular verb: assign it the feature reflexive

propagateReflexive(X) :-
	X has_feature aux ->  
	X has_feature reflexive,
	addFeature(subcat(vp$[morph(_,ed(2)),grid(_,[_X|_])],[reflexive]),X)
	; addFeature(reflexive,X).


subjCliticFeatures(MCase,[morphC(MCase), 
	adjR([goal(imposeNoCase(noCasemark(+),X),X)])]).

imposeNoCase(NoCaseFeature,X) :-
	\+ X has_feature NoCaseFeature,
	addFeature(NoCaseFeature,X).
	
advCliticFeatures([adjoin(v,left),morphC(obl),
adjR([not(feature(morph(_,ed(2)))), addFeature(transparent(ecp))])]).
	
subjClFeaturesInv(MCase,[adjoin(v,right),morphC(MCase),
		 adjR([addFeatures([wh,adjR([goal(isMatrixLevel,_)])])])]).

takesDe(CP) :-
	CP has_feature pAttachR(feature(control))
	if (cat(CP,c2), IP complement_of CP, IP has_feature inf([])).

% [C que] assigns acc Case to its spec except when the spec is Op

quirkyAcc(XP) :-
	XP has_feature ec(Type),
	XP has_feature case(Case) with_constraint accExceptOp(Case,Type).

 accExceptOp(Case,Type) :-
	var(Case) -> 
	true
    ;   ((Case == nom, Type == op) ->
	true
    ;   Case = acc).


% accExceptOp(Case,Type) :-
%	(Case == nom, Type == op) ->
%	    true
%	; Case = acc.

% [C qui] cannot co-occur with Op 
% It assigns nom Case to its spec

quirkyNom(XP) :-
	XP has_feature ec(Type) with_constraint Type \== op,
	XP has_feature case(nom).

ensureSubj(CP) :-
	cat(CP,c2) ->
	IP complement_of CP,
	cat(IP,i2),
	(IP has_feature subj(+);IP has_feature inf([]))
    ; true.

ensureInf(CP) :-
	cat(CP,c2) ->
	IP complement_of CP,
	cat(IP,i2),
	IP has_feature inf([])
    ; true.
