/* File: lhip_intf.pl
 * Created:  6-Dec-93
 * Last Update: 
 * Author: Afzal Ballim
 * Purpose: public interface to the lhip system.
 * Needs: lhip_int.pl
 */

/*

    The syntax of lhip rules is given below in a psuedo DCG form.  In
    addition to these definitions, each lhip_body must contain at least one
    clause that is either a normal or head rule.

    lhip_rule --> std_lhip_rule.
    lhip_rule --> delete_lhip_rule.

    std_lhip_rule    -->  atom ~~> lhip_body.
    delete_lhip_rule --> -atom ~~> lhip_body.

    lhip_body --> lhip_clause.           %
    lhip_body --> lhip_clause,lhip_body. % relavtive positioning
    lhip_body --> lhip_clause:lhip_body. % adjacency
    lhip_body --> lhip_body;lhip_body.   % disjunction

    lhip_clause --> atom.               % normal rule
    lhip_clause --> *atom.              % head rule
    lhip_clause --> -atom.              % specific delete rule
    lhip_clause --> @ terminal          % terminal
    lhip_clause --> [].                 % general delete
    lhip_clause --> ~lhip_body.         % negated rule
    lhip_clause --> (? lhip_body ?).    % optional
    lhip_clause --> !.                  % prolog cut
    lhip_clause --> {P}.                % embedded prolog code

*/

/************************************************************
 *
 * Public interface
 *
 ************************************************************/

/*===========================================================
 * Predicate: lhip_phrase/2
 * Modes: lhip_phrase(+Cat,+Sent)
 * 
 * Purpose: succeeds if Sent is a sentence of type Cat in
 *          a lhip grammar. This is actually an interface.
 *          Similar syntax to the prolog DCG phrase/2.
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_phrase(Cat,Sent)   
:-
    lhip_phrase(Cat,Sent,_S,_E,_C).

/*===========================================================
 * Predicate: lhip_phrase/5
 * Modes: lhip_phrase(+Cat,+Sent,-Start,-End,-Cover)
 * 
 * Purpose: As for lhip_phrase/2, but returns the span of
 *          the phrase in the sentence (indexed from zero, 
 *          End being one beyond the actual end of the span)
 *          and the number of tokens covered in this span.
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update: 25-Nov-93
 *=========================================================*/

lhip_phrase(Cat,Sent,S,E,C)
:-
    var(Cat),!,
    format("~q illegal arguments~n",[lhip_phrase(Cat,Sent,S,E,C)]),
    fail.

lhip_phrase(Cat,Sent,SS,SE,C)
:-
    lhip_clear_chart,
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    lhip_clause(Cat,Goal,[FSent,1,End,SS,SE,C,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: lhip_clear_chart/0
 * Modes: 
 * 
 * Purpose: clears the chart and resets the success counter.
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_clear_chart :-
    retractall(lhip_known(_,_,_,_,_)),
    retractall(lhip_known_fail(_,_)),
    lhip_success_ref(0).

/*===========================================================
 * Predicate: lhip_phrase_rp/2,lhip_phrase_rp/5
 * Modes: 
 * 
 * Purpose: as lhip_phrase/2, but does not delete information
 *          from previous parses.
 * 
 * Licensed Use: 
 * 
 * Created: 18-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_phrase_rp(Cat,Sent)
:-
    lhip_phrase_rp(Cat,Sent,_SS,_SE,_C).

lhip_phrase_rp(Cat,Sent,SS,SE,C)
:-
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    lhip_clause(Cat,Goal,[FSent,1,End,SS,SE,C,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: lhip_mc_phrases/4
 * Modes: lhip_mc_phrase(+Cat,+Sent,-Cover,-P)
 * 
 * Purpose: find the maximal coverage set of Sent by Cat
 * 
 * Licensed Use: 
 * 
 * Created: 18-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_mc_phrases(Cat,Sent,Cover,P)
:-
    setof((Cat,S,E,Cover),
          lhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    lhip_max_cov(P1,0,[],Cover,P).

lhip_max_cov([],C,P,C,P):- !.
lhip_max_cov([(HCat,S,E,HCov)|T],Mi,Pi,C,P)
:-
    HCov = Mi,!,
    lhip_max_cov(T,Mi,[(HCat,S,E)|Pi],C,P);
    HCov < Mi,!,
    lhip_max_cov(T,Mi,Pi,C,P);
    lhip_max_cov(T,HCov,[(HCat,S,E)],C,P).


/*===========================================================
 * Predicate: lhip_minmax_phrases/4
 * Modes: lhip_minmax_phrase(+Cat,+Sent,-Span,-MinMax)
 * 
 * Purpose: MinMax is the set of minimal spanning but maximum
 *          coverage parses of Sent under Cat.
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_minmax_phrases(Cat,Sent,Span,MinMax)
:-
    setof((Cat,S,E,Cover),
          lhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    lhip_max_cov(P1,0,[],Cover,Pc),
    lhip_min_span(Pc,_,[],Span,MinMax).

lhip_min_span([],MS,P,MS,P).
lhip_min_span([(Cat,S,E)|Cats],MS,MM,Span,MinMax)
:-
    ThisSpan is E-S,
    (ThisSpan = MS,!,
     lhip_min_span(Cats,MS,[(Cat,S,E)|MM],Span,MinMax)
    ;ThisSpan < MS,!,
     lhip_min_span(Cats,ThisSpan,[(Cat,S,E)],Span,MinMax)
    ;ThisSpan > MS,!,
     lhip_min_span(Cats,MS,MM,Span,MinMax)
    ).


/*===========================================================
 * Predicate: lhip_cv_phrase/2
 * Modes: 
 * 
 * Purpose: As lhip_phrase, but ensures complete coverage
 * 
 * Licensed Use: 
 * 
 * Created: 19-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_cv_phrase(Cat,Sent)
:-
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    lhip_clause(Cat,Goal,[FSent,1,End,_SS,_SE,Ntok,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: lhip_seq_phrase/3
 * Modes: lhip_seq_phrase(+Cat,+Sent,-Seq
 * 
 * Purpose: Seq is a seqence of parses across Sent, e.g.,
 *          Seq=[S0,S1,S2] means that S0 covers s0 to e0
 *          in Sent, etc, and ei <= s(i+1).
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_seq_phrase(Cat,Sent,Seq)
:-
    setof((Cat,S,E,Cover),
          lhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    lhip_seq_in(P1,0,Seq).

lhip_seq_in(P,Ep,[(Pr,S,E,C)|Seq])
:-
    member((Pr,S,E,C),P),
    Ep =< S,
    lhip_delete_dup((Pr,S,E,C),P,Prest),
    lhip_seq_in(Prest,E,Seq).

lhip_seq_in(_,X,[]) :- X =\= 0.

lhip_delete_dup(_,[],[]).
lhip_delete_dup(P,[P|T],Td)
:-  
    !,lhip_delete_dup(P,T,Td).
lhip_delete_dup(P,[Q|T],[Q|Td])
:-
    !,lhip_delete_dup(P,T,Td).

/*===========================================================
 * Predicate: lhip_maxT_phrases/3
 * Modes: lhip_maxT_phrases(+Cat,+Sent,-MaxT)
 * 
 * Purpose: returns phrases that cover the maximum threshold.
 *          On backtracking it returns the phrases that cover
 *          the next highest threshold.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_maxT_phrases(P,Sent,MaxT)
:-
    lhip_clear_chart,
    setof((T-P),
        (A,B,C)^(lhip_phrase_rp(P,Sent,A,B,C),
                 T is C/(B-A)),
        AllPT),
    keysort(AllPT,TPs),
    lhip_maxt(TPs,0,[],MaxT).

% last
lhip_maxt([],_,MaxT,MaxT).
% first
lhip_maxt([T-P|TPs],0,[],MaxT)
:-
    !,lhip_maxt(TPs,T,[P,T],MaxT).
%next same
lhip_maxt([T-P|TPs],T,Ps,MaxT)
:-
    lhip_maxt(TPs,T,[P|Ps],MaxT).
% next different
lhip_maxt([T0-P|TPs],T,_Ps,MaxT)
:-
    T0 =\= T,
    lhip_maxt(TPs,T0,[P,T0],MaxT).
lhip_maxt([T0-_P|_TPs],T,MaxT,MaxT)
:-
    T0 =\= T.

/*===========================================================
 * Predicate: lhip_success/1
 * Modes: lhip_success(?Num)
 * 
 * Purpose: lists the successful calls to clause Num of the grammar.
 *          If Num is unbound then lists all successful clauses.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_success :- lhip_success(_).
lhip_success(Num)
:-
    setof((Num,Succ),(A,B,C)^lhip_known(Num,A,B,C,Succ),Succs),
    lhip_list(Succs).

/*===========================================================
 * Predicate: lhip_ms_success/0,/1,/2
 * Modes: lhip_ms_success(?Num)
 * 
 * Purpose: as lhip_success but only lists those clauses that
 *          are not used in other succesful clauses.
 * 
 * Licensed Use: 
 * 
 * Created:  6-Dec-93
 * Last Update: 
 *=========================================================*/

% lhip_ms_success/0
lhip_ms_success :- lhip_ms_success(_).

% lhip_ms_success/1
lhip_ms_success(Num)
:-
    lhip_ms_success(Num,Succs),
    lhip_list(Succs).

lhip_ms_success(Num,Succs)
:-
	setof(X,
		  (A,B,C,D,E)^(
			lhip_known(A,B,C,D,E),
			member(X,C)),
		  Incs),
	lhip_success_counter(Cntr),
	lhip_ms_succ_nincs(0,Cntr,Num,Incs,SuccsN),
	sort(SuccsN,Succs).

% reached the limit
lhip_ms_succ_nincs(Limit,Limit,_,_,[]) :- !.

% current is included
lhip_ms_succ_nincs(X,Limit,Num,[X|Incs],Succs)
:-
	NewX is X+1,!,
	lhip_ms_succ_nincs(NewX,Limit,Num,Incs,Succs).

% current is not included, so a potential match
lhip_ms_succ_nincs(X,Limit,Num,[Y|Incs],Succs)
:-
	!,
	(var(Num),
	 lhip_known(TNum,X,_I,_F,S),
	 lhip_orig_args(S,So,SS,SE,FS,FE,Cov),
	 Succs=[(TNum,Cov,So,SS,SE,FS,FE)|Sc],
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Sc)
	;
	 lhip_known(Num,X,_I,_F,S),
	 lhip_orig_args(S,So,SS,SE,FS,FE,Cov),
	 Succs=[(Num,Cov,So,SS,SE,FS,FE)|Sc],
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Sc)
	;
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Succs)).

% all rest are not included
lhip_ms_succ_nincs(X,Limit,Num,[],Succs)
:-
	!,
	(var(Num),
	 lhip_known(TNum,X,_I,_F,S),
	 lhip_orig_args(S,So,SS,SE,FS,FE,Cov),
	 Succs=[(TNum,Cov,So,SS,SE,FS,FE)|Sc],
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[],Sc)
	;
	 lhip_known(Num,X,_I,_F,S),
	 lhip_orig_args(S,So,SS,SE,FS,FE,Cov),
	 Succs=[(Num,Cov,So,SS,SE,FS,FE)|Sc],
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[],Sc)
	;
	 NewX is X+1,
	 lhip_ms_succ_nincs(NewX,Limit,Num,[],Succs)).
	
% auxillary function
lhip_list([]).
lhip_list([(N,Cov,Co,SS,SE,FS,FE)|T])
:-
	format("(~q) ~q->[~q--~q)<-~q /~q ~~~~> ~q~n",
		[N,SS,FS,FE,SE,Cov,Co]),
    lhip_list(T).

/*===========================================================
 * Predicate: lhip_failure/1
 * Modes: lhip_failure(?Num)
 * 
 * Purpose: lists the failed calls to clause Num of the grammar. If
 *          Num is unbound then lists all failed clauses.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_failure:-lhip_failure(_).
lhip_failure(Num)
:-
    setof((Num,Fail),lhip_known_fail(Num,Fail),Fails),
    lhip_list2(Fails).


% failure aux. func
lhip_list2([]).
lhip_list2([(N,C)|T])
:-
	lhip_orig_args(C,Co,SS,SE,_FS,_FE,_Cov),
	format("(~q) ~q-><-~q ~~~~> ~q~n",
		[N,SS,SE,Co]),
    lhip_list2(T).

% auxillary, strip off the extra arguments
lhip_orig_args(P,Po,SS,SE,FS,FE,Cov)
:-
	functor(P,Pf,Nargs),
	ONargs is Nargs-8,
	functor(Po,Pf,ONargs),
	NSS is Nargs-6,arg(NSS,P,SS),
	NSE is Nargs-5,arg(NSE,P,SE),
	NFS is Nargs-4,arg(NFS,P,FS),
	NFE is Nargs-3,arg(NFE,P,FE),
	NCov is Nargs-2,arg(NCov,P,Cov),
	lhip_copyargs2(ONargs,P,Po).

lhip_copyargs2(0,_,_).
lhip_copyargs2(N,P,Po)
:-
	arg(N,P,A),arg(N,Po,A),
	NewN is N-1,
	lhip_copyargs2(NewN,P,Po).

/*===========================================================
 * Predicate: lhip_threshold/1
 * Modes: lhip_threshold(?T)
 * 
 * Purpose: the global threshold value is T
 * 
 * Licensed Use: 
 *      lhip_threshold(+T)  set threshold value to T
 *      lhip_threshold(-T)  T is the current threshold value
 * 
 * Created: 25-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_threshold(T)
:-
    var(T),
    lhip_threshold_value(T).

lhip_threshold(T)
:-
    nonvar(T),
    retract(lhip_threshold_value(_)),
    assert(lhip_threshold_value(T)).

