/*      TXTOUT.C - write the contents of a word_template to a text file
 ***************************************************************************
 *
 *	void init_txtout(outfp,filename,ochg,ambig)
 *	FILE *outfp;
 *	char *filename;
 *	struct change_list *ochg;
 *	int ambig;
 *
 *	void txtout(wp)
 *	struct word_template *wp;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	30-OCT-82	D. Weber/Bob Kasper
 *	13-MAR-85	hab/djw
 *	 7-APR-85	djw
 *	12-Nov-85	SRMc
 *	 3-JUN-86	hab
 *	 4-Apr-88	hab - cleanup after lint
 *	15-Apr-88	hab - do not do orthochange for words not
 *                              originally analyzed
 *      15-May-89       SRMc - rewrite for STAMP to eliminate global extern's
 *                           - write init_txtout()
 *      13-Jul-89       hab  - de-"lint" the source
 * 1.0f 10-Jan-90 ALB Add capability to recap from bitfield
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 * 1.1f 02-Aug-90 hab Add lower-upper word formation character handling
 *                    De-lint the source
 *	28-Dec-90	SRMc - BUG!! don't output anything for trailing
 *				nonalphabetics unless there was something
 *				in the orig_word field
 *	 5-Jan-91	SRMc - adjust for memory allocation with accounting
 *	 2-Mar-91	SRMc - put back to non-accounting memory allocation
 *	 3-Jan-92	SRMc - change argument list for apply_cc()
 ***************************************************************************
 * Copyright 1991, 1992 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "opaclib.h"
#include "class.h"
#include "envir.h"
#include "template.h"
#include "strlist.h"
#include "change.h"

extern char *strcpy();
extern char *mystrdup(), *apply_cc();
extern struct strlist *squeeze_strlist();
extern void myfree();

static FILE *txtoutfp = stdout;
static char *outfilename = "<stdout>";
static struct change_list *ortho_chg;		/* Text out change table */
static char ambig_char = '%';

static void recap();

/*************************************************************************
 * NAME
 *    init_txtout
 * ARGUMENTS
 *    outfp    - output FILE pointer
 *    filename - pointer to output filename string
 *    ochg     - pointer to output orthography changes
 *    ambig    - ambiguous output marker character
 * DESCRIPTION
 *    Initialize some parameters for use later by txtout().
 * RETURN VALUE
 *    none
 */
void init_txtout(outfp,filename,ochg,ambig)
FILE *outfp;
char *filename;
struct change_list *ochg;
int ambig;
{
txtoutfp    = outfp;		/* save these for later use by txtout() */
outfilename = filename;
ortho_chg   = ochg;
ambig_char  = ambig;
}

/*************************************************************************
 * NAME
 *    txtout
 * ARGUMENTS
 *    wp - pointer to word_template structure
 * DESCRIPTION
 *    Write the contents of a word_template structure to an output text
 *    file, restoring all the formatting information associated with the
 *    word.  Either multiple output possibilities exist as a strlist in
 *    the new_words field, or a single "failed" possibility exists as a
 *    string in the word field.
 * RETURN VALUE
 *    none
 */
void txtout(wp)
struct word_template *wp;
{
register struct strlist *slp;
register char *sp;
char buffer[BUFSIZE];
/*
 *  write the leading format information
 */
if (wp->format)
    fputs(wp->format, txtoutfp);
/*
 *  check for normal, possibly ambiguous output (stored as strlist)
 */
if (wp->new_words)
    {
    /*
     *  apply orthography changes to the word and recapitalize it
     */
    for (slp = wp->new_words; slp; slp = slp->slink)
	{				/* every ambiguity gets changed */
	sp = apply_cc(slp->stri, ortho_chg );
	myfree(slp->stri);
	slp->stri = sp;
	recap(slp->stri, wp->capital);          /* Recapitalize */
	}
    /*
     *  eliminate identical results
     */
    wp->new_words = squeeze_strlist( wp->new_words );
    /*
     *  write the word to the file
     */
    if (wp->new_words->slink)		/* If still ambiguity */
	{
	fprintf(txtoutfp, "%c%d%c",
		ambig_char, size_of_strlist(wp->new_words), ambig_char );
	write_strlist( wp->new_words, ambig_char, txtoutfp );
	fprintf(txtoutfp, "%c", ambig_char);
	}
    else
	fputs(wp->new_words->stri, txtoutfp);
    }
/*
 *  no valid output, so try some "invalid" output
 */
else if (wp->orig_word && *(wp->orig_word))
    {					/* first try the original word */
    recap(wp->orig_word, wp->capital);
    fprintf(txtoutfp, "%c0%c%s%c",
			ambig_char, ambig_char, wp->orig_word, ambig_char );
    }
else if (wp->word && *(wp->word))
    {					/* how about an orthochanged word? */
    recap(wp->word, wp->capital);
    fprintf(txtoutfp, "%c0%c%s%c",
			ambig_char, ambig_char, wp->word, ambig_char );
    }
/*
 *  now write the trailing nonalphabetics to the output file
 */
if (wp->non_alpha)
    fputs(wp->non_alpha,txtoutfp);
else if (wp->orig_word && *(wp->word))
    putc(' ', txtoutfp);	/* if word exists, default to a single space */

full_disk( txtoutfp, "TXTOUT", outfilename );
}

/*************************************************************************
 * NAME
 *    recap
 * ARGUMENTS
 *    cp    - pointer to string to recapitalize
 *    recap - flag for which kind of capitalization is wanted
 * DESCRIPTION
 *    Try to reimpose capitalization as it was in the original input text.
 * RETURN VALUE
 *    none
 */
static void recap(cp,recap)
register char *cp;
int recap;
{
register char c;
int bitmask;

switch (recap)
    {
    case INITCAP:
	/*
	 *  capitalize the first character which makes sense to `capitalize'
	 */
	for ( c = *cp ; c ; c = *++cp )
	    {
	    if (myislower(c))
		{ *cp = mytoupper(c); return; }
	    else if (myisupper(c))
		return; /* already capitalized */
	    }
	break;

    case ALLCAP:
	/*
	 *  capitalize every character which makes sense to `capitalize'
	 */
	for ( c = *cp ; c ; c = *++cp )
	    {
	    if (myislower(c))
		*cp = mytoupper(c);
	    }
	break;

    case NOCAP:
	/*
	 *  no capitalization wanted, so nothing to do
	 */
	break;

    default:
	/*
	 *  Bitfield, recap as indicated
	 */
	bitmask = 4;
	for ( c = *cp ; c ; c = *++cp )     /* For each char */
	    {
	    if (myisalpha(c))                 /* If alpha */
		{
		if ( recap & bitmask )      /* If its bitfield says cap */
		    *cp = mytoupper(c);       /* Recap it */
		bitmask <<= 1;              /* Shift bitmask up */
		}
	    }
	break;
    }
}
