/*      TRIE.C - functions for storing information in a trie
 ***************************************************************************
 *
 *      TRIE *add_to_trie( trp, key, info, linkinfo )
 *      TRIE *trp;
 *      char *key;
 *      void *info;
 *      (void *)(*linkinfo)();
 *
 *      void *search_trie(trp, key)
 *      TRIE *trp;
 *      char *key;
 *
 *      void show_trie(trp, showinfo)
 *      TRIE *trp;
 *      void (*showinfo)();
 *
 ***************************************************************************
 *      EDIT HISTORY
 *      20-May-89       Steve McConnel - loosely based on TRISET.C and
 *                              TRIACC.C from AMPLE
 *      13-Jul-89       hab  - de-"lint" the source
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 ***************************************************************************
 * Copyright 1989 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <string.h>
#include "trie.h"

#define NUL '\0'

extern char *strcpy(), *strcat();
extern char *myalloc(), *mystrdup();
extern void myfree();

static void *  newinfo;			/* these are recursion invariant */
static void *(*link_function)();	/*    parameters */
static void  (*show_function)();

static TRIE *enter_entry();
static void show_entries(), trie_indent();

/*************************************************************************
 * NAME
 *    add_to_trie
 * ARGUMENTS
 *    trp      - pointer to head of trie
 *    key      - pointer to insertion key (character string)
 *    info     - pointer to generic information
 *    linkinfo - pointer to function for adding infomation to existing list
 *    maxlevel - maximum depth to which the trie can go
 * DESCRIPTION
 *    Add info to the trie, using the insertion key.  The trie is cut off at
 *    a depth of maxlevel.
 * RETURN VALUE
 *    pointer to the head of the modified trie
 */
TRIE *add_to_trie( trp, key, info, linkinfo, maxlevel )
TRIE *trp;
char *key;
void *info;
void *(*linkinfo)();
int maxlevel;
{
newinfo = info;			/* save the invariant parameters */
link_function = linkinfo;
return( enter_entry( trp, key, maxlevel ) );
}

/*************************************************************************
 * NAME
 *    enter_entry
 * ARGUMENTS
 *    trp      - pointer to head of (sub)trie
 *    key      - pointer to insertion key (character string)
 *    levels   - number of levels deep we can still go
 * DESCRIPTION
 *    Add info to the trie, using the insertion key.
 *    This is a recursive function.
 * RETURN VALUE
 *    pointer to the head of the modified trie
 */
static TRIE *enter_entry( trp, key, levels )
TRIE *trp;
char *key;
int levels;
{
register TRIE *tp;
char *plet;
char newlet[2];		/* short string for new key letter */
int pos;
/*
 *  check for running out of key or running out of trie depth
 */
if ((*key == NUL) || (levels <= 0))
    {
    if (trp)
	{			/* link new stuff to existing information */
#ifdef THINK_C
	trp->trieinfo = (void *)(*link_function)(newinfo, trp->trieinfo);
#else
	(void *)trp->trieinfo = (void *)(*link_function)(newinfo, trp->trieinfo);
#endif
	return( trp );
	}
    else
	{			/* this level doesn't exist, so create it */
	tp = (TRIE *)myalloc( sizeof(TRIE) );
	tp->letters  = (char *)NULL;
	tp->children = (TRIE *)NULL;
	tp->siblings = (TRIE *)NULL;
	tp->trieinfo = newinfo;
	return( tp );
	}
    }
/*
 *  more key and trie depth to go, so set up newlet[] for key letter copying
 */
newlet[0] = *key;
newlet[1] = NUL;
if (trp)
    {
    /*
     *  trie exists, so check for children
     */
    if (trp->children)
	{
	/*
	 *  child tries exist, so look for the one we want
	 */
	if ((pos = strpos(trp->letters, newlet[0])) >= 0)
	    {
	    /*
	     *  appropriate child trie exists, so scan to it and recurse
	     */
	    for ( tp = trp->children ; pos ; --pos, tp = tp->siblings )
		;
	    enter_entry( tp, key+1, levels-1 );
	    return( trp );
	    }
	else
	    {
	    /*
	     *  no appropriate child trie, so create one recursively
	     */
	    plet = strcpy( myalloc((unsigned)strlen(trp->letters)+2),
				 trp->letters );
	    myfree( trp->letters );
	    trp->letters = strcat(plet, newlet);
	    for ( tp = trp->children ; tp->siblings ; tp = tp->siblings )
		;
	    tp->siblings = enter_entry((TRIE *)NULL, key+1, levels-1);
	    return( trp );
	    }
	}
    else
	{
	/*
	 *  no children, so create child trie recursively
	 */
	trp->letters  = mystrdup( newlet );
	trp->children = enter_entry((TRIE *)NULL, key+1, levels-1);
	return( trp );
	}
    }
else
    {
    /*
     *  no trie exists at this level, so create this level's node, creating
     *    the child node recursively
     */
    tp = (TRIE *)myalloc( sizeof(TRIE) );
    tp->letters  = mystrdup( newlet );
    tp->children = enter_entry((TRIE *)NULL, key+1, levels-1 );
    tp->siblings = (TRIE *)NULL;
#ifdef THINK_C
    tp->trieinfo = (void *)NULL;
#else
    (void *)tp->trieinfo = (void *)NULL;
#endif
    return( tp );
    }
}

/*************************************************************************
 * NAME
 *    search_trie
 * ARGUMENTS
 *    trp - pointer to the head of the trie
 *    key - pointer to the key character string
 * DESCRIPTION
 *    Search the trie for information stored using the key for access.
 * RETURN VALUE
 *    pointer to the generic information found in the trie, or NULL if
 *    the search fails
 */
void *search_trie(trp, key)
TRIE *trp;
char *key;
{
register TRIE *tp;
int pos;
/*
 *  check for nonexistent trie
 */
if (!trp)
    return( (void *)NULL );		/* no trie to search in */
/*
 *  check for running out of key:  ==> we've found it!
 */
if (!key || (*key == NUL))
    return( (void *)trp->trieinfo );

if (trp->letters)
    {
    /*
     *  check for first key letter in children's letters
     */
    if ((pos = strpos(trp->letters, *key)) >= 0)
	{
	/*
	 *  appropriate child trie exists, so scan to it and recurse
	 */
	for ( tp = trp->children ; pos ; --pos, tp = tp->siblings )
	    ;
	return( search_trie(tp, key+1) );
	}
    else
	return( (void *)NULL );		/* not in the trie */
    }
else
    return( (void *)trp->trieinfo );		/* end of search */
}

/*************************************************************************
 * NAME
 *    show_trie
 * ARGUMENTS
 *    trp      - pointer to the head of the trie
 *    showinfo - pointer to a function for displaying the information lists
 * DESCRIPTION
 *    Walk through a trie, displaying the information stored at each node.
 * RETURN VALUE
 *    none
 */
void show_trie(trp, showinfo)
TRIE *trp;
void (*showinfo)();
{
/*
 *  check for nonexistent trie or function
 */
if (!trp || !showinfo)
    return;			/* nothing to do */

show_function = showinfo;	/* store recursion invariant parameter */

show_entries( trp, 0 );		/* recursive showing, with indentation */
}

/*************************************************************************
 * NAME
 *    show_entries
 * ARGUMENTS
 *    trp    - pointer to the head of the trie
 *    indent - indentation counter (depth of recursion)
 * DESCRIPTION
 *    Recursively show the information in the trie, using a function
 *    supplied by the original caller to display the information at each
 *    node.
 * RETURN VALUE
 *    none
 */
static void show_entries(trp, indent)
TRIE *trp;
int indent;
{
register TRIE *tp;
register char *p;
/*
 *  show the information stored at this level in the trie
 */
if (trp->trieinfo)
    (*show_function)(trp->trieinfo, 3*indent);
/*
 *  recursively display the children trie nodes
 */
for (p = trp->letters, tp = trp->children ; tp ; tp = tp->siblings)
    {
    trie_indent(indent);
    printf("trie key: %c", *p++ );
    show_entries(tp, indent+1 );
    }
}

/*************************************************************************
 * NAME
 *    trie_indent
 * ARGUMENTS
 *    num - number of levels of indentation desired
 * DESCRIPTION
 *    Print a newline followed by num iterations of the indentation
 *    spacing.
 * RETURN VALUE
 *    none
 */
static void trie_indent(num)
int num;
{
putchar('\n');
while (num--)
    printf("   ");
}
