/*      STRCLA.C - process string classes
 ***************************************************************************
 *
 *	void init_scl()
 *
 *	void add_scl(line)
 *	char *line;
 *
 *	struct string_class *find_scl(name)
 *	char *name;
 *
 *	int scl_member(string,class)
 *	char *string;
 *	struct string_class *class;
 *
 *	int scl_lmatch(string,class)
 *	char *string;
 *	struct string_class *class;
 *
 *	int scl_rmatch(string,class)
 *	char *string;
 *      struct string_class *class;
 *
 *      show_scl()
 *
 ***************************************************************************
 *      EDIT HISTORY
 *       4-May-88       Steve McConnel
 *       9-May-88       SRMc - added show_scl()
 *                           - fixed scl_match() to return length of
 *                              successful match (still zero if unsuccessful)
 *      28-Jul-88       SRMc - replace ssalloc() with malloc() and realloc()
 *      24-Aug-88       SRMc - regularize error messages
 *      30-Aug-88       SRMc - remove quotes from output of show_scl()
 *      21-Oct-88       SRMc - reorganize the file header comments
 *      24-Oct-88       SRMc - trace down Aztec C code generation bug (UGH!!)
 *      10-Nov-88       SRMc - replace free() with myfree()
 *       3-Apr-89       SRMc - allow previously defined string classes to be
 *                              used in defining string classes
 *      16-May-89       SRMc - revise #include's to share code with STAMP
 *                           - renamed load_scl() to add_scl()
 *      13-Jul-89       hab  - de-"lint" the source
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 * 1.6e 02-Aug-90 hab Change index to strchr and rindex to strrchr
 *	17-Jan-91	SRMc - add ANSI-fied function prototypes
 *			     - use strcmp() instead of streq()
 *			     - declare bad_sc() as static void
 *			     - declare show_scl() as void
 *	29-Jan-91	SRMc - merged in AMPLE 1.6f sources
 *			     - fixed return(strlen())'s for THINK C
 *	11-Mar-91	SRMc - remove #define strchr -- it's in OPACLIB.H
 ***************************************************************************
 * Copyright 1988, 1991 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#endif
#include "opaclib.h"
#include "class.h"
#include "strlist.h"		/* defines strlist structure */

#ifdef __STDC__
#define P(s) s
#else
#define P(s) ()
#endif

/* strcla.c */
void init_scl P((void ));
void add_scl P((char *line ));
struct string_class *find_scl P((char *name ));
int scl_member P((char *string , struct string_class *class ));
int scl_lmatch P((char *string , struct string_class *class ));
int scl_rmatch P((char *string , struct string_class *class ));
void show_scl P((void ));

/* getwd.c */
extern char *skipwhite P((char *cp ));
extern char *getwd P((char *cp ));

/* myallo.c */
/* extern char *myalloc P((unsigned size )); */
extern void myfree P((char *s ));

/* stcmp.c */
int stcmp P((char *s , char *t ));

/* rstcmp.c */
int rstcmp P((char *s , char *t ));

#undef P

struct string_class *find_scl();

/****************************************************************************
 *  data used for string classes
 *
 *  notice that the list of classes per se are invisible outside this file
 */
static struct string_class *classes = NULL;

/*************************************************************************
 * NAME
 *    init_scl
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Initialize the string classes.
 * RETURN VALUE
 *    none
 */
void init_scl()
{
register struct strlist *sp;
register struct string_class *cp;
register struct strlist *nsp;
register struct string_class *ncp;

for ( cp = classes ; cp != (struct string_class *)NULL ; cp = ncp )
    {
    ncp = cp->sc_next;
    myfree( cp->sc_name );
    for (sp = cp->sc_members ; sp != (struct strlist *)NULL ; sp = nsp )
	{
	nsp = sp->slink;
	myfree( sp->stri );
	myfree( (char *)sp );
	}
    myfree( (char *)cp );
    }
classes = (struct string_class *)NULL;
}

/***************************************************************************
 * NAME
 *    bad_sc
 * ARGUMENTS
 *    mcp  - pointer to string class structure
 *    head - first part of the error message
 *    body - second part of the error message
 *    tail - third part of the error message
 * DESCRIPTION
 *    Print an error message, then erase all of the elements stored for a
 *    string class.
 * RETURN VALUE
 *    none
 */
static void bad_sc(mcp, head, body, tail)
struct string_class *mcp;
char *head, *body, *tail;
{
register struct strlist *sp, *np;
/*
 *  print the error message
 */
printf("%s%s%s", head, body, tail );
/*
 *  erase any class members which may have been stored
 */
for ( sp = mcp->sc_members ; sp ; sp = np )
    {
    np = sp->slink;
    myfree( (char *)sp->stri );
    myfree( (char *)sp );
    }
mcp->sc_members = (struct strlist *)NULL;
}

/*************************************************************************
 * NAME
 *    add_scl
 * ARGUMENTS
 *    line - pointer to "string class" line from analysis data file
 * DESCRIPTION
 *    Add a string class defined by the user to the list of string classes.
 * RETURN VALUE
 *    none
 */
void add_scl(line)
char *line;
{
register struct string_class *cp;
register struct strlist *sp;
register char *p;
register char *name;
char *q;
struct string_class *mp;
struct strlist *tp;
static char errhead[] = "\nSTRING CLASS: ";
/*
 *  split off the string class name and check that no string class by
 *  this name has been defined yet
 */
name = skipwhite(line);
if ((line == NULL) || (*name == NUL))
    {
    printf("%sEmpty definition", errhead );
    return;
    }
line = getwd(name);
if (*line == NUL)
    {
    printf("%sNo strings given in class %s", errhead, name );
    return;
    }
for ( cp = classes ; cp != (struct string_class *)NULL ; cp = cp->sc_next )
    {
    if (strcmp(name, cp->sc_name)==0)
	{
	printf("%s%s already defined", errhead, name );
	return;
	}
    }
/*
 *  allocate space and store the string class name
 */
cp = (struct string_class *)myalloc(sizeof(struct string_class));
cp->sc_name = strcpy( myalloc((unsigned)strlen(name)+1), name );
cp->sc_members = (struct strlist *)NULL;
/*
 *  get the strings and store them
 */
for ( p = line ; *p != '\0' ; p = line )
    {
    line = getwd(line);			/* split off the leading token */
    if (*p == '[')
	{
	/*
	 *  we have a [, indicating a string class reference
	 *  look for the matching ]
	 */
	++p;
	if (*p == NUL)
	    {
	    p = line;
	    line = getwd(line);
	    }
	if ((p == (char *)NULL) || (*p == NUL))
	    {
	    bad_sc(cp,errhead,"Dangling '[' in class definition of ",name);
	    break;
	    }
	q = strchr(p,']');
	if (q == (char *)NULL)
	    {
	    q = line;
	    if ((q == (char *)NULL) || (*q == NUL) || (*q != ']'))
		{
		bad_sc(cp,errhead,"Missing ']' in class definition of ",name);
		break;
		}
	    line = getwd(line);
	    }
	*q++ = NUL;
	if (*q != NUL)
	    {
	    bad_sc(cp,errhead,
		"No whitespace following ']' in class definition of ",name);
	    break;
	    }
	/*
	 *  we have a complete string class reference, with the string
	 *	class name pointed to by p and terminated by a NUL
	 *  check for a valid, already defined string class
	 */
	mp = find_scl(p);
	if (mp == (struct string_class *)NULL)
	    {
	    bad_sc(cp,errhead,"Unknown class ",p);
	    printf(" referenced in definition of %s", name);
	    break;
	    }
	/*
	 *  copy the elements from the old string class to the new
	 */
	for ( tp = mp->sc_members ; tp ; tp = tp->slink )
	    {
	    sp = (struct strlist *)myalloc(sizeof(struct strlist));
	    p = tp->stri;
	    sp->stri = strcpy(myalloc((unsigned)strlen(p)+1),p);
	    sp->slink = cp->sc_members;
	    cp->sc_members = sp;
	    }
	}
    else if (*p == ']')
	{
	bad_sc(cp, errhead, "Unexpected ']' in class definition of ", name);
	break;
	}
    else
	{
	sp = (struct strlist *)myalloc(sizeof(struct strlist));
	sp->stri = strcpy( myalloc((unsigned)strlen(p)+1), p );
	sp->slink = cp->sc_members;	/* link into the string list */
	cp->sc_members = sp;
	}
    }
/*
 *  link this string class into the list (only if it has members)
 */
if (cp->sc_members == (struct strlist *)NULL)
    {
    myfree( (char *)cp->sc_name );
    myfree( (char *)cp );
    }
else
    {
    cp->sc_next = classes;
    classes = cp;
    }
}

/*************************************************************************
 * NAME
 *    find_scl
 * ARGUMENTS
 *    name - pointer to string class's name
 * DESCRIPTION
 *    Search for a specific string class by name.
 * RETURN VALUE
 *    pointer to the string class node, or NULL if not found
 */
struct string_class *find_scl(name)
char *name;
{
register struct string_class *cp;

for ( cp = classes ; cp != (struct string_class *)NULL ; cp = cp->sc_next )
    {
    if (strcmp(name,cp->sc_name)==0)
	return(cp);
    }
return((struct string_class *)NULL);
}

/*************************************************************************
 * NAME
 *    scl_member
 * ARGUMENTS
 *    string - string to look for
 *    class  - pointer to string class node
 * DESCRIPTION
 *    Search a string class for a specific string.
 * RETURN VALUE
 *    1 if found, 0 if not found
 */
int scl_member(string,class)
char *string;
struct string_class *class;
{
register struct strlist *sp;

for ( sp = class->sc_members ; sp != (struct strlist *)NULL ; sp = sp->slink )
    {
    if (strcmp(string,sp->stri)==0)
	return( 1 );
    }
return( 0 );
}

/*************************************************************************
 * NAME
 *    scl_lmatch
 * ARGUMENTS
 *    string - string to match against
 *    class  - pointer to string class node
 * DESCRIPTION
 *    Search a string class for a specific string, matching at the left
 *    (beginning).
 * RETURN VALUE
 *    length of successful match if found, 0 if not found
 */
int scl_lmatch(string,class)
char *string;
struct string_class *class;
{
register struct strlist *sp;

for ( sp = class->sc_members ; sp != (struct strlist *)NULL ; sp = sp->slink )
    {
    if (stcmp(string,sp->stri))
	return( (int)strlen(sp->stri) );
    }
return( 0 );
}

/*************************************************************************
 * NAME
 *    scl_rmatch
 * ARGUMENTS
 *    string - string to match against
 *    class  - pointer to string class node
 * DESCRIPTION
 *    Search a string class for a specific string, matching at the right
 *    (end).
 * RETURN VALUE
 *    length of successful match if found, 0 if not found
 */
int scl_rmatch(string,class)
char *string;
struct string_class *class;
{
register struct strlist *sp;

for ( sp = class->sc_members ; sp != (struct strlist *)NULL ; sp = sp->slink )
    {
#ifdef MSDOS
#ifdef AZTEC
#ifndef BIGMEM
#define BAD_COMPILER_CODE_GENERATION
#endif
#endif
#endif
    if (rstcmp(string,sp->stri))
	return( (int)strlen(sp->stri) );
#ifdef BAD_COMPILER_CODE_GENERATION
    strcpy("","");
#endif
    }
return( 0 );
}

/*************************************************************************
 * NAME
 *    show_scl
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Display the contents of all the string classes.
 * RETURN VALUE
 *    none
 */
void show_scl()
{
register struct string_class *scl;
register struct strlist *sp;

printf("\nString classes:");
for ( scl = classes ; scl ; scl = scl->sc_next )
    {
    printf( "\n    %s {", scl->sc_name);
    for ( sp = scl->sc_members ; sp ; sp = sp->slink )
	printf(" %s", sp->stri );
    printf( " }" );
    }
}
