/*      ENVLEX.C - lexical scan for environment constraint expressions
 ***************************************************************************
 *
 *	elex_init(line)
 *	char *line;
 *
 *	int elex_get()
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	 7-May-88	Steve McConnel
 *	19-May-88	SRMc - add "~_" as a token
 *	 9-Sep-88	SRMc - add "(" and ")" as tokens
 *			     - use \ as a verbatim quote
 *	21-Oct-88	SRMc - reorganize the file header comments
 *	28-Oct-88	SRMc - shorten ELEX_ to EL_ in symbol names in order
 *                              to satisfy the UNIX PC System V compiler
 *       4-Jan-89       SRMc - fix for Microsoft C
 *      26-Jul-89       hab  - added #include "class.h"
 *                             replace struct strlist in defs.h with
 *                              STAMP's strlist.h
 * 1.6b 22-Jun-90 ALB Fix bug of not accepting 8-bit chars in env (isspace)
 *	20-Dec-90	SRMc - add #define index #ifndef BSD
 *			     - replace stcmp() with match_begin()
 *	11-Mar-91	SRMc - replace index with strchr throughout
 *			     - add #include <string(s).h>
 ***************************************************************************
 * Copyright 1988, 1989 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#ifndef isascii
#define isascii(x) (! (x & ~0177))
#endif
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#endif
#include "opaclib.h"
#include "class.h"
#include "envir.h"
#include "strlist.h"

#ifdef __STDC__
# define	P(s) s
#else
# define P(s) ()
#endif

/* envlex.c */
int elex_init P((char *line ));
int elex_get P((void ));
int main P((void ));

/* matchbegin.c */
extern int match_begin P((char *str, char *sub));

static int elex_tok P((void));
#undef P
/*
 *  values for elex_state
 */
#define EL_INIT	0	/* freshly initialized */
#define EL_WORK	1	/* merely going ahead working */
#define EL_HLD1	2	/* finished with one, holding a '/' */
#define EL_HLD2	3	/* finished with one, holding a '+/' */
#define EL_EOF	-1	/* at end of input string */

#define MAXLEXSIZE	50
static short elex_state;	/* state of the lexical analyzer */
char *elex_ptr = NULL;		/* pointer to the lexical string */
char elex_string[MAXLEXSIZE+1];	/* stores current lexical token */

/*************************************************************************
 * NAME
 *    elex_init
 * ARGUMENTS
 *    line - pointer to a line which is to be lexically scanned
 * DESCRIPTION
 *    Initialize for a lexical scan of an environment constraint string.
 * RETURN VALUE
 *    none
 */
elex_init(line)
char *line;
{
elex_state = EL_INIT;
elex_ptr = line;
}

/*************************************************************************
 * NAME
 *    elex_get
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Get the next lexical token, storing a string if needed.
 * RETURN VALUE
 *    integer code for token type (including end of environment), or EOF
 *    if end of input
 */
int elex_get()
{
register int type;

if (elex_ptr == (char *)NULL)
    return( EOF );
switch (elex_state)
    {
    case EL_INIT:		/* just starting an environment */
	type = elex_tok();
	elex_state = (type==EOF) ? EL_EOF : EL_WORK;
	return( type );

    case EL_WORK:		/* in the middle of an environment */
	switch (type = elex_tok())
	    {
	    case EOF:		elex_state = EL_EOF;	return( ENV_END );
	    case ENV_STRENV:	elex_state = EL_HLD1;	return( ENV_END );
	    case ENV_MORPH:	elex_state = EL_HLD2;	return( ENV_END );
	    default:					return( type );
	    }

    case EL_HLD1:		/* saw a '/' previously */
	elex_state = EL_WORK;
	return( ENV_STRENV );

    case EL_HLD2:		/* saw a '+/' previously */
	elex_state = EL_WORK;
	return( ENV_MORPH );

    default:			/* assume EL_EOF state */
	elex_ptr = (char *)NULL;
	return( EOF );
    }
}

/*************************************************************************
 * NAME
 *    elex_tok
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Load the next lexical token into elex_string.
 *    This is the only lexical function which modifies elex_ptr.
 * RETURN VALUE
 *    type of token found (ENV_...)
 */
static int elex_tok()
{
register char *p;
register int ch;

while (myisspace(*elex_ptr))      /* skip leading white space */
    ++elex_ptr;

ch = *elex_ptr++;

if (ch == NUL)
    {
    elex_string[0] = ch;	/* end of the input string */
    --elex_ptr;			/* stay on the NUL */
    return( EOF );
    }
else if (strchr("/_#[]()", ch) != (char *)NULL)
    {
    elex_string[0] = ch;	/* special single character token */
    elex_string[1] = NUL;
    switch (ch)
	{
	case '/':	return( ENV_STRENV );
	case '_':	return( ENV_MARK );
	case '#':	return( ENV_BOUND );
	case '[':	return( ENV_LBRACK );
	case ']':	return( ENV_RBRACK );
	case '(':	return( ENV_LPAREN );
	case ')':	return( ENV_RPAREN );
	default:	return( EOF );			/* "can't happen" */
	}
    }
else if ((ch == '~') && (*elex_ptr == '_'))
    {
    elex_string[0] = ch;	/* special two character token */
    elex_string[1] = *elex_ptr++;
    elex_string[2] = NUL;
    return( ENV_NOTMARK );
    }
else if ((ch == '+') && (*elex_ptr == '/'))
    {
    elex_string[0] = ch;	/* special two character token */
    elex_string[1] = *elex_ptr++;
    elex_string[2] = NUL;
    return( ENV_MORPH );
    }
else if (ch == '~')
    {				/* special single character token */
    elex_string[0] = ch;
    elex_string[1] = NUL;
    return( ENV_NOT );
    }
else if ((ch == '.') && (*elex_ptr == '.') && (*(elex_ptr+1) == '.'))
    {
    elex_string[0] = ch;	/* special three character token */
    elex_string[1] = *elex_ptr++;
    elex_string[2] = *elex_ptr++;
    elex_string[3] = NUL;
    return( ENV_ELLIPSIS );
    }
else
    {				/* literal string */
    p = elex_string;
    do  {
	if (ch == '\\')
	    {					/* get quoted character */
	    if ((*elex_ptr != NUL) && !myisspace(*elex_ptr))
		ch = *elex_ptr++;
	    }
	if (p < (elex_string + MAXLEXSIZE))	/* store the character */
	    *p++ = ch;
	ch = *elex_ptr++;			/* get next character */
	} while ((ch != NUL) &&
		 (!myisspace(ch)) &&
		 (strchr("/~_#[]()",ch) == (char *)NULL) &&
		 !((ch == '+')&&(*elex_ptr == '/')) &&
		 !((ch == '.')&& match_begin(elex_ptr,"..")) );
    --elex_ptr;			/* back up to terminating character */
    *p++ = NUL;			/* terminate the literal string */
    return( ENV_LITERAL );
    }
}

#ifdef TESTING
/*************************************************************************
 * NAME
 *    main
 * ARGUMENTS
 *    none (that this program uses)
 * DESCRIPTION
 *    test program for the lexical scanning implemented above
 * RETURN VALUE
 *    none
 */
main()
{
register int k;
char buf[100];

while (fgets(buf,sizeof buf, stdin) != (char *)NULL)
    {
    elex_init(buf);
    while ((k = elex_get()) != EOF)
	{
	printf("Lexical token value = %d ", k );
	switch (k)
	    {
	    case ENV_END:	printf("(ENV_END)");		break;
	    case ENV_STRENV:	printf("(ENV_STRENV)");		break;
	    case ENV_MORPH:	printf("(ENV_MORPH)");		break;
	    case ENV_BOUND:	printf("(ENV_BOUND)");		break;
	    case ENV_ELLIPSIS:	printf("(ENV_ELLIPSIS)");	break;
	    case ENV_LBRACK:	printf("(ENV_LBRACK)");		break;
	    case ENV_RBRACK:	printf("(ENV_RBRACK)");		break;
	    case ENV_MARK:	printf("(ENV_MARK)");		break;
	    case ENV_NOT:	printf("(ENV_NOT)");		break;
	    case ENV_LITERAL:
		printf("(ENV_LITERAL)");
		printf(" = \"%s\"", elex_string);
		break;
	    case ENV_LPAREN:	printf("(ENV_LPAREN)");		break;
	    case ENV_RPAREN:	printf("(ENV_RPAREN)");		break;
	    default:		printf("<<UNKNOWN>>");		break;
	    }
	printf("\n");
	}
    }
}
#endif
