/*      ENVCHK.C - check string environment constraints
 ***************************************************************************
 *
 *	int strenv_check(leftstring,leftsize,rightstring,ec)
 *	char *leftstring;
 *	int leftsize;
 *	char *rightstring;
 *	struct env_cond *ec;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	10-May-88	Steve McConnel
 *	20-May-88	SRMc - add ec_flags field to env_cond structure, with
 *				E_NOT bit
 *	 9-Jun-88	SRMc - revise struct allomorph, revise struct sd_affix
 *				(rename SD_ to sd_), and define struct
 *				sd_infix.
 *			     - Allow unlimited number of categories for roots,
 *				or category pairs for affixes.
 *			     - Allow more information for infixes.
 *	28-Jul-88	SRMc - replace ssalloc() with malloc() and realloc()
 *       7-Sep-88       SRMc - split into ENVCHK.C and MENVCK.C
 *       9-Sep-88       SRMc - handle E_OPTIONAL
 *      21-Oct-88       SRMc - reorganize the file header comments
 *      10-Nov-88       SRMc - replace free() with myfree()
 *      17-May-89       SRMc - remove unneeded #include "dict.h" and "adefs.h"
 *                              to share this file with STAMP
 *                           - changed argument list for strenv_check() to
 *                              allow more flexibility in splitting the left
 *                              and right environments
 *      13-Jul-89       hab  - de-"lint" the source
 *      19-Jul-89       hab  - modify to check all members of a string
 *                              class that succeed (at first) and not
 *                              just assume that the first one that matches
 *                              is correct
 *      26-Jul-89       hab  - removed conditional compile for SYNTHESIS
 *      27-Jul-89       hab  - add extern char *cpystr()
 *                             add Copyright 1989
 * 1.6a 21-Jun-90 BK  Fix up for THINKC on MAC
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 *	17-Jan-91	SRMc - replace cpystr() with strcpy()
 *			     - add #ifdef BSD for <strings.h>
 *			     - add ANSI-fied function prototypes
 *	29-Jan-91	SRMc - merged in AMPLE 1.6f sources
 ***************************************************************************
 * Copyright 1988, 1991 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#endif
#include "opaclib.h"
#include "strlist.h"
#include "class.h"
#include "envir.h"

#ifdef __STDC__
#define P(s) s
#else
#define P(s) ()
#endif

/* envchk.c */
int strenv_check P((char *leftstring , int leftsize , char *rightstring , struct env_cond *ec ));

/* myallo.c */
/* char *myalloc P((unsigned size )); */
void myfree P((char *s ));

/* rstcmp.c */
int rstcmp P((char *s , char *t ));

/* stcmp.c */
int stcmp P((char *s , char *t ));

/* local functions */
static int senv_left P((char *left , struct env_item *env));
static int senv_right P((char *right , struct env_item *env));

#undef P

/*************************************************************************
 * NAME
 *    shorten
 * ARGUMENTS
 *    string - pointer to a NUL-terminated string
 *    size   - number of bytes
 * DESCRIPTION
 *    Chop the indicated number of bytes from the end of the string.
 * RETURN VALUE
 *    pointer to the string, which is now shorter
 */
static char *shorten(string,size)
char *string;
int size;
{
int len;

if ((string == (char *)NULL) || (size <= 0))
    return(string);		/* check for reasonable arguments */

len = strlen(string) - size;	/* find out where to chop */
if (len <= 0)
    *string = NUL;		/* nothing left of the string... */
else
    *(string + len) = NUL;
return(string);
}

/*************************************************************************
 * NAME
 *    chk_senv_left
 * ARGUMENTS
 *    size - size of left-hand string that matched
 *    left - NUL-terminated left-hand string to be checked
 *    env  - pointer to list of left-hand side env_item's
 *    match - portion of string that was matched
 *            This is needed only for chk_senv_left, since the original string
 *            is (partly) destroyed by shorten
 * DESCRIPTION
 *    Check what to do based on size of matched string.
 * RETURN VALUE
 *    TRUE if successful, FALSE if unsuccessful
 */
static int chk_senv_left(size,left,env,match)
int size;
char *left;
struct env_item *env;
char *match;
{
    /*
     *  figure out what to do based on success (stored in size) , E_NOT,
     *  and E_OPTIONAL
     */
if (    (!size && (env->ei_flags & E_NOT)) ||
        (size && !(env->ei_flags & E_NOT)) )
    {					/* happily progress along */
    if (senv_left( shorten(left,(size ? size : 1)), env->ei_link ))
       return( TRUE );
					/* restore matched string that was
					   deleted by shorten */
    strcpy((left + strlen(left)), match);
    }
else if (env->ei_flags & E_OPTIONAL)
    {
    if (senv_left( left, env->ei_link ))
        return( TRUE );			/* didn't need it after all */
    }

return( FALSE );
}

/*************************************************************************
 * NAME
 *    chk_senv_right
 * ARGUMENTS
 *    size - size of right-hand string that matched
 *    right - NUL-terminated left-hand string to be checked
 *    env  - pointer to list of left-hand side env_item's
 * DESCRIPTION
 *    Check what to do based on size of matched string.
 * RETURN VALUE
 *    TRUE if successful, FALSE if unsuccessful
 */
static int chk_senv_right(size,right,env)
int size;
char *right;
struct env_item *env;
{
    /*
     *  figure out what to do based on success (stored in size) , E_NOT,
     *  and E_OPTIONAL
     */
if (    (size && !(env->ei_flags & E_NOT)) ||
	(!size && (env->ei_flags & E_NOT)) )
    {
    if (senv_right( right + (size ? size : 1), env->ei_link ))
        return( TRUE );
    }
else if (env->ei_flags & E_OPTIONAL)
    {
    if (senv_right( right, env->ei_link ))
        return( TRUE );
    }
return( FALSE );
}

/*************************************************************************
 * NAME
 *    senv_left
 * ARGUMENTS
 *    left - NUL-terminated left-hand string to be checked
 *    env  - pointer to list of left-hand side env_item's
 * DESCRIPTION
 *    Check whether the given string matches the given environment items.
 *    This is a recursive function.
 * RETURN VALUE
 *    nonzero if successful, zero if unsuccessful
 */
static int senv_left(left,env)
char *left;
struct env_item *env;
{
int size;
char *myleft;
int mylength;
struct strlist *sp;

if (env == (struct env_item *)NULL)
    return( TRUE );		/* no (more) environment */
if (left == (char *)NULL)
    return( FALSE );		/* no string at all?? */
/*
 *  If we had the DECUS C function salloc() or the 4.2BSD Unix function
 *  alloca() guaranteed to be available [these allocate storage on the
 *  stack, which gets reclaimed automatically when the function exits],
 *  this would be a virtual mirror image of senv_right() below.  Alas,
 *  this is not guaranteed, so I allocate space explicitly and use
 *  myriads of goto's instead of return's in order to guarantee that the
 *  space is released.
 */
mylength = strlen(left)+1;
myleft = strcpy(myalloc((unsigned)mylength),left);
/*
 *  match the current environment against what's wanted
 */
for (;; strcpy(left,shorten(myleft,1)) )
    {
    /*
     *  handle hitting the beginning of the word
     */
    if (*left == NUL)
	{
	if ((env->ei_flags&E_CLASS) || (env->ei_val.ei_string!=(char *)NULL))
	    {
	    if ((env->ei_flags & E_OPTIONAL) && senv_left(left,env->ei_link))
		goto goodleft;
	    if (    (env->ei_flags & E_NOT) &&
		    (env->ei_link==(struct env_item *)NULL) )
		goto goodleft;		/* boundary is "NOT something" */
	    else
		goto badleft;		/* had to match something more */
	    }
	else
	    {				/* boundary */
	    if (env->ei_flags & E_NOT)
		goto badleft;		/* didn't want this boundary */
	    else
		goto goodleft;		/* wanted this boundary */
	    }
	}
    /*
     *  handle having something to match against
     */
    if ( env->ei_flags & E_CLASS )
	{				/* check against a string class */
	size = 0;			/* assume it won't be in the class */
					/* for each element that matches the 
					   string, see if it will work */
        for ( sp = env->ei_val.ei_scl->sc_members ;
	      sp != (struct strlist *)NULL ;
	      sp = sp->slink )
	   {
#ifdef MSDOS
#ifdef AZTEC
#ifndef BIGMEM
#define BAD_COMPILER_CODE_GENERATION
#endif
#endif
#endif
           if (rstcmp(left,sp->stri))
              {
	      size = strlen(sp->stri);
              if (chk_senv_left (size, left, env, sp->stri) ) goto goodleft;
              }
#ifdef BAD_COMPILER_CODE_GENERATION
           strcpy("","");
#endif
            }
					/* if did not match the class */
        if (size == 0 &&		/* check for negation, etc. */
            chk_senv_left (0, left, env, (char *)NULL) ) goto goodleft;
        goto badleft;
	}
    else if (env->ei_val.ei_string != (char *)NULL)
	{				/* check against a literal string */
	size = (rstcmp(left,env->ei_val.ei_string)) ?
				strlen(env->ei_val.ei_string) : 0;
	                                /* check for validity */
        if (chk_senv_left(size, left, env, (char *)NULL) ) goto goodleft;
	}
    else
	{				/* check word initial condition */
	if (	((env->ei_flags & E_NOT) && (*left != NUL)) ||
		(!(env->ei_flags & E_NOT) && (env->ei_flags & E_ELLIPSIS)) ||
		(!(env->ei_flags & E_NOT) && (*left == NUL)) )
	    goto goodleft;		/* word boundary condition okay */
	else
	    goto badleft;
	}
    /*
     *  now check for E_ELLIPSIS case
     */
    if (!(env->ei_flags & E_ELLIPSIS))
	goto badleft;			/* had to match here */
    /* if we reach here, we try again further out (due to '...') */
    }

goodleft:
myfree( myleft );			/* too bad we don't have salloc() */
return( TRUE );

badleft:
myfree( myleft );			/*   or alloca() to use!! */
return( FALSE );
} /* end of senv_left() */

/*************************************************************************
 * NAME
 *    senv_right
 * ARGUMENTS
 *    right - NUL-terminated right-hand string to be checked
 *    env   - pointer to list of right-hand side env_item's
 * DESCRIPTION
 *    Check whether the given string matches the given environment items.
 *    This is a recursive function, thanks to ellipses.
 * RETURN VALUE
 *    nonzero if successful, zero if unsuccessful
 */
static int senv_right(right,env)
char *right;
struct env_item *env;
{
int size;
struct strlist *sp;

if (env == (struct env_item *)NULL)
    return( TRUE );		/* no (more) environment */
if (right == (char *)NULL)
    return( FALSE );		/* no string at all?? */
/*
 *  match the current environment against what's wanted
 */
for (;; ++right )
    {
    /*
     *  handle hitting the end of the word
     */
    if (*right == NUL)
	{
	if ((env->ei_flags&E_CLASS) || (env->ei_val.ei_string!=(char *)NULL))
	    {
	    if ((env->ei_flags&E_OPTIONAL) && senv_right(right,env->ei_link))
		return( TRUE );
	    if (    (env->ei_flags & E_NOT) &&
		    (env->ei_link==(struct env_item *)NULL) )
		return( TRUE );		/* boundary is "NOT something" */
	    else
		return( FALSE );	/* had to match something more */
	    }
	else
	    {
	    if (env->ei_flags & E_NOT)
		return( FALSE );	/* didn't want this boundary */
	    else
		return( TRUE );		/* wanted this boundary */
	    }
	}
    /*
     *  handle having something to match against
     */
    if ( env->ei_flags & E_CLASS )
	{				/* check against a string class */
	size = 0;			/* assume it won't match */
					/* for each element that matches the 
					   string, see if it will work */
        for ( sp = env->ei_val.ei_scl->sc_members ;
              sp != (struct strlist *)NULL ;
              sp = sp->slink )
           {
           if (stcmp(right,sp->stri))
              {
              size = strlen(sp->stri);
              if (chk_senv_right (size, right, env) ) return( TRUE );
              }
           }
					/* if it did not match the class */
	if (size == 0 &&		/*  check for negation, etc. */
	    chk_senv_right (0, right, env) ) return( TRUE );
        return( FALSE );
	}
    else if (env->ei_val.ei_string != (char *)NULL)
	{				/* check against a literal string */
	size = (stcmp(right,env->ei_val.ei_string)) ?
				strlen(env->ei_val.ei_string) : 0;
	                                /* check for validity */
        if (chk_senv_right(size, right, env) ) return( TRUE );
	}
    else
	{				/* check word final condition */
	if (env->ei_flags & E_NOT)
	    return( *right != NUL );	/* ellipses don't matter for ~# */
	if (env->ei_flags & E_ELLIPSIS)
	    return( TRUE );		/* we always eventually end! */
	return( *right == NUL );
	}
    /*
     *  now check for E_ELLIPSIS case
     */
    if (!(env->ei_flags & E_ELLIPSIS))
	return( FALSE );		/* had to match here */
    /* if we reach here, we try again further out (due to '...') */
    }
} /* end of senv_right() */

/*************************************************************************
 * NAME
 *    strenv_check
 * ARGUMENTS
 *    leftstring  - pointer to the leftside environment string
 *    leftsize    - length of the leftside environment string
 *    rightstring - pointer to the rightside environment string
 *    ec          - pointer to the allomorph's string environment
 * DESCRIPTION
 *    Check if this allomorph is allowed by it's surface (string)
 *    environment.
 *	(Note: the allomorph string begins at leftstring+leftsize and
 *				  and ends at rightstring;
 *	       the length of the allomorph string is
 *			rightstring - (leftstring+leftsize)  )
 * RETURN VALUE
 *    nonzero if okay, zero if the environment is wrong
 */
int strenv_check(leftstring,leftsize,rightstring,ec)
char *leftstring;
int leftsize;
char *rightstring;
struct env_cond *ec;
{
char *left;
int val;
/*
 *  handle null arguments safely
 */
if (ec == (struct env_cond *)NULL)
    return( TRUE );		/* no environment constraint at all */
if (leftstring == (char *)NULL)
    {
    leftstring = "";
    leftsize   = 0;
    }
if (rightstring == (char *)NULL)
    rightstring = "";
/*
 *  allocate space for a copy of the leftside string
 */
left = myalloc( (unsigned)leftsize+1 );
/*
 *  go through the list of string environment conditions, quitting at the
 *  first one that succeeds
 */
for ( val = FALSE ; ec != (struct env_cond *)NULL ; ec = ec->ec_link )
    {
    strncpy(left, leftstring, leftsize);   /* need fresh copy each time */
    left[leftsize] = NUL;		/* Ensure null termination */
    val = senv_left( left, ec->ec_left) &&
	  senv_right( rightstring, ec->ec_right );
    if (ec->ec_flags & E_NOT)		/* check for negative environment */
	val = !val;
    if (val)
	break;				/* both sides okay, so quit */
    }
myfree( left );				/* free the allocated space */
return( val );
} /* end of strenv_check() */
