/*      DTBOUT.C - database format output function for AMPLE program
 ***************************************************************************
 *
 *      dtbout( wtp, outfp, outfilename )
 *	struct word_template *wtp;
 *      FILE *outfp;
 *      char *outfilename;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	30-OCT-82	D. Weber/Bob Kasper
 *	27-MAR-85	hab/djw
 *	12-Nov-85	SRMc - port CQAP to Unix and MSDOS
 *	24-Mar-88	hab - remove CTRL-X as EOR
 *      15-Apr-88       hab - allow multiple ' ' in output
 *      26-May-88       SRMc - change order of output:
 *					\a - analysis
 *					\d - decomposition
 *					\w - original word
 *					\f - preceding stuff
 *					\c - capitalization
 *                                      \n - trailing stuff
 *                              note that every field except the first one
 *                              (\a) is optional
 *      28-May-88       SRMc - use ambigchar instead of hardwired '%'
 *                           - handle empty anlist with empty orig_word
 *      28-Jul-88       SRMc - replace ssalloc() with malloc() and realloc()
 *      21-Oct-88       SRMc - reorganize the file header comments
 *      26-Oct-88       SRMc - mark failures with, e.g., %0% instead of %1%
 *                           - merge in ENCODE.C, rewrite encode() as
 *                              encoded() to allow arbitrarily sized format
 *                              and trailing nonalphabetic (\f and \n) fields
 *      10-Nov-88       SRMc - replace free() with myfree()
 *       4-Jan-89       SRMc - fix for Microsoft C
 *      26-Jul-89       hab  - replace struct strlist in defs.h with
 *                              STAMP's strlist.h
 *      27-Jul-89       hab  - add extern void myfree()
 *      28-Sep-89       ALB  - fix bug of showing \w (null) at eof with \f
 * 1.5a  2-Jan-90 ALB Add code to output final category of word
 * 1.5d  8-Jan-90 ALB Change free list to use free_strlist()
 * 1.5f  8-Jan-90 ALB Finish \fd output code
 *      02-Aug-90 hab De-lint the source
 *	27-Feb-91	SRMc - bring up to date with STAMP's TEMPLATE.h
 *			     - change name of function to dtbout()
 *			     - remove global extern variables except for
 *				ambigchar
 *	 2-Mar-91	SRMc - rename ->pcatlist to ->catlist
 *			     - make ambigchar an argument to dtbout rather
 *				than a global variable
 *			     - remove reference to myfree() -- no longer used
 ***************************************************************************
 * Copyright 1991 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <string.h>
#include "opaclib.h"
#include "template.h"
#include "strlist.h"
#ifdef __STDC__
# define	P(s) s
#else
# define P(s) ()
#endif
/* dtbout.c */
void dtbout P((struct word_template *wtp , FILE *outfp , char *outfilename , int do_orig , int ambigchar ));
/* fuldsk.c */
int full_disk P((FILE *fp , char *proc , char *fname ));
#undef P

/****************************************************************************
 * NAME
 *    dtbfmt
 * ARGUMENTS
 *    outfp     - output file pointer
 *    label     - field label
 *    list      - string list structure pointer
 *    failed    - string to use if list is NULL
 *    ambigchar - marker for ambiguities and failures
 * DESCRIPTION
 *    formats a field of the analysis database, marking failures and
 *    ambiguities
 * RETURN VALUE
 *    none
 */
static void dtbfmt( outfp, label, list, failed, ambigchar)
FILE *outfp;
char *label;
struct strlist *list;
char *failed;
int ambigchar;
{
register struct strlist *slp;
register int i;

fputs( label, outfp);
if (list == (struct strlist *)NULL)
    {                 /* mark failure */
    fprintf(outfp, "%c0%c%s%c",
	    ambigchar, ambigchar, (failed) ? failed : "", ambigchar );
    }
else if (list->slink == (struct strlist *)NULL)
    fputs(list->stri, outfp);		/* single analysis */
else
    {					/* ambiguous analysis */
    for ( i = 0, slp = list ; slp ; slp = slp->slink, ++i )
	;
    fprintf(outfp, "%c%d%c", ambigchar, i, ambigchar);
    for (slp = list; slp; slp = slp->slink)
	fprintf(outfp, "%s%c", slp->stri, ambigchar);
    }
putc('\n',outfp);			/* terminate line */
}

/***************************************************************************
 * NAME
 *    encoded
 * ARGUMENTS
 *    outfp - pointer to output FILE
 *    str   - pointer to string
 * DESCRIPTION
 *    Replace certain control characters in the old string with backslash
 *    escape sequences in the file output.
 * RETURN VALUE
 *    none
 */
static void encoded( outfp, str)
FILE *outfp;
char *str;
{
register char *p;
register int c;
register int nl_last;

for ( p = str, nl_last = 0 ; c = *p++ ; )
    {
    if (nl_last && (c != '\n'))
	{			/* we've had a \n followed by more stuff */
	fputs("\n\t",outfp);	/*  so give another line in output (safely) */
	nl_last = 0;		/*  (multiple \n's in a row allowed) */
	}
    switch ( c )
	{
	case '\b':   fputs( "\\b",  outfp);			break;
	case '\f':   fputs( "\\f",  outfp);			break;
	case '\n':   fputs( "\\n",  outfp);	nl_last = 1;	break;
	case '\r':   fputs( "\\r",  outfp);			break;
	case '\t':   fputs( "\\t",  outfp);			break;
	case '\\':   fputs( "\\\\", outfp);			break;
	default:      putc( c,      outfp);			break;
	};
    }
}

/****************************************************************************
 * NAME
 *    dtbout
 * ARGUMENTS
 *    wtp         - pointer to word_template structure
 *    outfp       - output FILE pointer
 *    outfilename - output filename
 *    do_orig     - flag to output original word (\w field)
 *    ambigchar   - character to use to separate ambiguities and mark failures
 * DESCRIPTION
 *    Format the results of analysis as a database.
 *    each word is a record with these fields:
 *	\a   = analysis (ambiguities and failures marked)
 *      \d   = morpheme decomposition (ambiguities and failures marked)
 *      \cat = final category of word
 *      \w   = original word
 *      \f   = preceding format marks
 *      \c   = capitalization
 *      \n   = trailing nonalphabetics
 *    Ambiguities are marked as %n%ANAL1%ANAL2% ... %ANALn%
 *    Failures are marked as %0%original_word%
 * RETURN VALUE
 *    none
 */
void dtbout( wtp, outfp, outfilename, do_orig, ambigchar )
struct word_template *wtp;
FILE *outfp;
char *outfilename;
int do_orig;
int ambigchar;
{
/*
 *  write the analyzed word
 */
dtbfmt(outfp, "\\a ", wtp->anlist, wtp->orig_word, ambigchar);
/*
 *  write the morpheme decomposition
 */
if (wtp->dclist)
    dtbfmt(outfp, "\\d ", wtp->dclist, wtp->orig_word, ambigchar);

if (wtp->catlist)
    dtbfmt(outfp, "\\cat ", wtp->catlist, wtp->orig_word, ambigchar);

if (wtp->fdlist)
    dtbfmt(outfp, "\\fd ", wtp->fdlist, wtp->orig_word, ambigchar);

if (do_orig && wtp->orig_word)
    fprintf(outfp, "\\w %s\n", wtp->orig_word);
/*
 *  write any preceding format marks
 */
if (wtp->format)
    {
    fprintf(outfp, "\\f ");
    encoded(outfp, wtp->format);
    fprintf(outfp, "\n");
    }
/*
 *  write the capitalization flag
 */
if (wtp->capital)
    fprintf(outfp, "\\c %d\n", wtp->capital);
/*
 *  write any trailing non-alphabetics
 */
if (wtp->non_alpha && (strcmp(wtp->non_alpha, " ") != 0))
    {
    fprintf(outfp, "\\n ");
    encoded(outfp, wtp->non_alpha);
    fprintf(outfp, "\n");
    }
/*
 *  output blank line between records
 */
fputs( "\n", outfp);
full_disk( outfp, "DTBOUT", outfilename );
}
