/*      CATEG.C - handle categories and category classes
 ***************************************************************************
 *
 *	void init_cat()
 *
 *	add_cat(line)					add_cat
 *	char *line;
 *
 *	int find_cat(catname)				search_cats_name
 *	char *catname;
 *
 *	show_cats()
 *
 *	char *cat_name(catnum)				search_cats_num
 *	int catnum;
 *
 *	init_ccl()
 *
 *	add_ccl(line)					add_ccl
 *	char *line;
 *
 *      struct cat_class *find_ccl(name)                search_ccl
 *	char *name;
 *
 *	show_ccl()
 *
 *	int ccl_member(cat, class)			member_ccl
 *	int cat;
 *	struct cat_class *class;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	28-APR-88	Steve McConnel
 *	 4-May-88	SRMc - rework for new cat_class structure
 *	 6-May-88	SRMc - add the category functions, rename the file
 *				from CATCLA.C to CATEG.C
 *	12-May-88	SRMc - 8-bit category storage
 *	28-Jul-88	SRMc - replace ssalloc() with malloc() and realloc()
 *						
 *	24-Aug-88	SRMc - need to increment num_cats!!
 *			     - regularize error messages
 *	30-Aug-88	SRMc - remove quotes from output of show_ccl()
 *      19-Oct-88       SRMc - reorganize the file header comments
 *      10-Nov-88       SRMc - replace free() with myfree()
 *       3-Apr-89       SRMc - allow previously defined category classes to
 *                              be used in defining category classes
 *      16-May-89       SRMc - added show_cats() to display all categories
 *                           - renamed load_cat() to add_cat(), and renamed
 *                              load_ccl() to add_ccl()
 *      13-Jul-89       hab  - de-"lint" the source
 *      27-Jul-89       hab  - fix extern void myfree()
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 * 1.6e 02-Aug-90 hab Change index to strchr and rindex to strrchr
 *	17-Jan-91	SRMc - replace streq() with strcmp()
 *			     - add ANSI-fied extern function prototypes
 *			     - explicitly declare init_cat() as void
 *	29-Jan-91	SRMc - merged in AMPLE 1.6f sources
 *	11-Mar-91	SRMc - remove #define strchr -- it's in OPACLIB.H
 ***************************************************************************
 * Copyright 1988, 1991 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#endif
#include "opaclib.h"
#include "class.h"
#include "strlist.h"

#ifdef __STDC__
#define P(s) s
#else
#define P(s) ()
#endif

/* categ.c */
void init_cat P((void ));
void add_cat P((char *line ));
int find_cat P((char *name ));
void show_cats P((void ));
char *cat_name P((int catnum ));
void init_ccl P((void ));
void add_ccl P((char *line ));
struct cat_class *find_ccl P((char *name ));
int ccl_member P((int cat , struct cat_class *class ));
void show_ccl P((void ));
/* strlist.c */
void write_strlist P((struct strlist *list , int sep , FILE *outfp ));

/* getwd.c */
char *skipwhite P((char *cp ));
char *getwd P((char *cp ));

/* myallo.c */
/* char *myalloc P((unsigned size )); */
void myfree P((char *s ));

/* local functions */
struct cat_class *find_ccl P((char *name));
int find_cat P((char *name));
static void bad_cc P((char *mcp , char *head , char *body , char *tail));
#undef P

/****************************************************************************
 *  data used for categories
 *
 *  Categories are stored internally as a linked list of category names.
 *  This list is stored in the order that categories are loaded, and thus
 *  a tail pointer as well as a head pointer are maintained.
 *  Externally, categories are represented by a small integer, which
 *  effectively gives the position in the internal list.  The limit on the
 *  number of categories allowed is imposed by the size of the (unsigned)
 *  integer which stores this external representation.
 *
 *  Notice that the list of categories itself is invisible outside this file.
 */
#define MAXCATS 255			/* cannot use zero */
static struct strlist *cat_head = NULL;
static struct strlist *cat_tail = NULL;
static int num_cats = 0;

/*************************************************************************
 * NAME
 *    init_cat
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Initialize the categories, by freeing up any space that may have
 *    been allocated, and clearing all pointers and counters.
 * RETURN VALUE
 *    none
 */
void init_cat()
{
register struct strlist *s, *ns;

for ( s = cat_head ; s != (struct strlist *)NULL ; s = ns )
    {
    ns = s->slink;
    myfree( s->stri );
    myfree( (char *)s );
    }
cat_head = cat_tail = (struct strlist *)NULL;
num_cats = 0;
}

/*************************************************************************
 * NAME
 *    add_cat
 * ARGUMENTS
 *    line - pointer to "category" line from analysis data file
 * DESCRIPTION
 *    Add one or more categories defined by the user to the list of
 *    categories.
 * RETURN VALUE
 *    none
 */
void add_cat(line)
char *line;
{
char *cat;
register struct strlist *s;
static char errhead[] = "\nCATEGORY: ";
/*
 *  check for a null line
 */
line = skipwhite(line);
if ((line == (char *)NULL) || (*line == NUL))
    {
    printf("%sEmpty definition", errhead);
    return;
    }
for ( cat = line ; *cat != '\0' ; cat = line )
    {
    line = getwd(cat);
    if (find_cat(cat))
	printf("%s%s already defined", errhead, cat );
    else if (++num_cats > MAXCATS)
	{
	printf("%sToo many categories - cannot define %s", errhead, cat );
	return;
	}
    else
	{
	/*
	 *  allocate space for this category and link it to the end of the
	 *  list of categories
	 */
	s = (struct strlist *)myalloc(sizeof(struct strlist));
	s->stri = strcpy( myalloc((unsigned)strlen(cat)+1), cat);
	s->slink = (struct strlist *)NULL;
	if (cat_head == (struct strlist *)NULL)
	    cat_head = s;
	else
	    cat_tail->slink = s;
	cat_tail = s;
	}
    }
}

/*************************************************************************
 * NAME
 *    find_cat
 * ARGUMENTS
 *    name - pointer to category's name
 * DESCRIPTION
 *    Search for a specific category by name.
 * RETURN VALUE
 *    integer value of the category, or zero if not found
 */
int find_cat(name)
char *name;
{
register struct strlist *s;
register int k;
/*
 *  check for an empty name
 */
if ((name == (char *)NULL) || (*name == '\0'))
    return( 0 );
/*
 *  search down the list of category names
 */
for ( k = 1, s = cat_head ; s != (struct strlist *)NULL ; s = s->slink, ++k )
    {
    if (strcmp(name,s->stri)==0)
	return(k);
    }
return( 0 );
}

/*************************************************************************
 * NAME
 *    show_cats
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Display the names of all the defined categories.
 * RETURN VALUE
 *    none
 */
void show_cats()
{
printf("\nCategories: ");
write_strlist( cat_head, ' ', stdout );
}

/*************************************************************************
 * NAME
 *    cat_name
 * ARGUMENTS
 *    catnum - category number
 * DESCRIPTION
 *    Search for the name of given category.
 * RETURN VALUE
 *    pointer to the category name, or NULL if nonexistent
 */
char *cat_name(catnum)
int catnum;
{
register struct strlist *s;

catnum &= 0377;			/* may have been sign-extended */
if (catnum == 0)
    return( (char *)NULL );	/* zero is invalid */
/*
 *  move to the catnum'th entry in the list, and return that category name
 */
for (s = cat_head ; --catnum && (s != (struct strlist *)NULL) ; s = s->slink)
    ;
if (s == (struct strlist *)NULL)
    return( (char *)NULL );		/* invalid category number */
else
    return( s->stri );
}

/****************************************************************************
 *  data used for category classes
 *
 *  Category classes are stored internally as a linked list, with each node
 *  in the linked list containing the name of the category class, and a
 *  NUL-terminated byte array of the category numbers of the members of the
 *  class.  This imposes a limit of 255 categories.
 *
 *  Notice that the list of classes itself is invisible outside this file.
 */
#if MAXCATS > 255
CRASH AND BURN!!!!!!!!		/* produce a compiler error */
#endif
static struct cat_class *classes = NULL;

/*************************************************************************
 * NAME
 *    init_ccl
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Initialize the category classes.
 * RETURN VALUE
 *    none
 */
void init_ccl()
{
register struct cat_class *cp, *ncp;

for ( cp = classes ; cp != (struct cat_class *)NULL ; cp = ncp )
    {
    ncp = cp->cc_next;
    myfree( cp->cc_name );
    myfree( cp->cc_members );
    myfree( (char *)cp );
    }
classes = (struct cat_class *)NULL;
}

/*************************************************************************
 * NAME
 *    add_ccl
 * ARGUMENTS
 *    line - pointer to "category class" line from analysis data file
 * DESCRIPTION
 *    Add a category class defined by the user to the list of category
 *    classes.
 * RETURN VALUE
 *    none
 */
void add_ccl(line)
char *line;
{
char *name;
char members[256];
register int i, j;
register char *p;
register struct cat_class *cp;
char *q;
struct cat_class *mp;
static char errhead[] = "\nCATEGORY CLASS: ";
/*
 *  split off the category class name and check that no category class by
 *  this name has been defined yet
 */
name = skipwhite(line);
if ((line == NULL) || (*name == NUL))
    {
    printf("%sEmpty definition", errhead );
    return;
    }
line = getwd(name);
if (*line == '\0')
    {
    printf("%sNo categories given in class %s", errhead, name );
    return;
    }
for ( cp = classes ; cp != (struct cat_class *)NULL ; cp = cp->cc_next )
    {
    if (strcmp(name, cp->cc_name)==0)
	{
	printf("%s%s already defined", errhead, name );
	return;
	}
    }
/*
 *  allocate space and store the category class name
 */
cp = (struct cat_class *)myalloc(sizeof(struct cat_class));
cp->cc_name = strcpy( myalloc((unsigned)strlen(name)+1), name );
/*
 *  get the categories
 */
for ( i = 0, p = line ; *p != '\0' ; p = line )
    {
    line = getwd(line);			/* split off the leading token */
    if (*p == '[')
	{
	/*
	 *  we have a [, indicating a category class reference
	 *  look for the matching ]
	 */
	++p;
	if (*p == NUL)
	    {
	    p = line;
	    line = getwd(line);
	    }
	if ((p == (char *)NULL) || (*p == NUL))
	    {
	    bad_cc(members,errhead,
			"Dangling '[' in class definition of ",name);
	    break;
	    }
	q = strchr(p,']');
	if (q == (char *)NULL)
	    {
	    q = line;
	    if ((q == (char *)NULL) || (*q == NUL) || (*q != ']'))
		{
		bad_cc(members,errhead,
			"Missing ']' in class definition of ",name);
		break;
		}
	    line = getwd(line);
	    }
	*q++ = NUL;
	if (*q != NUL)
	    {
	    bad_cc(members,errhead,
		"No whitespace following ']' in class definition of ",name);
	    members[0] = NUL;
	    break;
	    }
	/*
	 *  we have a complete category class reference with the category
	 *	class name pointed to by p and terminated by a NUL
	 *  check for a valid, already defined category class
	 */
	mp = find_ccl(p);
	if (mp == (struct cat_class *)NULL)
	    {
	    bad_cc(members,errhead,"Unknown class ",p);
	    printf(" referenced in definition of %s", name);
	    break;
	    }
	/*
	 *  copy the elements from the old category class to the new
	 */
	for ( q = mp->cc_members ; (q != (char *)NULL) && (*q != NUL) ; ++q )
	    {
	    members[i] = NUL;
	    if (strchr(members,*q) == (char *)NULL)
		members[i++] = *q;
	    }
	}
    else if (*p == ']')
	{
	bad_cc(members,errhead,"Unexpected ']' in class definition of ",name);
	break;
	}
    else
	{
	j = find_cat(p);
	if (j == 0)
	    printf("%sUndefined category %s in class %s", errhead,p,name);
	else
	    members[i++] = j;
	}
    }
/*
 *  terminate the class membership list and store it
 */
if (members[0] == NUL)
    {
    myfree( cp->cc_name );
    myfree( (char *)cp );
    }
else
    {
    members[i] = '\0';
    cp->cc_members = strcpy(myalloc((unsigned)strlen(members)+1), members);
    /*
     *  link this category class into the list
     */
    cp->cc_next = classes;
    classes = cp;
    }
}

/*************************************************************************
 * NAME
 *    find_ccl
 * ARGUMENTS
 *    name - pointer to category class's name
 * DESCRIPTION
 *    Search for a specific category class by name.
 * RETURN VALUE
 *    pointer to the category class node, or NULL if not found
 */
struct cat_class *find_ccl(name)
char *name;
{
register struct cat_class *cp;
/*
 *  check for an empty name
 */
if ((name == (char *)NULL) || (*name == '\0'))
    return((struct cat_class *)NULL);
/*
 *  search down the list of category classes
 */
for ( cp = classes ; cp != (struct cat_class *)NULL ; cp = cp->cc_next )
    {
    if (strcmp(name,cp->cc_name)==0)
	return(cp);
    }
return((struct cat_class *)NULL);
}

/*************************************************************************
 * NAME
 *    ccl_member
 * ARGUMENTS
 *    cat   - category number
 *    class - pointer to category class node
 * DESCRIPTION
 *    Search a category class for a specific category.
 * RETURN VALUE
 *    1 if found, 0 if not found
 */
int ccl_member(cat,class)
int cat;
struct cat_class *class;
{
return( strchr(class->cc_members,cat) != (char *)NULL );
}

/*************************************************************************
 * NAME
 *    show_ccl
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Display the contents of all the category classes.
 * RETURN VALUE
 *    none
 */
void show_ccl()
{
register struct cat_class *ccl;
register char *p;

printf("\nCategory classes:");
for ( ccl = classes ; ccl ; ccl = ccl->cc_next )
    {
    printf( "\n    %s {", ccl->cc_name);
    for ( p = ccl->cc_members ; *p ; ++p )
	printf(" %s", cat_name((*p)&0377) );
    printf( " }" );
    }
}

/***************************************************************************
 * NAME
 *    bad_cc
 * ARGUMENTS
 *    mcp  - pointer to category string
 *    head - first part of the error message
 *    body - second part of the error message
 *    tail - third part of the error message
 * DESCRIPTION
 *    Print an error message, then erase all of the elements stored for a
 *    category class.
 * RETURN VALUE
 *    none
 */
static void bad_cc(mcp, head, body, tail)
char *mcp;
char *head, *body, *tail;
{
/*
 *  print the error message
 */
printf("%s%s%s", head, body, tail );
/*
 *  erase any class members which may have been stored
 */
*mcp = NUL;
}
