#ifdef PC
#include <alloc.h>
#endif

/*
 *   Name: klstr.c - string list manipulation routines
 *
 *   This module contains the routines which manipulate lists of strings.
 *   These manipulations are used while we are in the process of expanding
 *   all strings which contain alternative and/or optional values.
 *
 *   The following utility routines are supported:
 *
 *       lstradd     add an lstr to front of list of lstrs
 *       lstralt     form cross product of two sets of lstrs
 *       lstropt     add an empty string to a list of lstrs
 *       lstrapp     append a list of lstrs onto another list
 *       freestr     free a lstr
 *       dumplstr    dump a linked list of lstrs
 *       segset_name    retunr name of a segment set
 *       segment_name   return name of a segment or subset
 *       dumpstr     dump a string of segsets
 *       in_segset   is a lex/surf pair in a segset
 *       is_a        is one set a subset of another
 *       tstrcpy     copy a 0 terminated string of shorts
 *       tstrcat     concatenate onto a 0 terminated string of shorts
 *       tstrlen     return number of shorts in 0 terminated string
 */

#include <stdio.h>

#define EXTERN extern
#include "kgen.h"

/*
 *  Add a new string onto the front of a (possibly empty) list of
 *  strings.
 */

lstradd(lstr, str)
struct lstr **lstr;
short *str;
{
   struct lstr *tmp = (struct lstr *)malloc(sizeof (struct lstr));

   tstrcpy(tmp->text, str);

   if (*lstr == NULL) 
      tmp->nextstr = NULL;
   else 
      tmp->nextstr = *lstr;

   *lstr = tmp;
}


/*
 *  Form a string list containing all combinations of items from list 1
 *  followed by items from list 2.
 */

struct lstr *
lstralt(in1, in2, dofree)
struct lstr *in1;
struct lstr *in2;
int dofree;         /* if true free all of in1 and in2 */
{
   struct lstr *tmp = NULL;
   struct lstr *p;

   for ( ; in1!=NULL; in1=in1->nextstr) 
      for (p=in2; p!=NULL; p=p->nextstr) {
         lstradd(&tmp, in1->text);
         tstrcat(tmp->text, p->text);
      }

   if (dofree) {
      freestr(in1);
      freestr(in2);
   }

   return tmp;
}



/*
 *  If a set of string options does not allready include the empty
 *  string, add it.
 */

struct lstr *
lstropt(in1)
struct lstr *in1;
{
   struct lstr *p;
   struct lstr *tmp = NULL;

   for (p=in1; p!=NULL; p=p->nextstr)
      if (tstrlen(p->text) == 0)
         return in1;

   tmp = (struct lstr *)malloc(sizeof(struct lstr));
   tmp->text[0] = 0;
   tmp->nextstr = in1;

   return tmp;
}


/*
 *  Append list of string specified by in2 to end of list
 *  in1
 */
struct lstr *
lstrapp(in1, in2)
struct lstr *in1;
struct lstr *in2;
{
   struct lstr *p;

   for (p=in1; p->nextstr!=NULL; p=p->nextstr) {}
   p->nextstr = in2;

   return in1;
}


/*
 *  Free the list of lstr's pointed to by p
 */

freestr(p)
struct lstr *p;
{
   struct lstr *q;

   while (p != NULL) {
      q = p->nextstr;
      free(p);
      p = q;
   }
}


/*
 *  Dump a linked list of strings
 */

dumplstr(hdr, lstr)
char *hdr;
struct lstr *lstr;
{
   short *p;

   printf("%s:\n\n", hdr);
   while (lstr != NULL) {
      for (p = lstr->text; *p; ++p)
         printf("%s", segset_name(*p));
      printf("\n");
      lstr = lstr->nextstr;
   }
   printf("\n");
}


/*
 *  Return a pointer to a character string naming a given segment set.
 *  If the segment is a named set, return the name of the set.
 *  If the segment is an unamed set, return the set members in {}.
 */

char *
segset_name(segsetind)
int segsetind;            /* segment set index */
{
   static char name[128];
   struct segset *segp = &segset[segsetind & INDEXMASK];

   strcpy(name, segment_name(segp->lexseg));
   strcat(name, ":");
   strcat(name, segment_name(segp->surfseg));

   if (segsetind & ALTERNATIVE)
      strcat(name, "|");
   if (segsetind & REPEAT)
      strcat(name, "*");
   if (segsetind & COMMIT)
      strcat(name, ">");   
   strcat(name, " ");

   return name;
}


char *
segment_name(seg)
int seg;
{
   static char name[128];

   if (seg & COMPLEMENT)
      strcpy(name, "@");
   else if (seg & SUBSET)
      strcpy(name, subset[seg & INDEXMASK].subname);
   else {
      name[0] = seg & INDEXMASK;
      name[1] = 0;
   }

   return name;
}


/*
 *  Output a string of segment pair names
 */

dumppairs(txt, str)
char *txt;
short *str;
{
   int i;
  
   printf("%s : ", txt);
   for (; *str; ++str)
      printf("%c:%c ", all_lex[*str], all_surf[*str]);
   printf("\n");
}



/*
 *  Output a string of segset names
 */

dumpstr(txt, str)
char *txt;
short *str;
{
   int i;
  
   printf("%s : ", txt);
   for (; *str; ++str)
      printf("%s", segset_name(*str));
   printf("\n");
}



/*
 *  Return true if the specified pair is a member of the specified
 *  segment set.
 */

in_segset(pair, set)
int pair;
int set;
{
   short *src = segset[set & INDEXMASK].segs;
   int ans = 0;

   for (; *src && !ans; ++src)
      if (*src == pair)
         ans = 1;

   return ans;
}


/*
 *  Return true if set1  is a subset of set2.
 */

is_a(set1, set2)
int set1;
int set2;
{
   short *src = segset[set1 & INDEXMASK].segs;
   int ans = 1;

   for (; *src && ans; ++src)
      if (! in_segset(*src, set2) )
         ans = 0;

   if (debug >= 4) {
      printf ("is_a %s,", segset_name(set1));
      printf ("%s = %d\n", segset_name(set2), ans);
   }
   return ans;
}



/*
 *  Copy a 0 terminated string of shorts.
 */

tstrcpy(dst, src)
short *dst;
short *src;
{
   while (*src)
      *dst++ = *src++;
   *dst = 0;
}


/*
 *  Concatenate onto a zero terminated string of shorts.
 */

tstrcat(dst, src)
short *dst;
short *src;
{
   while (*dst)
      ++dst;
   tstrcpy(dst, src);
}


/*
 *  Return number of elements in a zero terminated string of shorts.
 */

tstrlen(src)
short *src;
{
   int len = 0;

   while (*src++)
      ++len;
   return len;
}


