/* fEopen.c -- Copyright 1988,1992 Liam R. E. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* $Id: fEopen.c,v 1.2 92/04/29 13:39:19 lee Exp $
 * Error checking versions of fopen() and fclose(), taken from Liam Quin's
 * unreleased error handling library and stripped down somewhat.
 */

#include "globals.h"
#include "emalloc.h"
#include "error.h"

#include <stdio.h>
#include <malloc.h>	/* This declares malloc(), realloc() etc... */

#include <errno.h>
#ifdef SYSV
# include <string.h>
#else
# include <strings.h>
#endif

/** error-checking fopen and fclose...
 **
 ** Contents:
 **
 **
 ** FILE *fEopen(int Severity; char *Name; char *What; char *Mode);
 **	This opens the Named file in the given mode; otherwise, it
 **	calls Error() with the appropriate severity.  "What" is a short
 **	description of the file -- e.g. "boys' names".  Error
 **	messages include this, for example
 **	prog: fatal error: couldn't open "name" (boys' names): file not found
 **
 ** void fEclose(int severity; FILE *fp; char *Name; char *What);
 **/

FILE *
fEopen(Severity, Name, What, Mode)
    int Severity;
    char *Name;
    char *What;
    char *Mode;
{
    char *FindMissingPart();
    FILE *Result;
    char *Format = "couldn't open \"%s\" (%s)";
    char *Problem;

    if (!Name) {
	if (What && *What) {
	    Error(Severity|E_INTERNAL, Format, "[NULL name]", What);
	} else {
	    Error(Severity|E_INTERNAL, "fEopen(%d, 0, 0, %s) illegal",
			Severity, (Mode ? ((*Mode)? Mode : "\"\"" ) : "0"));
	}
	return (FILE *) 0;
    }

    if (!Mode || !*Mode) {
	if (!What || !*What) {
	    Error(Severity|E_INTERNAL,
		 "fEopen: couldn't open \"%s\" (%s), since no Mode was given",
		 Name, What);
	} else {
	    Error(Severity|E_INTERNAL,
		 "fEopen: couldn't open \"%s\", since no Mode was given",
		 Name);
	}
	return (FILE *) 0;
    }

    if (IsDir(Name)) {
	Error(Severity, "\"%s\" (%s) is a directory (not opened)",
			Name, What);
	return (FILE *) 0;
    }

    errno = 0;
    if ((Result = fopen(Name, Mode)) != (FILE *) 0) {
	return Result;
    }

    /* Try to find out the exact problem by scrutinising errno...
     * We know that Name is not null, so 
     */
    switch (errno) {
    case EFAULT:
	Error(Severity|E_INTERNAL,
		"fEopen called with invalid file-name (EFAULT)");
	/* NOTE: don't print "what" out in this case lest it's broken too */
	return (FILE *) 0;
    case E2BIG:
#ifdef ENAMETOOLONG
    case ENAMETOOLONG:
#endif
	Error(Severity|E_SYS, "Can't open file \"%1024.1024s...\" (%s)",
		Name, What);
	return (FILE *) 0;
    case EPERM:
    case ENOENT:
    case EACCES:
    case ENOTDIR:

	if ((Problem = FindMissingPart(Name)) != (char *) NULL) {
	    Error(Severity,
		    "\"%s\" (%s) not opened -- can't find directory \"%s\"",
				    Name, What, Problem);
	    return (FILE *) 0;
	}
    }


    /* if we get here, there was another problem opening the file */
    Error(Severity|E_SYS, "Can't open file \"%s\" (%s)", Name, What);
    return (FILE *) 0;
}

char *
FindMissingPart(BadPath)
    char *BadPath;
{
    int e = errno;
    char *Path;

    register char *Start, *End;
    /* We must look to find out which component of the path is missing.
     * We can be sure that the first is not there.
     * Note that the parameter BadPath might be read-only...
     */
    
    if (!BadPath || !*BadPath) return (char *) NULL;

    if ((Path = malloc(strlen(BadPath) + 1)) == (char *) NULL) {
	errno = e;
	return (char *) NULL;
    }

    errno = e;

    for (Start = End = BadPath; *End; End++) {
	if (*End == '/') Start = End;
    }

    if (Start == BadPath) { /* no / in the name -- it's a simple file */
	(void) free(Path);
	errno = e;
	return BadPath;
    }

    while (Start != BadPath) {
	(void) strncpy(Path, Start, End - Start);
	if (IsDir(Path)) {
FoundIt:
	    /* We have found the first extant path component...
	     * so let's put one more in as the first that does
	     * not exist.
	     */

	    if (*(Start = End) == '/') {
		++Start; ++End;
	    }

	    for (; *End; End++) {
		if (*End == '/') break;
	    }
	    (void) strncat(Path, Start, End - Start);

	    errno = e;
	    return Path;
	}
    }

    if (*Path == '/') {
	Start = Path;;
	End = &Path[1];
	goto FoundIt;
    }
    return BadPath;
}

void
fEclose(Severity, fp, Name, What)
    int Severity;
    FILE *fp;
    char *Name;
    char *What;
{
    int Val;

    if (!Name) {
	if (fp == stdin) Name = "standard input";
	else if (fp == stdout) Name = "standard output";
	else if (fp == stderr) Name = "standard error";
    }

    if (!fp) {
	Error(Severity|E_INTERNAL, "fEclose %s [%s] with NULL fp forbidden",
	    (What ? What : "stream"), (Name ? Name : " (unnamed) "));
    } else if (!Name) {
	Error(Severity|E_INTERNAL, "fEclose %s with NULL Name forbidden",
		What ? What : "a file");
    }

    if ((Val = fclose(fp)) != 0) {
	Error(Severity|E_SYS, "%s: error closing %s",
					Name, What ? What : "file");
    }
}
