/* FileList.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/*
 *
 * FileList -- operations on the list of files.  This is the Document
 * Directory part of lq-Text.
 *
 * $Id: FileList.c,v 1.16 92/07/30 22:23:55 lee Exp $
 *
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>

#include "smalldb.h"
#include "fileinfo.h"
#include "emalloc.h"
#include "numbers.h"

/** Unix system calls that need to be declared: **/
extern int stat();
extern int open(), close(), creat();
extern void exit();
extern int read(), write();
extern unsigned alarm();
/** library functions that need to be declared: */
extern int lockf();
extern unsigned sleep();
extern void perror();
extern long atol();

/** other (lqtext) functions **/
t_FID GetNextFID();
t_FileInfo *GetFileInfo();
void lqGetFileModes();
/** **/

t_FID
GetMaxFID()
{
    extern int errno;

    int fd;
    struct stat StatBuf;
    char Buffer[20];

    /* ensure that the file is there */
    if (stat(FidFile, &StatBuf) == -1) {
	return 0;
    }

    if ((fd = open(FidFile, O_RDONLY, 0)) < 0) {
	int e = errno;

	fprintf(stderr, "%s: GetMaxFid: Can't read FID file ", progname);
	errno = e;
	perror(FidFile);
	exit(1);
    }

    /* Read the file */
    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "%s: GetMaxFid: Can't read from \"%s\"\n",
		progname, FidFile);
	exit(1);
    }

    (void) close(fd);

    Buffer[StatBuf.st_size] = '\0';

    return atol(Buffer);
}

/*ARGSUSED*/
t_FID
GetNextFID(Size)
    long Size; /* to let it keep short FIDs for huge files, execpt I don't */
{
    extern int errno;
    extern long atol();
    extern long lseek();
    int Flags, Modes;

    int fd;
    char Buffer[21];
    struct stat StatBuf;
    t_FID Result;

    lqGetFileModes(&Flags, &Modes);

    /* ensure that the file is there */
    if (stat(FidFile, &StatBuf) == -1) {
	fprintf(stderr, "Creating FID file \"%s\"\n", FidFile);
	if ((fd = open(FidFile, Flags|O_CREAT|O_RDWR, Modes)) < 0) {
	    fprintf(stderr, "Can't create FID file \"%s\" (%d, %d)\n",
			FidFile, Flags, Modes);
	    exit(1);
	}
	(void) close(fd);
	return GetNextFID(Size);

	/*NOTREACHED*/
    }


    if ((fd = open(FidFile, Flags, Modes)) < 0) {
	int e = errno;

	fprintf(stderr, "%s: Can't open FID file %s Flags %d Modes 0%o: ",
		progname, FidFile, Flags, Modes);
	errno = e;
	perror("system error");
	exit(1);
    }

    errno = 0;

    /* Read the file */
    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "Can't read from \"%s\"\n", FidFile);
	exit(1);
    }

    Buffer[StatBuf.st_size] = '\0';

    Result = atol(Buffer);

    if (Result == 0L || *Buffer == '-') {
	Result = 1L;
    }

    (void) sprintf(Buffer, "%lu\n", Result + 1);

    /* Move to the start of the file and write the new value.
     * No need to truncate the file, because it didn't shrink!
     */
    (void) lseek(fd, 0, 0L); /* TODO: error checking */
    (void) write(fd, Buffer, (unsigned int) strlen(Buffer)); /**/
    (void) close(fd); /* TODO: error checking */

    return Result;
}

typedef struct {
    unsigned int Size;
    char *Buffer;
} t_PhysicalIndexEntry;
# define EXTRABYTES (sizeof(long)*5) /* enough for all the "members" */

t_PhysicalIndexEntry *
FileInfo2Phys(FileInfo)
    t_FileInfo *FileInfo;
{
    t_PhysicalIndexEntry *PIE;
    char *p;
    register int NameLength;

    if (!FileInfo || !FileInfo->Name) return (t_PhysicalIndexEntry *) 0;

    NameLength = strlen(FileInfo->Name);

    PIE = (t_PhysicalIndexEntry *) emalloc(
				sizeof(t_PhysicalIndexEntry) + NameLength + 1);

    p = PIE->Buffer = (char *) emalloc(NameLength + EXTRABYTES + 1);

    sWriteNumber(&p, FileInfo->FID);
    sWriteNumber(&p, FileInfo->Date);
    sWriteNumber(&p, FileInfo->FilterType);
    sWriteNumber(&p, (long) NameLength);
    sWriteNumber(&p, FileInfo->FileSize);
    (void) strcpy(p, FileInfo->Name);

    PIE->Size = (long) (&p[NameLength] - PIE->Buffer);
    return (t_PhysicalIndexEntry *) erealloc((char *)PIE, PIE->Size);
}

t_FileInfo *
Phys2FileInfo(PIE)
    t_PhysicalIndexEntry *PIE;
{
    t_FileInfo *FileInfo;
    char *p;
    unsigned long NameLength;

    if (!PIE || !PIE->Buffer) return (t_FileInfo *) 0;

    FileInfo = (t_FileInfo *) emalloc(sizeof(t_FileInfo));

    p = PIE->Buffer;

    /* The order of these must match the order of the sWriteNumber calls in
     * FileInfo2Phys() above!
     */
    FileInfo->FID = sReadNumber(&p);
    FileInfo->Date = sReadNumber(&p);
    FileInfo->FilterType = sReadNumber(&p);
    NameLength = sReadNumber(&p);
    FileInfo->FileSize = sReadNumber(&p);

    FileInfo->Stream = (FILE *) 0;

    if (NameLength) {
	FileInfo->Name = emalloc(NameLength + 1);
	(void) strncpy(FileInfo->Name, p, NameLength);
	FileInfo->Name[NameLength] = '\0';

#if 0
	/* with this in place, wordinfo spends over 40% of its time
	 * in stat!
	 */
	if ((doc = FindFile(FileInfo->Name)) != (char *) 0) {
	    /* hence, we never retrieve non-existent files */
	    FileInfo->Name = erealloc(FileInfo->Name, strlen(doc) + 1);
	    (void) strcpy(FileInfo->Name, doc);
	}
#endif
    } else {
	FileInfo->Name = (char *) 0;
    }

    return FileInfo;
}

int
SaveFileInfo(FileInfo)
    t_FileInfo *FileInfo;
{
    t_PhysicalIndexEntry *PIE;
    datum key, data;
    DBM *db;
    int RetVal;
    char Buffer[20];
    char *p;

    if (!FileInfo) return -1;

    if ((PIE = FileInfo2Phys(FileInfo)) == (t_PhysicalIndexEntry *) 0) {
	return -1;
    }

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	return -1;
    }

    p = Buffer;
    sWriteNumber(&p, FileInfo->FID);
    data.dptr = Buffer;
    data.dsize = p - Buffer;

    if (FileInfo->Name && *(FileInfo->Name)) {
	int KeyLen = strlen(FileInfo->Name);
	key.dptr = emalloc(KeyLen + 2); /* +2: "\375" and \0 */
	*(key.dptr) = '\375';
	(void) strcpy(&(key.dptr[1]), FileInfo->Name);
	key.dsize = KeyLen + 1;
		/* length of name + length of "\375" -- the nul at the end
		 * is not included.
		 */

	(void) dbm_store(db, key, data, DBM_REPLACE);
	(void) efree(key.dptr);
    }

    key.dptr = data.dptr;
    key.dsize = data.dsize;

    data.dptr = PIE->Buffer;
    data.dsize = PIE->Size;

    RetVal = dbm_store(db, key, data, DBM_REPLACE);
    (void) efree(PIE->Buffer);
    (void) efree((char *) PIE);

    enddb(db);
    return RetVal;
}

void
lqRenameFile(OldName, NewName)
    char *OldName;
    char *NewName;
{
    t_FID FID;
    t_FileInfo *FileInfo;
    DBM *db;
    datum key, result;
    extern long atol();
    int n;
    t_FID Name2FID();

    FID = Name2FID(NewName);
    if (FID) {
	Error(E_WARN,
	    "lqReNameFile: rename %s to %s failed, new name has entry (%ld)\n",
	    OldName, NewName, FID
	);
	return;
    }

    FID = Name2FID(OldName);
    FileInfo = GetFileInfo(FID);

    n = strlen(NewName);
    key.dsize = strlen(OldName);
    /* see previous routine for comments about this +2 ugliness */
    if (n > key.dsize) {
	key.dptr = emalloc(n + 2);
    } else {
	key.dptr = emalloc(key.dsize + 2);
    }
    *(key.dptr) = '\375';
    (void) strcpy(&(key.dptr[1]), OldName);
    key.dsize += 1; /* for the cookie; we don't include the \0 */

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	Error(E_WARN|E_SYS,
	    "lqRenameFile can't get FID for %s (database \"%s\")",
	    OldName,
	    FileIndex
	);
	(void) efree(key.dptr);
	return;
    }
    result = dbm_fetch(db, key);

    /* If it was there, delete it */
    if (result.dptr != (char *) 0 && result.dsize != 0) {
	(void) dbm_delete(db, key);
    }

    /* Now make a new one */
    *(key.dptr) = '\375';
    (void) strcpy(&(key.dptr[1]), NewName);
    key.dsize = n + 1; /* for the cookie; we don't include the \0 */
    (void) dbm_store(db, key, result, DBM_REPLACE);
    /* Now change the reverse map */
    (void) strcpy(key.dptr, NewName);
    key.dsize = n;
    (void) efree(key.dptr);
    enddb(db);

    FileInfo->Name = NewName;
    SaveFileInfo(FileInfo);
}

t_FID
Name2FID(Name)
    char *Name;
{
    DBM *db;
    datum key, result;
    extern long atol();

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	Error(E_WARN|E_SYS,
	    "Name2FID can't get FID for %s (database \"%s\")",
	    Name,
	    FileIndex
	);
	(void) efree(key.dptr);
	return -1;
    }

    key.dsize = strlen(Name);
    /* see previous routine for comments about this +2 ugliness */
    key.dptr = emalloc(key.dsize + 2);
    *(key.dptr) = '\375';
    (void) strcpy(&(key.dptr[1]), Name);
    key.dsize += 1; /* for the cookie; we don't include the \0 */

    result = dbm_fetch(db, key);

    (void) efree(key.dptr);

    if (result.dptr == (char *) 0 || result.dsize == 0) {
	enddb(db);
	return (t_FID) 0;
    } else {
	t_FID FID;
	char *p = result.dptr;
	FID = (t_FID) sReadNumber(&p);
	enddb(db);
	return FID;
    }
}

t_FileInfo *
GetFileInfo(FID)
    t_FID FID;
{
    t_FileInfo *FileInfo;
    datum key, data;
    DBM *db;
    char Buffer[20];
    char *p = Buffer;
    t_PhysicalIndexEntry *PIE;

    sWriteNumber(&p, FID);
    key.dptr = Buffer;
    key.dsize = p - Buffer;

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	return (t_FileInfo *) 0;
    }

    data = dbm_fetch(db, key);

    if (data.dptr == (char *) 0 || data.dsize == 0) {
	enddb(db);
	return (t_FileInfo *) 0;
    }

    PIE = (t_PhysicalIndexEntry *) emalloc(sizeof(t_PhysicalIndexEntry));
    PIE->Size = data.dsize + 1; /* +1 because we don't store the \0 */
    PIE->Buffer = data.dptr;
    FileInfo = Phys2FileInfo(PIE);
    (void) efree((char *) PIE);

    enddb(db);
    return FileInfo;
}
