//--------------------------------------------------------------------
// Program    : neudl.C  -  Neural-Network Description Language
// Version    : 0.2
// 
// Author     : Joey Rogers - jrogers@buster.eng.ua.edu
// Date       : July 30, 1993
// Last Update: May 4, 1994
// Purpose    : This program parses NeuDL code and converts it to 
//              an object format.  Once converted, the objects can either
//              be executed or translated into C++.
//--------------------------------------------------------------------------
// NOTE:  This program and all of its supporting programs are for educational
//	  use only.  If NeuDL is distributed, it must be done so free of
//	  charge.  
//	
//	  NeuDL is still a prototype and no gaurantees are made about its
//	  validity; however, if problems are found, please notify the
//	  author (jrogers@buster.eng.ua.edu).
//--------------------------------------------------------------------------

#include<stdio.h>
#include"neudl.h"
#include"state.h"
#include"baseinst.h"
#include"coreinst.h"
#include"datainst.h"
#include"netinst.h"

#define P_UNKNOWN 0
#define P_SYMBOL 1
#define P_STRING 2
#define P_NUMBER 3
#define P_EXPRESS 4
#define P_SUBSCRIPT 5

#define NOT_RW 0
#define BEGIN 1
#define END 2
#define DECLARE_INT 3
#define IF 4
#define WHILE 5
#define PROGRAM 6
#define PRINT 7
#define NEWLINE 8
#define SKIP 9
#define FOR 10
#define READ 11

#define CREATE_NETWORK 12
#define FULLY_CONNECT 13
#define PARTIALLY_CONNECT 14
#define LOAD_DATA 15
#define FIND_HIGH_LOW 16
#define RESET_TO_HEAD 17
#define GET_DATA 18
#define COPY_ARRAY 19
#define BP_TRAIN 20
#define NORMALIZE_DATA 21
#define CREATE_DATA 22
#define ADD_DATA_BEFORE_CURR 23
#define CONNECT_WEIGHT 24
#define LOAD_NETWORK 25
#define DUMP 26
#define FORWARD_PASS 27
#define RUN_NETWORK 28
#define BEEP 29
#define PRINT_NETWORK 30
#define BP_TRAIN1 31
#define GET_WEIGHT 32
#define SET_WEIGHT 33
#define REMOVE_WEIGHT 34
#define DENORMALIZE_DATA 35
#define RESET_CURRENT_WEIGHT 36
#define GET_CURRENT_WEIGHT 37
#define NEXT_WEIGHT 38
#define NEXT_DATA 39
#define BACKWARD_PASS 40
#define GET_ERROR 41
#define RESET_ERROR 42
#define GET_NETWORK_HIGH_LOW 43
#define RESET_TO_TAIL 44
#define ADD_DATA_AFTER_CURR 45
#define SAVE_DATA 46
#define DELETE_DATA 47
#define PRINT_DATA 48
#define DECLARE_FLOAT 49
#define SAVE_NETWORK 50
#define BP_TRAIN2 51
//----------------------------------------------------------------------------
// Function Prototypes:

void Get_Token( char *&buffer, char token[], int &token_type );
void Error( char s1[] );
void Read_Inst( char buffer[] );
int Blank_Line( char buffer[] );
Instruction *Process_Instruction( void );
int is_reserved_word( char s[] );
void Store_Buffer( char buff[] );
void Initialize_NeuDL( State &s );
void Initialize_Translate( char filename[] );
void Prepare_Parameters( char *&buff_ptr, char token[]);

//-----------------------------------------------------------------------------
// Global Variables:

int curr_line=0;     // Current Line being parsed

char *reserved_word[]={"{","}","int","if","while",
			"program","print","newline","skip","for","read",
			"Create_Network","Fully_Connect","Partially_Connect",
			"Load_Data","Find_High_Low","Reset_To_Head",
			"Get_Data","Copy_Array","BP_Train","Normalize_Data",
			"Create_Data","Add_Data_Before","Connect_Weight",
			"Load_Network","Dump","Forward_Pass","Run_Network",
			"Beep","Print_Network","BP_Train1","Get_Weight",
			"Set_Weight","Remove_Weight","Denormalize_Data",
			"Reset_Current_Weight","Get_Current_Weight",
			"Next_Weight","Next_Data","Backward_Pass",
			"Get_Error","Reset_Error","Get_Network_High_Low",
		        "Reset_to_Tail","Add_Data_After","Save_Data",
			"Delete_Data","Print_Data","float","Save_Network",
		        "BP_Train2"};

int reserved_word_count=51;     // Number of reserved words

State s;		// Program state
Instruction *inst;      // Program Instruction
int mode=EXECUTE;	// Execution mode (can be EXECUTE,DEBUG,or TRANSLATE)
FILE *in;		// Input file pointer
int buff_flag=0;	// Global storage for parsing
char storage_buff[257];

//----------------------------------------------------------------------

void main( int argc, char *argv[] )
	{
	if (argc<2)   // Make sure an input file was given on command line
		{
		cout << "Usage: " << argv[0] << " <input file> <Argument 1>";
		cout << " <Argument 2> ... <Argument N> \n\n";
		exit(0);
		}

	int pos=0;			// Move command line argument to state
	for (int i=1; i<argc; i++)
		if (strcmp(argv[i],"-debug")==0) mode=DEBUG;
		else if (strcmp(argv[i],"-translate")==0) mode=TRANSLATE;
		     else
			{
			s.Set_Environment_String(pos,argv[i]);
			pos++;
			}

	in=fopen(argv[1],"r");   // Open input file
	if (in==NULL)
		{
		cout << "Could not open input file: " << argv[1] << "\n\n";
		exit(0);
		}

	Initialize_NeuDL(s);	     // Initialize System Variables
	inst=Process_Instruction();  // Get Program instruction (first pass)

	inst->Prepare(s);   // Second Pass Preparation


	if (mode==TRANSLATE)   // Translate to C++ code
		{
		Initialize_Translate(argv[1]);
		inst->Translate(s);
		cout << "}\n\n";
		}
	else inst->Execute(s);    // Interpret Code
	}
//-----------------------------------------------------------------------------
// Initialize_Translate - This function prints header information when 
// 			  translate mode is used

void Initialize_Translate( char filename[] )
	{
	cout << "// Program: " << filename << "\n\n" <<
		"#include <stdio.h>\n" <<
		"#include <iostream.h>\n" <<
		"#include \"bp.h\"\n" <<
		"#include \"data.h\"\n\n" <<

		"#define MAX_MIDDLE_LAYERS " << MAX_MIDDLE_LAYERS << "\n" <<
		"#define MAX_DATA_LISTS " << MAX_DATA_LISTS << "\n\n" <<


		"int TRAINING=0, TESTING=1;\n";

	cout << "char ___filename[80];\n" <<
		"int ___lcv;\n" <<
		"double *___temp_array;\n" <<
		"BPNET net;\n" <<
		"Data_Class *Data_List[" << MAX_DATA_LISTS << "];\n" <<
		"int Layer_Nodes[MAX_MIDDLE_LAYERS+2];\n" <<
		"int Output_Layer=0,Input_Layer=0,Layer_Count=0," <<
		"Weight_Count=0;\n" <<
		"int Data_Count[MAX_DATA_LISTS];\n" <<
		"int Data_Inputs[MAX_DATA_LISTS];\n" <<
		"int Data_Outputs[MAX_DATA_LISTS];\n";
	cout << "int ___temp1,___temp2,___temp3,___temp4;\n";
	cout << "double Learning_Rate=" << DEFAULT_LEARNING_RATE << ";\n" <<
		"double Momentum=" << DEFAULT_MOMENTUM << ";\n" <<
		"int Check_Test_Set=" << DEFAULT_CHECK_TEST_SET << ";\n" <<
		"int Min_Iterations=" << DEFAULT_MIN_ITERATIONS << ";\n" <<
		"int Max_Iterations=" << DEFAULT_MAX_ITERATIONS << ";\n" <<
		"int Display_Rate=" << DEFAULT_DISPLAY_RATE << ";\n" <<
		"double Tolerance=" << DEFAULT_TOLERANCE << ";\n" <<
		"\n\n\n" <<
		"void main( int argc, char *argv[] )\n";
	cout << "{\nfor (___lcv=0; ___lcv<MAX_DATA_LISTS; ___lcv++) {\n" <<
		"    Data_List[___lcv]=NULL;  Data_Count[___lcv]=0;\n"  <<
		"    Data_Inputs[___lcv]=0;   Data_Outputs[___lcv]=0;}\n";
		
	}
//-----------------------------------------------------------------------------
// Inialize_NNDL - Place system variables in state

void Initialize_NeuDL( State &s )
	{
	s.Declare_Allocate_Variable("Layer_Nodes",INT,0,MAX_MIDDLE_LAYERS+2);

	s.Declare_Allocate_Variable("TRAINING",INT,0);
	s.Declare_Allocate_Variable("TESTING",INT,1);

	s.Declare_Allocate_Variable("Learning_Rate",FLOAT,
				    DEFAULT_LEARNING_RATE);
	s.Declare_Allocate_Variable("Momentum",FLOAT,DEFAULT_MOMENTUM);
	s.Declare_Allocate_Variable("Check_Test_Set",INT,
				    DEFAULT_CHECK_TEST_SET);
	s.Declare_Allocate_Variable("Min_Iterations",INT,
				    DEFAULT_MIN_ITERATIONS);
	s.Declare_Allocate_Variable("Max_Iterations",INT,
				    DEFAULT_MAX_ITERATIONS);
	s.Declare_Allocate_Variable("Display_Rate",INT,DEFAULT_DISPLAY_RATE);
	s.Declare_Allocate_Variable("Tolerance",FLOAT,DEFAULT_TOLERANCE);

	s.Declare_Allocate_Variable("Data_Count",INT,0,MAX_DATA_LISTS);
	s.Declare_Allocate_Variable("Data_Inputs",INT,0,MAX_DATA_LISTS);
	s.Declare_Allocate_Variable("Data_Outputs",INT,0,MAX_DATA_LISTS);

	s.Declare_Allocate_Variable("Output_Layer",INT,0);
	s.Declare_Allocate_Variable("Input_Layer",INT,0);
	s.Declare_Allocate_Variable("Layer_Count",INT,0);

	s.Declare_Allocate_Variable("Weight_Count",INT,0);
	}
//-----------------------------------------------------------------------------
// Process Instruction - This recursive function converts an instruction to the
// 			 appropriate instruction class declarations.

Instruction *Process_Instruction( void )
	{
	Instruction *inst1,*inst2, *inst3;
	Compound_Inst *compound;
	char buffer[257];
	char *buff_ptr,token[257],*token_ptr,*token_ptr1;
	int token_type;
	char name[257];
	char exp[257];
	int size;
	int paren_count;
	int init_line;
	int type;

	Read_Inst(buffer); // Get input from file

	if (mode==DEBUG)   // If Debug Mode, print data as it is parsed
		{
		buff_ptr=buffer;
		while (isspace(*buff_ptr)) buff_ptr++;
		cout << setw(3) <<curr_line<<": " << buff_ptr << "\n" << flush;
		}

	buff_ptr=buffer;
	Get_Token(buff_ptr,token,token_type);

	switch(is_reserved_word(token))
	   {
	   case PROGRAM: return Process_Instruction();

	   case DECLARE_FLOAT:	 
	   case DECLARE_INT: 
			 switch(is_reserved_word(token))
				{
				case DECLARE_FLOAT: type=FLOAT;
					            break;
				case DECLARE_INT:   type=INT;
					            break;
				}
			 Get_Token(buff_ptr,token,token_type);
			 strcpy(name,token);

			       
			 if (*buff_ptr=='[')  // Get dimen, if any 
				{
				Get_Token(buff_ptr,token,token_type);
				return new Declare_Inst(curr_line, name, 
						        type, token);
				}
			 else 
				return new Declare_Inst(curr_line, name, type);



	   case NOT_RW: strcpy(name,token);    // Assignment Statement
			if (*buff_ptr=='[')
			    {
			    Get_Token(buff_ptr,token,token_type);
			    buff_ptr++;
			    }
			else strcpy(token,"0"); // Assume first if not given 
			while (isspace(*buff_ptr)) buff_ptr++;
			switch(*buff_ptr)
			   {
			   case '=': buff_ptr++;
				     strcpy(exp,buff_ptr);
				     break;
			   case '+':
			   case '-':
			   case '*':
			   case '/': strcpy(exp,name);
				     if (strcmp(token,"0")!=0)
					{
					strcat(exp,"[");
					strcat(exp,token);
					strcat(exp,"]");
					}
				     char temp[2];
				     temp[0]=*buff_ptr;
				     temp[1]='\0';
				     strcat(exp,temp);
				     buff_ptr++;
				     switch (*buff_ptr)
				       {
				       case '+':
				       case '-': strcat(exp,"1");
						 break;
				       default:  strcat(exp,"(");
						 strcat(exp,++buff_ptr);
						 strcat(exp,")");
						 break;
				       }
				     break;

			   default:
	 		     Error("Error in assignment - Invalid Operator");
			   }		

			return new 
				Assignment_Inst(curr_line,name,token,exp);

	   case PRINT:  Prepare_Parameters(buff_ptr,token);
			return new Print_Inst(curr_line,token);

	   case NEWLINE: return new NewLine_Inst(curr_line);

	   case SKIP   : return new Skip_Inst(curr_line);

	   case BEEP   : return new Beep_Inst(curr_line);

	   case PRINT_NETWORK  : return new Print_Network_Inst(curr_line);

	   case DUMP   : return new Dump_Inst(curr_line);

	   case IF:     Get_Token(buff_ptr,token,token_type);
			if (token_type!=P_EXPRESS)
 			   Error("No expression in conditional");
			strcpy(exp,token);
			buff_ptr++;
			init_line=curr_line;
			if (!Blank_Line(buff_ptr))
				Store_Buffer(buff_ptr);
			inst1=Process_Instruction();
			Read_Inst(buffer);
			buff_ptr=&buffer[0];
			Get_Token(buff_ptr,token,token_type);
			if (strcmp(token,"else")!=0)  // no else branch 
				{
				Store_Buffer(buffer); 
				inst2=new Skip_Inst(curr_line);
				}
			else			      // with else branch 
				{
				if (!Blank_Line(buff_ptr)) 
					Store_Buffer(buff_ptr);
				inst2=Process_Instruction();
				}
			return new Conditional_Inst(init_line,exp,inst1,inst2);

	   case WHILE:  Get_Token(buff_ptr,token,token_type);
			if (token_type!=P_EXPRESS)
			   Error("No expression in while loop");
			strcpy(exp,token);
			buff_ptr++;
			init_line=curr_line;
			if (!Blank_Line(buff_ptr))
				Store_Buffer(buff_ptr);
			inst1=Process_Instruction();
			return new While_Inst(init_line,exp,inst1);

							// Compound Inst
	   case BEGIN:  compound=new Compound_Inst(curr_line,mode); 
			inst1=Process_Instruction();
			while (inst1!=NULL)
				{
				compound->Add_Instruction(inst1);
				inst1=Process_Instruction();
				}
			return compound;

	   case FOR:    init_line=curr_line;
			while (*buff_ptr!='\0')
				{
				if (*buff_ptr=='(') break;
				buff_ptr++;
				}
			if (*buff_ptr!='(')
				Error("Missing Expression in For Loop");
			
			buff_ptr++;
			Store_Buffer(buff_ptr);
			inst1=Process_Instruction();   // loop initializer 
			Read_Inst(buffer);
			strcpy(exp,buffer);            // Test expression 
			Read_Inst(buffer);
			paren_count=1;
			buff_ptr=buffer;
			token_ptr=token;
			while (*buff_ptr!=(unsigned char)'\0')
				{
				if (*buff_ptr=='(') paren_count++;
				else if (*buff_ptr==')') 
					{
					paren_count--;
					if (paren_count==0)
						break;
					}
				*token_ptr=*buff_ptr;
				token_ptr++;
				buff_ptr++;
				}
			*token_ptr='\0';
			buff_ptr++;
			Store_Buffer(token);
			inst2=Process_Instruction();    // loop incrementer 
			if (!Blank_Line(buff_ptr))
				Store_Buffer(buff_ptr);
			inst3=Process_Instruction();    // loop body 
			return new For_Inst(init_line,inst1,exp,inst2,inst3);
			
	   case READ:   while (*buff_ptr!='\0')
				{
				if (*buff_ptr=='(') break;
				buff_ptr++;
				}
			if (*buff_ptr!='(')
				Error("Invalid Read Statement");
			*buff_ptr++;
			Get_Token(buff_ptr,token,token_type);
			if (token_type!=P_STRING)
				Error("Invalid identifier in Read");
			strcpy(name,token);
			if (*buff_ptr=='[')
				Get_Token(buff_ptr,token,token_type);
			else
				strcpy(token,"0");

			return new Read_Int_Inst(curr_line,name,token);
			

	   case END:    return NULL;  // End of Compound Stat.


	   case CREATE_NETWORK: Prepare_Parameters(buff_ptr,token);
  			    return new Create_Network_Inst(curr_line,token);

	   case FULLY_CONNECT:
			return new Fully_Connect_Inst(curr_line);

	   case PARTIALLY_CONNECT:
			return new Partially_Connect_Inst(curr_line);

	   case LOAD_DATA: Prepare_Parameters(buff_ptr,token);
   			   return new Load_Data_Inst(curr_line,token);

	   case SAVE_DATA: Prepare_Parameters(buff_ptr,token);
   			   return new Save_Data_Inst(curr_line,token);

	   case FIND_HIGH_LOW: Prepare_Parameters(buff_ptr,token);
			return new Find_High_Low_Inst(curr_line,token);

	   case BP_TRAIN:  Prepare_Parameters(buff_ptr,token);
			return new BP_Train_Inst(curr_line,token,0);

	   case BP_TRAIN1:   Prepare_Parameters(buff_ptr,token);
			return new BP_Train_Inst(curr_line,token,1);

	   case BP_TRAIN2:   Prepare_Parameters(buff_ptr,token);
			return new BP_Train_Inst(curr_line,token,2);

	   case DELETE_DATA:  Prepare_Parameters(buff_ptr,token);
			return new Delete_Data_Inst(curr_line,token);

	   case PRINT_DATA:  Prepare_Parameters(buff_ptr,token);
			return new Print_Data_Inst(curr_line,token);

	   case RESET_TO_HEAD:  Prepare_Parameters(buff_ptr,token);
			return new Reset_To_Head_Inst(curr_line,token);

	   case RESET_TO_TAIL:  Prepare_Parameters(buff_ptr,token);
			return new Reset_To_Tail_Inst(curr_line,token);

	   case GET_DATA: Prepare_Parameters(buff_ptr,token);
			return new Get_Data_Inst(curr_line,token);

	   case COPY_ARRAY: Prepare_Parameters(buff_ptr,token);
			return new Copy_Array_Inst(curr_line,token);

	   case NORMALIZE_DATA: Prepare_Parameters(buff_ptr,token);
			return new Normalize_Data_Inst(curr_line,token);

	   case DENORMALIZE_DATA: Prepare_Parameters(buff_ptr,token);
			return new Denormalize_Data_Inst(curr_line,token);

	   case CREATE_DATA: Prepare_Parameters(buff_ptr,token);
			return new Create_Data_Inst(curr_line,token);

	   case ADD_DATA_BEFORE_CURR: Prepare_Parameters(buff_ptr,token);
			return new Add_Data_Before_Curr_Inst(curr_line,token);

	   case ADD_DATA_AFTER_CURR: Prepare_Parameters(buff_ptr,token);
			return new Add_Data_After_Curr_Inst(curr_line,token);

	   case CONNECT_WEIGHT:  Prepare_Parameters(buff_ptr,token);
			return new Connect_Weight_Inst(curr_line,token);

	   case LOAD_NETWORK:  Prepare_Parameters(buff_ptr,token);
			return new Load_Network_Inst(curr_line,token);

	   case SAVE_NETWORK:  Prepare_Parameters(buff_ptr,token);
			return new Save_Network_Inst(curr_line,token);

	   case FORWARD_PASS:  Prepare_Parameters(buff_ptr,token);
			return new Forward_Pass_Inst(curr_line,token);

	   case RUN_NETWORK:  Prepare_Parameters(buff_ptr,token);
			return new Run_Network_Inst(curr_line,token);  

	   case GET_WEIGHT: Prepare_Parameters(buff_ptr,token);
			return new Get_Weight_Inst(curr_line,token);  

	   case SET_WEIGHT: Prepare_Parameters(buff_ptr,token);
			return new Set_Weight_Inst(curr_line,token);  

	   case REMOVE_WEIGHT:  Prepare_Parameters(buff_ptr,token);
			return new Remove_Weight_Inst(curr_line,token);  

	   case RESET_CURRENT_WEIGHT:
			return new Reset_Current_Weight_Inst(curr_line); 

	   case GET_CURRENT_WEIGHT: Prepare_Parameters(buff_ptr,token);
			return new Get_Current_Weight_Inst(curr_line,token);  

	   case NEXT_WEIGHT: 
			return new Next_Weight_Inst(curr_line);  

	   case NEXT_DATA: Prepare_Parameters(buff_ptr,token);
			return new Next_Data_Inst(curr_line,token);  

	   case BACKWARD_PASS: Prepare_Parameters(buff_ptr,token);
			return new Backward_Pass_Inst(curr_line,token);  

	   case GET_ERROR: Prepare_Parameters(buff_ptr,token);
			return new Get_Error_Inst(curr_line,token);  

	   case RESET_ERROR: return new Reset_Error_Inst(curr_line);  

	   case GET_NETWORK_HIGH_LOW: Prepare_Parameters(buff_ptr,token);
			return new Get_Network_High_Low_Inst(curr_line,token); 
	   }
	return NULL;
	}
//-----------------------------------------------------------------------------
// Store_Buffer - This function reinserts an instruction back into the input
// 		  file.  The instruction it stores will be the next one read.

void Store_Buffer( char buff[] )
	{
	buff_flag=1;
	strcpy(storage_buff,buff);
	}
//-----------------------------------------------------------------------------
// Read_Inst - This function reads an instruction from the input file, 
// 	       updates the line counter, and echos the line to the screen

void Read_Inst( char buffer[] )
	{
	int i;
	char c;
	char *buff_ptr;
	static char Brace_Flag=0;
	int comment_flag=0;
	int count=0;
	
	if (buff_flag==1)   // if text was inserted back into the file
		{	    // use it instead of reading new text
		buff_flag=0;
		strcpy(buffer,storage_buff);
		return;
		}

	if (feof(in)) return;

	if (Brace_Flag=='}' || Brace_Flag=='{')
		{
		buffer[0]=Brace_Flag;
		buffer[1]='\0';
		Brace_Flag=0;
		}
	else
		{
		buff_ptr=buffer;
		c=fgetc(in);
	    	while (c!='}' && c!='{' && c!=';' && !feof(in))
			{
		     // Remove Commented Text
			if (c=='/' && comment_flag==0) comment_flag=1;
			else if (c=='/' && comment_flag==1) 
				{
				comment_flag=0;
				if (count>0) 
					{
					buff_ptr--;
					count--;
					}
				c=fgetc(in);
				while (c!='\n' && !feof(in)) 
					c=fgetc(in);
				}
 
			if (c!='\n')
				{
				*buff_ptr=c;
				buff_ptr++;
				count++;
				}
			else curr_line++;
			c=fgetc(in);
			}
		*buff_ptr='\0';
		if (c=='{'  || c=='}')
			Brace_Flag=c;
		}

	if (Blank_Line(buffer))
		Read_Inst(buffer);
	}
//-----------------------------------------------------------------------------
// Blank_Line - This function identifies blank lines

int Blank_Line( char buffer[] )
	{
	for (int i=0; i<strlen(buffer); i++)
		if (!isspace(buffer[i])) return 0;
	return 1;
	}
//-----------------------------------------------------------------------------
// Get_Token - This function gets a token from the input line

void Get_Token( char *&buffer_ptr, char token[], int &token_type )
	{
	char *tokenPtr=token;
	token_type=P_UNKNOWN;

	while (isspace(*buffer_ptr))
		buffer_ptr++;

	if (strchr(":;#",*buffer_ptr)!=NULL)    // Symbol
		{
		*tokenPtr=*buffer_ptr;
		tokenPtr++;
		buffer_ptr++;
		token_type=P_SYMBOL;
		}           // Variable or Reserved Word
	else if (isalpha(*buffer_ptr) || *buffer_ptr=='}' || 
		 *buffer_ptr=='{' || *buffer_ptr=='_')
		{
		token_type=P_STRING;
		while (isalpha(*buffer_ptr) || isdigit(*buffer_ptr) ||
		       *buffer_ptr=='{' || *buffer_ptr=='}' || 
		       *buffer_ptr=='_')
			{
			*tokenPtr=*buffer_ptr;
			tokenPtr++;
			buffer_ptr++;
			}
		}
	else if (isdigit(*buffer_ptr))    // number
		{
		token_type=P_NUMBER;
		while (isdigit(*buffer_ptr) || *buffer_ptr=='.')
			{
			*tokenPtr=*buffer_ptr;
			tokenPtr++;
			buffer_ptr++;
			}
		}
	else if (*buffer_ptr=='(')     // expression
		{
		int paren_count=1;
		int qcount=0;
		token_type=P_EXPRESS;
		buffer_ptr++;
		while (*buffer_ptr!='\0')
			{
			if (*buffer_ptr=='\"') qcount=!qcount;
			else if (qcount!=1 && *buffer_ptr=='(') paren_count++;
			else if (qcount!=1 && *buffer_ptr==')')
				{
				paren_count--;
				if (paren_count==0) break;
				}
			*tokenPtr=*buffer_ptr;
			tokenPtr++;
			buffer_ptr++;
			}
		if (*buffer_ptr!=')')
			Error("Missing Parenthesis in expression");
		}
	else if (*buffer_ptr=='[')     // subscript
		{
		int bracket_count=1;
		token_type=P_SUBSCRIPT;
		buffer_ptr++;
		while (*buffer_ptr!='\0')
			{
			if (*buffer_ptr=='[') bracket_count++;
			else if (*buffer_ptr==']') 
				{
				bracket_count--;
				if (bracket_count==0) break;
				}
			*tokenPtr=*buffer_ptr;
			tokenPtr++;
			buffer_ptr++;
			}
		if (*buffer_ptr!=']')
			Error("Missing Bracket in Subscript");
		
		}
	else
		Error("Unable to find token");

	*tokenPtr='\0';
	}
//-----------------------------------------------------------------------------
// Error - This function performs error handling

void Error( char s1[] )
	{
	cout << "Error - Line " << curr_line << "\n";
	cout << s1 << "\n";
	exit(0);
	}
//-----------------------------------------------------------------------------
// is_reserved_word - This function determines is a string is a reserved 
//		      word or not.  It is not case sensitive.

int is_reserved_word( char s[] )
	{
	int size=strlen(s);
	int flag;
	int i,k;
	for (i=0; i<reserved_word_count; i++)
	     {
	     if (size!=strlen(reserved_word[i])) continue;
  	     flag=0;
	     for (k=0; k<size; k++)
		if (toupper(s[k])!=toupper(reserved_word[i][k]))
			{
			flag=1;   break;
			}
	     if (flag==0) return i+1;
	     }
	return 0;
	}
//-----------------------------------------------------------------------------
// Prepare_Parameters - This function prepares input for a param_instruction
//			class instruction

void Prepare_Parameters( char *&buff_ptr, char token[] )
	{
	int token_type;
	Get_Token(buff_ptr,token,token_type);
	if (token_type!=P_EXPRESS)
		Error("No Parameters for Instruction");
	}






