//----------------------------------------------------------------------
// Program: express.C  -  Expression Class
//
// Purpose: This class will perform validation and evaluation of an
//          integer or boolean expression.  This class also handles
//          relational operators (==,>,>=,<,<=,!=) and unary 
//	    operators (+,-).
//
// Name   : Joey Rogers
// Date   : June 1, 1993
//----------------------------------------------------------------------
// express.C

#include"express.h"

//-------------------------------------------------------------------------
// This constructor copies the input string into the class' private storage

Expression::Expression( char exp[] )
	{
	if (exp==NULL) exp_string=NULL;
	else
		exp_string=strdup(exp);
	}
//-------------------------------------------------------------------------
// This function is used for both the validation and evaluation functions.
// When it is used for validation, an error code is set according to the
// error causing the problem.  If the function is used for evaluation, the
// expression is assumed to be valid and the value is returned.

double Expression::Evaluate( State &s, int &code )
	{
	double result;

	code=VALID;
	open_parentheses=0;
	if (exp_string==NULL)
		{
		code=EMP_EXPRESS;
		return 0;
		}

	EquationPtr=exp_string;
	Get_Token();

	if (*CurrTokenText)
		result=Relation(s,code);
	else
		{
		result=0.0;
		code=EMP_EXPRESS;
		}

	if (open_parentheses!=0) code=UNMATCHED_PAR;
	return result;
	}
//----------------------------------------------------------------
// This function gets a token from the input string

void Expression::Get_Token( void )
	{
	char *tokenPtr = CurrTokenText;
	CurrTokenType= TT_UNKNOWN;

	while (isspace(*EquationPtr))
		EquationPtr++;

	if (NULL!=strchr("+-%*/()&|<>=!",*EquationPtr))
		{
		CurrTokenType=TT_DELIM;
		*tokenPtr = *EquationPtr;
		tokenPtr++;
		EquationPtr++;
		switch (CurrTokenText[0])
			{
			case '!':
			case '=':
			case '>':
			case '<': if (*EquationPtr=='=')
				     {
				     *tokenPtr = *EquationPtr;
				     tokenPtr++;
				     EquationPtr++;
				     }

			case '&': if (*EquationPtr=='&')
				     {
				     *tokenPtr = *EquationPtr;
				     tokenPtr++;
				     EquationPtr++;
				     }

			case '|': if (*EquationPtr=='|')
				     {
				     *tokenPtr = *EquationPtr;
				     tokenPtr++;
				     EquationPtr++;
				     }
			}
		}

	else if (isdigit(*EquationPtr))
		{
		CurrTokenType=TT_NUMBER;
		while (isdigit(*EquationPtr) || *EquationPtr=='.')
			{
			*tokenPtr = *EquationPtr;
			tokenPtr++;
			EquationPtr++;
			}
		}

	else if (isalpha(*EquationPtr) || *EquationPtr=='_')
		{
		CurrTokenType=TT_VARIABLE;
		while (isalpha(*EquationPtr) || isdigit(*EquationPtr) ||
			*EquationPtr=='_')
			{
			*tokenPtr=*EquationPtr;
			tokenPtr++;
			EquationPtr++;
			}
		if (*EquationPtr=='[')
			{
			CurrTokenType=TT_SUB_VARIABLE;
			sub_count=0;
			int bracket=1;
			*EquationPtr++;
			sub_start=EquationPtr;
			while (*EquationPtr!='\0')
				{
				if (*EquationPtr=='[') bracket++;
				if (*EquationPtr==']') 
					{
					bracket--;
					if (bracket==0) break;
					}
				EquationPtr++;
				sub_count++;
				}

			if (*EquationPtr==']') EquationPtr++;
			}		
		}
	*tokenPtr='\0';
	if (CurrTokenText[0]==')') open_parentheses--;
	else if (CurrTokenText[0]=='(') open_parentheses++;
	}
//----------------------------------------------------------------
// This function performs relation operation on the expression

double Expression::Relation(State &s, int &code)
	{
	double result = AndOrOr(s,code);
	char oper=CurrTokenText[0];
	char oper1=CurrTokenText[1];

	while (oper=='>' || oper=='<' || oper=='=' || oper=='!')
		{
		Get_Token();
		double value = AndOrOr(s,code);

		if (oper=='>' && oper1=='\0')      // Perform correct operation
			result=result > value;
		else if (oper=='>' && oper1=='=')
			result=result >= value;
		else if (oper=='<' && oper1=='\0')
			result=result < value;
		else if (oper=='<' && oper1=='=')
			result=result <= value;
		else if (oper=='=' && oper1=='=')
			result=result == value;
		else if (oper=='!' && oper1=='=')
			result=result != value;
		else
			{
			code=BAD_OPER;
			result=0;
			}

		oper=CurrTokenText[0];
		oper1=CurrTokenText[1];
		}
	return result;
	}
//----------------------------------------------------------------
// This function evaluates plus or minus operations

double Expression::PlusOrMinus(State &s, int &code)
	{
	double value;
	int value_type;
	double result = MultOrDiv(s,code);
	char oper=CurrTokenText[0];

	while ((oper=='+') || (oper=='-') || (oper=='%'))
		{
		Get_Token();
		value = MultOrDiv(s,code);

		if (oper=='+')
			result+=value;
		else if (oper=='-')
			result-=value;
		else result=(int)result%(int)value;   // modulus
		oper=CurrTokenText[0];
		}
	return result;
	}
//----------------------------------------------------------------
// This function evaluates 'and' and 'or' operations

double Expression::AndOrOr(State &s,int &code)
	{
	double value;
	int value_type;
	double result = PlusOrMinus(s,code);
	char oper=CurrTokenText[0];

	while ((oper=='&') || (oper=='|'))
		{
		Get_Token();
		value = PlusOrMinus(s,code);

		if (oper=='&')
			result=result && value;
		else
			result=result || value;

		oper=CurrTokenText[0];
		}
	return result;
	}
//----------------------------------------------------------------
// This function evaluates mulitply and divide operations

double Expression::MultOrDiv(State &s, int &code )
	{
	int value_type;
	double value;
	double result = Unary(s,code);
	char oper=CurrTokenText[0];

	while ((oper=='*') || (oper=='/'))
		{
		Get_Token();
		value = Unary(s,code);

		if (oper=='*')
			result*=value;
		else
			if (value==0)
				result=0;
			else
				result/=value;

		oper=CurrTokenText[0];
		}
	return result;
	}
//----------------------------------------------------------------
// This function evaluates Unary operations

double Expression::Unary(State &s, int &code )
	{
	int value_type;
	double value;
	double result;
	char oper=CurrTokenText[0];

	if (oper=='+' || oper=='-')
		Get_Token();

	value = Parenthesis(s,code);

	if (oper=='-')
		result=-value;
	else
		result=value;
	return result;
	}
//----------------------------------------------------------------
// This function handles parenthisized expressions

double Expression::Parenthesis(State &s, int &code)
	{
	double result;

	if (CurrTokenType==TT_DELIM && CurrTokenText[0]=='(')
		{
		Get_Token();
		result=Relation(s,code);

		if (CurrTokenText[0]!=')')
			{
			code=MISSING_PAR;
			return 0;
			}
		}
	else if (CurrTokenType==TT_NUMBER)
		{
		result = atof(CurrTokenText);
		}
	else if (CurrTokenType==TT_VARIABLE)
		{
		if (s.Get_Value(CurrTokenText,result,0)==0)
			code=UNDEC_VAR;
		}
	else if (CurrTokenType==TT_SUB_VARIABLE)
		{
		*(sub_start+sub_count)='\0';
		Expression sub(sub_start);
		*(sub_start+sub_count)=']';
		int code1;
		int index=(int)sub.Evaluate(s,code1);
		if (code1!=VALID) code=code1;
		int var_id=s.Get_Variable_ID(CurrTokenText);
		if (var_id==NOT_FOUND)
			code=UNDEC_VAR;
		else
			{
			if (index<0 || index>=s.Get_Size(var_id))
			   code=SUB_OUT_RANGE;
			else
			   s.Get_Value(CurrTokenText,result,index);
			}
		}
	else
		{
		code=SYNTAX_ERROR;
		result=0;
		}
	Get_Token();

	return result;
	}
//----------------------------------------------------------------
void Expression_Error( int code, int id, Expression *exp )
	{
	cout << "Line " << id << " - Expression: " << 
		exp->Get_Expression() << ":\n";
	switch (code)
	   {
	   case EMP_EXPRESS:
		cout << "Empty Expression";
		break;
	   case UNDEC_VAR:
		cout << "Undelcared Variable in Expresssion";
		break;
	   case TYPE_MIS:
		cout << "Type Mismatch";
		break;
	   case SYNTAX_ERROR:
		cout << "Syntax error in Expression";
		break;
	   case MISSING_PAR:
		cout << "Missing Parenthesis in Expression";
		break;
	   case UNMATCHED_PAR:
		cout << "Unmatched Parenthesis in Expression";
		break;
	   case BAD_OPER:
		cout << "Unknown Operator";
		break;
	   default:
		cout << "Unknown Error in Expression";
	   }
	}


/*

void main( void )
	{
	int code;
	State s;
	int i;

	s.Declare_Variable("i",0.8684);
	s.Declare_Variable("j",0.7545);

	Expression c("j<i");

	cout << "\n";
	cout << "Result:" << c.Evaluate(s,code) << "  code: " <<
		code << "\n\n" ;
	}



*/












