//----------------------------------------------------------------------
// Program: data.C  -  Data Class
//
// Author : Joey Rogers
// Date   : July 30, 1993
//
// Purpose: The data class maintains a linked list of data elements.  Each data
//	    element has an input array, and output array, and an 
//	    id number.  Various statistical and retrieval operations are
//	    supported by this class.
//
//----------------------------------------------------------------------

#include"data.h"

//----------------------------------------------------------------------------
// This function finds the high and low values of each input and 
// output component in the data list

void Data_Class::Find_High_Low( double high_in[], double low_in[],
			   double high_out[], double low_out[]  )
	{
	for (int i=0; i<in_size; i++)    // Initialize high/low values to first
		{
		high_in[i]=head->in[i];
		low_in[i]=head->in[i];
		}

	for (i=0; i<out_size; i++)
		{
		high_out[i]=head->out[i];
		low_out[i]=head->out[i];
		}
					       // Search through rest
	for (DATA *temp=head->next; temp!=NULL; temp=temp->next)
		{
		for (i=0; i<in_size; i++)
			{
			if (temp->in[i]>high_in[i]) high_in[i]=temp->in[i];
			if (temp->in[i]<low_in[i])  low_in[i]=temp->in[i];
			}
		for (i=0; i<out_size; i++)
			{
			if (temp->out[i]>high_out[i]) high_out[i]=temp->out[i];
			if (temp->out[i]<low_out[i])  low_out[i]=temp->out[i];
			}
		}
	}
//----------------------------------------------------------------------------
// This function save the data to a file

void Data_Class::Save_Data( char filename[] )
	{
	FILE *out=fopen(filename,"w");
	if (out==NULL)
		{
		cout << "Unable to open file for output: " << filename <<
			filename << "\n";
		exit(0);
		}
	
	fprintf(out,"Format: ! ");
	for (int i=0; i<in_size; i++)
		fprintf(out,"I ");
	for (i=0; i<out_size; i++)
		fprintf(out,"O ");
	fprintf(out,"\n");
	DATA *temp=head;
	while (temp!=NULL)
		{
		fprintf(out,"# %d  ",temp->id); 	
		for (i=0; i<in_size; i++)
        	    fprintf(out,"%lf ",temp->in[i]);
		for (i=0; i<out_size; i++)
		    fprintf(out,"%lf ",temp->out[i]);
		fprintf(out,"\n"); 	
		temp=temp->next;
		}
	fprintf(out,"*\n");
	fclose(out);		
	}
//----------------------------------------------------------------------------
// This function creates a list and loads a data file into it

void Data_Class::Load_Data( char filename[], int &data_count,
			     int &in, int &out )
	{
	int i;
	int Format[MAX_COLUMNS];
	int Format_Count;
	int Ipos,Opos;
	double temp_double;
	char ch;
	int id;
	DATA *temp;
	int output_nodes;

	FILE *infile=fopen(filename,"r");   // Open File 
	if (infile==NULL)
		{
		cout << "Unable to open file: " << filename << "\n";
		exit(0);
		}

	fscanf(infile,"Format:");      // Read format line 

	Format_Count=0;
	ch=0;  in=0;  out=0;

	while (ch!='\n')
		{
		fscanf(infile,"%c",&ch);
		if (ch!=' ')
			{
			Format[Format_Count]=ch;
			Format_Count++;
			if (ch=='I') in++;
			else if (ch=='O') out++;
			}
		}

	data_count=0;
	in_size=in;     out_size=out;   

	fscanf(infile,"%c",&ch);
	while (!feof(infile) && ch=='#')
		{
		temp=Create_Data();
		id=0;  Ipos=0;  Opos=0;
		for (i=0; i<Format_Count; i++)
			{
			switch(Format[i])
				{
				case'!':             // Identification Num
					fscanf(infile,"%d ",&id);
					break;
				case'*':	     // Omit
					fscanf(infile,"%lf ",&temp_double);
					break;
				case'i':
				case'I':             // Input
					fscanf(infile,"%lf ",&temp_double);
					temp->in[Ipos]= temp_double;
					Ipos++;
					break;
				case'o':
				case'O':             // Output
					fscanf(infile,"%lf ",&temp_double);
					temp->out[Opos]= temp_double;
					Opos++;
					break;
				}
			}

		temp->id=id;
		Add_Data_After_Curr(temp);
		fscanf(infile,"\n");
		fscanf(infile,"%c",&ch);
		}
	fclose(infile);

	data_count=count;
	}
//----------------------------------------------------------------------------
// This function normalizes each input and output component between the
// range of 0.0 and 1.0

void Data_Class::Normalize_Data( double h_i[], double l_i[],
		  	        double h_o[], double l_o[] )
	{
	if (high_in==NULL)
		{
		delete high_in;  delete low_in;
		delete high_out; delete low_out;
		}

	high_in=new double[in_size];
	low_in=new double[in_size];
	high_out=new double[out_size];
	low_out=new double[out_size];

	for (int i=0; i<in_size; i++)
		{
		high_in[i]=h_i[i];
		low_in[i]=l_i[i];
		}
	for (i=0; i<out_size; i++)
		{
		high_out[i]=h_o[i];
		low_out[i]=l_o[i];
		}

	for (DATA *temp=head; temp!=NULL; temp=temp->next)
		{
		for (i=0; i<in_size; i++)
			if (high_in[i]==low_in[i])
				temp->in[i]=1;
			else
				temp->in[i]=(temp->in[i]-low_in[i])/
				    (high_in[i]-low_in[i]);

		for (i=0; i<out_size; i++)
			if (high_out[i]==low_out[i])
				temp->out[i]=1;
			else
				temp->out[i]=(temp->out[i]-low_out[i])/
				    (high_out[i]-low_out[i]);
		}
	}

//----------------------------------------------------------------------------
// This function reverses the effects of Normalize_Data

void Data_Class::Denormalize_Data( void )
	{
	if (high_in==NULL) return;

	int i;
	for (DATA *temp=head; temp!=NULL; temp=temp->next)
		{
		for (i=0; i<in_size; i++)
			temp->in[i]=temp->in[i]*(high_in[i]-low_in[i])+
				    low_in[i];

		for (i=0; i<out_size; i++)
			temp->out[i]=temp->out[i]*(high_out[i]-low_out[i])+
				low_out[i];		
		}
	}
//-----------------------------------------------------------------------------
// This function creates a data item with the predetermined input/output sizes
	
DATA *Data_Class::Create_Data( void )
     {
     DATA *temp=new DATA;
     if (temp==NULL)
	{
	cout << "Cannot allocate Data Node.\n";
	exit(0);
	}

     temp->id=0;
     temp->in=new double[in_size];
     temp->out=new double[out_size];

     if (temp->in==NULL || temp->out==NULL)
	{
	cout << "Could Not Allocate Data Node\n";
	exit(0);
	}

     temp->next=NULL;

     return(temp);
     }
//-----------------------------------------------------------------------------
// Default Constructor for Data Class

Data_Class::Data_Class( void )
	{
	head=NULL;  tail=NULL;  curr=NULL; prev=NULL;
	high_in=NULL; low_in=NULL; high_out=NULL; low_out=NULL;
	count=0;
	in_size=0;  out_size=0;
	};
//-----------------------------------------------------------------------------
// Constructor for Data Class - User parameterized size

Data_Class::Data_Class( int input_size, int output_size )
	{
	head=NULL;   tail=NULL;  curr=NULL;  prev=NULL;
	high_in=NULL; low_in=NULL; high_out=NULL; low_out=NULL;
	count=0;
	in_size=input_size;  out_size=output_size;
	};
//-----------------------------------------------------------------------------
// Destructor for Data Class

Data_Class::~Data_Class( void )
	{
	DATA *data=head;
	DATA *temp;
	while (data!=NULL)
	    {
	    temp=data;
	    delete temp->in;  
	    delete temp->out;
	    data=data->next;
	    delete temp;
	    }
	};
//-----------------------------------------------------------------------------
// This function converts data into a DATA element

DATA *Data_Class::Convert_to_DATA(int id, double in[], double out[] )
	{
	DATA *data=Create_Data();
	for (int i=0; i<in_size; i++) data->in[i]=in[i];
	for (i=0; i<out_size; i++) data->out[i]=out[i];
	data->id=id;
	return(data);
	}

//-----------------------------------------------------------------------------
// This function adds a data element before the current element

void Data_Class::Add_Data_Before_Curr( int id, double in[], double out[])
	{
	DATA *data=Convert_to_DATA(id,in,out);
	Add_Data_Before_Curr(data);
	}
	
//-----------------------------------------------------------------------------
// This function adds a data element before the current element

void Data_Class::Add_Data_Before_Curr( DATA *data )
	{
	if (head==NULL)              // if list is empty
		{
		data->next=NULL;
		head=data;
		tail=data;
		}
	else if (curr==NULL)	     // if adding to end of list
		{
		tail->next=data;
		data->next=NULL;
		tail=data;
		}
	else if (curr==head)         // if adding to head of list
		{
		data->next=head;
		head=data;
		}
	else				// if adding to middle of list
		{
		prev->next=data;
		data->next=curr;
		}
		
	curr=data;
	DATA *temp=head;
	prev=NULL;             // Reset prev pointer
	while (temp!=curr) 
		{
		prev=temp;
		temp=temp->next;
		}
	count++;
	}
//-----------------------------------------------------------------------------
// This function adds a data element after the current element

void Data_Class::Add_Data_After_Curr( int id, double in[], double out[])
	{
	DATA *data=Convert_to_DATA(id,in,out);
	Add_Data_After_Curr(data);
	}
//-----------------------------------------------------------------------------
// This function adds a data element after the current element

void Data_Class::Add_Data_After_Curr( DATA *data)
	{
	if (head==NULL)              // if list is empty
		{
		data->next=NULL;
		head=data;
		tail=data;
		}
	else if (curr==NULL)	     // if adding to begining of list
		{
		data->next=head;
		head=data;
		}
	else 			     // if adding to middle of list	
		{
		data->next=curr->next;
		curr->next=data;
		if (tail==curr)      // if adding to end of list
			tail=data;
		}
	prev=curr;		
	curr=data;
	count++;
	}

//-----------------------------------------------------------------------------
// This function adds a data element after the current element

void Data_Class::Delete_Curr_Data( void )
	{
	DATA *temp;
	if (head==NULL || curr==NULL) return;   // if empty list or curr=NULL
	else if (curr==head)                    // if curr is begining of list
		{
		temp=head;
		head=head->next;
		if (head==NULL)	tail=NULL;	
		curr=head;
		}
	else 
		{
		prev->next=curr->next;
		temp=curr;
		curr=curr->next;

		if (curr==NULL)      // if last element is deleted
			{
			tail=head;
			while (tail->next!=NULL) 
				{
				prev=tail;
				tail=tail->next;
				}
			curr=tail;
			}
		}
	delete temp;
	count--;
	}
//-----------------------------------------------------------------------------

// This function retrieves the current data element in the list

void Data_Class::Get_Current_Data( int &id, double in[], double out[] )
	{
	if (curr==NULL) return;
	for (int i=0; i<in_size; i++)
		in[i]=curr->in[i];
	for (i=0; i<out_size; i++)
		out[i]=curr->out[i];
	id=curr->id;
	};
//-----------------------------------------------------------------------------
// This function prints the contents of the list

void Data_Class::Print_Data( void )
	{
	int i,j;

	for (DATA *temp=head; temp!=NULL; temp=temp->next)
		{
		cout << "ID: " << temp->id << "  Inputs: ";
		for (i=0; i<in_size; i++)
			cout << temp->in[i] << " ";
		cout << "  Outputs: ";
		for (i=0; i<out_size; i++)
			cout << temp->out[i] << " ";
		cout << "\n" << flush;
		}
	}
//-----------------------------------------------------------------------------
// This function retrieves the high/low information stored for the list

void Data_Class::Get_High_Low( double h_in[], double l_in[], 
			       double h_out[], double l_out[])
	{
	for (int i=0; i<in_size; i++) 
		{
		if (high_in!=NULL)
			{
			h_in[i]=high_in[i];
			l_in[i]=low_in[i];
			}
		else
			{
			h_in[i]=1.0;		// default values
			l_in[i]=0.0;
			}
		}

	for (i=0; i<out_size; i++)
		{
		if (high_in!=NULL)
			{
			h_out[i]=high_out[i];
			l_out[i]=low_out[i];
			}
		else
			{
			h_out[i]=1.0;		// default values
			l_out[i]=0.0;
			}
		}
	}







