//----------------------------------------------------------------------
// Program: coreinst.h  -  Core Instruction Class Header
//
// Author : Joey Rogers
// Date   : July 30, 1993
//
// Purpose: This is the header for the Core Instruction Classes.  The 
//	    core instruction are the C-like instruction used by NeuDL.	 
//----------------------------------------------------------------------

#ifndef COREINST
#define COREINST

#include"baseinst.h"

//============================================================================
// Assignment Instruction Class -
// This class allow variables to be assigned expressions

class Assignment_Inst : public Instruction
	{
	public:
		Assignment_Inst(int inst_id, char target_str[],
			char target_sub[], char source_str[]);
		void Execute( State &s);
		void Prepare( State &s );
                void Translate( State &s );
	private:
		char *target;
		Expression *source;
		Expression *sub;
		int var_id;
	};
//============================================================================
// Compound Instruction Class -
// This class allows instructions of any kind to be bundled together
// into one instruction. When this instruction is executed, all of the
// instructions in its make up are executed.  Instructions are stored in
// a linked list, so the maximum number of instructions per compound
// instruction is limited only by memory.

class Compound_Inst : public Instruction
	{
	public:
		Compound_Inst(int inst_id, int mode_type);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );
		void Add_Instruction( Instruction *inst );
	private:
		struct inst_list                // Instruction List Node
			{
			Instruction *inst;
			inst_list *next;
			};

		inst_list *head, *tail;
		int mode;   			// Execution mode
	};

//============================================================================
// Skip Instruction Class -
// This class provides the skip instruction, which does nothing when executed

class Skip_Inst : public Instruction
	{
	public:
		Skip_Inst(int inst_id):Instruction( inst_id ){};
		void Translate( State &s )
		       { cout << "// Skip - ID: " << id << "\n   ;\n";};
	};

//============================================================================
// Dump Instruction Class - Dumps program state to screen

class Dump_Inst : public Instruction
	{
	public:
		Dump_Inst(int inst_id):Instruction( inst_id ){};
		void Execute( State &s) {s.Dump();};
	};

//============================================================================
// Beep Instruction Class - makes a sound

class Beep_Inst : public Instruction
	{
	public:
		Beep_Inst(int inst_id):Instruction( inst_id ){};
		void Execute( State &s) { cout << "\a";};
		void Translate( State &s )
		       { cout << "// Beep - ID: " << id <<
			 "   cout << \"\\a\";\n";};
	};

//============================================================================
// Conditional Instruction Class -
// This class provides the conditional instruction.  It executes one of
// two instructions when executed depending on the evaluation of the
// test expression.

class Conditional_Inst : public Instruction
	{
	public:
		Conditional_Inst(int inst_id,char test[],
				 Instruction *thenbranch,
				 Instruction *elsebranch);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );

	private:
		Expression *test_exp;          // Test condition expression
		Instruction *then_inst;	       // Then branch instruction
		Instruction *else_inst;	       // Else branch instruction
	};
//============================================================================
// While Loop Instruction Class -
// It is a pretest loop and will execute the body of the loop
// until the test condition is false.

class While_Inst : public Instruction
	{
	public:
		While_Inst(int inst_id,char test[], Instruction *loop_body);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );
	private:
		Expression *test_exp;	 // Test condition expression
		Instruction *body;	 // Loop body instruction
	};
//============================================================================
// For Instruction Class -
// This class is the for loop instruction.  It is a pretest loop and will
// execute the body of the loop until the test condition is false.
// An incrementer instruction is executed at the end of each loop.

class For_Inst : public Instruction
	{
	public:
		For_Inst(int inst_id, Instruction *init, char test[], 
		     Instruction *increm, Instruction *loop_body);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );
	private:
		Instruction *initializer;   // Inst executed before loop begins
		Instruction *incrementer;   // Inst executed each iteration
		Expression *test_exp;	    // Test Condition expression
		Instruction *body;	    // Loop body instruction
	};
//============================================================================
// Print Instruction Class -
// This class provides an instruction to print out integer exps and strings

class Print_Inst : public Instruction
	{
	public:
		Print_Inst(int inst_id, char exp_str[]);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );
	private:
		struct print_list   // Print List Node
			{
			char *str;
			int type;
			print_list *next;
			int code;
			};

		char *string;
		print_list *head,*tail;  // Pointers to list of things to print
	};

//============================================================================
// Read Integer Instruction Class -
// This class provides an instruction to read in a value

class Read_Int_Inst : public Instruction
	{
	public:
		Read_Int_Inst(int inst_id,char target_str[],char target_sub[]);
		void Execute( State &s);
		void Prepare( State &s );
		void Translate( State &s );
	private:
		char *name;		// Variable name to read
		Expression *sub;        // Variables subscript expression
		int var_id;		// ID of Variable Name
	};

//============================================================================
// NewLine Instruction Class -
// This class provides an instruction to output a newline on the screen

class NewLine_Inst : public Instruction
	{
	public:
		NewLine_Inst(int inst_id):Instruction(inst_id){};
		void Execute( State &s) { cout << "\n";};
		void Translate( State &s )
			{ cout << "// Newline - ID: " << id << "\n";
			cout << "   cout << \"\\n\";\n"; };
	};
//============================================================================
// Declare Instruction Class -
// This class provides an instruction to declare a variable

class Declare_Inst : public Instruction
	{
	private:
		Expression *size;     // Size of Variable 
		char *name;	      // Variable Name
		int data_type;	      // Type of Data
	public:
		Declare_Inst(int inst_id, char *var_name, int type, 
			     char *size_exp=NULL);
		void Prepare( State &s );
		void Execute( State &s);
                void Translate( State &s );
	};
//============================================================================
// Copy Array Instruction Class - Copies one array to another

class Copy_Array_Inst : public Param_Instruction
	{
	public:
		Copy_Array_Inst(int inst_id, char parameters[] ):
		    Param_Instruction(inst_id,parameters,2){};
		void Execute( State &s);
		void Prepare( State &s);
		void Translate( State &s );
	private:
		int source,target;   // id numbers for source and dest arrays
	};



#endif








