//----------------------------------------------------------------------
// Program: coreinst.C  -  Core Instruction Classes
//
// Name   : Joey Rogers
// Date   : July 30, 1993
//
// Purpose: This file contains all of the core instrution classes.  The
//	    core instruction are the C-like instruction used by NeuDL.
//----------------------------------------------------------------------

#include"coreinst.h"

//============================================================================
// Constructor for Assignment Instruction Class

Assignment_Inst::Assignment_Inst( int inst_id, char target_str[],
	 char target_sub[], char source_str[]):Instruction(inst_id)
	{
	target=strdup(target_str);
	source = new Expression(source_str);
	if (strcmp("0",target_sub)==0)  // Optimized for non-subscripted vars
		sub=NULL;
	else
		sub=new Expression(target_sub);
	var_id=-1;
	}
//--------------------------------------------------------------------------
// This function prepares an assignment instruction

void Assignment_Inst::Prepare( State &s )
	{
	var_id=s.Get_Variable_ID(target);

	if (var_id==NOT_FOUND)
	   {
	   Error(id,"Target Variable in Assignment is Undeclared: ",target);
	   exit(0);
	   }
	}
//--------------------------------------------------------------------------
// This function executes an assignment instruction.

void Assignment_Inst::Execute( State &s )
	{
	int code;
	int pos;

	if (sub==NULL)            // if no subscript, pos is zero
		pos=0;
	else
		{
		pos=(int)sub->Evaluate(s,code);  // Evaluate subscript
		if (code!=VALID)
			{
			Error(id,"Run Time Error: expression error\n\n");
			Expression_Error(code,id,sub);
			exit(0);
			}
		}

	if (s.Set_Value(var_id,source->Evaluate(s,code),pos)==OUT_RANGE)
		{
		Error(id,
		  "Run Time Error: Subscript Out of Range in Assignment\n\n");
		exit(0);
		}

	if (code!=VALID)
		{
		Error(id,"Run Time Error: expression error\n\n");
		exit(0);
		}
	}
//--------------------------------------------------------------------------
// This function translates an assignment instruction

void Assignment_Inst::Translate( State &s )
	{
	cout << "// Assignment - ID: " << id <<"\n";
	cout << "   " << target;
	if (s.Get_Structure(var_id)==ARRAY) 
		{
		cout << "[(int)(";
		Convert_Expression(sub); 
		cout << ")]";
		}
	cout << "=";
	Convert_Expression(source);
	cout << ";\n";
	}
//============================================================================
// Constructor for Compound Instruction

Compound_Inst::Compound_Inst(int inst_id,int mode_type):Instruction(inst_id)
	{
	mode=mode_type;    // EXECUTE or DEBUG
	head=NULL;
	tail=NULL;
	}
//============================================================================
// This function allow instructions to be added to the compound instruction

void Compound_Inst::Add_Instruction( Instruction *inst )
	{
	inst_list *temp=new inst_list;
	temp->inst=inst;
	temp->next=NULL;

	if (tail!=NULL)
		tail->next=temp;
	else
		head=temp;

	tail=temp;
	}
//--------------------------------------------------------------------------
// This function executes the compound instruction

void Compound_Inst::Execute( State &s )
	{
	for (inst_list *temp=head; temp!=NULL; temp=temp->next)
		{
		if (mode==DEBUG) temp->inst->Inst_ID();
		temp->inst->Execute(s);
		}
	}
//--------------------------------------------------------------------------
// This function prepares the compound instruction

void Compound_Inst::Prepare( State &s )
	{
	for (inst_list *temp=head; temp!=NULL; temp=temp->next)
		{
		if (mode==DEBUG) temp->inst->Inst_ID();
		temp->inst->Prepare(s);
		}
	}
//--------------------------------------------------------------------------
// This function translates the compound instruction

void Compound_Inst::Translate( State &s )
	{
	cout << "// Compound - ID: " << id << "\n   {\n";
	for (inst_list *temp=head; temp!=NULL; temp=temp->next)
		{
		temp->inst->Translate(s);
		cout << flush;
		}
	cout << "   }     // Compound - ID: " << id << "\n";
	}
//============================================================================
// This is the constructor for the conditional instruction class

Conditional_Inst::Conditional_Inst(int inst_id, char test[],
			Instruction *thenbranch,Instruction *elsebranch)
			:Instruction(inst_id)
	{
	test_exp=new Expression(test);
	then_inst=thenbranch;
	else_inst=elsebranch;
	}
//--------------------------------------------------------------------------
// This function prepares a conditional instruction

void Conditional_Inst::Prepare( State &s )
	{
	then_inst->Prepare(s);
	else_inst->Prepare(s);
	}
//--------------------------------------------------------------------------
// This function executes a conditional instruction.  The then_instruction is
// executed if the test condition is true and the else_instruction is
// executed if the test condition is not true.

void Conditional_Inst::Execute( State &s )
	{
	int code;
	int cond_val;

	cond_val=test_exp->Evaluate(s,code);

	if (code!=VALID)   
		{
		Error(id,"Run Time Error: conditional\n\n");
		Expression_Error(code,id,test_exp);
		cout << "Code: " << code << "\n";
		s.Dump();
		exit(0);
		}

	if (cond_val!=0)  		    // if condition is true 
		then_inst->Execute(s);
	else                                // if condition is false
		else_inst->Execute(s);
	}
//--------------------------------------------------------------------------
// This function translates the Conditional statement

void Conditional_Inst::Translate( State &s )
	{
	cout << "// Conditional - ID: " << id << "\n" <<"   if (";
	Convert_Expression(test_exp);
	cout << ") {\n";
	then_inst->Translate(s);
	cout << "   }   else  {\n";
	else_inst->Translate(s);
	cout << "   }\n";
	}
//============================================================================
// This is the constructor for the while loop instruction

While_Inst::While_Inst(int inst_id, char test[], Instruction *loop_body):
	    Instruction(inst_id)
	{
	test_exp=new Expression(test);
	body=loop_body;
	}
//--------------------------------------------------------------------------
// This function prepares the while loop instruction

void While_Inst::Prepare( State &s )
	{
	body->Prepare(s);
	}
//--------------------------------------------------------------------------
// This function executes the while loop instruction

void While_Inst::Execute( State &s )
	{
	int code;
	int cond_val;

	cond_val=test_exp->Evaluate(s,code);

	while (code==VALID && cond_val!=0)  // while cond is true and valid
		{
		body->Execute(s);
		cond_val=test_exp->Evaluate(s,code);
		}

	if (code!=VALID)
		{
		Error(id,"Run Time Error: while loop\n");
		exit(0);
		}
	}
//--------------------------------------------------------------------------
// This function translates the While instruction

void While_Inst::Translate( State &s )
	{
	cout << "// While Loop - ID: " << id << "\n" <<"   while (";
	Convert_Expression(test_exp);
	cout << ") {\n";
	body->Translate(s);
	cout << "   }\n";
	}
//============================================================================
// This is the constructor for the for loop instruction

For_Inst::For_Inst(int inst_id, Instruction *init, char test[],
		     Instruction *increm, Instruction *loop_body):
		     Instruction(inst_id)
	{
	initializer=init;
	incrementer=increm;
	test_exp=new Expression(test);
	body=loop_body;
        }
//--------------------------------------------------------------------------
// This function prepares the for loop instruction

void For_Inst::Prepare( State &s )
	{
	initializer->Prepare(s);
	incrementer->Prepare(s);
	body->Prepare(s);
	}
//--------------------------------------------------------------------------
// This function executes the for loop instruction

void For_Inst::Execute( State &s )
	{
	int code;
	int cond_val;

	initializer->Execute(s);

	cond_val=test_exp->Evaluate(s,code);

	while (code==VALID && cond_val!=0)  // while expr is true and valid
		{
		body->Execute(s);
		incrementer->Execute(s);
		cond_val=test_exp->Evaluate(s,code);
		}
	if (code!=VALID)
		{
		Error(id,"Run Time Error: for loop\n");
		cout  << "Code: " << code;
		cout << "Expression: " << test_exp->Get_Expression() << "\n";
		exit(0);
		}
	}
//--------------------------------------------------------------------------
// This function translates the for instruction

void For_Inst::Translate( State &s )
	{
	cout << "// For Loop - ID: " << id << "\n";
	initializer->Translate(s);
	cout << "   while (";
	Convert_Expression(test_exp);
	cout << ")\n   {\n";
	body->Translate(s);
	incrementer->Translate(s);
	cout << "   }   // For Loop - ID: " << id << "\n";
	}
//============================================================================
// This is the constructor for the print instruction class

Print_Inst::Print_Inst(int inst_id, char exp_str[] ):Instruction(inst_id)
	{
	string=strdup(exp_str);
	head=NULL; tail=NULL;
	}
//--------------------------------------------------------------------------
// This function executes the print instruction

void Print_Inst::Execute( State &s )
	{
	int code;
	Expression *exp;

	for (print_list *temp=head; temp!=NULL; temp=temp->next)
		{
		if (temp->type==EXPRESSION_TYPE)
			{
			exp=new Expression(temp->str);
			cout << exp->Evaluate(s,code);
			
			if (code!=VALID)
			      {
			      Error(id,"Could not evaluate print expression");
			      exit(0);
			      }
			delete exp;
			}
		else cout <<temp->str;
		}
	cout << flush;
	}
//--------------------------------------------------------------------------
// This function translates the print instruction

void Print_Inst::Translate( State &s )
	{
	Expression *temp_exp;
	cout << "// Print - ID: " << id << "\n";
	for (print_list *temp=head; temp!=NULL; temp=temp->next)
		{
		cout << "   cout << ";
	if (temp->type==EXPRESSION_TYPE)
			{
			temp_exp=new Expression(temp->str);
			Convert_Expression(temp_exp);
			delete temp_exp;
			}
		else if (temp->code==NORMAL_STRING)
			  cout << "\"" << temp->str << "\"";
		     else cout << "argv[" << temp->code << "]";

		cout << ";\n";
		}
	}
//--------------------------------------------------------------------------
// This function prepares the print instruction

void Print_Inst::Prepare( State &s )
	{
	int code;
	char *ptr;
	print_list *ptemp;
	int type;
	char token[100];
	ptr=string;

	Get_Parameter_Token(id,ptr,token,type);
	while (type!=EMPTY)
		{
		ptemp=new print_list;
		ptemp->str=strdup(token);
		Convert_String(s,ptemp->str,ptemp->code);
		ptemp->next=NULL;
		ptemp->type=type;
		if (head==NULL)
			{
			head=ptemp;
			tail=ptemp;
			}
		else
			{
			tail->next=ptemp;
			tail=ptemp;
			}
		Get_Parameter_Token(id,ptr,token,type);
		}
	}
//============================================================================
// This is the constructor for the read instruction class

Read_Int_Inst::Read_Int_Inst(int inst_id, char target_str[],
			     char target_sub[] ):Instruction(inst_id)
	{
	name=strdup(target_str);
	if (strcmp("0",target_sub)==0)
		sub=NULL;
	else
		sub=new Expression(target_sub);
	}
//--------------------------------------------------------------------------
// This function executes the read integer instruction

void Read_Int_Inst::Execute( State &s )
	{
	double temp;
	int pos;
	int code;
	if (sub==NULL)
		pos=0;
	else
		{
		pos=(int)sub->Evaluate(s,code);
		if (code!=VALID)
			{
			Error(id,
			      "Run Time Error: expression error in Read\n\n");
			exit(0);
			}
		}

	cin >> temp;
	if (s.Set_Value(var_id,temp,pos)==OUT_RANGE)
		{
		Error(id,"Run Time Error: Subscript Out of Range in Read\n\n");
		exit(0);
		}
	}
//--------------------------------------------------------------------------
// This function prepares the read instruction

void Read_Int_Inst::Prepare( State &s )
	{
	var_id=s.Get_Variable_ID(name);
	delete name;
	if (var_id==NOT_FOUND)
		{
		cout << "Undeclared identifier in Read: " << name << "\n";
		exit(0);
		}
	}
//--------------------------------------------------------------------------
// This function translates the read instruction

void Read_Int_Inst::Translate( State &s )
	{
	cout << "// Read - ID: " << id << "\n" <<
		"   cin >> " << s.Get_Name(var_id);
	if (sub!=NULL)
		{
		cout << "[(int)(";
		Convert_Expression(sub);
		cout << ")]";
		}
	cout << ";\n";
	}
//============================================================================
// This is the constructor for the Declare Instruction class

Declare_Inst::Declare_Inst(int inst_id, char *var_name, int type, 
			   char *size_exp):Instruction(inst_id)
	{
	name=strdup(var_name);
	if (size_exp==NULL) size=NULL;
	else size=new Expression(size_exp);
	data_type=type;
	}
//----------------------------------------------------------------------------
// This function prepare the Declare instruction

void Declare_Inst::Prepare( State &s )
	{					 // declare of 0 size
	if (size==NULL)
	  s.Declare_Variable(name,data_type);
	else
	  s.Declare_Variable(name,data_type,size->Get_Expression());
	}
//----------------------------------------------------------------------------
// This function executes the declare instruction

void Declare_Inst::Execute( State &s )
	{
	int code;
	if (size==NULL)
	  {
	  s.Allocate_Variable(s.Get_Variable_ID(name),1);
	  }
	else
	  {
	  s.Allocate_Variable(s.Get_Variable_ID(name),size->Evaluate(s,code));

	  if (code!=VALID)
		{
		Expression_Error(id,code,size);
		exit(0);
		}
	  }
	}
//----------------------------------------------------------------------------
// This function translates the declare instruction

void Declare_Inst::Translate( State &s )
	{
	cout << "// Declare - ID: " << id << "\n";
	switch (s.Get_Type(s.Get_Variable_ID(name)))
	  {
	  case INT: cout << "    int ";
		    break;

	  case FLOAT: cout << "    double ";
		      break;
	  }

	if (s.Get_Structure(s.Get_Variable_ID(name))==ARRAY)
		{
		cout << "*" << name << "=new ";
	        switch (s.Get_Type(s.Get_Variable_ID(name)))
         	  {
	          case INT: cout << "int";
		            break;

            	  case FLOAT: cout << "double";
		            break;
	          }

		cout << "[(int)(";
		Convert_Expression(size);
		cout << ")];\n   for (___lcv=0; ___lcv<";
		Convert_Expression(size);
		cout << "; ___lcv++)\n" << "      " << name << "[___lcv]=0;\n";
		}
	else	
		cout << name << "=0;\n";
	}
//============================================================================
// This function will execute the copy array instruction

void Copy_Array_Inst::Execute( State &s )
	{
	int size=s.Get_Size(source);
	if (size!=s.Get_Size(target))
		{
		Error(id,"Run Time Error: Arrays must be of the same size");
		exit(0);
		}
	double value;
	for (int i=0; i<size; i++)
		{
		s.Get_Value(source,value,i);
		s.Set_Value(target,value,i);
		}
	}
//--------------------------------------------------------------------------
// This fucntion will prepare the copy array instruction

void Copy_Array_Inst::Prepare( State &s )
	{
	target=s.Get_Variable_ID(param[0]);
	source=s.Get_Variable_ID(param[1]);
	if (s.Get_Structure(target)!=ARRAY || s.Get_Structure(source)!=ARRAY
             || s.Get_Type(target)!=s.Get_Type(source))
		{
		Error(id,"Parameters must be arrays of the same type");
		exit(0);
		}
	Clear_Param();
	}
//--------------------------------------------------------------------------
// This instruction will translate the copy array instruction

void Copy_Array_Inst::Translate( State &s )
	{
	cout << "// Copy Array - ID: " << id << "\n" <<
		"   for (___lcv=0; ___lcv<(int)(";
	Expression *temp=new Expression(s.Get_Size_Expression(target));
	Convert_Expression(temp);
	delete temp;
	cout << ");___lcv++)\n      " << s.Get_Name(target) << "[___lcv]=" <<
		s.Get_Name(source) << "[___lcv];\n";
	}






