//----------------------------------------------------------------------------
// Program: bp.C  -  Neural Network and Backpropagation Neural Network Classes
//
// Author : Joey Rogers
// Date   : July 30, 1993
//
// Purpose: This class provides a generic forward pass neural network class,
//          NET, and a derived backpropagation neural network class, BPNET.
//
//----------------------------------------------------------------------------

#include"bp.h"

//----------------------------------------------------------------------------
// Constructor for Generic Forward Pass Neural Network

NET::NET( void )
	{
	node_counter=0;
	high_in=NULL;  high_out=NULL; low_in=NULL; low_out=NULL;
	weight_count=0;
	}
//----------------------------------------------------------------------------
void NET::Set_Node_Type( int layer, int node, int type )
	{
	

	}
//----------------------------------------------------------------------------
// This function creates a network in memory; it is supplied with the 
// number of layer to create and an array containing the number of
// nodes to create for each layer

void NET::Create_Layers( int layers, int layer_nodes[] )
	{
	layer=new LAYER[layers];
	if (layer==NULL)
		{
		fprintf(stderr,"Cannot Allocate Layers.\n");
		exit(0);
		}

	for (int i=0; i<layers; i++)    // Initialize Layer Info
	    {
	    layer[i].node=Create_Nodes(layer_nodes[i]);
	    layer[i].node_count=layer_nodes[i];
	    }

	in_count=layer_nodes[0];
	mid_count=layers-2;
	out_count=layer_nodes[layers-1];
	out=layers-1;
	layer_count=layers;

	high_in=new double[in_count];
	low_in=new double[in_count];
	high_out=new double[out_count];
	low_out=new double[out_count];

	for (i=0; i<in_count; i++)
		{ high_in[i]=1.0;  low_in[i]=0.0; }

	for (i=0; i<out_count; i++)
		{ high_out[i]=1.0;  low_out[i]=0.0; }

	}
//--------------------------------------------------------------------------
// This function creates a network node

NODE *NET::Create_Nodes( int number )
	{
	NODE *temp=new NODE[number];
	if (temp==NULL)
		{
		fprintf(stderr,"Cannot Allocate Network Nodes.\n");
		exit(0);
		}

	for (int i=0; i<number; i++)    // Initialize Weight Info
	    {
	    temp[i].id=node_counter;
	    node_counter++;
	    temp[i].value=0;
	    temp[i].error=0;
	    temp[i].work=0;
	    temp[i].bias=Random();
	    temp[i].first_weight=NULL;
	    }
	return(temp);
	}
//----------------------------------------------------------------------------
// This function creates a network weight

WEIGHT *NET::Create_Weight( void )
	{
	WEIGHT *temp=new WEIGHT;
	if (temp==NULL)
		{
		fprintf(stderr,"Cannot Allocate Weight.\n");
		exit(0);
		}

	temp->layer=-999;         // Initialize Weight Info
	temp->node=-999;
	temp->input_node=NULL;
	temp->next=NULL;
	temp->value=Random();
	temp->prev_delta=0;
	return(temp);
	}
//-------------------------------------------------------------------------
// This function creates a double array and initializes its values

double *NET::Create_Array( int size, double init_value )
	{
	if (size<=0) return NULL;
	double *temp=new double[size];
	if (temp==NULL)
	   {
	   fprintf(stderr,"Cannot allocate a double array.\n");
	   exit(0);
	   }

	for (int i=0; i<size; i++)      // Initialize Array 
	    temp[i]=init_value;

	return(temp);
	}
//-------------------------------------------------------------------------
// This function loads a network file from disk and creates network
// in memory

void NET::Load_Network(char filename[] )
	{
	int i,j;
	double value;
	WEIGHT *temp;
	int layer1,layer2;
	int node1,node2;
	int *layer_nodes;

	FILE *infile=fopen(filename,"r");     // Open Network file 
	if (infile==NULL)
		{
		cout << "Cannot open network file: " << filename<< "\n";
		exit(0);
		}

	fscanf(infile,"* Layers: %d\n",&layer_count);  // Read network size
	layer_nodes=new int[layer_count];
	fscanf(infile,"* Nodes: ");
	for(i=0; i<layer_count; i++)
		fscanf(infile,"%d",&layer_nodes[i]);
	in_count=layer_nodes[0];
	out_count=layer_nodes[layer_count-1];
	fscanf(infile,"\n");
	out=layer_count-1;
	Create_Layers(layer_count,layer_nodes);
	if (high_in!=NULL)     // free up high low memory if already allocated
		{
		delete high_in;  delete low_in;
		delete high_out; delete low_out;
		}

	high_in=Create_Array(layer_nodes[0],0);
	high_out=Create_Array(layer_nodes[out],0);
	low_in=Create_Array(layer_nodes[0],0);
	low_out=Create_Array(layer_nodes[out],0);

	fscanf(infile,"* High/Low (in): ");    // Read High/Low Values 
	for(i=0; i<layer_nodes[0]; i++)
		fscanf(infile,"%lf %lf",&high_in[i],&low_in[i]);
	fscanf(infile,"\n");

	fscanf(infile,"* High/Low (out): ");
	for(i=0; i<layer_nodes[out]; i++)
		fscanf(infile,"%lf %lf",&high_out[i],&low_out[i]);
	fscanf(infile,"\n");

	fscanf(infile,"* Bias Terms:\n");      // Read Bias Terms
	for (i=1; i<layer_count; i++)
		for (j=0; j<layer[i].node_count; j++)
			{
			fscanf(infile,"# %d  %d  %lf\n",&layer1,&node1,&value);
			layer[layer1].node[node1].bias=value;
			}

	fscanf(infile,"* Weights:\n");     // Read Weights
	char header_char; 
	while(!feof(infile))
		{
		fscanf(infile,"%c",&header_char);
		if (header_char!='#') break;
		fscanf(infile," %d %d %d %d %lf\n",
		    &layer1,&node1,
		    &layer2,&node2,&value);

		temp=Create_Weight();
		temp->layer=layer2;
		temp->node=node2;
		temp->value=value;
		Connect_Weight(temp,layer2,node2,layer1,node1);
		}
	fclose(infile);

	delete layer_nodes;
	}
//----------------------------------------------------------------------------
// This function produces a random number between -0.5 and 0.5

double NET::Random(void)
	{
	double temp= (double)((rand()%(int)(((1000)+1)-(-1000)))+(-1000));
	temp/= 1000.0-1.0;
	return temp/2.0;
	}
//----------------------------------------------------------------------------
// This function performs a forward pass on the network

void NET::Forward_Pass( void )
	{
	int i,j;
	WEIGHT *k;
	double sum;
	NODE *curr_node;
	
	for (i=1; i<layer_count; i++)
  	    for (j=0; j<layer[i].node_count; j++)
		{
		curr_node=&layer[i].node[j];
		curr_node->work=0;
		sum=0;
		for (k=curr_node->first_weight; k!=NULL; k=k->next)
			sum+=k->value*k->input_node->value;
		curr_node->value=Transfer_Function(sum+curr_node->bias);
		}
	}
//----------------------------------------------------------------------------
// This function performs a backward pass on the network, adjusting the
// network weights to better produce the current input

void BPNET::Backward_Pass( DATA *data, double learning_rate, double momentum )
	{
	int i,j,k;
	WEIGHT *l;
	NODE *node;
	double sum;
	double delta;

	node=layer[out].node;     // Compute Output layer error 
	for (i=0; i<layer[out].node_count; i++)
	    node[i].error=node[i].value*(1.0-node[i].value)*
		          (data->out[i]-node[i].value);

	for (i=out-1; i>0; i--)   // Compute Middle layer error 
		{
		node=layer[i+1].node;
		for (k=0; k<layer[i+1].node_count; k++)
			for (l=node[k].first_weight; l!=NULL; l=l->next)
				l->input_node->work+=node[k].error*l->value;

		for (k=0; k<layer[i].node_count; k++)
			{
			node=&layer[i].node[k];
			node->error=node->value*(1.0-node->value)*node->work;
			network_error+=pow(node->error,2);
			}
		}

	for (i=1; i<=out; i++)    // Adjust Weights 
		{
		node=layer[i].node;

		for (j=0; j<layer[i].node_count; j++)
		    {
		    node[j].bias+=learning_rate*node[j].error;  // update bias 

		    for (l=node[j].first_weight; l!=NULL; l=l->next)
				{
				delta=learning_rate*node[j].error*
					l->input_node->value;
				l->value+=delta+(momentum*l->prev_delta);
				l->prev_delta=delta;
				}
		    }
		}
	}
//----------------------------------------------------------------------------
// This is the network transfer function

double NET::Transfer_Function( double value) 
	{
        return 1.0/(1.0+exp(SIG_GAIN*-value));
	}
//----------------------------------------------------------------------------
// This function retrieves the high/low values for the network

void NET::Get_High_Low( double h_i[], double l_i[], 
			double h_o[], double l_o[] )
	{
	for (int i=0; i<in_count; i++) 
		{
		h_i[i]=high_in[i];
		l_i[i]=low_in[i];
		}
	for (i=0; i<out_count; i++)
		{
		h_o[i]=high_out[i];
		l_o[i]=low_out[i];
		}
	}
//----------------------------------------------------------------------------
// This function sets the high/low values for the network

void NET::Set_High_Low( double h_i[], double l_i[], 
			double h_o[], double l_o[] )
	{
	if (high_in!=NULL)
		{
		delete high_in; delete low_in;
		delete high_out; delete low_out;
		}

	high_in=new double[in_count];
	low_in=new double[in_count];
	high_out=new double[out_count];
	low_out=new double[out_count];

	for (int i=0; i<in_count; i++) 
		{
		high_in[i]=h_i[i];
		low_in[i]=l_i[i];
		}
	for (i=0; i<out_count; i++)
		{
		high_out[i]=h_o[i];
		low_out[i]=l_o[i];
		}
	}
//----------------------------------------------------------------------------
// This function trains a backpropagation neural network

void BPNET::Train( int mode )
	{
	DATA *data_item;
	NODE *node;
	double temp_error;
	double total_error;
	double best_test_error=MAX_VALUE;
	double high_error;
	int temp; 
	int good=0, best_good=0,t_good,t_best_good=0;
	double best_good_error, t_best_good_error;	

	if (high_in!=NULL)
		{
		delete high_in;  delete low_in;
		delete high_out; delete low_out;
		}

	high_in=new double[in_count];
	low_in=new double[in_count];
	high_out=new double[out_count];
	low_out=new double[out_count];

	if (train_data!=NULL) 
		train_data->Get_High_Low( high_in,low_in,high_out,low_out);
	else
		{
		for (int i=0; i<in_count; i++)
			{ high_in[i]=1.0;  low_in[i]=0.0;}
		for (i=0; i<out_count; i++)
			{ high_out[i]=1.0;  low_out[i]=0.0;}
		}

	iteration=0;
	test_error=MAX_VALUE;
	best_good_error=MAX_VALUE;
	int saved_flag=0;

	while (!Finished(best_good)
	       && iteration<max_iterations)
		{
		for (data_item=train_data->Get_Data_Head(); data_item!=NULL; 
		     data_item=data_item->next)
			{
			Load_Input_Layer(data_item->in,in_count);
			Forward_Pass();
			Backward_Pass(data_item,beta,alpha);
			}

                                // Compute Error on Testing Set if needed 
		if (iteration % check_test_set==0 )
			{
			switch (mode)
			  {
			  case 0: Test_Testing_Set(good);
			          if (good>best_good) 
				     {
				     best_good=good;
				     best_good_error=test_error;
				     Save_Weights(network_file);  
				     saved_flag=1;
			 	     }
				  break;

 			  case 1: Test_Testing_Set1(good);
				  
			          if (good>best_good) 
				     {
				     best_good=good;
				     best_good_error=test_error;
				     Save_Weights(network_file);  
				     saved_flag=1;
				     }
				  break;

			  case 2: Test_Testing_Set2(good);
		 	          if (best_test_error>test_error) 
				     {
				     best_test_error=test_error;
				     Save_Weights(network_file);  
				     saved_flag=1;
				     }
				  break;
			  }


			}

	    // Display current state 
		if (display_rate>0 && iteration % display_rate==0)
		  {
		  cout << setw(4) << iteration << ".   Err: " << setw(12) 
			<< test_error << "  Good:" << setw(3) << good << 
			" (" << setw(3) << best_good << ")   " <<
			(int)((double)good/(double)test_data->Get_Count()
				*100.0+.5)<< "%";

		   if (saved_flag==1) { cout << " *"; saved_flag=0;}
		   cout << "\n" << flush;
		   }	

		iteration++;
		}

	if (good==best_good && best_good_error>=low_test_error) 
		Save_Weights(network_file);
	}
//----------------------------------------------------------------------------
// This function determines if Train is finished training

int BPNET::Finished( int test_correct )
	{
	if (iteration<min_iterations) return 0;
	if (test_correct==test_data->Get_Count()) return 1;
	else return 0;
	}
//----------------------------------------------------------------------------
// This function return the values in the output layer

void NET::Get_Output( double output[] )
	{
	for (int i=0; i<layer[out].node_count; i++)
		output[i]=layer[out].node[i].value;
	}
//----------------------------------------------------------------------------
// This function saves the current network state

void NET::Save_Weights( char filename[] )
	{
	int i,j;
	WEIGHT *k;

	FILE *outfile=fopen(filename,"w");
	if (outfile==NULL)
	   {
	   cout << "Output file cannot be opened: " << filename << "\n";
	   exit(0);
	   }

	fprintf(outfile,"* Layers: %d\n",layer_count);
	fprintf(outfile,"* Nodes: ");
	for(i= 0; i<layer_count; i++)
		fprintf(outfile,"%d ",layer[i].node_count);
	fprintf(outfile,"\n");
	fprintf(outfile,"* High/Low (in): ");
	for(i=0; i<layer[0].node_count; i++)
		fprintf(outfile,"%lf %lf ",high_in[i],low_in[i]); 
	fprintf(outfile,"\n");
	fprintf(outfile,"* High/Low (out): ");
	for (i=0; i<layer[out].node_count; i++)
		fprintf(outfile,"%lf %lf ",high_out[i],low_out[i]);
	fprintf(outfile,"\n"); 

	fprintf(outfile,"* Bias Terms:\n");
	for (i=1; i<layer_count; i++)
		for (j=0; j<layer[i].node_count; j++)
			fprintf(outfile,"# %d  %d     %30.20lf\n", 
				i,j,layer[i].node[j].bias);

	fprintf(outfile,"* Weights:\n");
	for (i=0; i<layer_count; i++)
	    for (j=0; j<layer[i].node_count; j++)
		for (k=layer[i].node[j].first_weight; k!=NULL; k=k->next)
		    fprintf(outfile,"# %d %d %d %d %30.20lf\n",
				    i,j,k->layer,k->node,k->value);
	fclose(outfile);
	}
//----------------------------------------------------------------------------
// This function prints the network: nodes values and weight values
void NET::Print_Network( void )
	{
	int i,j,k;
	WEIGHT *temp;

	cout << "\nLayer Count: " << layer_count << "\n";

	for(i=0; i<layer_count; i++)
		{
		cout << "Layer: " << i << "\n";
		for(j=0; j<layer[i].node_count; j++)
		    {
		    cout << "    Node: " <<j<< "(" << layer[i].node[j].id<< ")"
		         << " -  Value: " << layer[i].node[j].value 
			 << "  Bias: " << layer[i].node[j].bias << "\n";
		    temp=layer[i].node[j].first_weight;
		    k=0;
		    while (temp!=NULL)
				{
				cout << "     (" << temp->layer << "," << 
				     temp->node << ")  " << 
				     temp->input_node->id << "  Value: " <<
				     temp->value << "\n";
				k++;
				temp=temp->next;
				}
		    }
		}
	cout << "\n" << flush;
	}
//----------------------------------------------------------------------------
// This function fully connects a network - each node is connected to each
// node in each succeeding layer

void NET::Fully_Connect_Network( void )
	{
	int i,j,l,k;

	for (i=1; i<layer_count; i++)
		for (j=0; j<layer[i].node_count; j++)
			for (k=i-1; k>=0; k--)
			    for (l=0; l<layer[k].node_count; l++)
				Connect_Weight(Create_Weight(),k,l,i,j);

	}
//----------------------------------------------------------------------------
// This function partially connects a network - each node is connected to each
// node in its succeeding layer

void NET::Partially_Connect_Network( void )
	{
	int i,j,l;

	for (i=1; i<layer_count; i++)
		for (j=0; j<layer[i].node_count; j++)
			for (l=0; l<layer[i-1].node_count; l++)
				Connect_Weight(Create_Weight(),i-1,l,i,j);

	}
//----------------------------------------------------------------------------
// This function connects a weight between two network nodes

void NET::Connect_Weight( WEIGHT *wgt, int from_layer, int from_node, 
			    int to_layer, int to_node )
	{
	wgt->layer=from_layer;
	wgt->node=from_node;
	if (!(Valid_Node(from_layer,from_node)&&Valid_Node(to_layer,to_node)))
		{
		cout << "Unable to make Connection: (" << from_layer << 
			"," << from_node << ")->(" << to_layer << "," <<
			to_node << ")\n";
		exit(0);
		}

	wgt->input_node=&layer[from_layer].node[from_node];
	wgt->next=layer[to_layer].node[to_node].first_weight;
        layer[to_layer].node[to_node].first_weight=wgt;
	weight_count++;
	}
//----------------------------------------------------------------------------
// This function returns a specific network weight value

double NET::Get_Weight(int from_layer, int from_node,int to_layer, int to_node)
	{
	if (!(Valid_Node(to_layer,to_node)&&Valid_Node(from_layer,from_node)))
		{
		cout << "Invalid Node in Connection: (" << from_layer << 
			"," << from_node << ")->(" << to_layer << "," <<
			to_node << ")\n";
		exit(0);
		}

	WEIGHT *temp=layer[to_layer].node[to_node].first_weight;
	while (temp!=NULL)
		{
		if (temp->layer==from_layer && temp->node==from_node)
			return temp->value;
		temp=temp->next;
		}

	cout << "Connection does not exist: (" << from_layer << 
		"," << from_node << ")->(" << to_layer << "," <<
		to_node << ")\n";
		
	exit(0);
	return 0;
	}
//----------------------------------------------------------------------------
// This function sets a specific network weight value

void NET::Set_Weight(int from_layer, int from_node,int to_layer, int to_node,
		     double value)
	{
	if (!(Valid_Node(to_layer,to_node)&&Valid_Node(from_layer,from_node)))
		{
		cout << "Invalid Node in Connection: (" << from_layer << 
			"," << from_node << ")->(" << to_layer << "," <<
			to_node << ")\n";
		exit(0);
		}

	WEIGHT *temp=layer[to_layer].node[to_node].first_weight;
	while (temp!=NULL)
		{
		if (temp->layer==from_layer && temp->node==from_node)
			{
			temp->value=value;
			return;
			}
		temp=temp->next;
		}

	cout << "Connection does not exist: (" << from_layer << 
		"," << from_node << ")->(" << to_layer << "," <<
		to_node << ")\n";
		
	exit(0);
	}
//----------------------------------------------------------------------------
// This function removes a weighted connection from the network

void NET::Remove_Weight(int from_layer,int from_node,int to_layer, int to_node)
	{
	if (!(Valid_Node(to_layer,to_node)&&Valid_Node(from_layer,from_node)))
		{
		cout << "Invalid Node in Connection: (" << from_layer << 
			"," << from_node << ")->(" << to_layer << "," <<
			to_node << ")\n";
		exit(0);
		}

	WEIGHT *temp=layer[to_layer].node[to_node].first_weight;
	WEIGHT *prev=NULL;
	while (temp!=NULL)
		{
		if (temp->layer==from_layer && temp->node==from_node)
			{
			if (prev==NULL)
				layer[to_layer].node[to_node].first_weight=
					temp->next;
			else
			        prev->next=temp->next;
			weight_count--;
			delete temp;
			return;
			}
		prev=temp;
		temp=temp->next;
		}

	cout << "Connection does not exist: (" << from_layer << 
		"," << from_node << ")->(" << to_layer << "," <<
		to_node << ")\n";
		
	exit(0);
	}
//----------------------------------------------------------------------------
// This function test to see if a node is really in the network

int NET::Valid_Node( int node_layer, int node_pos )
	{
	if (node_layer<0 || node_layer>=layer_count ||
	    node_pos<0 || node_pos>=layer[node_layer].node_count )
		return 0;
	return 1;
	}
//----------------------------------------------------------------------------
// This function tests the test set to see how many element are correctly
// learned

void BPNET::Test_Testing_Set( int &good_cnt )
	{
	double high_error,temp_error;
	double total_error;
	int good=0;
	int i;

     // Compute Error on Testing Set 
	for (DATA *data_item=test_data->Get_Data_Head(); data_item!=NULL;
	     data_item=data_item->next)
		{
		Load_Input_Layer(data_item->in,in_count);
		Forward_Pass();
		high_error=fabs(layer[out].node[0].value-data_item->out[0]);
		total_error=pow(high_error,2);
		for (i=1;i<layer[out].node_count;i++)
		   {
		   temp_error=layer[out].node[i].value-data_item->out[i];
		   total_error+=pow(temp_error,2);
		   if (fabs(temp_error)>high_error)
			 high_error=fabs(temp_error);
		   }
		if (high_error<tolerance) good++;
		}

	good_cnt=good;
	test_error=total_error/2.0;
	}
//----------------------------------------------------------------------------
// This is an alternative to the previous function

void BPNET::Test_Testing_Set1( int &good_cnt )
	{
	DATA *data_item;
	NODE *node;
	double low_imp=MAX_VALUE;	
	double high_auth=MIN_VALUE;
	int imp=0;
	int auth=0;
	int good=0;

	for (data_item=test_data->Get_Data_Head(); data_item!=NULL;
	     data_item=data_item->next)
	     if (data_item->out[0]==1)
		{
		Load_Input_Layer(data_item->in,in_count);
	        Forward_Pass();
	        node=layer[out].node;
                if (low_imp>node[0].value) low_imp=node[0].value;
		imp++;
		}

	for (data_item=test_data->Get_Data_Head(); data_item!=NULL;
	     data_item=data_item->next)
	     if (data_item->out[0]!=1)
		{
		Load_Input_Layer(data_item->in,in_count);
	        Forward_Pass();
	        node=layer[out].node;
                if (node[0].value<low_imp) good++;
		if (node[0].value>high_auth) high_auth=node[0].value;
		auth++;
		}

	good_cnt=good+imp;
	test_error=auth-good;
	if (iteration % display_rate==0)
	  	cout << "    High Auth: " << high_auth <<  "  Low Imp: " <<
		     low_imp << "    Gap: " << low_imp-high_auth<< "\n";
	}
//----------------------------------------------------------------------------
// This is an alternative to the previous function


void BPNET::Test_Testing_Set2( int &good_cnt )
	{
	double high_error,temp_error;
	double total_error,total_error1=0;
	int good=0;
	int i;

     // Compute Error on Testing Set 
	for (DATA *data_item=test_data->Get_Data_Head(); data_item!=NULL;
	     data_item=data_item->next)
		{
		Load_Input_Layer(data_item->in,in_count);
		Forward_Pass();
		high_error=fabs(layer[out].node[0].value-data_item->out[0]);
		total_error=pow(high_error,2);
		total_error1+=high_error;
		if (high_error<tolerance) good++;
		}

	good_cnt=good;
	test_error=total_error1;


	}
//---------------------------------------------------------------------------
// This functin loads data into the input layer

void NET::Load_Input_Layer( double data[], int size )
	{
	for (int i=0; i<size; i++)
		layer[0].node[i].value=data[i];	
	}
//---------------------------------------------------------------------------
// This fucntion takes a data list and runs it though the network.  The 
// output can be written to file or printed on the screen.

void NET::Run_Network( Data_Class *data, char filename[] )
	{
	double temp,temp_net;
	int i,j;
	int mode=0;
	FILE *outfile;

	if (strcmp(filename,"")==0) mode=1;   // do not store to file
	else
		{
		outfile=fopen(filename,"w");
		if (outfile==NULL)
		   {
		   cout << "Unable to open output file: " << filename << "\n";
		   exit(0);
		   }
		}

	for (DATA *curr_data=data->Get_Data_Head(); curr_data!=NULL; 
		curr_data=curr_data->next)
		{
		Load_Input_Layer( curr_data->in, in_count);
		Forward_Pass();
		if (mode==1) cout <<"ID: "<< curr_data->id <<"  Outputs: ";
			else fprintf(outfile,"%d   ",curr_data->id);
		for (i=0; i<layer[out].node_count; i++)
			{
			temp=high_out[i]-low_out[i];
			temp_net=(layer[out].node[i].value)*temp+low_out[i];
			if (mode==1)
			     cout << temp_net << " (" << 
				  layer[out].node[i].value << ")  ";
			else
			     fprintf(outfile,"%lf %lf   ",temp_net,
			          layer[out].node[i].value);
			}

		if (mode==0) fprintf(outfile,"\n");
		   else cout << "\n" << flush;
		}
	if (mode==0) fclose(outfile);
	}
//---------------------------------------------------------------------------
// This function sets the current weight pointer to the first weight in network

void NET::Reset_Current_Weight( void )
	{
	int i,j;
	for (i=0; i<=out; i++)
		for (j=0; j<layer[i].node_count; j++)
			if (layer[i].node[j].first_weight!=NULL)
				{
				curr_weight=layer[i].node[j].first_weight;
				curr_weight_layer=i;
				curr_weight_node=j;
				break;
				}
	}
//---------------------------------------------------------------------------
// This function returns the information from the current weight

int NET::Get_Current_Weight( int &from_layer, int &from_node,
			      int &to_layer, int &to_node, double &value )
	{
	if (curr_weight==NULL) return 0;
	from_layer=curr_weight->layer;
	from_node=curr_weight->node;
	to_node=curr_weight_node;
	to_layer=curr_weight_layer;
	value=curr_weight->value;
	return 1;
	}
//---------------------------------------------------------------------------
// This function advances the current weight 

void NET::Next_Weight( void )
	{
	int i, j;

	if (curr_weight->next!=NULL)    // if there is another weight in node
		{
		curr_weight=curr_weight->next;
		return;
		}

	do   // Go through nodes until a weight is found
	  {
	  Next_Node(curr_weight_layer,curr_weight_node);
	  curr_weight=
	    layer[curr_weight_layer].node[curr_weight_node].first_weight;
	  } while (curr_weight==NULL);
	}
//---------------------------------------------------------------------------
// This function finds the next valid network node

void NET::Next_Node( int &lay, int &node )
	{
	node++;
	if (node>=layer[lay].node_count)
		{
		lay++;
		node=0;
		}

	if (lay>=layer_count)
		lay=0;
	}	













