
functor Sexp () : SEXP =
struct
    structure P = Parser
    open P

    infix 5 --;
    infix 3 >>;
    infix 0 ||;

    structure Symbols = SymbolTable ()
    structure Sym  = Symbols.Sym

    val symbol = Symbols.symbol

    datatype sexp = 
	Nil
      | Bool of bool
      | Sym of Sym.symbol 
      | Int of int    
      | Char of int
      | Str of string 
      | Cons of sexp * sexp

    (* improper lists *)

    fun improper ([e1],e2) = Cons(e1,e2)
      | improper ((e1::t),e2) = Cons(e1,improper(t,e2))

    (* proper lists *)

    fun proper  [] = Nil
      | proper (h::t) = Cons(h,proper t)

    (* parse an s-expression *)

    fun sexp tokens =
	(sym 
      || num 
      || boolean 
      || char 
      || str 
      || quote -- sexp      >> (fn (_,y) => (Cons (Sym (symbol "quote"), y)))
      || lpar -- rpar       >> (fn (_,_) => Nil)
      || lpar -- list -- dot -- sexp -- rpar >> (fn ((((_,l),_),e),_) => 
						 improper (l,e))
      || lpar -- list -- rpar >> (fn ((_,l),_) => proper l)
	 ) tokens 

    and sym tokens =  
	case S.head tokens of
	    SYM s => (Sym (symbol s), S.tail tokens)
	  | _ => raise SyntaxError "Symbol expected"

    and num tokens =
	case S.head tokens of
	    NUM n => (Int n, S.tail tokens) 
	  | _ => raise SyntaxError "Integer expected"

    and boolean tokens =
	case S.head tokens of
	    BOOL b => (Bool b, S.tail tokens) 
	  | _  =>  raise SyntaxError "Boolean expected" 

    and char tokens =
	case S.head tokens of
	    CHAR c => (Char c, S.tail tokens) 
	  | _ => raise SyntaxError "Character expected"

    and str tokens =
	case S.head tokens of
	    STR s => (Str s, S.tail tokens)
	  | _ => raise SyntaxError "String expected"

    and quote tokens =
	case S.head tokens of 
	    QUOTE => (Nil, S.tail tokens) 
	  | _ =>  raise SyntaxError "\"'\" expected"

    and lpar tokens =
	case S.head tokens of
	    LPAR => (Nil, S.tail tokens) 
	  | _ =>  raise SyntaxError "\"(\" expected"

    and rpar tokens =
	case S.head tokens of
	    RPAR => (Nil, S.tail tokens)
	  | _ => raise SyntaxError "\")\" expected"

    and dot tokens =
	case S.head tokens of
	    DOT => (Nil, S.tail tokens)
	  | _ => raise SyntaxError "\".\" expected"

    (* list : sequence of one or more expressions *)

    and list tokens =
	(sexp -- (repeat sexp)       >> (fn (h,t) => (h::t))
	 ) tokens
	    

    exception Read of int * string  (* line number, message *)
    exception EOF

    fun read (instr : instream) : sexp =
        (case sexp (lex instr) of (exp, _) => exp)
	handle SyntaxError (msg) => raise Read (0,msg)
	     | LexError (ln,msg) => raise Read (ln,msg)
	     | S.EmptyStream => raise EOF

    fun write_string os s =
     let fun write_char "\\" = output(os,"\\\\")
	   | write_char "\"" = output(os,"\\\"")
           | write_char c = output(os,c)
     in
	 output(os,"\"");
	 List.app write_char (explode s); 
	 output(os,"\"")
     end

    (* protect a string *)

    fun protect s =
     let fun makechar "\\" = "\\\\"
	   | makechar "\"" = "\\\""
           | makechar c = c
     in
	 implode (("\""::(List.map makechar (explode s)))@["\""])
     end

   fun identity x = x

  (* s2b: sexp to block, for pretty printing *)

  local open Pretty 
  in
      fun format exp ( f : string -> string) =
	  let  fun s2b Nil = string "()"
		 | s2b (Sym s) = string (Symbols.string s)
		 | s2b (Bool true) = string "#t"
		 | s2b (Bool false) = string "#f"
		 | s2b (Int i) = string (Integer.makestring i)
		 | s2b (Char 10) = string "#\\newline"
		 | s2b (Char 32) = string "#\\space"
		 | s2b (Char i) = string ("#\\"^(chr i))
		 | s2b (Str s) = string (f s)
		 | s2b (Cons(Sym ("quote",_),e1)) = block (0,[string "'", s2b e1])
		 | s2b (l as Cons(e1,e2)) =
		   let fun members (Cons(e1,Nil)) = [s2b e1]
			 | members (Cons(e1,r as (Cons(_,_)))) =  
			       (s2b e1)::(break 1)::(members r)
			 | members (Cons(e1,e2)) = [s2b e1,string " . ", s2b e2]
		   in
		       block (4, (string "("::((members l)@[string ")"])))
		   end
	  in
	      s2b exp
	  end

      fun write os exp = Pretty.print os (format exp protect) 
      fun display os exp = Pretty.print os (format exp identity) 
  end

(* Equality of sexps is defined here to have the meaning of Scheme's
equal?  predicate, which should no be mistaken for eq? or eqv?
Neither eq? nor eqv? can be implemented in ML, as they compare the
locations of cons cells and strings in store, rather than their
values. *)

   fun equal Nil Nil = true 
     | equal (Bool x) (Bool y) = (x = y)
     | equal (Sym x) (Sym y) = (Sym.eq x y)
     | equal (Int x) (Int y) = (x = y)
     | equal (Char x) (Char y) = (x = y)
     | equal (Str x) (Str y) = (x = y)
     | equal (Cons(x1,y1)) (Cons(x2,y2)) = 
         (equal x1 x2) andalso (equal y1 y2)
     | equal _ _ = false

end  (* Sexp functor *)

(* Test Read-Eval-Print loop.

structure S = Sexp ()
open S

val prompt = ref "";
    
fun rep () =
   let val _ =  output(std_out, !prompt)
       val result = read std_in 
   in
       output(std_out,"\n");
       write std_out result;
       output(std_out,"\n");
       flush_out std_out;
       rep ()
   end handle 
	Read (i,s) => 
	    (output(std_out, "(error \"line "^
                              (Integer.makestring i)^": "^s^"\")\n");
	    flush_out std_out;
	    rep ())
      | Overflow => 
		(output(std_out, "(error \"bignums not supported\")\n");
		flush_out std_out;
		rep ())
      | EOF => ()

exportFn ("rep",(fn (_,_) => rep()))

*)
	    
