
functor Search (S : SPACE) : SEARCH =
struct
    open S

    structure Sol : STREAM = Stream()

    datatype problem =
	prblm of { start : node,
		  space : space,
		  goal  : state -> bool }

    fun problem s sp g = prblm { start=s, space=sp, goal=g }

    fun start (prblm {start=s, ...}) = s
    fun problemSpace (prblm {space=s, ...}) = s
    fun goal (prblm {goal=g, ...}) = g

    type strategy = problem -> node Sol.stream

    fun strategy (s : problem -> node Sol.stream) = s

    fun search p s = s p

    fun path (n : node) : label list =
	if isroot n then
	    []
	else
	    (path (parent n))@[label n]

    val trace : bool ref = ref false
    val traceFile : outstream ref = ref std_out

    val expanded : int ref = ref 0
   
    fun reset () = expanded := 0

   (* Some Utilities *)

   fun expand (n : node) (s : space) : node S.Succ.stream =
       (inc expanded;
	if !trace then
	     (output(!traceFile, Integer.makestring (!expanded));
	      output(!traceFile, ". ");
	      put_node (!traceFile) n;
	      output(!traceFile, "\n\n"))
	else
	    ();
	apply s n)

   (* CS : Set of closed nodes *)
       
   structure CS : SET = ListSet(struct 
				    type element = node
				    fun eq (n1 : node) (n2 : node) =
					eq_state (state n1) (state n2)
				    val format = format
				    val put = put_node
				end)
       
   
   fun hillClimbing (no_worse : node -> node -> bool)
       (prblm {start=n, space=sp, goal=goalp})  =
       (reset ();
	let val closed_nodes = ref CS.empty_set

	    fun quick_sort (str : node S.Succ.stream) : node list =
		let fun sort [] = []
		      | sort [x] = [x]
		      | sort (a::rest) = (* the head "a" is the pivot *)
			let fun split(lft,rht,[]) = sort lft @ (a::sort rht)
			      | split(lft,rht,x::l) =
				if no_worse x a  then split(x::lft,rht,l)
				else split(lft,x::rht,l)
			in split([],[],rest)
			end
		in 
		    sort (S.Succ.stream2list str)
		end;

	(* choose_successor selects the first successor which is no_worse
	 than the parent and has not already been expanded.  The list from
	 which the selection is made is expected to have been sorted according 
         to the no_worse relation.  Once a successor is considered which is
	 worse than the parent, no other successors are considered. *)


	    fun loop (n : node) : node Sol.stream =
		let fun choose_successor [] = NONE
		      | choose_successor (first::rest) =
			if no_worse first n then
			    if not (CS.member first (!closed_nodes)) then
				(closed_nodes := CS.add first (!closed_nodes);
				 SOME first)
			    else
				choose_successor rest
			else
			    NONE
		in
		    if goalp(state n) then 
			Sol.singleton n
		    else
			case choose_successor (quick_sort (expand n sp)) of
			    NONE => Sol.empty_stream ()
		          | SOME next_node => loop next_node
		end
	in
	    loop n
	end)
       
(* type bnode : depth-bounded node *)
 
  datatype bnode = bot | bnode of { depth: int, node:node }

   exception BNode

   fun depth bot = raise BNode
     | depth (bnode {depth=i, ...}) = i

   fun node_of bot = raise BNode
     | node_of (bnode {node=n, ...}) = n

   fun format_bnode bot = Pretty.string "bottom"
     | format_bnode bn = format (node_of bn)

   fun put_bnode os bot = output(os,"bottom ")
     | put_bnode os bn = put_node os (node_of bn)

   fun eq_bnode bot bot = true
     | eq_bnode bn1 bn2 = 
          eq_node (node_of bn1) (node_of bn2)

   fun bnode_expand (d : int) (bn : bnode) (sp : space) 
       : bnode S.Succ.stream =
       S.Succ.map (fn (n : node) => bnode { depth=d, node=n }) 
       (expand (node_of bn) sp)

   (* NS : A Node Set for managing closed bnodes *)

   structure NS : SET =  ListSet(struct 
				     type element = bnode
				     fun eq (n1  : element) (n2 : element) =
					 eq_state (state (node_of n1))
					          (state (node_of n2))
				     
                                     val format = format_bnode

				     fun put os (e : bnode) = 
					 put_node os (node_of e)
				 end)

(* 
  depthFirst resets the monitor,
  depthFirst' doesn't. 
 *)

   fun depthFirst' 
       (limit : int) 
       (prblm {start=n, space=sp, goal=goalp})  =
   let val closed_nodes = ref NS.empty_set
       val root = bnode { depth=0, node=n }
       val open_nodes = ref (S.Succ.singleton root)

       fun loop () : node Sol.stream =
	   if S.Succ.eos (!open_nodes) then
	       Sol.empty_stream ()
	   else
	       let val n = S.Succ.head (!open_nodes)
		   val in_closed = NS.member n (!closed_nodes)
	       in
		   open_nodes := S.Succ.tail (!open_nodes);
		   if not in_closed then
		       closed_nodes := NS.add n (!closed_nodes)
		   else
		       ();
		   if goalp (state (node_of n)) then
		       Sol.stream (node_of n, Sol.delay (fn () => loop ()))
		   else if not in_closed andalso (depth n < limit) then
		       (open_nodes := S.Succ.append (bnode_expand (depth n + 1)
						             n sp, 
						!open_nodes);
			loop ())
		   else 
		       loop ()
	       end
   in
       loop ()
   end
  
   fun depthFirst l p =
	    (reset ();  depthFirst' l p)
	                   
   fun breadthFirst (limit : int) (prblm {start=n, space=sp, goal=goalp})  =
   let val closed_nodes = ref NS.empty_set
       val root = bnode { depth=0, node=n }
       val open_nodes = ref (S.Succ.singleton root)

       fun loop () : node Sol.stream =
	   if S.Succ.eos (!open_nodes) then
	       Sol.empty_stream ()
	   else
	       let val n = S.Succ.head (!open_nodes)
		   val in_closed = NS.member n (!closed_nodes)
	       in
		   open_nodes := S.Succ.tail (!open_nodes);
		   if not in_closed then
		       closed_nodes := NS.add n (!closed_nodes)
		   else
		       ();
		   if goalp (state (node_of n)) then
		       Sol.stream (node_of n, Sol.delay (fn () => loop ()))
		   else if not in_closed andalso (depth n < limit) then
		   (* This next line is the only difference between breadth
		      first and depth first search *)
		       (open_nodes := S.Succ.append (!open_nodes,
						bnode_expand (depth n + 1)
						             n sp);
			loop ())
		    else 
		        loop ()
	       end
   in
       reset (); loop ()
   end

   fun iterativeDeepening (init: int) (step : int) (limit : int)  p =
       (reset ();
	let fun loop (max : int) =
	    if max > limit then
		Sol.empty_stream ()
	    else
		Sol.append (depthFirst' max p, loop (max + step))
	in
	    loop init
	end)

   fun for_each [] _ = ()
     | for_each (h::t) f = (f h; for_each t f)
       
   structure D =
       struct
	   type element = bnode
	   val bottom = bot 
	   val format = format_bnode
	   val put = put_bnode
	   val eq = eq_bnode
       end
	   
   structure PQ = PQueue(D)

   fun bestFirst (limit : int) 
                 (better : node -> node -> bool)
	         (prblm {start=n, space=sp, goal=goalp}) =

   let fun prior bn1 bn2 = better (node_of bn1) (node_of bn2)
       val closed_nodes = ref NS.empty_set
       val root = bnode { depth=0, node=n }
       val open_nodes = PQ.empty_queue prior

       fun loop () : node Sol.stream =
	   if PQ.empty open_nodes then
	       Sol.empty_stream ()
	   else
	       let val n = PQ.dequeue open_nodes
		   val in_closed = NS.member n (!closed_nodes)
	       in
		   if not in_closed then
		       closed_nodes := NS.add n (!closed_nodes)
		   else
		       ();
		   if goalp (state (node_of n)) then
		       Sol.stream (node_of n, Sol.delay (fn () => loop ()))
		   else if not in_closed andalso (depth n < limit) then
		       (for_each (S.Succ.stream2list (bnode_expand (depth n + 1)
						                n sp))
			         (fn n1 => (PQ.enqueue open_nodes n1));
			loop ())
		    else 
		        loop ()
	       end
   in    
       reset(); 
       PQ.enqueue open_nodes root;
       loop ()
   end

   (* A Star strategy *)
  
   local
   
       datatype asn =   (* a star node *)
	   bot 
	 | asn of { depth: int ref, 
		   node: node,
		   parent : asn ref,
		   successors : asn list ref,
		   g : real ref,
		   f : real ref}

       exception Asn

       (* selectors *)

       fun depth bot = raise Asn
	 | depth (asn {depth=i, ...}) = i

       fun node_of bot = raise Asn
	 | node_of (asn {node=n, ...}) = n

       fun parent bot = raise Asn
	 | parent (asn {parent=bp, ...}) = bp

       fun changeParent bot _  = raise Asn
	 | changeParent n1 n2 =
	           (parent n1 := n2;
		    setParent (node_of n1) (node_of n2))
	   
       fun successors bot = raise Asn
	 | successors (asn {successors=s, ...}) = s

       fun g bot = raise Asn
	 | g (asn {g=x, ...}) = x

       fun f bot = raise Asn
	 | f (asn {f=x, ...}) = x

       (* printing and equality *)

       fun format_asn bot = Pretty.string "bottom"
	 | format_asn n = format (node_of n)

       fun put_asn os bot = output(os,"bottom ")
	 | put_asn os n = put_node os (node_of n)

       fun asn_eq bot bot = true
	 | asn_eq asn1 asn2 = eq_node (node_of asn1) (node_of asn2)


       structure D =
	   struct
	       type element = asn
	       val bottom = bot
	       fun eq (n1  : element) (n2 : element) =
		   eq_state (state (node_of n1))
		            (state (node_of n2))

               val format = format_asn

	       fun put os (e : asn) = 
		   put_node os (node_of e)
	   end

       structure AS : SET =  ListSet(D)
       structure PQ = PQueue(D)

       datatype 'a option = None | Some of 'a

   in
       fun A (h : state -> real)
	     (cost : label -> real)
	     (limit : int)	     
	     (prblm {start=n, space=sp, goal=goalp}) =

       let fun prior (asn {f=ref f1, ...}) (asn {f=ref f2, ...}) =
	            f1 < f2
	     | prior _ _ = false

	   val closed_nodes = ref AS.empty_set
	   val root = asn {depth=ref 0, 
			   node=n, 
			   parent=ref bot,
			   successors=ref [],
			   g=ref 0.0,
			   f=ref (h (state n))}

	   val open_nodes = PQ.empty_queue prior
	   
	   fun on_open (n : node) : (int * asn) option =
	       let val result = PQ.position open_nodes
		                     (fn asn1 => eq_node (node_of asn1) n)
	       in
		   Some result
	       end handle PQ.Position => None

	   fun on_closed (n : node) : asn option =
	       let fun loop s =
		   if AS.empty s then
		       None 
		   else
		       let val (asn1,rest) = AS.choose s 
		       in
			   if eq_node n (node_of asn1) then
			       Some asn1 
			   else
			       loop rest
		       end
	       in
		   loop (!closed_nodes)
	       end

	   fun propagate _ [] = ()
	     | propagate p1 
	       ((child as (asn {g=g1,
				f=f1,
				successors=ref s,
				parent=ref p2,...}))
		::tail) =
	   
	       let val g2 = !(g p1) + cost (label (node_of child))

		   (* update_parent:  if cost from new parent, g2, is
		    less than cost from old parent, g1, switch link to
		    the new parent and continue propagation *)

                   fun update_parent () : bool =
		       if g2 < (!g1) then
			   (changeParent child p1;
			    true)
		       else
			   false
	       in		       
		   if (asn_eq p1 p2) orelse (update_parent ()) then
		       (g1 := g2;
			f1 := h (state (node_of child)) + g2;
			propagate child s)   (* continue depth first *)
		   else
		       ();    (* current path is better, so stop *)
		   propagate p1 tail
	       end

	   fun process_successor best_node sn =
	   let val sng = !(g best_node) + cost (label sn)
	       val snf = h (state sn) + sng
	   in
	       case on_open sn of
		   Some (i,old) =>
		       (successors best_node :=
			          old::(!(successors best_node));
			if !(g old) <= sng then 
			    ()
			else
			    (changeParent old best_node;
			     setLabel (node_of old) (label sn);
			     depth old := !(depth best_node)+1;
			     g old := sng;
			     f old := snf;
			     PQ.repair open_nodes i))
		 | None =>
		       case on_closed sn of
			   Some old => 
			       (successors best_node :=
				        old::(!(successors best_node));
				if !(g old) <= sng then 
				    ()
				else
				    (changeParent old best_node;
				     setLabel (node_of old) (label sn);
				     depth old := !(depth best_node)+1;
				     g old := sng;
				     f old := snf;
				     propagate old (!(successors old))))
			 | None => 
			       let val successor = 
				   asn {depth=ref (1+(!(depth best_node))),
					node=sn,
					parent=ref best_node,
					successors=ref [],
					g=ref sng,
					f=ref snf}  
			       in
				   successors best_node := 
				        successor::(!(successors best_node));
				    PQ.enqueue open_nodes successor
			       end
	   end

	   fun loop () : node Sol.stream =
	       if PQ.empty open_nodes then
		   Sol.empty_stream ()
	       else
		   let val best_node = PQ.dequeue open_nodes
		   in
		       closed_nodes := AS.add best_node (!closed_nodes);
		       if goalp (state (node_of best_node)) then
			   Sol.stream (node_of best_node, 
				       Sol.delay (fn () => loop ()))
		       else if !(depth best_node) < limit then
			   (for_each (S.Succ.stream2list 
				          (expand (node_of best_node) sp))
			             (process_successor best_node);
			    loop ())
		       else 
			   loop ()
		   end
       in    
	   reset(); 
	   PQ.enqueue open_nodes root;
	   loop ()
       end  (* fun A *)
   end  (* local *)

end;  (* Search *)
        
