
signature SEARCH =
sig
    include SPACE

    (* SEARCH includes two stream substructures.  Succ, inherited from SPACE,
     are used to define the search space.  Sol, here, is used in strategies and
     solution streams.   Reason for this additional complexity:  When there 
     are only finitely many transitions from every node, it may make sense to 
     implement Succ using lists (i.e. with the ListStream functor).  But, to 
     avoid searching the whole space at once, Sol should probably always be 
     implemented using delayed evaluation, with the Stream functor. *)

    structure Sol : STREAM

    type problem
    val problem : node -> space -> (state -> bool) -> problem

    val start : problem -> node
    val problemSpace : problem -> space
    val goal : problem -> (state -> bool)

    type strategy 

    val strategy : (problem -> node Sol.stream) -> strategy

    val search : problem -> strategy -> node Sol.stream

    (* path returns the path from the starting node to the goal node.
The user can also use the parent selector to access the complete nodes
from the goal to the starting node, instead of just the labels.  This
function is intended to handle just the usual case.  *)

    val path : node -> label list

    (* Monitoring Facilities *)

    val trace : bool ref
    val traceFile : outstream ref
    val expanded : int ref
    
    (* Standard Strategies *)

    (* For hill climbing, the functional argument may define a partial 
       order on nodes.  In some domains, such as the eight puzzle, hill
       climbing may fail to find a solution, even though there is one, if 
       the order is strict, because it may be necessary to move "laterally", 
       to a position no better than the current one. *)
 
    val hillClimbing : (node -> node -> bool) -> strategy
    val depthFirst : int -> strategy
    val breadthFirst : int -> strategy
    val iterativeDeepening : int -> int -> int ->  strategy

    (* For best-first search, each node may have only finitely many
     successors.  One could transform infinite spaces into finite ones        
     by truncating the space after the first n successors, for some n. 
     The functional argument must define a strict order on nodes. *)           

    val bestFirst : int -> (node -> node -> bool) -> strategy 

    (* Algorithm A.  Applying A to a heuristic evaluation function which
     always underestimates the cost of reaching a goal node from the node 
     being evaluated yields the famous A* algorithm. *)

    val A : (state -> real) -> (label -> real) -> int -> strategy
end


