
(* 
Pretty printing.  From Paulson, ML for the Working Programmer.
Page 312.  A few things have been changed, to adjust them to my taste:

    1) The type T is now called "block"
    2) The Pretty module is now a structure, instead of functor, to make it 
easy to pretty print with blocks composed out of types from different structures.
    3) The print procedure no longer prints a newline after pretty printing
the block.   This allows blocks to be pretty printed "in line".

author: Tom Gordon 
*)


structure Pretty  : PRETTY =
    struct
	datatype block = 
	    Block of block list * int * int 	(*indentation, length*)
	  | String of string
	  | Break of int;			(*length*)

(* Add the lengths of the expressions until the next Break; if no
Break then include "after", to account for text following this block.
*)

	val indent = ref 4  (* spaces *)
	val linelength = ref 78

	fun breakdist (Block(_,_,len)::bs, after) = 
	         len + breakdist(bs, after)
	  | breakdist (String s :: bs, after) = 
	         size s + breakdist (bs, after)
	  | breakdist (Break _ :: bs, after) = 0
	  | breakdist ([], after) = after;

	fun print os b  =
	    let val space = ref (!linelength)

		fun blanks 0 = ()
		  | blanks n = (output(os," ");  space := !space - 1; 
				blanks(n-1))

		fun newline () = (output(os,"\n");  space := (!linelength))

		fun printing ([], _, _) = ()
		  | printing (b::bs, blockspace, after) =
		    (case b of
			 Block(bbs,indent,len) =>
			     printing(bbs, !space-indent, 
				      breakdist(bs,after))
		       | String s => (output(os,s);   space := !space - size s)
		       | Break len => 
			     if len + breakdist(bs,after) <= !space 
				 then blanks len
			     else (newline(); 
				   blanks((!linelength)-blockspace));
				 printing (bs, blockspace, after))
	    in  
		printing([b], (!linelength), 0)
	    end;

	fun length (Block(_,_,len)) = len
	  | length (String s) = size s
	  | length (Break len) = len;

	val string = String  and  break = Break;

	fun block (indent,bs) =
	    let fun sum([], k) = k
		  | sum(b::bs, k) = sum(bs, length b + k)
	    in  Block(bs,indent, sum(bs,0))  end;
    end;


