(* author: Tom Gordon
   based on: F.W. Burton, "An efficient functional implementation 
   of FIFO queues", Information Processing Letters, Vol 14, pp. 205-206, as 
   described in Paulson, "ML for the Working Programmer", pg. 239. *)

functor BurtonQueue (E : ELEMENT): QUEUE =
struct
	structure E = E
     	type queue =  (E.element list * E.element list)
	   
	exception EmptyQueue

        fun norm ([],tails) = (rev tails,[])
          | norm  q = q
	   
	val empty_queue = ([],[])

	fun empty ([],_) = true
	  | empty _ = false

	fun enqueue (heads,tails) x = 
	    norm(heads, x::tails)

	fun dequeue (x::heads,tails) = (x, norm (heads,tails))
	  | dequeue ([],_) = raise EmptyQueue

	fun front ([],_) = raise EmptyQueue
	  | front (x::_,_) = x

	fun put os ([],[]) = ()
          | put os (hd::tl,rest) = (E.put os hd; put os (tl,rest))
	  | put os q = put os (norm q)

        local open Pretty 
	in
	    fun format (heads, tails) =
		let fun f [] = []
		      | f [h] = [E.format h]
		      | f (h::t) = (E.format h)::(break 1)::(f t)
		    fun front () =
			if null heads then 
			    []
			else
			    (f heads)@[break 1]
		in
		   block(!indent, 
			 (string "(")::(front ())@(f tails)@[string ")"])
	       end       
	end
end;
	   
	   
		
		
		
