;;; -*- Syntax: ANSI-Common-Lisp; Package: CL-LIB; Mode: LISP -*-
;;; Copyright (C) 1994, 1992, 1985 by Bradford W. Miller, miller@cs.rochester.edu
;;; Unlimited non-commercial use is granted to the end user, other rights to
;;; the non-commercial user are as granted by the GNU LIBRARY GENERAL PUBLIC LICENCE
;;; version 2 which is incorporated here by reference.

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the Gnu Library General Public License as published by
;;; the Free Software Foundation; version 2.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; Gnu Library General Public License for more details.

;;; You should have received a copy of the Gnu Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
(in-package cl-lib)
;;;; RE/DFA package by Brad Miller (originally written as part of the RHET project)

;;; changes since 1985 other than bug fixes:
;;; 10/29/92 miller - change [] operator to () <grouping> so [] can be used for collection or range.
;;;                   change lispm characters to be appropriate for allegro, etc. (ascii)
;;;

;;;
;;; A string should be given as an arg to Convert-RE-To-DFA which will return a DFA.
;;; Calling CompatibleP on a string and the DFA will return T if the DFA accepts the string.
;;;
;;; Note that each character is treated individually. So, if you want to match FOO, you probably want to call
;;; (Convert-RE-To-DFA "F&O&O")
;;;
;;; The routine caches the dfas it generates. Call (clear-dfa-cache) to clear the cache.

;;;
;;;
(defconstant possible-literals
	     '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M
	       #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z
	       #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m
	       #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z
	       #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9
	       #\< #\> #\- #\= #\! #\% #\/ #\SPACE)
  "These are the possible values for literals in a RE.")

(defconstant +operators+ '(#\& #\( #\) #\[ #\] #\| #\/ #\, #\. #\$ #\^))

(defparameter *debug* nil)	  ;if t make things easier to analyze
(defvar *local-re*)

(defvar *cached-dfas* nil
  "A hash table consisting of the dfa's we've encountered")

(defun clear-dfa-cache ()
    (declare (optimize (speed 3) (safety 0)))
  (cond
    ((hash-table-p *cached-dfas*)
     (clrhash *cached-dfas*))
    (t
     (setq *cached-dfas* (make-hash-table :test #'equal)))))

(eval-when (load eval)
  (clear-dfa-cache))

;;;
;;;
(defun compatiblep (string1 dfa)
  "compatiblep returns t if the DFA could have generated the string.
Get the compiled DFA from convert-RE-to-DFA."
  (declare (type string string1)
	   (type compiled-function dfa)
	   (optimize (speed 3) (safety 0)))

  ;; this is easy, given the dfa... 
  (funcall dfa string1))

(defun expand-re (re)
  (let (result)
    (while re
      (cond
       ((member (car re) possible-literals)
        (push (car re) result)
        (if (member (cadr re) possible-literals)
            (push #\& result)))
       (t
        (push (car re) result)))
      (pop re))
    (nreverse result)))

;;;
;;;
(defun convert-re-to-dfa (re)
  "This function converts REs to DFAs (compiled)."
  (declare (type string re)
	   (optimize (speed 3) (safety 0)))

  ;; We have to take the RE supplied and precidence parse it into a tree, then
  ;; convert that into an NFA with epsilon moves, then convert that to an NFA without
  ;; epsilon moves, and finally convert THAT into a DFA. The function returned should be
  ;; A compiled lisp function implementing the DFA and accepting one argument (a
  ;; string). It either returns T or NIL depending on whether the string matches
  ;; the DFA or not.
  ;;
  ;; For the moment, we will accept #\&, slash, epsilon, and * as operators, other symbols (A-Z and some
  ;; non alpha chars) as literals. We can add 'not' (or +) later as time permits.
  ;;
  ;; This is hard!
  ;;
  ;; Find Expression will return an NFA constructed from the RE. It does NOT return a lisp
  ;; function. Eliminate epsilon will remove epsilon rules from that NFA. It does NOT return a lisp
  ;; function. NFA to DFA will convert the NFA w/o epsilon rules to a DFA. It does NOT return a
  ;; lisp function. Optimize DFA will optimize the DFA. It does NOT return a lisp function. DFA to
  ;; function turns my representation of a DFA into a lisp function of one arg (the thing to match)
  ;; This is then compiled.
  (cond
    ((gethash re *cached-dfas*))
    (t
     (let ((*local-re* (expand-re (coerce (format nil "~a" re) 'list))))
       (declare (special *local-re*))     ;to use the HU algorithm, need these.

       (cond
	 (*debug*
	  (gensym 0)))			       ;reset counter
       (setf (gethash re *cached-dfas*)
	     (compile nil
		      (dfa-to-function
			(optimize-dfa
			  (nfa-to-dfa
			    (eliminate-epsilon
			      (find-expression)))))))))))

;;;; non-exported functions...

;;;
;;; Structure of our internal rep of the rules
;;;
(defstruct rule
  (from nil :type symbol)
  (on nil :type character)
  (to nil :type symbol))

(defstruct fa
  (start nil :type symbol)
  (end nil :type (or list symbol)) ;could be a list of symbols
  (rules nil :type list))	   ;list of rule structures.

;;;
;;;

(defun find-expression ()
  "This function is from HU _Introduction to Automata Theory, Languages, and Computation_
NFAs look like (Start End (rules)) where rules are ((from char-to-match).to)"
  (declare (optimize (speed 3) (safety 0)))

  (let ((retval (find-product)))    ;what we will return
    (loop
      (cond
	((not (eql (car *local-re*) #\|))
	 (return))
	(t
	 (pop *local-re*)
	 (let ((newval (find-product))	   ;new nfa we will or in
	       (newstart (gensym "q"))
	       (newend (gensym "q")))
	   (setq retval
		 (make-fa :start newstart :end newend
			  :rules (list*
				   (make-rule :from newstart :on #\/ :to (fa-start retval))
				   (make-rule :from newstart :on #\/ :to (fa-start newval))
				   (make-rule :from (fa-end retval) :on #\/ :to newend)
				   (make-rule :from (fa-end newval) :on #\/ :to newend)
				   (nconc (fa-rules retval) (fa-rules newval)))))))))
    retval))

;;;
;;;
(defun find-product ()
  "this function is also from hu"
  (declare (optimize (speed 3) (safety 0)))

  (let ((retval (find-term)))	    ;what we will return

    (loop
      (cond
	((not (eql (car *local-re*) #\&))
	 (return))
	(t
	 (pop *local-re*)
	 (let ((newval (find-term)))	   ;new nfa we will and in
	   (setq retval (make-fa :start (fa-start retval) :end (fa-end newval)
				 :rules (list*
					  (make-rule :from (fa-end retval) :on #\/ :to (fa-start newval))
					  (nconc (fa-rules retval) (fa-rules newval)))))))))
  
  retval))

;;;
;;;
(defun find-term ()
  "this too is from HU
Represent an NFA a startnode, endnode(s) rules..."
  (declare (optimize (speed 3) (safety 0)))

  (let ((retval nil));what we will return
    (cond
      ((member (car *local-re*) possible-literals)
       (setq retval
	     (make-fa :start (gensym "q") :end (gensym "q")))   ;build nfa
       (setf (fa-rules retval) (list
				 (make-rule :from (fa-start retval)
					    :on (car *local-re*)
					    :to (fa-end retval))))
       (pop *local-re*))
      ((eql (car *local-re*) #\()
       (pop *local-re*)
       (setq retval (find-expression))
       (cond
	 ((eql (car *local-re*) #\))
	  (pop *local-re*))
	 (t
	  (error "re is not properly formed")))))
    (loop
     (cond
       ((eql (car *local-re*) #\*)
	(pop *local-re*)
	(let ((origstart (fa-start retval))
	      (origend (fa-end retval))
	      (newstart (gensym "q"))
	      (newend (gensym "q")))
	  (setq retval (make-fa :start newstart :end newend
				:rules (list*
					 (make-rule :from origend :on #\/ :to origstart)
					 (make-rule :from newstart :on #\/ :to origstart)
					 (make-rule :from newstart :on #\/ :to newend)
					 (make-rule :from origend :on #\/ :to newend)
					 (fa-rules retval))))))
       (t
	(return))))
     
    retval))

;;;
;;;
(defun eliminate-epsilon (nfa-with-epsilon)
  "this function takes the internal rep of an nfa and eliminates the epsilon rules, using HU's algorithm.
Note, this function may return more than one end state in it's NFA rep."
  (declare (type fa nfa-with-epsilon)
	   (optimize (speed 3) (safety 0)))

  ;; given the input NFA with epsilon rules, the basic thing is to be able to compute the epsilon
  ;; closure of each state. Note that if the epsilon closure of the original's start state includes
  ;; the end state, we must make sure that we allow the new NFA's start and finish state to both be
  ;; "finish" states.

  (let* ((start-state (fa-start nfa-with-epsilon))
	 (old-finish (fa-end nfa-with-epsilon))
	 (rules (fa-rules nfa-with-epsilon))
	 (states-used (delete-duplicates (find-states rules)))
	 (e-closures (make-hash-table :test #'eq :size (list-length states-used) :rehash-threshold 0.99)))
    ;; first, calculate the e-closures for each state.
    (calculate-e-closures rules e-closures states-used)
    (make-fa
      :start
      start-state       ;start state will be the same
      :end
      (cond ((member old-finish (gethash start-state e-closures))
		  (list start-state old-finish))
		 (t
		  (list old-finish)))			
      ;; for each state/literal pair, compute the new transition(s)
      :rules
      (delete nil (mapcan
		    #'(lambda (x) (find-lit-transitions x (important-rules rules) e-closures))
		    states-used)))))	 ;for each state, look up all the transitions based on the literals used.

;;;
;;;
(defun find-lit-transitions (state rules e-hash-table)
  "this function takes a particular state, and returns the state transition
rules from this state to any other state."
  (declare (type atom state)
	   (type list rules)
	   (type hash-table e-hash-table)
	   (optimize (speed 3) (safety 0)))

  ;; for this state, and all states in it's epsilon closure, look up rules in RULES
  ;; that use a literal to move, and return those rules.
  (delete nil
	  (mapcan
	    #'(lambda (x) (test-match-rule state x e-hash-table))
	    rules)))

;;;
;;;
(defun test-match-rule (STATE XRULE E-HASH-TABLE)
  "this function is the guts of checking for rules."
  (declare (type atom state)
	   (type hash-table e-hash-table)
	   (type rule xrule)
	   (optimize (speed 3) (safety 0)))

  ;; if any of the states are the source node in rule, then return a copy of the rule
  ;; substituting state for whatever state used to be present in from.
  (cond
    ((and
       (not (eql (rule-on xrule) #\/))
       (member (rule-from xrule) (cons state (gethash state e-hash-table))))
     (mapcar
       #'(lambda (x) (make-rule :from state :on (rule-on xrule) :to x))
       (cons (rule-to xrule) (gethash (rule-to xrule) e-hash-table))))
    (t nil)))

;;;
;;;
(defun important-rules (rules)
  "this function takes a list of rules and ruturns only those that contain a literal."
  (declare (type list rules)
	   (optimize (speed 3) (safety 0)))

  (let ((result nil))
    (dolist
      (x rules)

      (cond
	((not (eql #\/ (rule-on x)))
	 (push x result))))
    result))

;;;
;;;
(defun calculate-e-closures (rules e-hash-table states)
  "This function takes a list of rules and finds all e-closures for the rules.
It expects to add these to a passed hash table. (destructive!!)"
  (declare (type list rules states)
	   (type hash-table e-hash-table)
	   (optimize (speed 3) (safety 0)))

  ;; for each rule in the rules passed, add the to node to the hash table for the from node, if epsilon
  ;; move. Then, recursively add nodes from the epsilon closures of each node in my closure, but don't
  ;; add a node twice. We can do this by deleting fully expanded nodes from the passed list of
  ;; states. note that we could not have (as a side effect of the NFA created) a node with an epsilon
  ;; move to another node with an epsilon move back to it. So we shouldn't have to do anything too
  ;; hard....
  (mapc #'(lambda (x) (ec-internal x e-hash-table)) rules) ;process the table first
  (let ((*states* (copy-list states)))			   ;I HAD TO DO IT!!
    ;;it has to be special!!!  (reason is that subroutines may modify,	and we want THIS ONE
    ;;to be the one modified). Passing both up and down would be worse, and inefficient, too.
    (declare (special *states*))
    ;;we don't iterate over the *states* because each call to E-CLOSE may eliminate more than one
    ;;state in *states*
    (loop 
      (cond	  
	((null *states*)
	 (return))
	(t
	 (e-close (car *states*) e-hash-table)))))   ;call the recursive function to take care of it.
  (values))							

;;;
;;;
(defun ec-internal (xrule e-hash-table)
  "This function processes the rules into the e-hash-table, 
by checking to see if a rule is an epsilon move, and if so, adding the to node
to the hash of the from node."
  (declare (type rule xrule)
	   (type hash-table e-hash-table)
	   (optimize (speed 3) (safety 0)))

  (cond
    ((eql (rule-on xrule) #\/)
     (setf
       (gethash (rule-from xrule) e-hash-table)
       (cons (rule-to xrule) (gethash (rule-from xrule) e-hash-table)))))
  (values))
  
;;;
;;;
(defun e-close (state e-hash-table)
  "this function processes the states passed, recursively, and adds their complete closure to the table.
It relies on the fact that there are no cycles of epsilon shifts."
  (declare (type list *states*)
	   (special *states*)
	   (type hash-table e-hash-table)
	   (optimize (speed 3) (safety 0)))

  (cond
    ((member state *states*);more to do....
     (setq *states* (delete state *states*))
     (setf (gethash state e-hash-table)
	   (append (gethash state e-hash-table)
		   (car (delete nil
				(mapcar #'(lambda (x) (e-close x e-hash-table))
					(gethash state e-hash-table))))))))

  (gethash state e-hash-table))

;;;
;;;
(defun find-states (rules)
  "this function scans a list of NFA or DFA rules, and returns a list of all the states present."
  (declare (type list rules)
	   (optimize (speed 3) (safety 0)))

  (cond
    ((null rules)
     nil)
    (t
     (cons (rule-from (car rules))     ;start state of rule
	   (cons (rule-to (car rules))	    ;end state of rule
		 (find-states (cdr rules)))))))	  ;rest of rules
     
;;;
;;;
(defun nfa-to-dfa (nfa-without-epsilon)
  "this function takes the internal rep of an NFA without epsilon rules and turns it into a DFA
using HU's algorithm."
  (declare (type fa nfa-without-epsilon)
	   (optimize (speed 3) (safety 0)))

  ;; starting with the nfa's start state, we create a rule in our DFA for each state or
  ;; combinations of states a move in our NFA w/o epsilons can get us to on a certain input. If we
  ;; are working on a "combination" state, our moves out are regulated by the union of the moves out
  ;; the of the individual component states in the NFA.

  ;; we return an DFA which is of essentially the same representation as the NFA we started with.
  (let* ((*nfa-rules* (fa-rules nfa-without-epsilon))
	 (nfa-end (fa-end nfa-without-epsilon))
	 (nfa-start (fa-start nfa-without-epsilon))
	 (*nfa-literals* (remove-duplicates (find-literals *nfa-rules*)))
	 ;; the number of states in the dfa could be 2**number of states in the nfa, but
	 ;; typically is much smaller. estimate it at the number of nfa rules.
	 (*dfa-rules* (make-hash-table :test #'equal :size (list-length *nfa-rules*) :rehash-threshold 0.99))
	 (dfa-finish nil);end states for our dfa
	 (dfa-start (nfa-state-to-dfa-state nfa-start))
	 (*dfa-states*
	   (make-hash-table :test #'eq :size (list-length *nfa-rules*) :rehash-threshold 0.99))) ;the states in our dfa
    (declare (special *dfa-rules* *nfa-rules* *dfa-states* *nfa-literals*))	 ;sub funs will update...
										      
    ;; get the closure of the start state, which will recursively get the closure of all states the
    ;; start state can reach (small optimization)
    (lit*-closure nfa-start)
    
    ;; process the states of our nfa for potential end-states
    (maphash #'(lambda (key val)
		 (cond ((intersection val nfa-end)
			(setq dfa-finish (cons key dfa-finish)))))
	     *dfa-states*)
    
    ;; only dump the dfa states accessible from the start state (many were subs and won't be).
    (make-fa
      :start dfa-start
      :end dfa-finish
      :rules (follow-dfa-table dfa-start *dfa-rules*))))

;;;
;;;
(defun lit*-closure (nfa-symbol-to-close)
  "THis function takes a symbol and a list of literals and finds it's closure in the nfa passed,
updating our DB of DFA rules and states."
  (declare (type symbol nfa-symbol-to-close)
	   (type list *nfa-literals*)
	   (type hash-table *dfa-states*)
	   (special *dfa-states* *nfa-literals*)	;
	   (optimize (speed 3) (safety 0)))

  (let ((dfa-symbol-to-close (nfa-state-to-dfa-state nfa-symbol-to-close)))
    
    (cond
      ((gethash dfa-symbol-to-close *dfa-states*);already computed
       nil)
      (t
       ;; for each literal we know about, look up all the states we can get to from the passed
       ;; state
       (mapc
	 #'(lambda (x)
	     (lit-closure nfa-symbol-to-close dfa-symbol-to-close x))
	 *nfa-literals*))))
  (values))

;;;
;;;
(defun lit-closure (nfa-symbol-to-close dfa-symbol-to-close literal)
  "this function is like lit*-closure, but for a single literal (called by lit*-closure)."
  (declare (type symbol nfa-symbol-to-close)
	   (type atom dfa-symbol-to-close)
	   (type character literal)
	   (type list *nfa-rules*)
	   (special *nfa-rules*)
	   (optimize (speed 3) (safety 0)))

  (let* ((state-set (nfa-accessible nfa-symbol-to-close literal *nfa-rules*))
	 (new-state (nfa-state-to-dfa-state state-set)))

    (cond (state-set ;if there's something to expand on...
	   (dfa-from-nfa-internal new-state state-set dfa-symbol-to-close literal))))
  (values))

;;;
;;;
(defun dfa-unify-states (union-state literal)
  "this function takes a dfa state that is the union of many NFA states and a literal, and creates
a transition rule for the DFA union state of all the NFA states to a new set of states. (each
NFA state will already be interned as separate DFA transition rules, so we essentially look each
up, and then set the passed DFA state to be a transition on this literal to the union of all of
them."
  (declare (type atom union-state)
	   (type character literal)
	   (type hash-table *dfa-states* *dfa-rules*)
	   (special *dfa-states* *dfa-rules*)
	   (optimize (speed 3) (safety 0)))

  (let* ((new-union-state-set
	   (delete-duplicates
	     (remove nil
		     (mapcar ;the transition rules for all states in this closure
		       #'(lambda (x) (gethash x *dfa-rules*))
		       (mapcar	  ;keys of state/literal pairs
			 #'(lambda (x) (list (nfa-state-to-dfa-state x) literal))
			 (gethash union-state *dfa-states*))))))
	 (new-union-state (nfa-state-to-dfa-state new-union-state-set)))

    (cond (new-union-state-set		  ;were there any for this literal?
	   (dfa-from-nfa-internal new-union-state new-union-state-set union-state literal))))
  (values))
    
;;;
;;;
(defun dfa-from-nfa-internal (new-union-state new-union-state-set union-state literal)
  "setup dfa rule
internal function that takes the new DFA state, the list of NFA states it represents, the
literal used for the transition, and the DFA state we are going from,
and updates the DFA rules appropriately."
  (declare (type atom new-union-state union-state)
	   (type list *nfa-literals*)
	   (type character literal)
	   (type hash-table *dfa-states* *dfa-rules*)
	   (special *nfa-literals* *dfa-states* *dfa-rules*)
	   (optimize (speed 3) (safety 0)))

  ;; first, store the translation of statename to set of dfa states it represents, if we need to.
  (cond
    ((null (gethash new-union-state *dfa-states*))
     (setf (gethash new-union-state *dfa-states*) new-union-state-set)
     (setf (gethash (list union-state literal) *dfa-rules*) new-union-state)
     ;; and get the closure of this new state
     (mapc #'(lambda (x) (lit*-closure x)) new-union-state-set)
     (mapc #'(lambda (x) (dfa-unify-states new-union-state x)) *nfa-literals*))
    (t
     ;; since it was already there, just add the transition.
     (setf (gethash (list union-state literal) *dfa-rules*) new-union-state)))
  (values))

;;;
;;;
(defun nfa-accessible (state literal nfa-rules)
  "this function takes an nfa state, and a literal, and returns all the states that can be gotten
to via the literal from that state."
  (declare (type symbol state)
	   (type character literal)
	   (type list nfa-rules)
	   (optimize (speed 3) (safety 0)))

  (let ((result nil))
    (dolist (xrule nfa-rules)
      (cond
	((and (eq state (rule-from xrule)) (eql literal (rule-on xrule)))
	 (push (rule-to xrule) result))))
    result))

;;;
;;;
(defun find-literals (rules)
  "this function scans a list of nfa w/o epsilon move rules, and returns a list of all the literals present."
  (declare (type list rules)
	   (optimize (speed 3) (safety 0)))

  (cond
    ((null rules)
     nil)
    (t
     (cons (rule-on (car rules)) (find-literals (cdr rules))))))

;;;
;;;
(defun follow-dfa-table (from-node dfa-table &optional (*history* nil))
  "This function returns the real rules from a DFA rule hashtable 
by tree walking it starting with the start node. (it isn't really a tree, so...)"
  (declare (type symbol from-node)	  ;chart from this node
	   (type hash-table dfa-table)	  ;the rules
	   (type list *history*)	  ;which nodes should not be followed.
	   (special *history*)
	   (optimize (speed 3) (safety 0)))

  (let ((retval nil))			  ;list of rules
    (cond
      ((member from-node *history*)
       nil)				  ;don't follow, it's already output.
      (t
       (push from-node *history*)
       (maphash #'(lambda (key val)
		    (cond
		      ((eq (car key) from-node)
		       (push (make-rule :from (car key) :on (cadr key) :to val) retval)
		       (setq retval (nconc (follow-dfa-table val dfa-table *history*) retval)))
		      (t nil)))
		dfa-table)))
    retval))

;;;
;;;
(defun nfa-state-to-dfa-state (nstates)
  "takes an nfa state or set of states and returns the dfa state that covers it. 
(intern if necessary)"
  (declare (optimize (speed 3) (safety 0)))

  (cond ((atom nstates)
	 (intern (concatenate 'simple-string "d" (subseq (string nstates) 1)) (find-package 'user)))
	(t
	 (intern (apply #'concatenate
			(list* 'simple-string "d"
			       (stable-sort
				 (mapcar #'(lambda (x) (subseq (string x) 1)) nstates)
				 #'string<))) (find-package 'user)))))

;;;
;;;
(defun optimize-dfa (dfa)
  "this function takes the internal rep of a DFA and optimizes it (eliminates duplicate states)
It uses the algorithm described on pg. 70 of HU. to find the inequivalent states in the FA."
  (declare (type fa dfa)
	   (optimize (speed 3) (safety 0)))

  dfa) ;for the moment, just return it

;  (let* ((q (remove-duplicates (find-states (fa-rules dfa))))	;position of state in this list is critical!
;	 (mark-array (make-array (list (list-length q) (list-length q)) :initial-element nil))
;	 (f (fa-end dfa))
;	 (q-f (set-difference q f))
;	 (trans (fa-rules dfa))
;	 (alpha (find-literals trans))
;	 )
;
;    (do ((q1 q (cdr q1))					;assign to each atom in q a value.
;	 (n1 0 (+ n1 1)))
;	((null q1))
;      (set (car q1) n1))
;
;    ;;mark pairs of final and nonfinal states which can't possibly be equivalent.
;    (dolist (p1 f)
;      (dolist (q1 q-f)
;	(mark p1 q1 mark-array)))
;
;    ;; for each pair of distinct states (p,q) in fxf or (q-f)x(q-f) do
;    (mapc
;      #'(lambda (p1 q1)
;	  ;; if for some input symbol a, ((p, a), (q, a)) is marked
;	  (cond
;	    ((plusp
;	       (reduce #'+
;		       (mapcar
;			 #'(lambda (x)
;			     (apply #'aref
;				    `(mark-array 
;				       ,(sort
;					  (eval (find-newstate p1 x trans))
;					  (eval (find-newstate q1 x trans))
;					  #'<))))
;			 alpha)))
;	     ;; mark (p, q), and recursively mark all unmarked pairs on the list for (p,q) and on
;	     ;; the lists of other pairs that are marked at this step
;	     (mark p1 q1 mark-array)
;	     (mark-list p1 q1 mark-array))
;	     
;	    (t
;	     (mapcar
;	       #'(lambda (x)
;		   (cond
;		     ((not (eq (find-newstate p1 x trans) (find-newstate q1 trans)))
;		      (push (list p1 q1)
;			    (apply #'aref
;				   `(mark-array
;				   ,(sort
;				      (eval (find-newstate p1 x trans))
;				      (eval (find-newstate q1 x trans))
;				      #'<)))))))
;	       alpha))))
;      
;      (append (cross f f) (cross q-f q-f)))
;
;   (dotimes (q1 (listlength q)) ;assign to each atom in q it's equivalence class
;      (do ((q2 (+ q1 1) (+ q2 1)))
;	  ((eql q2 (listlength q)))
;	(cond
;	  ((not (eq 't (car (aref mark-array q1 q2))))
;	   (set (nth q1 q) (cons (nth q2 q) (eval (nth q1 q))))
;	   (set (nth q2 q) (cons (nth q1 q) (eval (nth q2 q))))))))
;      
;
;    ;; now that the table is set up, output the new dfa, combining the redundant states
;    (make-fa
;      :start (fa-start dfa)
;      :end f
;      :rules (minimal-rules trans mark-array))))
;  
;;;;
;;;; for each pair stored in list for p,q passed, mark it.
;;;;
;(defun mark-list (p q mark-array)
;  (declare (type atom p q)
;	   (type array mark-array)
;	   (optimize (speed 3) (safety 0)))
;
;  (dolist (instance (apply #'aref `(mark-array ,(sort (eval p) (eval q) #'<))))
;    (cond
;      ((and (not (eq (car instance) 't)) (not (eq instance nil)))
;       (apply #'mark (append instance mark-array))
;       (apply #'mark-list (append instance mark-array))))))	 ;recursively
;						  
;;;;
;;;; this function marks the appropriate point in the array (sorts the evaled args, etc.)
;;;;
;(defun mark (x y xarray)
;  (declare (type atom x y)
;	   (type array mark-array)
;	   (optimize (speed 3) (safety 0)))
;
;  (push 't (apply #'aref `(mark-array ,(sort (eval x) (eval y) #'<)))))
;
;;;
;;;
(defun find-newstate (state literal rules)
  "this function takes a state and a literal, and finds the newstate in the list of rules (dfa simulator)."
  (declare (type atom state)
	   (type character literal)
	   (optimize (speed 3) (safety 0)))

  (dolist (xrule rules)
    (cond
      ((and (eq state (rule-from xrule))
	    (eql literal (rule-on xrule)))
       (return (rule-to xrule))))))

;;;;
;;;; this function takes two lists, and returns the cross product. it does not return duplicates.
;;;;
;(defun cross (x y)
;  (declare (type list x y)
;	   (optimize (speed 3) (safety 0)))
;
;  (let ((result nil))
;    (dolist (x1 x)
;      (dolist (y1 y)
;	(cond ((not (eq x1 y1))
;	       (push (list x1 y1) result)))))
;    (remove-duplicates result)))

;;;
;;;
(defun dfa-to-function (dfa)
  "this function takes the internal rep of a dfa and converts it to a lisp function.
Right now, just return a tree walker. In the future, if I am sufficiently ambitious, I
could compile the DFA into real lisp code."
  (declare (type fa dfa)
	   (optimize (speed 3) (safety 0)))

  `(lambda (x)
     (declare (type string x)
	      (optimize (speed 3) (safety 0)))

     ;; turn the passed dfa into a function that returns T if the string X matches, and NIL
     ;; if it doesn't (i.e. if it takes us to a final state or not)
     (let ((current-state (quote ,(fa-start dfa)))
	   (finish-states (quote ,(fa-end dfa)))
	   (rules (quote ,(fa-rules dfa)))
	   (current-input x))

       (loop
       (cond ((string= "" current-input)
	      (cond
		((member current-state finish-states)
		 (return 't))	 ;done, success!
		(t
		 (return nil)))) ;lossage!
	     (t
	      (cond
		((setq current-state
		       (find-newstate current-state
				      (char current-input 0)
				      rules))  ;no, sim step on the first char
		 (setq current-input (subseq current-input 1)))
		(t
		 (return nil)))))))))	       ;lossage!
