;;; -*- Mode: LISP; Syntax: ansi-common-lisp; Package: CL-LIB; Base: 10 -*-
;;; 
;;; Copyright (C) 1994, 1993, 1992 by Bradford W. Miller, miller@cs.rochester.edu
;;;                                and the Trustees of the University of Rochester
;;; Unlimited non-commercial use is granted to the end user, other rights to
;;; the non-commercial user are as granted by the GNU LIBRARY GENERAL PUBLIC LICENCE
;;; version 2 which is incorporated here by reference.

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU Library General Public License as published by
;;; the Free Software Foundation; version 2.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU Library General Public License for more details.

;;; You should have received a copy of the GNU Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;
;;; The following is contributed by miller@cs.rochester.edu

(in-package cl-lib)

;; additions for other versions of lisp are welcome!
(defmacro macro-indent-rule (symbol what)
  #-lep (declare (ignore symbol what))
  #+lep                         ;must  be 4.1 or later, have lep
  `(add-initialization ,(format nil "lep init for ~A" symbol)
                       '(lep::eval-in-emacs ,(concatenate 'string "(put '" (string-downcase (string symbol)) " 'fi:lisp-indent-hook " (string-downcase (format nil "'~S)" what))))
                       '(:lep))
  )

;;; yep, verrry similar to the one in zetalisp, so on the symbolics we use that one instead.
#+SYMBOLICS (EVAL-WHEN (COMPILE LOAD EVAL) (SETF (SYMBOL-FUNCTION 'MAPATOMS) #'ZL:MAPATOMS))
#-SYMBOLICS
(DEFUN MAPATOMS (FUNC &OPTIONAL (PACKAGE *PACKAGE*) (INHERITED-SYMBOLS-TOO T))
  "Maps the passed function over all symbols in the package, and if inherited-symbols-too is non-nil, then
over those symbols as well. Note that the function may be called >once on a symbol."
  (DO-SYMBOLS (SYM PACKAGE)
    (IF (OR INHERITED-SYMBOLS-TOO
	    (EQ PACKAGE (SYMBOL-PACKAGE SYM)))
	(FUNCALL FUNC SYM))))

;; This function was inspired by a similar function on the Symbolics lisp machine, which was 
;; used in the Rhet system. In fact, if we are on a symbolics, use that one. It will return more values, but
;; that's ok. Plus we get the input editor "for free :-)"
;; collect the characters until we hit a delimiter or eof, then turn it into a
;;  string and return!
#-SYMBOLICS
(DEFUN READ-DELIMITED-STRING (DELIMITERS &OPTIONAL (STREAM *STANDARD-INPUT*) (EOF-ERROR-P T) EOF-VALUE)
  "Read a stream until one of the delimiters (a list of characters) is found. Returns the characters so
read until the delimiter as a string, plus the additional values: EOF-VALUE, which is as passed if eof was
reached, and the delimiter that caused termination of the string. If EOF-ERROR-P is non-nil (the default),
then an EOF causes an error to be signalled instead of returning EOF-VALUE."
  (DECLARE (TYPE LIST DELIMITERS)
	   (TYPE STREAM STREAM))
  (LET (CHAR-LIST)
    (DECLARE (DYNAMIC-EXTENT CHAR-LIST))
    (DO ((READ-CHAR (READ-CHAR STREAM EOF-ERROR-P :EOF) (READ-CHAR STREAM EOF-ERROR-P :EOF)))
	((OR (MEMBER READ-CHAR DELIMITERS) (EQ READ-CHAR :EOF))
	 (VALUES (COERCE (NREVERSE CHAR-LIST) 'STRING)
                 (IF (EQ READ-CHAR :EOF) EOF-VALUE) READ-CHAR))
      (PUSH READ-CHAR CHAR-LIST))))

;;
(DEFMACRO DEFCLASS-X (TYPE SUPERTYPES SLOTS . STUFF)
  "Extended defclass, also creates a TYPE-P function and MAKE-TYPE function, like defstuct did."
  `(eval-when (compile load eval)
     (DEFCLASS ,TYPE ,SUPERTYPES ,SLOTS ,@STUFF)
     (DEFUN ,(INTERN (CONCATENATE 'STRING (STRING TYPE) "-P")) (TERM)
       (TYPEP TERM ',TYPE))
     (DEFUN ,(INTERN (CONCATENATE 'STRING "MAKE-" (STRING TYPE))) (&REST ARGS)
       (APPLY 'MAKE-INSTANCE ',TYPE ARGS))))

(macro-indent-rule defclass-x (like defclass))

#+ALLEGRO-V4.0
(EVAL-WHEN (COMPILE LOAD EVAL)
  (setf (symbol-function 'COMMON-LISP:hash-table-size) (symbol-function 'excl::hash-table-buckets)
	(symbol-function 'COMMON-LISP:hash-table-test) (symbol-function 'excl::hash-table-kind)))
;;
(DEFUN COPY-HASH-TABLE (OLD-HASH-TABLE)
  #+SYMBOLICS (CLI::COPY-TABLE OLD-HASH-TABLE)
  #-SYMBOLICS
  (let ((new-hash (make-hash-table
		    :test (hash-table-test old-hash-table)
		    :size (hash-table-size old-hash-table)
		    :rehash-size (hash-table-rehash-size old-hash-table)
		    :rehash-threshold (hash-table-rehash-threshold old-hash-table))))
    (maphash #'(lambda (key entry)
		 (setf (gethash key new-hash) entry))
	     old-hash-table)
    NEW-HASH))

;;; this is to support a field in a clos structrure called "flags", which is bit encoded. The testname can be used to see if the
;;; bit (defined by flagname - a constant) is set. It can also be setf to set or clear it. The type is the type of structure this
;;; test will handle, allowing multiple encodings of the flags field for different structures.
(DEFMACRO DEFFLAG (TESTNAME (TYPE FLAGNAME))
  `(PROGN (DEFMETHOD ,TESTNAME ((TERM ,TYPE))
	    (LOGTEST ,FLAGNAME (FLAGS TERM)))
	  (DEFMETHOD (SETF ,TESTNAME) (NEW-FLAG (TERM ,TYPE))
	    (SETF (FLAGS TERM) (IF NEW-FLAG
				   (LOGIOR (FLAGS TERM) ,FLAGNAME)
				   (LOGAND (FLAGS TERM) (LOGNOT ,FLAGNAME)))))))

;; similar to above, but for defstruct type thingos; we assume the accessor is "typename"-flags

;; fix to make sure we use typename in constant to avoid name collisions 7/30/92 bwm
(DEFMACRO DEFFLAGS (TYPENAME &BODY FLAGNAMES)
  (LET ((ACCESSOR (INTERN (FORMAT NIL "~A-FLAGS" TYPENAME)))
	(VARNAME1 (GENSYM))
	(VARNAME2 (GENSYM)))
    (DO* ((FLAG FLAGNAMES (CDR FLAG))
	  (FUNNAME (INTERN (FORMAT NIL "~A-~A-P" TYPENAME (CAR FLAG))) (INTERN (FORMAT NIL "~A-~A-P" TYPENAME (CAR FLAG))))
          (CONSTNAME (INTERN (FORMAT NIL "+~A-~A+" TYPENAME (CAR FLAG))) (INTERN (FORMAT NIL "+~A-~A+" TYPENAME (CAR FLAG))))
	  (COUNT 1 (* COUNT 2))
	  (CODE))
	((NULL FLAG) `(PROGN ,@CODE))
      (PUSH `(DEFSETF ,FUNNAME (,VARNAME1) (,VARNAME2)
	       `(SETF (,',ACCESSOR ,,VARNAME1) (IF ,,VARNAME2
						(LOGIOR (,',ACCESSOR ,,VARNAME1) ,',constname)
						(LOGAND (,',ACCESSOR ,,VARNAME1) (LOGNOT ,',constname)))))
	    CODE)
      (PUSH `(DEFUN ,FUNNAME (,VARNAME1)
	       (LOGTEST ,constname (,ACCESSOR ,VARNAME1)))
	    CODE)
      (PUSH `(DEFCONSTANT ,constname ,COUNT) CODE))))

(macro-indent-rule defflags 1)

;;; fix to not use "declare" options in non-let clause - 3/14/91 bwm
;; fix for optimized expansion when condition is already known (constant) nil or non-nil.

(DEFMACRO LET-MAYBE (CONDITION BINDINGS &BODY BODY)
  "Binds let arguments only if condition is non-nil, and evaluates body in any case."
  (cond
   ((null condition)
    `(PROGN ,@(IF (EQ (CAAR BODY) 'DECLARE) (CDR BODY) BODY)))
   ((eq condition t)
    `(let ,bindings ,@body))
   (t                                   ;defer to runtime
    `(IF ,CONDITION
         (LET ,BINDINGS
           ,@BODY)
       (PROGN ,@(IF (EQ (CAAR BODY) 'DECLARE) (CDR BODY) BODY))))))

(macro-indent-rule let-maybe ((1 1 quote) (0 2 1)))

(DEFUN ROUND-TO (NUMBER &OPTIONAL (DIVISOR 1))
  "Like Round, but returns the resulting number"
  (* (ROUND NUMBER DIVISOR) DIVISOR))
	  
(defun factorial (n)
  "Compute the factorial of an integer"
  (cond ((minusp n)
	 (cerror "Compute -(~D!) instead" "I can't do -~D!" n n)
	 (factorial (- n)))
	(t
	 (do ((x n (1- x))
	      (result 1))
	     ((zerop x) result)
	   (declare (fixnum x))
	   (setf result (* x result))))))

;;; The #'eql has to be quoted, since this is a macro. Also, when
;;; binding variables in a macro, use gensym to be safe.
(defmacro update-alist (item value alist &key (test '#'eql) (key '#'identity))
  "If alist already has a value for Key, it is updated to be Value. 
   Otherwise the passed alist is updated with key-value added as a new pair."
  (let ((entry (gensym))
        (itemv (gensym))
        (valuev (gensym)))              ; to assure proper evaluation order and single expansion
    `(let* ((,itemv ,item)
            (,valuev ,value)
            (,entry (assoc ,itemv ,alist :test ,test :key ,key)))
       (if ,entry
	   (progn (setf (cdr ,entry) ,valuev)
		  ,alist)
	   (setf ,alist (acons ,itemv ,valuev ,alist))))))

(macro-indent-rule update-alist 1)

;;; Faster definition. In old definition, length may wind up cdring
;;; down the list (depends on the lisp).
(defun truncate-keywords (input-list)
  "Many functions take multiple arguments, via &rest, that can cause
   problems when keyword arguments are also supplied. This function
   truncates a list at the first top-level keyword. Thus, '(A B C :FOO D)
   is returned as (A B C). Note that the new list is freshly consed to 
   avoid any stack problems with destroying a &rest argument."
  (declare (type list input-list)
	   (optimize (speed 3) (safety 0)))
  (ldiff input-list (member-if #'keywordp input-list)))
;;; Note: See also Remove-Keywords below in this file.


;;
;;

;;; any benefit of the position's :from-end is lost by the calls to length,
;;; so use member.
(defun extract-keyword (key arglist 
			    &optional (default nil) &key (no-value nil))
  "Searches the arglist for keyword key, and returns the following mark,
   or the default if supplied. If no-value is non-nil, then if nothing follows
   the key it is returned."
  (declare (type list arglist)
	   (type t default)
	   (type keyword key)
	   (optimize (speed 3) (safety 0)))
  (let ((binding (member key arglist)))
    (cond ((and (null binding) no-value)
	   no-value)
	  ((cdr binding)
	   (cadr binding))
	  (t
	   default))))

;;; Explicit tagbody, with end-test at the end, to be nice to poor
;;; compilers.
(defmacro while (test &body body)
  "Keeps invoking the body while the test is true;
   test is tested before each loop."
  (let ((end-test (gensym))
	(loop (gensym)))
    `(block nil
       (tagbody (go ,end-test) 
		,loop
		,@body
		,end-test
		(unless (null ,test) (go ,loop))
		(return)))))

(macro-indent-rule while 1)

(defmacro while-not (test &body body)
  "Keeps invoking the body while the test is false;
   test is tested before each loop."
  (let ((end-test (gensym))
	(loop (gensym)))
    `(block nil
       (tagbody (go ,end-test)
		,loop
		,@body
		,end-test
		(unless ,test (go ,loop))
		(return)))))

(macro-indent-rule while-not 1)

(defmacro let*-non-null (bindings &body body)
  "like let*, but if any binding is made to NIL, the let*-non-null immediately returns NIL."
#+symbolics  (declare lt:(arg-template ((repeat let)) declare . body))

  `(block lnn (let* ,(mapcar #'process-let-entry bindings)
                    ,@body)))

(macro-indent-rule let*-non-null (like let))

(defun process-let-entry (entry)
  "if it isn't a list, it's getting a nil binding, so generate a return. Otherwise, wrap with test."
  (declare (optimize (speed 3) (safety 0)))

  (if (atom entry)
      `(,entry (return-from lnn nil))
      `(,(car entry) (or ,@(cdr entry) (return-from lnn nil)))))

;;; add dynamic-extent declaration 3/8/91 - bwm
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO MAPC-DOTTED-LIST (FN &REST LISTS)
  "Like normal Mapc, but handles dotted lists, and will apply the fn to the dotted argument, unless it is NIL"
  (let ((arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists)))
        (fnv (gensym)))
    `(BLOCK mdl (let ((,fnv ,fn))       ; avoid multiple evaluation
                  (MAPLIST #'(lambda ,arglist
                               (funcall ,FNv ,@(mapcar #'(lambda (argname) `(car ,argname)) arglist))
                             ;; is cdr an atom
                             (COND
                              ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
                               (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist))
                               (RETURN-FROM mdl (VALUES)))))
                         ,@LISTS))
            (VALUES))))

(macro-indent-rule mapc-dotted-list (like mapc))

;;; add dynamic-extent declaration & use gensyms for local vars 3/8/91 - bwm
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO MAPCAR-DOTTED-LIST (FN &REST LISTS)
  "Like normal Mapcar, but handles dotted lists, and will apply the fn to the dotted argument, unless it is NIL"
  (LET ((RETURN-VAL (GENSYM))
        (last-retval (gensym))
        (last-retval1 (gensym))
	(LASTCDR (GENSYM))
        (arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists)))
        (fnv (gensym)))
    `(LET (,RETURN-VAL ,LASTCDR ,last-retval1 (,fnv ,fn)) ; avoid multiple evaluation
       (BLOCK mdl (MAPLIST #'(LAMBDA ,arglist
                               (let ((,last-retval (list (funcall ,FNv ,@(mapcar #'(lambda (argname) `(car ,argname)) arglist)))))
                                 (if ,return-val
                                     (nconc ,last-retval1 ,last-retval)
                                   (setq ,return-val ,last-retval))
                                 (setq ,last-retval1 ,last-retval))
                               ;; is cdr an atom
			       (COND
                                ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
                                 (SETQ ,LASTCDR (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist)))
                                 (RETURN-FROM mdl (values)))))
			   ,@LISTS))
       (NCONC ,RETURN-VAL ,LASTCDR))))

(macro-indent-rule mapcar-dotted-list (like mapcar))

;;; add dynamic-extent declaration & use gensyms for local vars 3/8/91 - bwm
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO MAPCAN-DOTTED-LIST (FN &REST LISTS)
  "Like normal Mapcan, but handles dotted lists, and will apply the fn to the dotted argument, unless it is NIL"
  (LET ((RETURN-VAL (GENSYM))
        (fnv (gensym))
        (arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists))))
     `(LET (,RETURN-VAL (,fnv ,fn))
	(BLOCK mdl (MAPLIST #'(lambda ,arglist
				(SETQ ,RETURN-VAL (NCONC ,RETURN-VAL (funcall ,FNv ,@(mapcar #'(lambda (argname) `(car ,argname)) arglist))))
				;; is cdr an atom
				(COND
				  ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
				   (SETQ ,RETURN-VAL (NCONC ,RETURN-VAL (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist))))
				   (RETURN-FROM mdl (VALUES)))))
			    ,@LISTS))
	,RETURN-VAL)))

(macro-indent-rule mapcan-dotted-list (like mapcan))
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO MAPLIST-DOTTED-LIST (FN &REST LISTS)
  "Like normal Maplist, but handles dotted lists, and will apply the fn to the dotted argument, unless it is NIL"
  (LET ((RETURN-VAL (GENSYM))
        (fnv (gensym))
        (arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists))))
     `(LET (,RETURN-VAL (,fnv ,fn))
	(BLOCK mdl (MAPLIST #'(lambda ,arglist
				(SETQ ,RETURN-VAL (nconc ,RETURN-VAL (list (funcall ,FNv ,@arglist))))
				;; is cdr an atom
				(COND
				  ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
				   (SETQ ,RETURN-VAL (nconc ,RETURN-VAL (list (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist)))))
				   (RETURN-FROM mdl (VALUES)))))
			    ,@LISTS))
	,RETURN-VAL)))

(macro-indent-rule maplist-dotted-list (like maplist))

;;; add dynamic-extent declaration 3/8/91 - bwm
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO SOME-DOTTED-LIST (FN &REST LISTS)
  "Like normal Some, but handles a single dotted list, and will apply the fn to the dotted argument, unless it is NIL"
  (let ((fnv (gensym))
        (arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists))))
    `(let ((,fnv ,fn))
       (BLOCK sdl (MAPLIST #'(lambda ,arglist
                               (IF (funcall ,FNv ,@(mapcar #'(lambda (argname) `(car ,argname)) arglist))
                                   (RETURN-FROM sdl T)
                                 ;; is cdr an atom
                                 (COND
                                  ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
                                   (IF (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist))
                                       (RETURN-FROM sdl T)
                                     (RETURN-FROM sdl NIL))))))
                           ,@LISTS)
              NIL))))                    ;fell thru maplist w/o return

(macro-indent-rule some-dotted-list (like some))

;;; add dynamic-extent declaration 3/8/91 - bwm
;;; rewrite for greater efficiency 5/28/93 - bwm
(DEFMACRO EVERY-DOTTED-LIST (FN &REST LISTS)
  "Like normal Every, but handles dotted lists, and will apply the fn to the dotted arguments, unless they are (all) NIL."
  (let ((fnv (gensym))
        (arglist `(,@(mapcar #'(lambda (l)
                                 (declare (ignore l))
                                 (gensym)) ; entry for each list passed.
                             lists))))
    `(let ((,fnv ,fn))
       (BLOCK Edl (MAPLIST #'(lambda ,arglist
                               (IF (funcall ,FNv ,@(mapcar #'(lambda (argname) `(car ,argname)) arglist))
                                   ;; is cdr an atom
                                   (COND
                                    ((or ,@(mapcar #'(lambda (arg) `(and (atom (cdr ,arg)) (cdr ,arg))) arglist))
                                     (IF (funcall ,FNv ,@(mapcar #'(lambda (argname) `(cdr ,argname)) arglist))
                                         (RETURN-FROM Edl T)
                                       (RETURN-FROM Edl NIL))))
                                 (RETURN-FROM Edl NIL)))
                           ,@LISTS)
              T))))                      ;fell thru maplist w/o return

(macro-indent-rule every-dotted-list (like every))

(defmacro msetq (vars value)
#+lispm  (declare (compiler:do-not-record-macroexpansions)
                  (zwei:indentation 1 1))
 `(multiple-value-setq ,vars ,value))

(macro-indent-rule msetq (like multiple-value-setq))

(defmacro mlet (vars value &body body)
#+lispm  (declare (compiler:do-not-record-macroexpansions)
                  (zwei:indentation 1 3 2 1))
   `(multiple-value-bind ,vars ,value ,@body))

(macro-indent-rule mlet (like multiple-value-bind))

;;; the following is contributed by quiroz@cs.rochester.edu with slight modifications by miller@cs.rochester.edu

(defmacro cond-binding-predicate-to (symbol &rest clauses)
  "(cond-binding-predicate-to symbol . clauses)                      [macro]
a COND-like macro.  The clauses are exactly as in COND.  In the body
of a clause, the SYMBOL is lexically bound to the value returned by the
test.  Example: 

  (cond-binding-predicate-to others
    ((member 'x '(a b c x y z))
     (reverse others)))

evaluates to
  
  (x y z)"
#+lispm  (declare (zwei:indentation 0 3 1 1))
  (check-type symbol symbol)
  `(let (,symbol)
     (cond ,@(mapcar #'(lambda (clause)
                         `((setf ,symbol ,(first clause))
                           ,@(rest clause)))
                     clauses))))

(macro-indent-rule cond-binding-predicate-to (like case))

(defmacro dosequence ((var sequence &optional result) &BODY body)
  "(dosequence (var sequence &optional result) &body body)      [macro]
This macro is like DOLIST \(q.v.), except that the iteration is over
any sequence, not necessarily a list."
#+lispm  (declare (zwei:indentation 1 1))
  (check-type var symbol)
  (let ((iter-index (gensym))
        (iter-limit (gensym)))
    `(do* ((,var)
           (,iter-limit (length ,sequence))
           (,iter-index 0 (+ ,iter-index 1)))
         ((= ,iter-index ,iter-limit)
          (setq ,var nil)
          ,result)
       (setq ,var (elt ,sequence ,iter-index))
       ,@body)))

(macro-indent-rule dosequence (like dolist))

;;; the following is contributed by baldwin@cs.geneseo.edu

(defun Force-String (Thing)
  "Generates a string representation of Thing. This representation
is the print name for symbols, otherwise whatever 'coerce' can do (which may
be to generate an error sometimes)."

  (cond
    ((symbolp Thing)  (symbol-name Thing))
    (t  (coerce Thing 'string))))

(defun Prefix? (Prefix Seq)
  "Prefix? - Checks to see if Prefix is really a prefix of Seq. Returns
T if it is, NIL otherwise. Just checks that Prefix is no longer than
Seq, then checks to see if the the initial subsequence of Seq that is
the same length as Prefix is equal to Prefix. Prefix is a real prefix
if and only if both conditions hold."

  (and (<= (length Prefix) (length Seq))
       (equalp (subseq Seq 0 (length Prefix)) Prefix)))


(defun Elapsed-Time-in-Seconds (Base Now)
  "Returns the time in seconds that has elapsed between Base and Now.
Just subtracts Base from Now to get elapsed time in internal time units,
then divides by the number of internal units per second to get seconds."

  (coerce  (/  (- Now Base)  internal-time-units-per-second)  'float))


(defun Bit-Length (N)
  " Computes the number of bits needed to represent integer N.
Assumes that 0 requires 1 bit to represent, positive numbers require
floor(log(N))+1 bits, and negative numbers require one bit more than
their positive counterparts (for the sign bit). This treatment of
negative integers is a little bit arbitrary, but seems as good as
anything else."

  (cond
    ((= N 0)  1)
    ((< N 0)  (+ (Bit-Length (- N)) 1))
    ((> N 0)  (+ (floor (log N 2)) 1))))


(defun Flatten (L)
  "Flattens list L, i.e., returns a single list containing the
same atoms as L but with any internal lists 'dissolved'. For example,
(flatten '(a (b c) d))  ==>  (a b c d)
Recursively flattens components of L, according to the following rules:
 - an atom is already flattened.
 - a list whose CAR is also a list is flattened by appending the
   flattened CAR to the flattened CDR (this is what dissolves internal
   lists).
 - a list whose CAR is an atom is flattened by just flattening the CDR
   and CONSing the original CAR onto the result.
These rules were chosen with some attention to minimizing CONSing."

  (cond
    ((null L )   '() )
    ((atom L)    L)
    ((consp L)
     (if (consp (car L))
	 (append (Flatten (car L)) (Flatten (cdr L)))
	 (cons (car L) (Flatten (cdr L)))))
    (t   L)))


(defun Sum-of-Powers-of-Two-Representation (N)
  "Figures out how to represent N as a sum of powers of two. Returns a list of exponents,
the idea being the N is the sum over E in this list of two raised to the E-th power. 
Requires N to be a positive integer, so that all exponents in the result list are integers."

  (declare (integer N))
  (assert (> N 0))
  (do ( (I 0 (+ I 1))
        (Exps '() (if (logbitp I N)
		      (cons I Exps)
		      Exps)) )
      ((>= I (integer-length N))  Exps)
    (declare (integer I) (list Exps))))


(defun Difference-of-Powers-of-Two-Representation (N)
  "Figures out how to represent N as the difference of a sequence of powers of 2 
 (e.g., 2^e1 - 2^e2 - ...). Returns a list of exponents, with e1 as the last and
the others in some arbitrary order. Requires N to be an integer greater than 0,
which simplifies the code but isn't absolutely necessary. Starts by figuring out 
The smallest power of two greater than or equal to N - this exponent becomes e1. 
Remaining exponents are just those of the greater power of two minus N."

  (declare (integer N))
  (assert (> N 0))
  (let* ((E1 (ceiling (log N 2)))
	 (Next-Power (expt 2 E1)))
    (declare (integer E1 Next-Power))
    (if (= Next-Power N)
	(list E1)
	(append (Sum-of-Powers-of-Two-Representation (- Next-Power N)) (list E1)))))


(defun Ordinal-String (N)
  " Generates a string representing N as an ordinal number (i.e., 1st, 2nd, etc.). 
Works by printing N and the appropriate suffix to a string - N is printed in decimal, 
the suffix is looked up based on the last digit of N (i.e., N mod 10)."

  (declare (integer N))
  (let ((Last-Digit (mod (abs N) 10))
	(Last-2-Digits (mod (abs N) 100)))
    (declare (integer Last-Digit))
    (format nil "~d~a" N (cond
			   ((or (= Last-2-Digits 11)
				(= Last-2-Digits 12)
				(= Last-2-Digits 13))         "th")
			   ((= Last-Digit 1)                  "st")
			   ((= Last-Digit 2)                  "nd")
			   ((= Last-Digit 3)                  "rd")
			   (t                                 "th")))))



(defun Between (Lo Hi)
  "Generates a list of integers between Lo and Hi, inclusive. 
Straightforward recursive definition, i.e., result is Lo consed onto
integers from Lo+1 to Hi, unless Lo is greater than Hi in which case
result is nil."

  (declare (integer Lo Hi))
  (cond
    ((> Lo Hi)  '() )
    (t   (cons Lo (Between (+ Lo 1) Hi)))))


;; More stuff by miller@cs.rochester.edu

(defun reverse-alist (alist &key (test #'eql))
  "Takes an alist of uniqe keys and non-unique values, and returns an alist of unique keys based on the values, whose values are
lists of the original keys."
  (let (result)
    (dolist (x alist)
      (let ((assoc-result (assoc (cdr x) result :test test)))
	(if assoc-result
	    (setf (cdr assoc-result) (cons (car x) (cdr assoc-result)))
	    (setq result (acons (cdr x) (list (car x)) result)))))
    result))

;; Fast versions of the commonlisp union and intersection, that want and return sorted lists.
;; rewrite for more speed 6/1/93 by miller.

(defun fast-union (list1 list2 predicate &key (test #'eql) (key #'identity))
  "Like Union (but no support for test-not) should be faster because list1 and list2 must be sorted.
Fast-Union is a Merge that handles duplicates. Predicate is the sort predicate."
  (declare (type list list1 list2))
  (let (result result1
	(wlist1 list1)
	(wlist2 list2))
    (while (and wlist1 wlist2)
      (cond
       ((funcall test (funcall key (car wlist1)) (funcall key (car wlist2)))
        (setq result1 (nconc result1 (list (pop wlist1))))
        (pop wlist2))
       ((funcall predicate (funcall key (car wlist1)) (funcall key (car wlist2)))
        (setq result1 (nconc result1 (list (pop wlist1)))))
       (t
        (setq result1 (nconc result1 (list (pop wlist2))))))
      (if (null result) (setq result result1)))
    (cond
      (wlist1
       (nconc result wlist1))
      (wlist2
       (nconc result wlist2))
      (t
       result))))

(defun fast-intersection (list1 list2 predicate &key (test #'eql) (key #'identity))
  "Like Intersection (but no support for test-not) should be faster because list1 and list2 must be sorted.
Fast-Intersection is a variation on Merge that handles duplicates. Predicate is the sort predicate."
  (declare (type list list1 list2))
  (let (result result1
	(wlist1 list1)
	(wlist2 list2))
    (while (and wlist1 wlist2)
      (cond
	((funcall test (funcall key (car wlist1)) (funcall key (car wlist2)))
	 (setq result1 (nconc result1 (list (pop wlist1))))
	 (pop wlist2))
	((funcall predicate (funcall key (car wlist1)) (funcall key (car wlist2)))
	 (pop wlist1))
	(t
	 (pop wlist2)))
      (if (null result) (setq result result1)))
    result))

(deftype alist () 'list)

(defun true-list-p (term)
  "Returns t if the term is a non-dotted list. Note that nil is a true list."
  (declare (optimize (speed 3) (safety 0)))
  (and (listp term)
       (not (cdr (last term)))))

;; scheme-stream (thanks to barmar@think.com)
;; Why use these instead of generators & series? Well, you can use both. Note that generators discard their output, while scheme-streams cache 
;; their output. What constitutes a tail-form isn't specified. So it's valid to put, e.g. a Water's generator there, or at least a fn
;; that conses up a new scheme-stream whose head is the result of the call on the generator, and whose tail is another call on this fn.

(defstruct scheme-stream
  head
  tail
  (tail-closure-p t))

(defmacro cons-scheme-stream (head tail-form)
  `(make-scheme-stream :head ,head
		       :tail #'(lambda () ,tail-form)))

(defmacro list-scheme-stream (&rest args)
  "Analogue to the cl list function, only the last arg is delayed."
  (assert (> (list-length args) 1) (args) "List-scheme-stream requires at least 2 args.")
  (let* ((revargs (nreverse args))
         (result `(cons-scheme-stream ,(second revargs) ,(pop revargs))))
    (pop revargs)
    (while revargs
      (setq result `(make-scheme-stream :head ,(pop revargs) :tail ,result :tail-closure-p nil)))
    result))

(defun ss-head (stream)
  "Return the head of scheme stream Stream. If Stream is not a stream, returns NIL (to allow the usual car of nil)."
  (declare (optimize (speed 3) (safety 0)))
  (if (scheme-stream-p stream)
      (scheme-stream-head stream)))     ;return nil for non-streams

(defun ss-tail (stream)
  "Return the tail of the scheme stream Stream. Invokes lazy evaluation if needed. 
If stream is not a scheme stream, return NIL (allows the usual cdr of nil)."
  (declare (optimize (speed 3) (safety 0)))
  (cond
   ((not (scheme-stream-p stream))
    nil)
   ((scheme-stream-tail-closure-p stream)
    (setf (scheme-stream-tail-closure-p stream) nil
          (scheme-stream-tail stream) (funcall (scheme-stream-tail stream))))
   (t (scheme-stream-tail stream))))

;; scheme force/delay model

(defstruct scheme-delay
  (first-time-p t)
  value)

(defmacro scheme-delay (form)
  `(make-scheme-delay :value #'(lambda () ,form)))

(defun scheme-force (delay)
  (cond ((scheme-delay-first-time-p delay)
	 (setf (scheme-delay-first-time-p delay) nil
	       (scheme-delay-value delay) (funcall (scheme-delay-value delay))))
	(t (scheme-delay-value delay))))

;; OK, how many times have you written code of the form
;;
;; (let ((retval (mumble)))
;;    (setf (slot retval) bletch)
;;    (setf (slot retval) barf)
;;    retval)
;;
;; or things of the sort? More than you care to remember most likely. Enter the utterly useful PROGFOO.
;; Think of it as a PROG1 with the value being bound to FOO. inside it's extent Lexically, of course.

(defmacro progfoo (special-term &body body)
  `(let ((foo ,special-term))
     ,@body
     foo))

(macro-indent-rule progfoo (like prog1))

(defmacro with-rhyme (body)
  "Well, there must be rhyme OR reason, and we now admit there is no reason, so...
Used to flag silly constructs that may need to be rewritten for best effect."
  body)

(macro-indent-rule with-rhyme (like progn))

;; and for common lisp fans of multiple values... FOO is the first value, you can access all the values as MV-FOO.
;; returns the multiple values, like multiple-values-prog1

(defmacro mv-progfoo (special-term &body body)
  `(let* ((mv-foo (multiple-value-list ,special-term))
          (foo (car mv-foo)))
     ,@body
     (values-list mv-foo)))

(macro-indent-rule mv-progfoo (like multiple-value-prog1))

;; from the net
;; From: Kerry Koitzsch <kerry@erg.sri.com>

(defun GET-COMPILED-FUNCTION-NAME (fn)
  "Returns the symbol name of a function. Covers the six major CL vendors."
  #+lispm
  (when (si:lexical-closure-p fn)
    (return-from get-compiled-function-name nil))
  (etypecase fn 
    (symbol fn)
    (compiled-function #+cmu(kernel:%function-header-name fn)
                       #+:mcl(ccl::function-name fn)
                       #+lispm(si:compiled-function-name fn)
                       #+akcl(system::compiled-function-name fn)
                       #+lucid
                       (when (sys:procedurep fn)
                         (sys:procedure-ref fn SYS:PROCEDURE-SYMBOL))
                       #+excl (xref::object-to-function-name fn)
                       #+lispworks (system::function-name fn)
                       )))

;; back to miller@cs.rochester.edu

;; This may seem like a silly macro, but used inside of other macros or code generation facilities it is very useful - you can see 
;; comments in the (one-time) macro expansion!

(defmacro comment (&rest anything)
  "Expands into nothing"
  (declare (ignore anything)))

;; define boolean control extensions to and, or, not... the bit operators are there, but not the more general short-circuting ones. 
;; Some of these will only make sense on two operands, and many can't short circuit (exclusive ops, for instance).

(defmacro xor (&rest predicates)
  "True only if exactly one predicate is true. Short circutes when it finds a second one is true. Returns the true predicate"
  (let ((result (gensym))
        (temp (gensym))
        (block-name (gensym)))
    `(block ,block-name
       (let ((,result ,(car predicates))
             ,temp)
         ,@(let (code-result)
             (dolist (pred (cdr predicates))
               (push `(cond
                       ((and (setq ,temp ,pred)
                             ,result)
                        (return-from ,block-name nil))
                       (,temp
                        (setq ,result ,temp)))
                     code-result))
             (nreverse code-result))
         ,result))))
         
(macro-indent-rule xor (like and))

(defmacro eqv (&rest predicates)
  "True iff all predicates produce the same result according to eql, or passed :test (exclusive nor if binary)"
  (let ((result (gensym))
        (real-preds (truncate-keywords predicates))
        (test-key (extract-keyword :test predicates #'eql))
        (block-name (gensym)))
      `(block ,block-name
         (let ((,result ,(car real-preds)))
           ,@(let (code-result)
               (dolist (pred (cdr real-preds))
                 (push `(if (not (funcall ,test-key ,pred ,result))
                            (return-from ,block-name nil))
                       code-result))
               (nreverse code-result))
           (or ,result t)))))

(macro-indent-rule eqv (like and))

(defmacro nand (&rest predicates)
  "True only if all predicates are not true. Short circutes when it finds one is false."
  (let ((block-name (gensym)))
    `(block ,block-name
       ,@(let (code-result)
           (dolist (pred predicates)
             (push `(if (not ,pred)
                        (return-from ,block-name t))
                   code-result))
             (nreverse code-result))
       nil)))
         
(macro-indent-rule nand (like and))

(defmacro nor (&rest predicates)
  "True only if all predicates are false. Short circutes when it finds a one is true."
  (let ((block-name (gensym)))
    `(block ,block-name
       ,@(let (code-result)
           (dolist (pred predicates)
             (push `(if ,pred
                        (return-from ,block-name nil))
                   code-result))
             (nreverse code-result))
       t)))
         
(macro-indent-rule nor (like and))

;; duplicate an array. 12/29/93 by miller
(defun copy-array (array &optional (element-copier #'identity))
  "Returns an (exact) copy of the passed array, with same size, fill pointer (if there is one), adjustable quality,
element type, and contents. Uses element-copier to copy elements (default #'identity)."
  (declare (optimize (speed 3) (safety 0) (debug 0)))
  (progfoo (make-array (array-dimensions array) 
                       :element-type (array-element-type array)
                       :adjustable (adjustable-array-p array)
                       :fill-pointer (and (array-has-fill-pointer-p array)
                                          (fill-pointer array)))
    (dotimes (i (array-total-size array))
      (setf (row-major-aref foo i) (funcall element-copier (row-major-aref array i))))))

(let (loaded-pathnames)
  (defun clear-load-once ()
    (setq loaded-pathnames nil))
  
  (defun load-once (pathname &rest load-opts)
    (let ((preloaded (assoc pathname loaded-pathnames :test #'equalp)))
      (cond
       ((and preloaded
             (> (file-write-date pathname) (cdr preloaded)))
        (apply #'load pathname load-opts)
        (setf (cdr preloaded) (file-write-date pathname)))
       ((null preloaded)
        (apply #'load pathname load-opts)
        (push (cons pathname (file-write-date pathname)) loaded-pathnames))))))
