;;; -*- Mode: LISP; Syntax: ansi-common-lisp; Package: CL-LIB; Base: 10 -*-
;;; 
;;; Copyright (C) 1994, 1993, 1992 by Bradford W. Miller, miller@cs.rochester.edu
;;; Unlimited non-commercial use is granted to the end user, other rights to
;;; the non-commercial user are as granted by the GNU LIBRARY GENERAL PUBLIC LICENCE
;;; version 2 which is incorporated here by reference.

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU Library General Public License as published by
;;; the Free Software Foundation; version 2.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU Library General Public License for more details.

;;; You should have received a copy of the GNU Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

(in-package cl-lib)
;;;
;;; The following is contributed by miller@cs.rochester.edu


(defmacro parser-error (stream format-string &rest format-args)
  "Process an error within the parse process, e.g. reader macros. On some systems (e.g. lisp machines) this will invoke
the rubout handler, and let you fix up your input."
  #+symbolics
  `(zl:::sys:read-error ,stream ,format-string ,@format-args)
  #+explorer
  (declare (ignore stream))
  #+explorer
  `(cerror :no-action nil 'sys:read-error-1 ,format-string ,@format-args)
  #+clim
  `(cond
    ((clim::input-editing-stream-p stream)
     (clim::simple-parse-error ,format-string ,@format-args))
    (t ;; either a file, or maybe just not running under clim.
     (error (concatenate 'string "Parse error: " ,format-string "~@[ at file position ~D~]") 
          ,@format-args
          (and (typep ,stream 'file-stream)
               (file-position ,stream)))))
  #-(or symbolics explorer clim)
  `(error (concatenate 'string "Parse error: " ,format-string "~@[ at file position ~D~]") 
          ,@format-args
          (and (typep ,stream 'file-stream)
               (file-position ,stream))))

(macro-indent-rule parser-error (like format))


;; this is probably a bit on the obscure side, but it comes up often enough. The idea is to have something that warns the user
;; once and gives him options on what to do. If he wants, further warnings will be supressed.
;; Likely the new continuation stuff would be a better encoding, but this was written before that was widely available.

(defvar *warn-or-error-cleanup-initializations* nil)

(defvar *general-warning-list* nil "For Warn-Or-Error, if the function doesn't want it's own checklist.")
(eval-when (compile load eval)
  (setq *warn-or-error-cleanup-initializations* nil)
  (setq *general-warning-list* nil))

;;
;;
(defun warn-or-error (item checklist continue-control-string proceed-control-string format-control-string &rest format-args)
  "Like Warn, returns non-nil if the user wishes to continue. A nil return implies failure.
A bit snazzier than just WARN or CERROR, this function does a warning, and then asks if the user wants to go ahead 
 (continue-control-string), error out, or go ahead and not ask again. Item and checklist is supplied by the
caller for just this functionality: if item is found on checklist, Warn-or-Error will return non-nil."
  (let ((*print-non-default-types* nil))
    (declare (special *print-non-default-types*))

    (unless (member item (eval checklist))
      (apply #'warn (cons format-control-string format-args))
      (if continue-control-string (format *debug-io* (concatenate 'string "Shall I " continue-control-string "?")))
      (case (loop (format *debug-io* "(Y(es), ~A; F(ail); D(ebug); P(roceed), ~A)"
                          continue-control-string proceed-control-string)
              (let* ((char (read-char *debug-io*))
                     (selection (assoc char '((#\Y . :go)
                                              (#\y . :go)
                                              (#\F . :fail)
                                              (#\f . :fail)
                                              (#\D . :debug)
                                              (#\d . :debug)
                                              (#\P . :proceed)
                                              (#\p . :proceed)))))
                (if selection
                    (return (cdr selection)))))
	    (:go
	      't)
	    (:fail
	      nil)
	    (:debug
	      (break))
	    (:proceed
	      (set checklist (cons item (eval checklist))))))))

;;
;; unlike cerror, assert doesn't let you fiddle with the new bindings of the passed vars. Via an "operation", check does:

(defmacro check (vars operation assertion format-string &rest format-args)
  "An enhanced version of assert, that takes an operation to perform on args, which can be refered to by the assertion as check."
  `(block check
     (loop
       (let ((check ,operation))
         (if ,assertion
             (return-from check check))
         (cerror ,(format nil "Change the value of ~{~A~^,~}." vars) ,format-string ,@format-args)
         ;; continued
         ,@(mapcan #'(lambda (var)
                      `((when (y-or-n-p ,(format nil "change the value of ~S from ~~S? " var) ,var)
                          (format *debug-io* ,(format nil "Enter new value of ~S: " var))
                          (setq ,var (read)))))
                  vars)))))

;; following contributed by George Ferguson (ferguson@cs.rochester.edu)

#+clim
(CLIM:define-application-frame error-frame ()
  ()
  (:panes (msg-pane :application
		    :height '(3 :line)
		    :min-width '(30 :character)
		    :scroll-bars nil
		    :initial-cursor-visibility nil)
	  (ok-pane CLIM:push-button
		   :label "Ok"
		   :width '(4 :character)
		   :activate-callback #'(lambda (button)
					  (CLIM:frame-exit
					   (CLIM:pane-frame button)))))
  (:layouts
   (default (CLIM:vertically () msg-pane ok-pane))))

#+clim
(defun popup-error (msg)
  "Pops up an error box displaying MSG."
  (let* ((frame (CLIM:make-application-frame 'error-frame
					     :pretty-name "Hola! Error!"))
	 (pane (CLIM:get-frame-pane frame 'msg-pane)))
    (CLIM:window-clear pane)
    (terpri pane)			; skip to 2nd line
    (format pane msg)
    (CLIM:change-space-requirements pane :resize-frame t :width (list (1+ (length msg)) :character))
    (CLIM:run-frame-top-level frame)))

