;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         ships.l
; Description:  Naval warfare simulator using frobs.
; Author:       Eric Muehle
; Created:      30-Apr-87
; Package:      USER
; RCS $Header: ships.l,v 1.4 88/03/19 15:58:01 jed Exp $
;
; (c) Copyright 1987, University of Utah, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Load in the system
; (require "frobs")

;;;; This is a simple naval warfare simulator.  There are 2 types of ships:
;;;; surface and submarine.  There are 3 kinds of combat: 
;;;; asw (anti submarine warfare), ssm (surface to surface missle), and
;;;; torpedo (sub vs ship).

;;; Basic components of all ships.
(def-class ship nil :slots (name strength owner))

;;; Determines if a ship is sunk
(def-method ({class ship} sunk?)()
  (when (> 1 (strength $self))
    (format t "The ~S has been sunk~%" (name $self))
    t))

;;; Adds damage to a ship
(def-method ({class ship} add-damage)(dam)
  (decf (strength $self) dam))


;;; The definition of a surface ship.  Surface ships can only use ssm and
;;; asw as forms of combat.  
(def-class surface-ship {class ship} 
  :slots (size ssm asw)
  ;; lets give a default value to the inherited slot STRENGTH
  :init ((nil (strength 10))))

;;; Function that makes a new surface ship
(def-method ({class surface-ship} make) 
  (name owner &key (size 'medium)(strength (strength $self)))
  (let ((ship (new-instance* $self :name name)))
    (setf (name ship) name)
    (setf (owner ship) owner)
    (setf (size ship) size)
    (setf (strength ship) strength)
    (setf (asw ship) (new-instance* {class asw}))
    (setf (ssm ship) (new-instance* {class ssm}))
    ship))

;;; Attack method for a surface ship attacking another ship
(def-method ({class ship} attack)(target)
  ;; if the target is a surface ship use SSM
  (cond ((eq (frob-type target) 'surface-ship)
	 (combat (ssm $self) target))
	;; otherwise it is a submarine
	(t (combat (asw $self) target))))


;;; The definition of a submarine.  Subs can only use torpedo and
;;; asw as forms of combat.
(def-class submarine {class ship} 
  :slots (torpedo asw) 
  ;; give a default value to the inherited slot STRENGTH
  :init ((nil (strength 12))))

;;; Function that makes a new submarine
(def-method ({class submarine} make)
  (name owner &key (strength (strength $self)))
  (let ((ship (new-instance* $self :name name)))
    (setf (name ship) name)
    (setf (owner ship) owner)
    (setf (strength ship) strength)
    (setf (asw ship) (new-instance* {class asw}))
    (setf (torpedo ship) (new-instance* {class torpedo}))
    ship))

;;; Attack method for a submarine attacking another ship
(def-method ({class submarine} attack)(target)
  ;; if the target is a surface ship use TORPEDO	    
  (cond ((eq (frob-type target) 'surface-ship)
	 (combat (torpedo $self) target))
	;; otherwise it is a submarine
	(t (combat (asw $self) target))))



;;; The ASW combat class.  There is a PROB probability of a hit and
;;; each hit inflicts DAM worth of damage points.
(def-class asw nil 
  :slots ((prob 80)
	  (dam 7)))

;;; The combat method for ASW attacks
(def-method ({class asw} combat)(target)
  (when (hit (prob $self))
    (let ((damage (1+ (random (dam $self)))))
      (format t "The ~S was hit with ~S points in an ASW attack~%"
	(name target) damage)
      (add-damage target damage)
      (sunk? target))))


;;; The SSM combat class.  There are 2 probabilities of hitting a target
;;; (depending on the size of the target) and each hit inflicts DAM worth
;;; of damage points.
(def-class ssm nil 
  :slots ((small-prob 40)
	  (large-prob 70)
	  (dam 11)))

;;; The combat method for SSM attacks
(def-method ({class ssm} combat)(target)
  (when (hit (get-prob $self target))
    (let ((damage (1+ (random (dam $self)))))
      (format t "The ~S was hit with ~S points in a ~S attack~%"
	(name target) damage (frob-type $self))
      (add-damage target damage)
      (sunk? target))))

;;; Returns the probability of an SSM hitting a target depending on size
;;; of the target.
(def-method ({class ssm} get-prob)(target)
  (case (size target)
    ((small medium) (small-prob $self))
    (large (large-prob $self))))


;;; The TORPEDO combat class.  It is the same as the SSM class but adds
;;; one more probability for hitting MEDIUM sized targets.
(def-class torpedo {class ssm} 
  :slots ((med-prob 50)))

;;; Returns the probability of a TORPEDO hitting a target depending on size
;;; of the target.
(def-method ({class torpedo} get-prob)(target)
  (cond ((eq (size target) 'medium)
	 (med-prob $self))
	;; if the target is not medium sized then
	;; use the SSM get-prob method
	(t (call-method (ssm get-prob) target))))

 

;;; Returns T if an attack hits its target.
(defun hit (x)
  (< (random 101) x))


;;; lets make some ships
(make {class surface-ship} 'kiev 'ussr :size 'large)
(make {class surface-ship} 'new-jersey 'us :size 'large :strength 20)
(make {class submarine} 'boston 'us)
(make {class submarine}  'pskov 'ussr)

;;; lets have a war
(attack {boston} {pskov})
(attack {pskov} {boston})

(attack {boston} {kiev})
(attack {kiev} {boston})

(attack {kiev} {new-jersey})
(attack {new-jersey} {kiev})


;;;; Instead of having ship vs ship lets have fleet vs fleet

;;; A fleet has an owner and a mode which indicates the current status
;;; of the fleet, and a list of ships.  MODE is private because we
;;; do not want the players to cheat and write code that would
;;; change their mode illegally.  Since this class is going to be locked
;;; with a key, the close-class keyword is given NIL which forces the
;;; class module to remain open.
(def-class fleet nil :slots (owner name) :private (mode) :mv (ships) 
  :close-class nil)

(def-method ({class fleet} make) (name owner &rest ships)
  (outfit (new-instance* $self :name name) name owner ships))

;;; Method to outfit a fleet
(def-method ({class fleet} outfit)(name owner ships)
  (setf (name $self) name)	    
  (setf (mode $self) 'attack)
  (setf (owner $self) owner)
  (assert-vals $self 'ships ships))

;;; This method is private.  This should not be able to be run outside
;;; of the FLEET module.
(def-method ({class fleet} remove-ship :private)(ship)
  (remove-val $self 'ships ship))

;;; The general combat method for fleets
(def-method ({class fleet} combat)(enemy)
  (let (target)
    (when (ships $self)
      (format t "The ~S fleet is attacking the ~S fleet~%"
	(name $self)(name enemy)))
    (dolist (s (ships $self))
      (setf target (nth (random (length (ships enemy))) (ships enemy)))
      (when (attack s target)
	(remove-ship enemy target)))
    (when (> (length (ships $self))(length (ships enemy)))
      (setf (mode enemy) 'retreat)
      (setf (mode $self) 'attack))
    (when (null (ships enemy))
      (format t "The ~S fleet has been sunk by the ~S fleet~%"
	(name enemy)(name $self)))))

;;; The attack method for fleets.  If both fleets are in the same mode
;;; then the first fleet attacks first.  Otherwise the fleet that attacks 
;;; first is the fleet with a MODE of ATTACK.
(def-method ({class fleet} attack)(enemy)
  (cond ((eq (mode $self)(mode enemy))
	 (combat $self enemy)
	 (combat enemy $self))
	((eq (mode $self) 'attack)
	 (combat $self enemy)
	 (combat enemy $self))
	(t
	 (combat enemy $self)
	 (combat $self enemy))))
	 

;;; Lets lock the class with a password.  We can view the FLEET class as
;;; the game server.  If the players could have access to the module
;;; they could alter the characteristics of the fleets and cheat.
(close-class game-keeper)

;;; lets make some ships
(make {class surface-ship} 'kiev 'ussr :size 'large)
(make {class surface-ship} 'new-jersey 'us :size 'large :strength 20)
(make {class submarine} 'boston 'us)
(make {class submarine}  'pskov 'ussr)

(make {class fleet} '7th 'us {boston} {new-jersey})
(make {class fleet} 'baltic 'ussr {pskov} {kiev})

;;; lets have a war
(attack {7th} {baltic})
(attack {baltic} {7th})
(attack {baltic} {7th})

;; End of file.
