;; Major mode for editing Quixote,
;; and for running Micro Quixote under GNU Emacs.
;;
;; Copyright (C) 1993 ICOT
;; written by gniibe

;;;;;;;;;;;;;;;;;;;;
;; This code is based on prolog.el in Mule distribution.
;; Actually, most part are verbatim copy. :-)
;;
;; Though this file is not part of GNU Emacs, it is distributed under
;; the conditions of GNU General Public License.

  ;; Major mode for editing Prolog, and for running Prolog under Emacs
  ;; Copyright (C) 1986, 1987 Free Software Foundation, Inc.
  ;; Author Masanobu UMEDA (umerin@flab.flab.fujitsu.junet)

  ;; Modified for rich indentation in Prolog mode inspired by c-mode.el
  ;;	Ken'ichi HANDA (handa@etl.go.jp) 1990.6.13, 90.7.25
;;
;;;;;;;;;;;;;;;;;;;;
(if (boundp 'MULE)
    (define-program-coding-system nil ".*mq.*" *internal*))
(if (boundp 'NEMACS)
    (define-program-kanji-code  nil ".*mq.*" 3) ; EUC
  )


(defvar quixote-mode-syntax-table nil)
(defvar quixote-mode-abbrev-table nil)
(defvar quixote-mode-map nil)
  
(defvar quixote-consult-string "&end.\n"
  "*(Re)Consult mode (for Micro Quixote). ")

(defvar inferior-quixote-program "mq"
  "*Program name for invoking an inferior Micro Quixote with `run-quixote'.")
(defvar inferior-quixote-startfile nil
  "*File to be sent to inferior Quixote process at startup time.")
(defvar inferior-quixote-switches nil
  "*List of arguments for invoking inferior Quixote process.")

;;; Customizable variables for indentation
(defvar quixote-indent-level 4
  "*Indentation of Prolog statements with respect to containing block.")
(defvar quixote-paren-offset 1
  "*Extra indentation for parens, compared with other text in same context.")
(defvar quixote-continued-statement-offset 4
  "*Extra indent for lines not starting new statements.")
(defvar quixote-first-argument-offset 4
  "*Extra indent for the first argument relative to a head.")
(defvar quixote-term-separator
  ".*\\(;;\\|,\\|<=\\)\\s *\\($\\|%\\)"
  "*Regexp to test if the last term of current line is terminated.")
(defvar quixote-clause-terminator
  ";;\\s *\\(%.*\\)*$"
  "*Regexp to search for termination of clause.")
(defvar quixote-tab-always-indent t
  "*Non-nil means TAB in Quixote mode should always reindent the current line,
regardless of where in the line point is when the TAB command is used.")

(if quixote-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)
    (modify-syntax-entry ?\\ "\\" table)
    (modify-syntax-entry ?% "<" table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?\[ "(]" table)
    (modify-syntax-entry ?\] ")[" table)
    (modify-syntax-entry ?\{ "(}" table)
    (modify-syntax-entry ?\} "){" table)
    (modify-syntax-entry ?\n ">" table)
    (modify-syntax-entry ?\f ">" table)
    (modify-syntax-entry ?/ ". 14" table)
    (modify-syntax-entry ?* ". 23" table)
    ;;
    (modify-syntax-entry ?\' "\"" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (setq quixote-mode-syntax-table table)))

(define-abbrev-table 'quixote-mode-abbrev-table ())

(defun quixote-mode-variables ()
  (set-syntax-table quixote-mode-syntax-table)
  (setq local-abbrev-table quixote-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^%%\\|^$\\|" page-delimiter)) ;'%%..'
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'quixote-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "\\(\\)\\(%+ *\\|/\\*+ *\\)") 
  (make-local-variable 'comment-column)
  (setq comment-column 48)
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'quixote-comment-indent))

(defun quixote-mode-commands (map)
  (define-key map "\t" 'quixote-indent-line)
  (define-key map "\e\C-x" 'quixote-consult-region)
  (define-key map "[" 'electric-quixote-paren)
  (define-key map "]" 'electric-quixote-paren)
  (define-key map "{" 'electric-quixote-paren)
  (define-key map "}" 'electric-quixote-paren)
  ;; (define-key map ";" 'electric-quixote-paren)
  ;; (define-key map ":" 'electric-quixote-paren)
  (define-key map "=" 'electric-quixote-eq)
  (define-key map "\e\C-q" 'quixote-indent-current-clause))

(if quixote-mode-map
    nil
  (setq quixote-mode-map (make-sparse-keymap))
  (quixote-mode-commands quixote-mode-map))

(defun quixote-mode ()
  "Major mode for editing Quixote code.
Blank lines and `%%...' separate paragraphs.  `%'s start comments.
Commands:
\\{quixote-mode-map}
Entry to this mode calls the value of quixote-mode-hook
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map quixote-mode-map)
  (setq major-mode 'quixote-mode)
  (setq mode-name "Quixote")
  (quixote-mode-variables)
  (run-hooks 'quixote-mode-hook))

(defun electric-quixote-paren (arg)
  "Insert character and correct line's indentation."
  (interactive "P")
  (if (and (not arg)
	   (save-excursion
	     (skip-chars-backward " \t")
	     (bolp)))
      (progn
	(insert last-command-char)
	(quixote-indent-line)
	(delete-char -1)))
  (self-insert-command (prefix-numeric-value arg)))

(defun electric-quixote-eq (arg)
  "Insert character and correct line's indentation."
  (interactive "P")
  (if (and (not arg)
	   (save-excursion
	     (beginning-of-line)
	     (looking-at "^[ \t]*<=$")))
      (progn
	(insert last-command-char)
	(quixote-indent-line))
    (self-insert-command (prefix-numeric-value arg))))

(defun quixote-comment-indent ()
  "Compute Quixote comment indentation."
  (cond ((looking-at "%%%") 0)
	((or (looking-at "%%") (looking-at "/\\*"))
	 (let ((indent (calculate-quixote-indent)))
	   (if (consp indent) (car indent) indent)))
	(t
	 (save-excursion
	       (skip-chars-backward " \t")
	       (max (1+ (current-column)) ;Insert one space at least
		    comment-column)))
	))

(defun beginning-of-clause ()
  "Move backward to beginning of clause."
  (interactive)
  (re-search-backward "^    [^%\t\n\f]" nil 'mv))

(defun end-of-clause ()
  "Move forward to end of clause."
  (interactive)
  (re-search-forward ";;" nil 'mv))

(defun quixote-indent-line ()
  "Indent current line as Quixote code.
Return the amount the indentation changed by."
  (interactive)
  (let ((indent (calculate-quixote-indent nil))
	start-of-block
	beg shift-amt
	(case-fold-search nil)
	(pos (- (point-max) (point))))
    (if (listp indent)
	(progn
	  (setq start-of-block (cdr indent))
	  (setq indent (car indent)))
      (setq start-of-block 0))
    (beginning-of-line)
    (setq beg (point))
    (setq indent
	  (cond ((eq indent nil) (current-indentation))
		((eq indent t) (calculate-quixote-indent-within-comment))
		(t
		 (skip-chars-forward " \t")
		 (cond ((looking-at "%%%") 0)
		       ((looking-at "%%") indent)
		       ((= (following-char) ?%) comment-column)
		       ((= (following-char) ?\]) start-of-block)
		       ((= (following-char) ?\[)
			(+ indent quixote-paren-offset))
		       (t indent)))))
    (skip-chars-forward " \t")
    (setq shift-amt (- indent (current-column)))
    (if (zerop shift-amt)
	(if (> (- (point-max) pos) (point))
	    (goto-char (- (point-max) pos)))
      (delete-region beg (point))
      (indent-to indent)
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (if (> (- (point-max) pos) (point))
	  (goto-char (- (point-max) pos))))
    shift-amt))

(defun calculate-quixote-indent (&optional parse-start)
  "Return appropriate indentation for current line as Quixote code.
In usual case returns an integer: the column to indent to.
Returns nil if line starts inside a string, t if in a comment,
\(indent . start-of-block\) if line is within a paren block."
  (save-excursion
    (beginning-of-line)
    (if (= (following-char) ?%)
	nil
      (let ((indent-point (point))
	    (case-fold-search nil)
	    state
	    containing-sexp)
	(if parse-start
	    (goto-char parse-start)
	  (beginning-of-clause)
	  (setq parse-start (point)))
	(setq state (parse-partial-sexp parse-start indent-point))
	(setq containing-sexp (car (cdr state)))
	(cond ((or (nth 3 state) (nth 4 state))
	       ;; return nil or t if should not change this line
	       (nth 4 state))
	      ((null containing-sexp)
	       ;; Line is at top level.
	       ;; Look at previous non-comment line to determine whether
	       ;; we are in top-level or within a clause definition.
	       ;; Set basic-indent accordingly.
	       (let (p)
		 (goto-char indent-point)
		 (if (or (bobp)
			 (not (re-search-backward "^    [^% \t\n\f]" nil 'mv)))
		     (progn
		       (goto-char indent-point)
		       (if (looking-at " *[&?]")
			   0
			 quixote-indent-level))
		   (setq p (point))
		   (while
		       (and (re-search-forward ";;[ \t\n\f]" indent-point 'mv)
			    (progn
			      (forward-char -1)
			      (setq state (parse-partial-sexp p (point)))
			      (or (nth 4 state) (nth 3 state) (nth 1 state)))))
		   (if (< (point) indent-point)
		       (progn
			 (goto-char indent-point)
			 (if (looking-at " *[&?]")
			     0
			   quixote-indent-level))
		     (goto-char parse-start)
		     (if (re-search-forward "<=" indent-point 'mv)
			 (+ (current-column)
			    quixote-paren-offset)
		       (+ quixote-indent-level
			  ;; Now add a little if this is a continuation line.
			(progn
			  (quixote-backward-to-noncomment parse-start)
			  (beginning-of-line)
			  (if (looking-at quixote-term-separator)
			      0
			    quixote-continued-statement-offset))))))))
	      ((or (/= (char-after containing-sexp) ?\[)
		   (= (char-syntax (char-after (1- containing-sexp))) ?w))
	       ;; line is argument, not statement.  Return a list.
	       (cons
		(if (nth 2 state)
		    ;; If not the start of first argument,
		    ;; indent the same amount as the first argument
		    (progn
		      (goto-char (1+ containing-sexp))
		      (skip-chars-forward " \t\n" indent-point)
		      (+ (current-column)
			 (progn (goto-char indent-point)
				(quixote-backward-to-noncomment containing-sexp)
				(beginning-of-line)
				(if (looking-at quixote-term-separator) 0
				  quixote-continued-statement-offset))))
		  ;; the first argument
		  ;; indent to the start of predicate + alpha
		  (goto-char (1- containing-sexp))
		  (re-search-backward "\\S_\\<" nil 'mv)
		  (forward-char 1)
		  (+ (current-column) quixote-first-argument-offset))
		(save-excursion
		  (goto-char containing-sexp)
		  (current-column))))
	      (t
	       ;; Statement level.  Return a list.
	       (let (current-block-indent block-paren)
		 (goto-char containing-sexp)
		 ;; At first, find indentation of current block
		 (setq block-paren
		       (car (cdr (parse-partial-sexp parse-start (point)))))
		 (setq current-block-indent
		       (save-excursion
			 (if (not block-paren)
			     (current-indentation)
			   (goto-char block-paren)
			   (quixote-forward-to-noncomment containing-sexp)
			   (current-column))))
		 (cons
		  ;; Is line a first statement after an open-paren?
		  (or
		   ;; If no, find that first statement and indent like it.
		   (save-excursion
		     (quixote-forward-to-noncomment indent-point)
		     ;; The first following code counts
		     ;; if it is before the line we want to indent.
		     (and (< (point) indent-point)
			  (+ (current-column)
			     (progn
			       (goto-char indent-point)
			       (quixote-backward-to-noncomment containing-sexp)
			       (beginning-of-line)
			       (if (looking-at quixote-term-separator)
				   0
				 quixote-continued-statement-offset)))))
		   ;; If no previous statement,
		   ;; indent it relative to line paren is on.
		   (+ current-block-indent quixote-first-argument-offset))
		  current-block-indent))))))))

(defun calculate-quixote-indent-within-comment ()
  "Return the indentation amount for line, assuming that
the current line is to be regarded as part of a block comment."
  (let (end star-start)
    (save-excursion
      (beginning-of-line)
      (skip-chars-forward " \t")
      (setq star-start (= (following-char) ?\*))
      (skip-chars-backward " \t\n")
      (setq end (point))
      (beginning-of-line)
      (skip-chars-forward " \t")
      (and (re-search-forward "/\\*[ \t]*" end t)
	   star-start
	   (goto-char (1+ (match-beginning 0))))
      (current-column))))

(defun quixote-backward-to-noncomment (lim)
  (let (opoint stop)
    (while (not stop)
      (skip-chars-backward " \t\n\f" lim)
      (setq opoint (point))
      (if (and (>= (point) (+ 2 lim))
	       (= (preceding-char) ?/) (= (char-after (- (point) 2)) ?*))
	  (search-backward "/*" lim 'mv)
	(let ((p (max lim (save-excursion (beginning-of-line) (point)))))
	  (if (nth 4 (parse-partial-sexp p (point)))
	      (search-backward "%" p 'mv)
	    (goto-char opoint)
	    (setq stop t)))))))

(defun quixote-forward-to-noncomment (lim)
  (forward-char 1)
  (while (progn
	   (skip-chars-forward " \t\n" lim)
	   (looking-at "%\\|/\\*"))
    ;; Skip over comments and labels following openparen.
    (if (= (following-char) ?\%)
	(forward-line 1)
      (forward-char 2)
      (search-forward "*/" lim 'mv))))

(defun mark-quixote-clause ()
  "Put mark at end of current quixote clause, point at beginning."
  (interactive)
  (push-mark (point))
  (end-of-clause)
  (push-mark (point))
  (beginning-of-clause))

(defun mark-quixote-clauses ()
  "Put mark at end of quixote clause group of the same
predicate, point at beginning."
  (interactive)
  (push-mark (point))
  (if (not (looking-at "^\\(\\sw\\|\\s_\\)+\\(\\[\\|/\\|;;\\)"))
      (re-search-backward "^\\(\\sw\\|\\s_\\)+\\([\\|/\\|;;\\)" nil t))
  (let ((predicate
	 (concat "^" (buffer-substring (match-beginning 0) (match-end 0)))))
    (while (re-search-forward predicate nil t)
      (end-of-clause))
    (push-mark (point))
    (while (re-search-backward predicate nil t))))

(defun quixote-indent-current-clause ()
  "Indent all lines in a curren Quixote clause."
  (interactive)
  (let (p)
    (save-excursion
      (end-of-clause)
      (setq p (point-marker))
      (beginning-of-clause)
      (while (< (point) p)
	(quixote-indent-line)
	(forward-line 1)))))

;;;
;;; Inferior quixote mode
;;;
(defvar inferior-quixote-mode-map nil)

(defun inferior-quixote-mode ()
  "Major mode for interacting with an inferior Micro Quixote process.

The following commands are available:
\\{inferior-quixote-mode-map}

Entry to this mode calls the value of quixote-mode-hook with no arguments,
if that value is non-nil.  Likewise with the value of shell-mode-hook.
quixote-mode-hook is called after shell-mode-hook.

You can send text to the inferior Micro Quixote from other buffers
using the commands send-region, send-string and \\[quixote-consult-region].

Commands:
Tab indents for Qrolog; with argument, shifts rest
 of expression rigidly with the current line.
Paragraphs are separated only by blank lines and '%%'. '%'s start comments.

Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.
\\[shell-send-eof] sends end-of-file as input.
\\[kill-shell-input] and \\[backward-kill-word] are kill commands, imitating normal Unix input editing.
\\[interrupt-shell-subjob] interrupts the shell or its current subjob if any.
\\[stop-shell-subjob] stops, likewise. \\[quit-shell-subjob] sends quit signal, likewise."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'inferior-quixote-mode)
  (setq mode-name "Inferior Micro Quixote")
  (setq mode-line-process '(": %s"))
  (quixote-mode-variables)
  (require 'shell)
  (if inferior-quixote-mode-map
      nil
    (setq inferior-quixote-mode-map (copy-keymap shell-mode-map))
    (quixote-mode-commands inferior-quixote-mode-map))
  (use-local-map inferior-quixote-mode-map)
  (make-local-variable 'last-input-start)
  (setq last-input-start (make-marker))
  (make-local-variable 'last-input-end)
  (setq last-input-end (make-marker))
  (make-variable-buffer-local 'shell-prompt-pattern)
  (setq shell-prompt-pattern "^\\(\\?- \\|.+> \\)") ;Set quixote prompt pattern
  (run-hooks 'shell-mode-hook 'quixote-mode-hook))

(defvar inferior-bc nil
  "BC - the calculator")

(defvar quixote-process nil
  "Micro Quixote")

(defun run-quixote ()
  "Run an inferior Micro Quxote process, input and output via buffer *quixote*."
  (interactive)
  (require 'shell)
  (let ((process-connection-type nil))
    (switch-to-buffer (make-shell "quixote"
				  inferior-quixote-program nil "-i")))
  (let* ((buffer (current-buffer))
	 (proc (get-buffer-process buffer)))
    (setq quixote-process proc)
    (if (boundp 'MULE)
	(set-process-coding-system proc *internal* *internal*))
    (if (boundp 'NEMACS)
	(set-process-kanji-code proc 3))
    (set-process-sentinel proc 'quixote-process-sentinel)
    (set-process-filter proc 'quixote-process-filter))
  (setq inferior-bc nil)
  (inferior-quixote-mode))

(defun quixote-process-sentinel (process status)
  (if inferior-bc
      (delete-process inferior-bc)))

(defun quixote-create-bc-process ()
  (setq inferior-bc (start-process "bc" nil "bc"))
  (process-kill-without-query inferior-bc)
  (set-process-sentinel inferior-bc 'bc-process-sentinel)
  (set-process-filter inferior-bc 'bc-process-filter))

(defun bc-process-sentinel (process status)
  (setq inferior-bc nil))

(defvar quixote-ext-display nil)

(defun bc-process-filter (process string)
  (cond ((= (aref string 0) ?y)
	 (if quixote-ext-display
	     (quixote-output-to-buffer quixote-process string))
	 (process-send-string quixote-process "\n"))
	((= (aref string 0) ?n)
	 (if quixote-ext-display
	     (quixote-output-to-buffer quixote-process string))
	 (process-send-string quixote-process "no.\n"))
	((>= (aref string 0) ?A)
	 ())
	(t
	 (if quixote-ext-display
	     (quixote-output-to-buffer quixote-process string))
	 (process-send-string quixote-process string))))

(defvar quixote-expect-unique-answers nil
  "")

(defun quixote-create-uniq-process ()
  (let ((process-connection-type nil))
    (setq inferior-uniq
	  (start-process "uniq" nil "sh" "-c" "sort | uniq"))
    (if (boundp 'MULE)
	(set-process-coding-system inferior-uniq *internal* *internal*))
    (if (boundp 'NEMACS)
	(set-process-kanji-code inferior-uniq 3))
    (process-kill-without-query inferior-uniq)
    (set-process-sentinel inferior-uniq 'uniq-process-sentinel)
    (set-process-filter inferior-uniq 'uniq-process-filter)))

(defun uniq-process-sentinel (process status)
  (setq inferior-uniq nil))

(defun uniq-process-filter (process string)
  (quixote-output-to-buffer quixote-process string))

(defun quixote-output-to-buffer (process string)
  (let ((opoint (point))
	(mark (process-mark process))
	(buffer (process-buffer process))
	(old-buffer (current-buffer)))
    (set-buffer buffer)
    (goto-char mark)
    (if (<= (point) opoint)
	(setq opoint (+ opoint (length string))))
    (insert string)
    (set-marker mark (point))
    (set-buffer old-buffer)))

(defun quixote-process-filter (process string)
  (cond ((string-match
	  "^EXT EXPRESSION\\?\\? \"\\(.+\\)\"\\[arg2=\\(.+\\),arg1=\\(.+\\)\\] \\?\\?> " string)
	 (if (> (match-beginning 0) 0)
	     (quixote-output-to-buffer
	      process
	      (substring string 0 (match-beginning 0))))
	 (quixote-output-to-buffer
	  process
	  (substring string (match-beginning 0) (match-end 0)))
	 (let* ((operator
		 (substring string (match-beginning 1) (match-end 1)))
		(arg2
		 (substring string (match-beginning 2) (match-end 2)))
		(arg1
		 (substring string (match-beginning 3) (match-end 3)))
		(message (concat arg1 operator arg2 "\n")))
	   (if (null inferior-bc)
	       (quixote-create-bc-process))
	   (process-send-string inferior-bc message)))
	((string-match
	  "^EXT CONSTRAINT\\?\\? \\(.+\\) \\([><=]+\\) \\(.+\\) \\?\\?> " string)
	 (if (> (match-beginning 0) 0)
	     (quixote-output-to-buffer
	      process
	      (substring string 0 (match-beginning 0))))
	 (if quixote-ext-display
	     (quixote-output-to-buffer
	      process
	      (substring string (match-beginning 0) (match-end 0))))
	 (let* ((a1
		 (substring string (match-beginning 1) (match-end 1)))
		(op
		 (substring string (match-beginning 2) (match-end 2)))
		(a2
		 (substring string (match-beginning 3) (match-end 3)))
		(message
		 (format "if (%s %s %s) print \"yes.\\n\" else print \"no.\\n\"\n" a1 op a2)))
	   (if (null inferior-bc)
	       (quixote-create-bc-process))
	   (process-send-string inferior-bc message)))
	((string-match
	  "^EXT CONSTRAINT\\?\\? \\(.+\\) regexp \\(.+\\) \\?\\?> " string)
	 (if (> (match-beginning 0) 0)
	     (quixote-output-to-buffer
	      process
	      (substring string 0 (match-beginning 0))))
	 (if quixote-ext-display
	     (quixote-output-to-buffer
	      process
	      (substring string (match-beginning 0) (match-end 0))))
	 (let* ((string1
		 (substring string (match-beginning 1) (match-end 1)))
		(regexp
		 (substring string (match-beginning 2) (match-end 2))))
	   (if (= (aref regexp 0) ?\")
	       (setq regexp (substring regexp 1 -1)))
	   (if (string-match regexp string1)
	       (progn
		 (process-send-string process "\n")
		 (if quixote-ext-display
		     (quixote-output-to-buffer process "yes.\n")))
	     (if quixote-ext-display
		 (quixote-output-to-buffer process "no.\n"))
	     (process-send-string process "no.\n"))))
	(quixote-expect-unique-answers
	 (if (eq quixote-expect-unique-answers t)
	     (if (string-match "^yes." string)
		 (let* ((s (substring string 5))
			(len (length s)))
		   (setq quixote-expect-unique-answers 'read-answer)
		   (quixote-create-uniq-process)
		     (if (> len 0)
			 (process-send-string inferior-uniq s))
		     (quixote-output-to-buffer process "yes.\n"))
	       (quixote-output-to-buffer process string))
	   (if (eq quixote-expect-unique-answers 'read-answer)
	       (if (string-match "^\\(\\?- \\|&> \\)" string)
		   (let ((prompt
			  (substring string (match-beginning 1) (match-end 1))))
		     (if (> (match-beginning 1) 0)
			 (process-send-string
			  inferior-uniq
			  (substring string 0 (match-beginning 1))))
		     (process-send-eof inferior-uniq)
		     (sit-for 1)  ; well Emacs is not complete system...
		     (accept-process-output inferior-uniq)
		     (quixote-output-to-buffer process prompt)
		     (setq quixote-expect-unique-answers t))
		 (process-send-string inferior-uniq string)))))
	(t
	 (if quixote-consulting
	     ()
	   (quixote-output-to-buffer process string)))))

(defvar quixote-consulting nil
  "")

(defun quixote-consult-region (compile beg end)
  "Send the region to the Quixote process made by M-x run-quixote."
  (interactive "P\nr")
  (save-excursion
    (send-region "quixote" beg end)
    (let ((quixote-consulting t))
      (accept-process-output quixote-process)) ))

(defun quixote-consult-region-and-go (compile beg end)
  "Send the region to the inferior Quixote, and switch to *quixote* buffer."
  (interactive "P\nr")
  (quixote-consult-region compile beg end)
  (switch-to-buffer "*quixote*"))
