// Imogene data structures
// See imgene.h for detailed genes 
// Copyright 1993 by Harley Davis and Nitsan Seniak (for byte code compiler)

#ifdef _DEBUG
#define MYTRACE(obj) obj->Trace();
#else
#define MYTRACE(obj)
#endif

                     
typedef enum 
{
	GN_NAGENE, RESTORE_XY,
 	GN_ADD, GN_SUB, GN_MUL, GN_DIV,
  	GN_AND, GN_OR, GN_XOR,
  	GN_X, GN_Y,
  	GN_RAND,
  	GN_MAX, GN_MIN,
  	GN_DISTXY,
  	GN_CONST,
  	GN_SIN, GN_COS,
  	GN_LN,
  	GN_NOT,
  	GN_CIRC,
  	GN_RAND_RANGE,
  	GN_AVG, GN_INV, GN_POLAR_1, GN_POLAR_2,
  	GN_MANH, GN_MDISTXY, 
  	GN_BITMAP  
}
Byte;                     
                     
union ByteCode
{
	Byte code;
	long data;
	HANDLE handle;
};
     
class CImGene;
class CImView;
     
class ByteCodeProgram
{            
	private:
		ByteCode program[MaxProgramLength];
		unsigned length;
		
	public:
		void AddCode(Byte b) { program[length++].code = b; }
		void AddData(long d) { program[length++].data = d; }
		void AddHandle(HANDLE h) { program[length++].handle = h; } 
		void Load(CImGene *); 
		void Run(long x1, long y1, long x2, long y2, 
				CDC* pDC, CDC* pMemDC, CImView* pView);
}; 	

class CImObject : public CObject
// base superclass
{
	protected:
		DECLARE_DYNCREATE(CImObject)
};

class CImGeneration;
class CImEvolution;
class CImMutation;
class CImMutation1;
class CImMutation2;

class CImGene : public CImObject
// a gene.
{
	protected:
		DECLARE_DYNCREATE(CImGene)
		virtual void CopyData(CImGene* pGene);
		virtual long LocalComplexity() { return 1; }
		
		int m_nIndex;				// index supplied by enumeration
	public:
	
#ifdef _DEBUG
		virtual void Trace();
#endif // debug

		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual void Compile(ByteCodeProgram *);
		virtual Byte GeneByte() { return GN_NAGENE; };
		virtual const int NArgs() { return -1; }
		virtual const char* Name() { return "Not a real gene!";}
		virtual long Complexity() { return LocalComplexity(); }
		virtual char* PrintToString();
		virtual void CreateRandom(CImEvolution* pEvolution, int nDepth);
		virtual int Enumerate(int nMaxIndex);
		virtual CImGene* SelectRandomSubgene(int nSize);
		virtual CImGene* CopyReplacingSubgene(int nIndex, CImGene* pSubgene);
		virtual void ReplaceSubgene(CImGene* pNewGene, int nIndex, CImGene* pSubgene);
		virtual BOOL IsTerminal() { return FALSE; }
		virtual CImGene* Mutate(CImEvolution* pEvolution);
		virtual CImGene* MapMutating(CImEvolution* pEvolution, int nPct);
		CImGene* Copy();
		virtual CImGene* Cross(CImGene* pGene2);
		CImGene();
};

class CImGene0 : public CImGene
// genes with 0 arguments (terminals)
{
	protected:
		DECLARE_DYNCREATE(CImGene0)
	public:
		virtual const int NArgs() { return 0; }
		virtual void CreateRandom(CImEvolution*, int nDepth);
		virtual char* PrintToString();
		virtual BOOL IsTerminal() { return TRUE; }
		virtual void CopyData(CImGene* pNewGene);
		virtual int Enumerate(int nMaxIndex);
		virtual CImGene* SelectRandomSubgene(int nSize);
		virtual void ReplaceSubgene(CImGene* pNewGene, int nIndex, CImGene* pSubgene);
		virtual CImGene* MapMutating(CImEvolution* pEvolution, int nPct);
		virtual void Compile(ByteCodeProgram *);		
#ifdef _DEBUG
		virtual void Trace();
#endif
};

class CImGene1 : public CImGene
// genes with 1 argument
{   
	protected:
		DECLARE_DYNCREATE(CImGene1)
		CImGene* m_pArg1;
	public:
		CImGene* GetArg1() { return m_pArg1; }
		void SetArg1(CImGene* pArg) { m_pArg1 = pArg; }
		virtual const int NArgs() { return 1; }
		virtual void CreateRandom(CImEvolution*, int nDepth);
		virtual char* PrintToString();
		virtual void CopyData(CImGene* pNewGene);
		virtual int Enumerate(int nMaxIndex);
		virtual CImGene* SelectRandomSubgene(int nSize);
		virtual void ReplaceSubgene(CImGene* pNewGene, int nIndex, CImGene* pSubgene);
		virtual CImGene* MapMutating(CImEvolution* pEvolution, int nPct);
		virtual long Complexity() { return LocalComplexity()+(m_pArg1->Complexity()); }
		virtual void Compile(ByteCodeProgram *);		
		
		CImGene1(); 
		~CImGene1(void);
		
#ifdef _DEBUG
		virtual void Trace();
#endif
};

class CImGene2 : public CImGene
// genes with 2 arguments
{
	protected:
		DECLARE_DYNCREATE(CImGene2)
		CImGene* m_pArg1;
		CImGene* m_pArg2;
	public:
		CImGene* GetArg1() { return m_pArg1; }
		CImGene* GetArg2() { return m_pArg2; }
		void SetArg1(CImGene* pArg) { m_pArg1 = pArg; }
		void SetArg2(CImGene* pArg) { m_pArg2 = pArg; } 
		virtual const int NArgs() { return 2; }
		virtual void CreateRandom(CImEvolution*, int nDepth);
		virtual char* PrintToString();
		virtual void CopyData(CImGene* pNewGene);
		virtual int Enumerate(int nMaxIndex);
		virtual CImGene* SelectRandomSubgene(int nSize);
		virtual void ReplaceSubgene(CImGene* pNewGene, int nIndex, CImGene* pSubgene);
		virtual CImGene* MapMutating(CImEvolution* pEvolution, int nPct);
		virtual long Complexity()
		{ return LocalComplexity()+(m_pArg1->Complexity())+(m_pArg2->Complexity()); }
		virtual void Compile(ByteCodeProgram *);		
		
		CImGene2();
		~CImGene2(void);
		
#ifdef _DEBUG
		virtual void Trace();
#endif
};

class CImOrganism : public CImObject
// One organism.
{
	protected:
		DECLARE_DYNCREATE(CImOrganism)
	protected:
		CImGene* m_pGenome;		// the genome
		int m_nFitness;
	
	public:
		CImOrganism();
		~CImOrganism();
		
#ifdef _DEBUG
		void Trace() { m_pGenome->Trace(); }
#endif // debug
		
		long Eval(long x, long y, CBitmap* pBitmap) { return m_pGenome->Eval(x, y, pBitmap); }
		char* PrintToString() { return m_pGenome->PrintToString(); }
		void SetFitness(int n) { m_nFitness = n; }
		int GetFitness() { return m_nFitness; }
		CImGene* GetGenome() { return m_pGenome; }
		long Complexity() { return m_pGenome->Complexity(); }
		
		virtual void CreateRandom(CImEvolution* pEvolution, int nMaxDepth);
		virtual void InitFromPrevGen(CImGeneration* pPrevGen, int nMaxDepth);
		virtual void DoCrossover(CImGeneration* pPrevGen);
		virtual void DoCopy(CImGeneration* pPrevGen);
		virtual void DoMutation(CImEvolution* pEvolution);
};

class CImEvolution : public CImObject
// Represents an entire evolution.
// corresponds to a document.
{
	protected:
		DECLARE_DYNCREATE(CImEvolution)
	protected:
		int m_nMaxGens;				// max generations.
		int m_nGenSize;				// size of each generation.
		int m_nMaxDepth;			// max genome depth.
		int m_nCurGen;				// current generation index.
		int m_nMaxComplexity;		// max genome complexity.
		CImGeneration* m_pCurGen;	// current generation.
		CImGeneration* m_pPrevGen;	// previous generation.
		CPtrArray* m_pGenomePool; 	// list of usable genome classes.
		int m_nGenomePoolSize;		// number of genome classes in pool.
		CPtrArray* m_pTerminals;	// list of terminals (0 arg genes).
		int m_nTerminalsSize;		// number of terminal classes
		CPtrArray* m_pMutations1;	// list of 1 arg mutation classes
		int m_nMutationsSize1;		// number of 1 arg mutation classes
		CPtrArray* m_pMutations2;	// list of 2 arg mutation classes
		int m_nMutationsSize2;		// number of 2 arg mutation classes
		CObList* m_pGenerations; 	// list of generations.
		int m_nCrossoverPct;		// % crossover vs. copy
		int m_nMutationPct;			// % mutation vs. leave alone
		char *m_sBitmapLib;			// directory where bitmaps are to be found.

		CRuntimeClass* GetRandomGeneClass();
		CRuntimeClass* GetRandomTerminalClass();
		CRuntimeClass* GetRandomMutationClass1();
		CRuntimeClass* GetRandomMutationClass2();
		void StartGeneration(BOOL bRandom);

	public:
		CImEvolution(int nMaxGens = 1000, int nGenSize = 9, int nMaxDepth = 2);
		~CImEvolution();
		int GetGenSize() 		{ return m_nGenSize; }
		int GetMaxDepth () 		{ return m_nMaxDepth; }
		CImGeneration* 	GetPrevGen () 	{ return m_pPrevGen; }
		int GetCrossoverPct() 	{ return m_nCrossoverPct; }
		int	GetMutationPct() 	{ return m_nMutationPct; }
		int GetMaxComplexity()  { return m_nMaxComplexity; }
		void SetCrossoverPct(int nPct)	{ m_nCrossoverPct = nPct; }
		void SetMutationPct(int nPct) 	{ m_nMutationPct = nPct; }
		void SetMaxComplexity(int nMax) { m_nMaxComplexity = nMax; }
		char *GetBitmapLib() { return m_sBitmapLib; }
		void SetBitmapLib(char *sNewLib) { free(m_sBitmapLib); m_sBitmapLib = sNewLib; }
		
		CImGene* CreateRandomGenome(int MaxDepth);
		CImGene* CreateRandomGenome();
		CImGene* CreateRandomTerminal(int MaxDepth);
		CImMutation1* CreateRandomMutation1();
		CImMutation2* CreateRandomMutation2();
		void NewGeneration();
		void RandomGeneration();
		void InitGenePool();
		void InitMutations();
		
		CImOrganism* GetOrganism(int i, int nGeneration = -1); 
};

class CImGeneration : public CImObject
// Represents one generation.
{	
	protected:
		DECLARE_DYNCREATE(CImGeneration)
	protected:
		int m_nGeneration;
		CObArray* m_pOrganisms;			// array of organisms
		CImEvolution* m_pEvolution;		// the evolution
		int m_nTotalFitness;			// the sum of all orgs' fitnesses
		BOOL m_bRandom;					// was the generation randomly generated?
	public:
		CImGeneration(CImEvolution* pEvolution, int nGeneration, BOOL bRandom);
		CImGeneration();
		~CImGeneration(void);
		virtual CImOrganism* GenerateOrganism(CImEvolution* pEvolution, CImGeneration* pPrevGen, 
												int nMaxDepth, int nMaxCom);
		CImOrganism* GetOrganism(int i);
		virtual int ComputeTotalFitness();
		virtual CImOrganism* SelectOrganism();
		CImEvolution* GetEvolution() { return m_pEvolution; }
};

// Mutation classes

class CImMutation : public CImObject
{
	protected:
		DECLARE_DYNCREATE(CImMutation)
	public:
		virtual const char* Name() { return "mutation"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene* pGene)
		{
			TRACE("** Bad mutation!\n");
			return NULL;
		}
};

class CImMutation1 : public CImMutation
{
	protected:
		DECLARE_DYNCREATE(CImMutation1)
	public:
		virtual const char* Name() { return "1 arg mutation"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene1* pGene)
		{
			TRACE("** Bad 1 arg mutation!\n");
			return NULL;
		}
};

class CImMutation2 : public CImMutation
{
	protected:
		DECLARE_DYNCREATE(CImMutation2)
	public:
		virtual const char* Name() { return "2 arg mutation"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene2* pGene)
		{
			TRACE("** Bad mutation2!");
			return NULL;
		}
};

class CImTakeArg1 : public CImMutation1
{
	protected:
		DECLARE_DYNCREATE(CImTakeArg1)
	public:
		virtual const char* Name() { return "Take argument 1"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene1* pGene);
};        

class CImTakeArg2 : public CImMutation2
{
	protected:
		DECLARE_DYNCREATE(CImTakeArg2)
	public:
		virtual const char* Name() { return "Take argument 2"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene2* pGene);
};

class CImAddLevel1 : public CImMutation1
{
	protected:
		DECLARE_DYNCREATE(CImAddLevel1)
	public:
		virtual const char* Name() { return "Add level 1"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene1* pGene);
};

class CImAddLevel2 : public CImMutation2
{
	protected:
		DECLARE_DYNCREATE(CImAddLevel2)
	public:
		virtual const char* Name() { return "Add level 2"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene2* pGene);
};

class CImReplaceFn1 : public CImMutation1
{
	protected:
		DECLARE_DYNCREATE(CImReplaceFn1)
	public:
		virtual const char* Name() { return "replace fn 1"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene1* pGene);
}; 

class CImReplaceFn2 : public CImMutation2
{
	protected:
		DECLARE_DYNCREATE(CImReplaceFn2)
	public:
		virtual const char* Name() { return "replace fn 2"; }
		virtual CImGene* Mutate(CImEvolution* pEvolution, CImGene2* pGene);
};

