// imgene.h:  Real gene classes
// Copyright 1993 by Harley Davis and Nitsan Seniak (for byte code methods)

class CImAddGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImAddGene)
	public:
		virtual const char* Name() { return "+"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_ADD; };		
};

#define BC_ADD 		{ 	long arg2 = StackPop(sp);						\
				  	  	StackSet(sp, lEvResult(StackTop(sp)+arg2));		\
				  		break; }

class CImSubGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImSubGene)
	public:
		virtual const char* Name() { return "-"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_SUB; };
};

#define BC_SUB		{ long arg2 = StackPop(sp);				\
				  	StackSet(sp, labs(StackTop(sp)-arg2));	\
				  	break; }

class CImMulGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImMulGene)
	public:
		virtual const char* Name() { return "*"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_MUL; };
};

#define BC_MUL		{ long arg2 = StackPop(sp);				\
				  	long arg1 = StackTop(sp); 				\
				  	StackSet(sp, lEvResult(arg1 * arg2));	\
				  	break; }		    

class CImDivGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImDivGene)
	public:
		virtual const char* Name() { return "/"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_DIV; };
};

#define BC_DIV		{ long arg2 = StackPop(sp)+1;								\
				  	long arg1 = StackTop(sp)+1;									\
				  	if (arg1 < arg2) StackSet(sp, (arg1*lResultMax_1)/arg2);	\
				  	else StackSet(sp, (arg2*lResultMax_1)/arg1);				\
				  	break; }
class CImAndGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImAndGene)
	public:
		virtual const char* Name() { return "&"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_AND; };
};

#define BC_AND		{ long arg2 = StackPop(sp);				\
				  	StackSet(sp, (StackTop(sp) & arg2));	\
				 	 break; }

class CImOrGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImOrGene)
	public:
		virtual const char* Name() { return "|"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_OR; };
};

#define BC_OR		{ long arg2 = StackPop(sp);				\
				  	StackSet(sp, (StackTop(sp) | arg2));	\
				  	break; }


class CImXorGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImXorGene)
	public:
		virtual const char* Name() { return "^"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_XOR; };
};

#define BC_XOR		{ long arg2 = StackPop(sp);				\
				 	StackSet(sp, (StackTop(sp) ^ arg2));	\
				  	break; }

class CImXGene : public CImGene0
{
	protected:
		DECLARE_DYNCREATE(CImXGene)
	public:
		virtual const char* Name() { return "x"; }
		virtual long Eval(long x, long, CBitmap* pBitmap) { return x; }
		virtual Byte GeneByte() { return GN_X; };
}; 

#define BC_X		{ StackPush(sp, x);	\
				 	 break; }

class CImYGene : public CImGene0
{
	protected:
		DECLARE_DYNCREATE(CImYGene)
	public:
		virtual const char* Name() { return "y"; }
		virtual long Eval(long, long y, CBitmap*) { return y; }
		virtual Byte GeneByte() { return GN_Y; };
};

#define BC_Y	{ StackPush(sp, y);	\
				  break; }	

class CImRandGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImRandGene)
	public:
		virtual const char* Name() { return "rand"; }
		virtual long Eval(long, long, CBitmap*);
		virtual Byte GeneByte() { return GN_RAND; };
};

#define BC_RAND		{ long arg = StackTop(sp);				\
				  	long r = (rand()%200) - 100;			\
				  	StackSet(sp, lEvResult(labs(arg+r)));	\
				  	break; }
class CImFiltGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImFiltGene)
		long nCorner, nEdge, nCenter, nSum;
	public:
		virtual const char* Name() { return "filt"; }
		virtual long Eval(long, long, CBitmap*);
		virtual long Complexity();
		CImFiltGene() 
		{

			nCorner = rand()%3; 
			nEdge = rand()%6; 
			nCenter = rand()%10; 
			nSum = nCorner+nEdge+nCenter;
			// TRACE("Made filt: %ld, %ld, %ld: %ld", nCorner, nEdge, nCenter, nSum);
		}
};

class CImMaxGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImMaxGene)
	public:
		virtual const char* Name() { return "max"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_MAX; };
};

#define BC_MAX 	{ long arg2 = StackPop(sp);					\
				  long arg1 = StackTop(sp); 				\
				  StackSet(sp, arg1 > arg2 ? arg1 : arg2); 	\
				  break; }

class CImMinGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImMinGene)
	public:
		virtual const char* Name() { return "min"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_MIN; };
};

#define BC_MIN	{ long arg2 = StackPop(sp);					\
				  long arg1 = StackTop(sp);					\
				  StackSet(sp, arg1 < arg2 ? arg1 : arg2);	\
				  break; }

class CImAvgGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImAvgGene)
	public:
		virtual const char* Name() { return "avg"; }
		virtual long Eval(long x, long y, CBitmap* pBitmap);
		virtual Byte GeneByte() { return GN_AVG; };
};

#define BC_AVG 	{ long arg2 = StackPop(sp);            		\
				  StackSet(sp, (arg2 + StackTop(sp))/2);  	\
				  break; }

class CImDistxyGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImDistxyGene)
		long nX, nY;
		virtual long LocalComplexity() { return 3; }
	public:
		virtual const char* Name() { return "dxy"; }
		virtual long Eval(long, long, CBitmap*);
		CImDistxyGene();             
		virtual Byte GeneByte() { return GN_DISTXY; };
		virtual void Compile(ByteCodeProgram *);
};

#define BC_DISTXY	{ long nx = ByteCodeData(pCurByte);			\
				  	long ny = ByteCodeData(pCurByte);         	\
				  	long dy = StackPop(sp) - ny;              	\
				  	long dx = StackTop(sp) - nx;              	\
				  	double dist = sqrt(double(dx*dx + dy*dy)); 	\
				  	StackSet(sp, lEvResult(long(dist)));       	\
				  	break; }
				  	
class CImMdistxyGene : public CImDistxyGene
{
	protected:
		DECLARE_DYNCREATE(CImMdistxyGene)
	public:
		virtual const char* Name() { return "mdxy"; }
		virtual long Eval(long, long, CBitmap*);
		CImMdistxyGene() {};
		virtual Byte GeneByte() { return GN_MDISTXY; }
};

#define BC_MDISTXY	{ long nx = ByteCodeData(pCurByte);			\
				  	long ny = ByteCodeData(pCurByte);         	\
				  	long dy = labs(StackPop(sp) - ny);         	\
				  	long dx = labs(StackTop(sp) - nx);          \
				  	StackSet(sp, (dx+dy)/2); 			      	\
				  	break; }

class CImConstGene : public CImGene0
{
	protected:
		DECLARE_DYNCREATE(CImConstGene)
		long m_nConst;
		char* m_sConst;
	public:
		virtual const char* Name() { return (const char*)m_sConst; }
		virtual long Eval(long, long, CBitmap*) { return m_nConst; }
		virtual void CopyData(CImGene* pNewGene);
		virtual Byte GeneByte() { return GN_CONST; };
		virtual void Compile(ByteCodeProgram *);
		CImConstGene();
		~CImConstGene();
};

#define BC_CONST	{ StackPush(sp, ByteCodeData(pCurByte));	\
				  	break; }

class CImSinGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImSinGene)
		virtual long LocalComplexity() { return 2; }
	public:
		virtual const char* Name() { return "sin"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_SIN; };
};

/*
#define BC_SIN	{ long arg = StackTop(sp);             			\
				  double ds = sin(double(arg)*ScaledPi);       	\
				  StackSet(sp, long((dMaxResult2-1)*ds + lResultMax2));  	\
				  break; }
*/

#define BC_SIN	{ StackSet(sp, lsin(StackTop(sp)));  	\
				  break; }

class CImCosGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImCosGene)
		virtual long LocalComplexity() { return 2; }
	public:
		virtual const char* Name() { return "cos"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_COS; };
};
				  
#define BC_COS	{ StackSet(sp, lcos(StackTop(sp)));	\
				  break; }
				  
class CImLnGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImLnGene)
		virtual long LocalComplexity() { return 2; }
	public:
		virtual const char* Name() { return "ln"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_LN; }
};

#define BC_LN	{ StackSet(sp, lln(StackTop(sp))); break; }


class CImNotGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImNotGene)
	public:
		virtual const char* Name() { return "~"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_NOT; };
};

#define BC_NOT	{ StackSet(sp, labs(~StackTop(sp)));	\
				  break; }

class CImInvGene : public CImGene1
{
	// Do Pickover's inversion.
	protected:
		DECLARE_DYNCREATE(CImInvGene)
	public:
		virtual const char* Name() { return "inv"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_INV; };
		virtual void Compile(ByteCodeProgram *);
};

#define BC_INV	{ StackPush(sp, x);                              	\
				  StackPush(sp, y);                               	\
				  double x2 = dScale(double(x), dResultMax, 2.0);  	\
				  double y2 = dScale(double(y), dResultMax, 2.0);  	\
				  double xy = (x2*x2) + (y2*y2);              		\
				  if(xy != 0)                              			\
				  {                                      			\
				  if(x2<xy)                                 		\
				  	x = long(x2*dResultMax/xy);             		\
				  else                                          	\
				  	x = long(xy*dResultMax/x2);                 	\
				  	                                          		\
				  if(y2<xy)                                  		\
					y = long(y2*dResultMax/xy);         			\
				  else                                  			\
					y = long(xy*dResultMax/y2);         			\
				  }				                        			\
				  break; }
class CImPolxGene : public CImGene1
{
	// polar to cartesian, where x is the radius.
	protected:
		DECLARE_DYNCREATE(CImPolxGene)
	public:
		virtual const char* Name() { return "polx"; }
		virtual long Eval(long x, long y, CBitmap*);
}; 

class CImPolyGene : public CImGene1
{
	// polar to cartesian, where y is the radius.
	protected:
		DECLARE_DYNCREATE(CImPolyGene)
	public:
		virtual const char* Name() { return "poly"; }
		virtual long Eval(long x, long y, CBitmap*);
};

class CImCircGene : public CImGene0
{
	protected:
		DECLARE_DYNCREATE(CImCircGene)
		long m_cx, m_cy;	// center of circle
		virtual long LocalComplexity() { return 2; }
	public:
		virtual const char* Name() { return "circ"; }
		virtual long Eval(long x, long y, CBitmap*);
		CImCircGene() { m_cx = rand()%lResultMax; m_cy = rand()%lResultMax; }
		virtual Byte GeneByte() { return GN_CIRC; }
		virtual void Compile(ByteCodeProgram *);
};

#define BC_CIRC	{ long nx = ByteCodeData(pCurByte);				\
				  long ny = ByteCodeData(pCurByte);           	\
				  double dy = double(y - ny);                   \
				  double dx = double(x - nx);                   \
				  double dist = hypot(dx, dy);  				\
				  StackPush(sp, lEvResult(long(dist)));       	\
				  break; }
				  
class CImManhGene : public CImCircGene
{
	protected:
		DECLARE_DYNCREATE(CImManhGene)
	public:
		virtual const char* Name() { return "manh"; }
		virtual long Eval(long x, long y, CBitmap*);
		CImManhGene() {};
		virtual Byte GeneByte() { return GN_MANH; }
};

#define BC_MANH { long nx = ByteCodeData(pCurByte);				\
				  long ny = ByteCodeData(pCurByte);           	\
				  StackPush(sp, (labs(y-ny)+labs(x-nx))/2);    	\
				  break; }
				  
class CImBitmapGene : public CImGene0
{
	protected:
		DECLARE_DYNCREATE(CImBitmapGene)
		const char *m_sFile;
		long m_nWidth;
		long m_nHeight;
		HANDLE m_pBits;
	public:
		virtual const char* Name() { return m_sFile; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_BITMAP; }
		virtual void Compile(ByteCodeProgram*);
		virtual void CreateRandom(CImEvolution*, int);
		virtual ~CImBitmapGene();
};

#define BC_BITMAP { long nw = ByteCodeData(pCurByte);				\
					long nh = ByteCodeData(pCurByte);				\
					HANDLE pBits = (HANDLE)ByteCodeData(pCurByte);	\
					LPSTR bits = (LPSTR)GlobalLock(pBits);			\
					StackPush(sp, bits[((x*nw)/lResultMax)+			\
									   ((y*nh)/lResultMax)]);		\
					GlobalUnlock(pBits); 							\
					break;											\
					} 

class CImAtanGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImAtanGene)
		virtual long LocalComplexity() { return 2; }
	public:
		virtual const char* Name() { return "atan"; }
		virtual long Eval(long x, long y, CBitmap*);
};

class CImRandRangeGene : public CImGene2
{
	protected:
		DECLARE_DYNCREATE(CImRandRangeGene)
	public:
		virtual const char* Name() { return "rndr"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual Byte GeneByte() { return GN_RAND_RANGE; };
};

#define BC_RAND_RANGE	{ long arg2 = StackPop(sp);					\
				  		long arg1 = StackTop(sp);					\
				  		if (arg1 == arg2) StackSet(sp, arg1);    	\
				  		else { long min = arg1<arg2 ? arg1 : arg2; 	\
				  		 	long max = arg1<arg2 ? arg2 : arg1;    	\
				  		 	StackSet(sp, min+(rand()%(max-min))); }	\
				  		break; }

class CImPolarGene : public CImGene1
{
	protected:
		DECLARE_DYNCREATE(CImPolarGene)
		long m_dx, m_dy; 	// center point
		int m_bRhoX;     	// treat rho as new x, or as new y?
	public:
		virtual const char* Name() { return "polar"; }
		virtual long Eval(long x, long y, CBitmap*);
		virtual void Compile(ByteCodeProgram *);
		CImPolarGene();
};

#define BC_POLAR_1	{ StackPush(sp, x);								\
				  	StackPush(sp, y);                   			\
				  	double dx = double(x-ByteCodeData(pCurByte));	\
				  	double dy = double(y-ByteCodeData(pCurByte)); 	\
				  	double rho = hypot(dx, dy);                   	\
				  	double theta = asin(dy/rho)+(pi/2); 			\
				  	x = long(rho)*lMaxDist/lResultMax;            	\
				  	y = long((theta * dResultMax)/pi);           	\
				  	break; }
				  	
#define BC_POLAR_2	{ StackPush(sp, x);                           	\
				  	StackPush(sp, y);                            	\
				  	double dx = double(x-ByteCodeData(pCurByte));  	\
				  	double dy = double(y-ByteCodeData(pCurByte));  	\
				  	double rho = hypot(dx, dy);                  	\
				  	double theta = asin(dy/rho)+(pi/2);       		\
				  	y = long(rho)*lMaxDist/lResultMax;         		\
				  	x = long((theta * dResultMax)/pi);         		\
				  	break; }
