// CIMOBJ (sic): Basic implementation of Imogene objects
// Copyright 1993 by Harley Davis

#include <math.h>
#include <dos.h>
#include <errno.h>
#include <direct.h>
#include <string.h>
#include "stdafx.h"
#include "iminl.h"
#include "imobj.h"
#include "imgene.h"
#include "showdib.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNCREATE(CImObject, CObject)
IMPLEMENT_DYNCREATE(CImGene, CImObject)
IMPLEMENT_DYNCREATE(CImOrganism, CImObject)
IMPLEMENT_DYNCREATE(CImEvolution, CImObject)
IMPLEMENT_DYNCREATE(CImGeneration, CImObject)

IMPLEMENT_DYNCREATE(CImGene2, CImGene)
IMPLEMENT_DYNCREATE(CImGene1, CImGene)
IMPLEMENT_DYNCREATE(CImGene0, CImGene)

// gene classes
IMPLEMENT_DYNCREATE(CImAddGene, CImGene2)
IMPLEMENT_DYNCREATE(CImSubGene, CImGene2)
IMPLEMENT_DYNCREATE(CImMulGene, CImGene2)
IMPLEMENT_DYNCREATE(CImDivGene, CImGene2)
IMPLEMENT_DYNCREATE(CImAndGene, CImGene2)
IMPLEMENT_DYNCREATE(CImOrGene, 	CImGene2)
IMPLEMENT_DYNCREATE(CImXorGene, CImGene2)
IMPLEMENT_DYNCREATE(CImDistxyGene, CImGene2)
IMPLEMENT_DYNCREATE(CImMdistxyGene, CImDistxyGene)
IMPLEMENT_DYNCREATE(CImMaxGene, CImGene2) 
IMPLEMENT_DYNCREATE(CImMinGene, CImGene2)
IMPLEMENT_DYNCREATE(CImAvgGene, CImGene2)
IMPLEMENT_DYNCREATE(CImRandRangeGene, CImGene2)
IMPLEMENT_DYNCREATE(CImSinGene,	CImGene1) 
IMPLEMENT_DYNCREATE(CImCosGene,	CImGene1)
IMPLEMENT_DYNCREATE(CImLnGene, CImGene1)
IMPLEMENT_DYNCREATE(CImNotGene,	CImGene1)
IMPLEMENT_DYNCREATE(CImRandGene, CImGene1)
IMPLEMENT_DYNCREATE(CImFiltGene, CImGene1)
IMPLEMENT_DYNCREATE(CImAtanGene, CImGene1)
IMPLEMENT_DYNCREATE(CImInvGene, CImGene1)
IMPLEMENT_DYNCREATE(CImPolxGene, CImGene1)
IMPLEMENT_DYNCREATE(CImPolyGene, CImGene1)
IMPLEMENT_DYNCREATE(CImPolarGene, CImGene1)
IMPLEMENT_DYNCREATE(CImXGene,	CImGene0)
IMPLEMENT_DYNCREATE(CImYGene,	CImGene0)
IMPLEMENT_DYNCREATE(CImConstGene, CImGene0)
IMPLEMENT_DYNCREATE(CImCircGene, CImGene0)
IMPLEMENT_DYNCREATE(CImManhGene, CImCircGene)
IMPLEMENT_DYNCREATE(CImBitmapGene, CImGene0)

IMPLEMENT_DYNCREATE(CImMutation, CImObject)
IMPLEMENT_DYNCREATE(CImMutation1, CImMutation)
IMPLEMENT_DYNCREATE(CImMutation2, CImMutation)
IMPLEMENT_DYNCREATE(CImTakeArg1, CImMutation1)
IMPLEMENT_DYNCREATE(CImTakeArg2, CImMutation2)
IMPLEMENT_DYNCREATE(CImAddLevel1, CImMutation1)
IMPLEMENT_DYNCREATE(CImAddLevel2, CImMutation2)
IMPLEMENT_DYNCREATE(CImReplaceFn1, CImMutation1)
IMPLEMENT_DYNCREATE(CImReplaceFn2, CImMutation2)

CImOrganism::CImOrganism()
{
	// construct an organism.
	m_nFitness = 1;	// fitness is at least one to give them all a chance.
}

CImOrganism::~CImOrganism()
{
	delete m_pGenome;
}

CImEvolution::CImEvolution(int nMaxGens, int nGenSize, int nMaxDepth)
{
	TRACE("Creating an evolution.\n");
	m_nMaxGens = nMaxGens;
	m_nGenSize = nGenSize;
	// m_nMaxDepth = nMaxDepth;
	m_nMaxDepth = 2; // 5
	m_nCurGen = 0;
	m_pCurGen = NULL;
	m_pPrevGen = NULL;
	m_nCrossoverPct = 100;	// was 80% crossover, 20% copy
	m_nMutationPct = 100;	// was 50% mutation rate
	m_nMaxComplexity = 100;
	m_sBitmapLib = _getcwd(NULL, 0);
	
	m_pGenerations = new CObList;
	
	InitGenePool();
	InitMutations();
}

CImEvolution::~CImEvolution()
{
	POSITION pos = m_pGenerations->GetHeadPosition();
	while (pos != NULL)
	{
		CImGeneration* pNextGen = (CImGeneration*)m_pGenerations->GetNext(pos);
		delete pNextGen;
	}
    
    delete m_pCurGen;
	delete m_pGenerations;
	free(m_sBitmapLib);
	
	m_pGenomePool->RemoveAll();
	delete m_pGenomePool;
	
	m_pTerminals->RemoveAll();
	delete m_pTerminals;
	
	m_pMutations1->RemoveAll();
	delete m_pMutations1;
	
	m_pMutations2->RemoveAll();
	delete m_pMutations2;
}

void CImEvolution::InitGenePool()
{
	// Add all the known genes to the genome pool
	m_pGenomePool = new CPtrArray;

	m_pGenomePool->Add(RUNTIME_CLASS(CImAddGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImSubGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImMulGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImDivGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImSinGene)); 
	m_pGenomePool->Add(RUNTIME_CLASS(CImCosGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImLnGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImAndGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImOrGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImXorGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImNotGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImDistxyGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImMdistxyGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImRandGene));	
	m_pGenomePool->Add(RUNTIME_CLASS(CImMaxGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImMinGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImAvgGene));
	m_pGenomePool->Add(RUNTIME_CLASS(CImRandRangeGene));	
	m_pGenomePool->Add(RUNTIME_CLASS(CImInvGene));

	m_pGenomePool->Add(RUNTIME_CLASS(CImPolarGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImPolxGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImPolyGene));
	m_nGenomePoolSize = m_pGenomePool->GetSize();
	
	// Losers:
	// m_pGenomePool->Add(RUNTIME_CLASS(CImXGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImYGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImCircGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImConstGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImFiltGene));
	// m_pGenomePool->Add(RUNTIME_CLASS(CImAtanGene));
	
	
	// Set up the list of terminals:  Genes with no arguments.
	m_pTerminals = new CPtrArray;
	m_pTerminals->Add(RUNTIME_CLASS(CImCircGene));
	m_pTerminals->Add(RUNTIME_CLASS(CImManhGene));
	m_pTerminals->Add(RUNTIME_CLASS(CImXGene));
	m_pTerminals->Add(RUNTIME_CLASS(CImYGene));
	m_pTerminals->Add(RUNTIME_CLASS(CImConstGene));
	// Sadly, not yet ready...
	// m_pTerminals->Add(RUNTIME_CLASS(CImBitmapGene));
	m_nTerminalsSize = m_pTerminals->GetSize();
}

void CImEvolution::InitMutations()
{
	// Initialize the list of mutation classes
	// 1 arg mutations
	m_pMutations1 = new CPtrArray;
	m_pMutations1->Add(RUNTIME_CLASS(CImTakeArg1));
	m_pMutations1->Add(RUNTIME_CLASS(CImReplaceFn1));
	m_pMutations1->Add(RUNTIME_CLASS(CImAddLevel1));
	m_nMutationsSize1 = m_pMutations1->GetSize();
	
	// 2 arg mutations
	m_pMutations2 = new CPtrArray;
	m_pMutations2->Add(RUNTIME_CLASS(CImTakeArg2));
	m_pMutations2->Add(RUNTIME_CLASS(CImReplaceFn2));
	m_pMutations2->Add(RUNTIME_CLASS(CImAddLevel2));
	m_nMutationsSize2 = m_pMutations2->GetSize();
}

CRuntimeClass* CImEvolution::GetRandomGeneClass()
{
	return (CRuntimeClass*)m_pGenomePool->GetAt(rand()%m_nGenomePoolSize);
}

CRuntimeClass* CImEvolution::GetRandomTerminalClass()
{
	return (CRuntimeClass*)m_pTerminals->GetAt(rand()%m_nTerminalsSize);
}

CRuntimeClass* CImEvolution::GetRandomMutationClass1()
{
	return (CRuntimeClass*)m_pMutations1->GetAt(rand()%m_nMutationsSize1);
}

CRuntimeClass* CImEvolution::GetRandomMutationClass2()
{
	return (CRuntimeClass*)m_pMutations2->GetAt(rand()%m_nMutationsSize2);
}

void CImEvolution::NewGeneration()
{
 	StartGeneration(FALSE);
}

void CImEvolution::RandomGeneration()
{
	StartGeneration(TRUE);
}

void CImEvolution::StartGeneration(BOOL bRandom)
{
	// new generation
	TRACE("CImEvolution: Making new generation.\n");
	m_nCurGen++;
	if(!m_pCurGen==NULL)
	{
		// not the first generation.
		m_pCurGen->ComputeTotalFitness();
		m_pGenerations->AddTail(m_pCurGen);
		m_pPrevGen = m_pCurGen;
	}
	m_pCurGen = new CImGeneration(this, m_nCurGen, bRandom);
}

// Creating random genomes from the gene pool.
// Need to be able to create both regular hierarchical genes
// and terminals.

CImGene* CImEvolution::CreateRandomGenome()
{
	CRuntimeClass* pGeneClass = GetRandomGeneClass();
	return (CImGene*)pGeneClass->CreateObject();
}

CImGene* CImEvolution::CreateRandomGenome(int MaxDepth)
{
	// TRACE("CImEvolution: Creating random genome at depth %d\n", MaxDepth);
	CRuntimeClass* pGeneClass = GetRandomGeneClass();
	CImGene* pGenome = (CImGene*)pGeneClass->CreateObject();
	pGenome->CreateRandom(this, rand()%(MaxDepth+1));
	return pGenome;
}

CImGene* CImEvolution::CreateRandomTerminal(int MaxDepth)
{
	// TRACE("CImEvolution: Creating random terminal at depth %d\n", MaxDepth);
	CRuntimeClass* pGeneClass = GetRandomTerminalClass();
	CImGene* pGenome = (CImGene*)pGeneClass->CreateObject();
	pGenome->CreateRandom(this, MaxDepth);
	return pGenome;
}

CImMutation1* CImEvolution::CreateRandomMutation1()
{
	TRACE("CImEvolution: Creating random 1 arg mutation.\n");
	CRuntimeClass* pMutationClass = GetRandomMutationClass1();
	CImMutation1* pMutation = (CImMutation1*)pMutationClass->CreateObject();
	TRACE("Created %s.\n", pMutation->Name());
	return pMutation;
}

CImMutation2* CImEvolution::CreateRandomMutation2()
{
	TRACE("CImEvolution: Creating 2 arg random mutation.\n");
	CRuntimeClass* pMutationClass = GetRandomMutationClass2();
	CImMutation2* pMutation = (CImMutation2*)pMutationClass->CreateObject();
	TRACE("Created %s.\n", pMutation->Name());
	return pMutation;
}

CImOrganism* CImEvolution::GetOrganism(int i, int nGeneration)
{
	// return the organism in the generation nGeneration (default m_nCurGen) at index i.
	if ((nGeneration==-1) || (nGeneration==m_nCurGen))	// default is -1, meaning current generation
		return m_pCurGen->GetOrganism(i);
	else
	{
		TRACE("Finding organism %d in generation %d\n", i, nGeneration);
		POSITION index = m_pGenerations->FindIndex(nGeneration-1);
		CImGeneration* pGen = (CImGeneration*)(m_pGenerations->GetAt(index));
		return pGen->GetOrganism(i);
	}
}	

CImGeneration::CImGeneration(CImEvolution* pEvolution, int nGeneration, BOOL bRandom)
{
	// create a generation 
	TRACE("Constructing generation %d.\n", nGeneration);
	m_nTotalFitness = 0;
	m_nGeneration = nGeneration;
	m_pEvolution = pEvolution;
	CImGeneration* pPrevGen = pEvolution->GetPrevGen();
	m_bRandom = bRandom;
	int nMaxDepth = pEvolution->GetMaxDepth();
	int nGenSize = pEvolution->GetGenSize();
	int nMaxCom = pEvolution->GetMaxComplexity();
	// create and fill in organism array.
	m_pOrganisms = new CObArray;
	m_pOrganisms->SetSize(nGenSize);
	for(int i = 0; i < nGenSize; i++)
	{
		CImOrganism* pNewOrg = GenerateOrganism(pEvolution, pPrevGen, nMaxDepth, nMaxCom);
		m_pOrganisms->SetAt(i, pNewOrg);
	}
}	

CImGeneration::CImGeneration() {}

CImGeneration::~CImGeneration()
{
	// destroy a generation
	for(int i = 0; i < m_pEvolution->GetGenSize(); i++)
	{
		delete m_pOrganisms->GetAt(i);
	}
	delete m_pOrganisms;
}

int CImGeneration::ComputeTotalFitness()
{
	m_nTotalFitness = 0;
	for(int i = 0; i < m_pEvolution->GetGenSize(); i++)
	{
		m_nTotalFitness += GetOrganism(i)->GetFitness();
	}
	TRACE("Total fitness = %d\n", m_nTotalFitness);
	return m_nTotalFitness;
}

CImOrganism* CImGeneration::SelectOrganism()
// Select a random individual from the generation based on
// proportional fitnesses.
{
	int nOrgIndex = rand()%m_nTotalFitness;
	int nSum = 0;
	for(int i = 0; i < m_pEvolution->GetGenSize(); i++)
	{
		CImOrganism* pOrg = GetOrganism(i);
		int nNewSum = nSum + pOrg->GetFitness();
		if((nOrgIndex>=nSum)&&(nOrgIndex<nNewSum))
		{
			// TRACE("Selected organism #%d\n", i);
			return pOrg;                      
		}
		nSum = nNewSum;
	}
	TRACE("** Couldn't select organism!\n");
	return NULL;
}	

CImOrganism* CImGeneration::GetOrganism(int i)
{
	// return organism at index i.
	return (CImOrganism*)(m_pOrganisms->GetAt(i));
}

CImOrganism* CImGeneration::GenerateOrganism(CImEvolution* pEvolution, CImGeneration* pPrevGen, 
												int nMaxDepth, int nMaxCom)
{
	// Generate an organism
	CImOrganism* pNewOrg = new CImOrganism;
	int trials = 0;

	start:
	
	trials++;
	TRACE("** Trial #%d...\n", trials);
	
	if((pPrevGen==NULL)||m_bRandom)
	{
		// the first or a random generation -- generate randomly.
		// TRACE("Creating random generation organism, MaxDepth = %d.\n", nMaxDepth);
		pNewOrg->CreateRandom(pEvolution, nMaxDepth);
	}
	else
	{
		// create from previous generation.
		// TRACE("Creating next generation organism, MaxDepth = %d.\n", nMaxDepth);
		pNewOrg->InitFromPrevGen(pPrevGen, nMaxDepth);
	}
	
	if ((pNewOrg->Complexity() > nMaxCom) && (trials < 10))
	{
		TRACE("* Organism too complex; regenerating.\n");
		delete pNewOrg;
		pNewOrg = new CImOrganism;
		goto start;
	}
	
	TRACE("** Generated '");
	MYTRACE(pNewOrg)
	TRACE("' in %d trials.\n\n", trials);
	return pNewOrg;
}

// Organism Operations

void CImOrganism::CreateRandom(CImEvolution* pEvolution, int MaxDepth)
// To create a random individual, we select a random gene from
// the gene pool and initialize it with CreateRandom.
// We use Koza's "grow" method, rather than the "full" or
// "ramped 1/2 & 1/2" methods.  See Koza, p. 92-93. 
{
	// TRACE("CImOrganism: CreateRandom with MaxDepth = %d\n", MaxDepth);
	m_pGenome = pEvolution->CreateRandomGenome(MaxDepth);	
}

void CImOrganism::InitFromPrevGen(CImGeneration* pPrevGen, int MaxDepth)
// Initializing an organism from the previous generation.
// Two choices:
//  1. Copy the individual.
//  2. Crossover between two individuals.
// After creating, see if mutation should occur. 
{
	int nCrossover = rand()%100;
	int nMutation = rand()%100;
	CImEvolution* pEvolution = pPrevGen->GetEvolution();
	
	if(nCrossover<=(pEvolution->GetCrossoverPct()))
	{
		DoCrossover(pPrevGen);
	}
	else
	{
		DoCopy(pPrevGen);
	}
	
	if(nMutation<=(pEvolution->GetMutationPct()))
	{
		DoMutation(pEvolution);
	} 
}

void CImOrganism::DoCrossover(CImGeneration* pPrevGen)
// pick 2 organisms from the previous generation and cross them.
// \\ this is not real crossover because we don't exchange, but just put
//    part of one parent in the other.
{
	TRACE("Doing crossover.\n");
	CImOrganism* pOrg1 = pPrevGen->SelectOrganism();
	CImOrganism* pOrg2 = pPrevGen->SelectOrganism();
	
	CImGene* pGene1 = pOrg1->GetGenome();
	CImGene* pGene2 = pOrg2->GetGenome();

	m_pGenome = pGene1->Cross(pGene2);
}

void CImOrganism::DoCopy(CImGeneration* pPrevGen)
// pick one organism from the previous generation and copy it.
{
	TRACE("Copying organism.\n");
	CImOrganism* pOrg = pPrevGen->SelectOrganism();
	m_pGenome = pOrg->GetGenome()->Copy();
}

void CImOrganism::DoMutation(CImEvolution* pEvolution)
// Mutate the organism.
{
	TRACE("Mutating.\n"); 
	m_pGenome = m_pGenome->Mutate(pEvolution);
	return;
}


// Gene operations

// CreateRandom:  Create a random genome using the grow method.

void CImGene::CreateRandom(CImEvolution*, int)
// \\ should be pure virtual? 
{   
	TRACE("** Creating random for CImGene.\n");
	return; }

void CImGene0::CreateRandom(CImEvolution*, int)
// normally, nothing to do for a terminal. 
{   
	// TRACE("Creating random terminal '%s'\n", Name());
	return; 
}

void CImGene1::CreateRandom(CImEvolution* pEvolution, int MaxDepth)
// if MaxDepth is at zero, create a terminal for the argument.
// otherwise, create a random genome with MaxDepth - 1. 
{
	// TRACE("Creating random gene1 '%s', depth = %d\n", Name(), MaxDepth);
	if (MaxDepth==0)
		{
			m_pArg1 = pEvolution->CreateRandomTerminal(MaxDepth);
		}
	else
		{
			m_pArg1 = pEvolution->CreateRandomGenome(MaxDepth-1);
		}
	return;
}

void CImGene2::CreateRandom(CImEvolution* pEvolution, int MaxDepth) 
{
	// TRACE("Creating random gene2 '%s', depth = %d\n", Name(), MaxDepth);
	if (MaxDepth==0)
		{
			m_pArg1 = pEvolution->CreateRandomTerminal(MaxDepth);
			m_pArg2 = pEvolution->CreateRandomTerminal(MaxDepth);
		}
	else
		{
			m_pArg1 = pEvolution->CreateRandomGenome(MaxDepth-1);
			m_pArg2 = pEvolution->CreateRandomGenome(MaxDepth-1);
		}
	return;
}

// Copy a genome.

CImGene* CImGene::Copy()
{
	// TRACE("Copying gene %s\n", Name());
	CImGene* pNewGene = (CImGene*)(GetRuntimeClass()->CreateObject());
	CopyData(pNewGene);
	return pNewGene;
}

void CImGene::CopyData(CImGene* pNewGene)
// copy data from this to pNewGene
{
	TRACE("** Copying raw gene data!\n");
}

void CImGene0::CopyData(CImGene* pNewGene)
{
	// TRACE("Copying gene0 %s\n", Name());
	return;
}

void CImGene1::CopyData(CImGene* pNewGenex)
{
	CImGene1* pNewGene = (CImGene1*)pNewGenex;
	// TRACE("gene1: Copying data from a %s to a %s\n", Name(), pNewGene->Name());
	pNewGene->m_pArg1 = (m_pArg1->Copy());
	ASSERT((pNewGene->m_pArg1)!=NULL);
	// TRACE("Arg1: %s -> %s\n", m_pArg1->Name(), (pNewGene->m_pArg1)->Name());
}

void CImGene2::CopyData(CImGene* pNewGenex)
{
	CImGene2* pNewGene = (CImGene2*)pNewGenex; 
	// TRACE("gene2: Copying data from a %s to a %s\n", Name(), pNewGene->Name());
	pNewGene->m_pArg1 = (m_pArg1->Copy());
	ASSERT((pNewGene->m_pArg1)!=NULL);
	// TRACE("Arg1: %s -> %s\n", m_pArg1->Name(), (pNewGene->m_pArg1)->Name());
	pNewGene->m_pArg2 = (m_pArg2->Copy());
	ASSERT((pNewGene->m_pArg2)!=NULL);
	// TRACE("Arg2: %s -> %s\n", m_pArg2->Name(), (pNewGene->m_pArg2)->Name());
}

void CImConstGene::CopyData(CImGene* pNewGenex)
{
	CImConstGene* pNewGene = (CImConstGene*)pNewGenex;
	// TRACE("const gene: copying data from %s\n", Name(), pNewGene->Name());
	delete pNewGene->m_sConst;	// delete old name. 
	pNewGene->m_nConst = m_nConst;
	pNewGene->m_sConst = new char[8];
	strcpy(pNewGene->m_sConst, (const char*)m_sConst);
	// TRACE("Arg1: %s -> %s\n", m_sConst, pNewGene->m_sConst);
}

// Crossover:
//  1. number all the genes in this and gene2
//  2. select a random gene from gene2.
//  3. select a random index from this.
//  4. copy this, replacing indexth gene with select part of gene2.

// This crossover is somewhat nonstandard, since we pick two parents
// and only produce one offspring, rather than switching the trees.
// This is probably ok for an entertainment program, but is quite bogus
// for real GP. 

CImGene* CImGene::Cross(CImGene* pGene2)
{
	TRACE("CImGene: Crossing ");
	MYTRACE(this)
	TRACE("\n and ");
	MYTRACE(pGene2)
	TRACE("\n");
	
	// enumerate both genes.
	int nSize1, nSize2;
	nSize1 = Enumerate(0);
	nSize2 = pGene2->Enumerate(0);
	
	// select subgene from gene2.
	int nSubIndex = rand()%(nSize2+1);
	CImGene* pSubgene = pGene2->SelectRandomSubgene(nSubIndex);
	ASSERT(pSubgene!=NULL);
	TRACE("Chose subgene %d (%s)\n", nSubIndex, pSubgene->Name());
	
	// select random index from this.
	int nCopyIndex = rand()%(nSize1+1);
	
	// copy this, replacing with subgene @ copy index.
	TRACE("Copying subgene.\n");
	CImGene* pSubgeneCopy = pSubgene->Copy();
	ASSERT(pSubgeneCopy!=NULL);
	TRACE("Copying this replacing w/subgene @ %d\n", nCopyIndex);
	CImGene* pResult = CopyReplacingSubgene(nCopyIndex, pSubgeneCopy);
	
	ASSERT(pResult!=NULL);
	TRACE("Produced ");
	MYTRACE(pResult)
	TRACE("\n");
	
	return pResult;
}

// Mutation

CImGene* CImGene::Mutate(CImEvolution* pEvolution)
{
	Enumerate(0);
	int nPct = pEvolution->GetMutationPct()/(m_nIndex+1);
	return MapMutating(pEvolution, nPct);
}

CImGene* CImGene::MapMutating(CImEvolution*, int)
{
	TRACE("** Mutating raw gene!\n");
	return NULL;
}

CImGene* CImGene0::MapMutating(CImEvolution* pEvolution, int nPct)
{
	if((rand()%100)<=nPct)
	{
		// only one choice: create random.
		delete this;
		return pEvolution->CreateRandomTerminal(0);
	}
	else
	{
		return this;
	}
}

CImGene* CImGene1::MapMutating(CImEvolution* pEvolution, int nPct)
{
	if((rand()%100)<=nPct)
	{
		// Choose a 1-arg mutation and execute it.
		CImMutation1* pMutation = pEvolution->CreateRandomMutation1();
		CImGene* pResult = pMutation->Mutate(pEvolution, this);
		delete pMutation;
		return pResult;
	}
	else
	{   
		m_pArg1 = m_pArg1->MapMutating(pEvolution, nPct);
		return this;
	}
}

CImGene* CImGene2::MapMutating(CImEvolution* pEvolution, int nPct)
{ 
	if((rand()%100)<=nPct)
	{
		// Choose a 2-arg mutation and execute it.
		CImMutation2* pMutation = pEvolution->CreateRandomMutation2();
		CImGene* pResult = pMutation->Mutate(pEvolution, this);
		delete pMutation;
		return pResult;	
	}
	else // continue mapping in the arguments.
	{
		m_pArg1 = m_pArg1->MapMutating(pEvolution, nPct);
		m_pArg2 = m_pArg2->MapMutating(pEvolution, nPct);
		return this;
	}
}
	

// Enumeration

int CImGene::Enumerate(int nIndex)
{
	TRACE("** Enumerating raw gene @ %d\n", nIndex);
	return -1;
}

int CImGene0::Enumerate(int nIndex)
{
	m_nIndex = nIndex;
	return nIndex;
}

int CImGene1::Enumerate(int nIndex)
{
	m_nIndex = nIndex;
	return m_pArg1->Enumerate(nIndex+1);
}

int CImGene2::Enumerate(int nIndex)
{
	int nIndex1 = m_pArg1->Enumerate(nIndex);
	m_nIndex = nIndex1+1;
	return m_pArg2->Enumerate(nIndex1+2);
}

// Select a random subgene.

CImGene* CImGene::SelectRandomSubgene(int Index)
{
	TRACE("** Selecting random gene @ %d\n", Index);
	return NULL;
}

CImGene* CImGene0::SelectRandomSubgene(int Index)
{
	if(m_nIndex==Index)
	{
		return this;
	}
	else
	{
		// This case should be an error!
		TRACE("** Wrong index for terminal %s: %d vs %d!\n", Name(), Index, m_nIndex);
		return NULL;
	}
}

CImGene* CImGene1::SelectRandomSubgene(int Index)
{
	if(m_nIndex==Index)
	{
		return this;
	}
	else
	{
		return m_pArg1->SelectRandomSubgene(Index);
	}
}

CImGene* CImGene2::SelectRandomSubgene(int Index)
{
	if(m_nIndex==Index)
	{
		return this;
	}
	else if (Index<m_nIndex)
	{
		return m_pArg1->SelectRandomSubgene(Index);
	}
	else
	{
		return m_pArg2->SelectRandomSubgene(Index);
	}
}

// Copy replacing at given index.
CImGene* CImGene::CopyReplacingSubgene(int Index, CImGene* Subgene)
{
	if(m_nIndex==Index)
	{
		return Subgene;
	}
	else
	{
		CImGene* pNewGene = (CImGene*)(GetRuntimeClass()->CreateObject());
		ReplaceSubgene(pNewGene, Index, Subgene);
		return pNewGene;
	}
}    

void CImGene::ReplaceSubgene(CImGene* NewGene, int Index, CImGene* Subgene)
{
	TRACE("** In %s, replacing raw subgene @ %d: %s", Name(), Index, Subgene->Name());
}

void CImGene0::ReplaceSubgene(CImGene* NewGene, int, CImGene*)
{
	CopyData(NewGene);
}

void CImGene1::ReplaceSubgene(CImGene* NewGene, int Index, CImGene* Subgene)
{
	((CImGene1*)NewGene)->m_pArg1 = m_pArg1->CopyReplacingSubgene(Index, Subgene);
}

void CImGene2::ReplaceSubgene(CImGene* NewGene, int Index, CImGene* Subgene)
{
	CImGene2* pNewGene = (CImGene2*)NewGene;
	if(Index<m_nIndex)
	{
		pNewGene->m_pArg1 = m_pArg1->CopyReplacingSubgene(Index, Subgene);
		pNewGene->m_pArg2 = m_pArg2->Copy();
	}
	else
	{
		pNewGene->m_pArg1 = m_pArg1->Copy();
		pNewGene->m_pArg2 = m_pArg2->CopyReplacingSubgene(Index, Subgene);
	}
}
			

// Evaluation of genes for a given point.

long CImGene::Eval(long, long, CBitmap*) { return -1; }

long CImAddGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return lEvResult(arg1+arg2);
}

long CImSubGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return labs(arg1-arg2);
}

long CImMulGene::Eval(long x, long y, CBitmap* pBitmap)
{   
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return lEvResult(arg1*arg2);
}

long CImDivGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap)+1;
	long arg2 = m_pArg2->Eval(x, y, pBitmap)+1;
    
    if((arg1==arg2)&&(arg1==0)) return 0;
	if(arg1<arg2) return (arg1*lResultMax_1)/arg2;
	return (arg2*lResultMax_1)/arg1;
}

long CImAndGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return lEvResult(arg1&arg2);
}

long CImOrGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return lEvResult(arg1|arg2);
}

long CImXorGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return lEvResult(arg1^arg2);
}

long CImDistxyGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long dx = (m_pArg1->Eval(x, y, pBitmap)) - nX;
	long dy = (m_pArg2->Eval(x, y, pBitmap)) - nY;
	double dist = sqrt(double(dx*dx + dy*dy));
	return lEvResult(long(dist));
}

long CImMdistxyGene::Eval(long x, long y, CBitmap* pBitmap)
// manhattan distance between (arg1,arg2) and (nX,nY).
// Divide by two to guarantee that we are in bounds.
{
	long dx = labs((m_pArg1->Eval(x, y, pBitmap)) - nX);
	long dy = labs((m_pArg2->Eval(x, y, pBitmap)) - nY);
	return (dx+dy)/2;
}

long CImCircGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long xx = x - m_cx;
	long yy = y - m_cy;
	double dist = sqrt(double(xx*xx + yy*yy));
	return lEvResult(long(dist));
}

long CImManhGene::Eval(long x, long y, CBitmap* pBitmap)
// Manhatten distance between (x,y) and (m_cx, m_cy)
{
	return (labs(x-m_cx)+labs(y-m_cy))/2;
}

long CImBitmapGene::Eval(long x, long y, CBitmap* pBitmap)
// Get the (x, y) palette entry for the bitmap in the gene.
{
	long bx = (x*m_nWidth)/lResultMax;
	long by = (y*m_nHeight)/lResultMax;
	LPSTR sBits = (LPSTR)GlobalLock(m_pBits);
	long result = (long)(sBits[bx+(m_nWidth * by)]);
	GlobalUnlock(m_pBits);
	return result;
}

long CImSinGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long nArgVal = m_pArg1->Eval(x,y,pBitmap);
	double lrm2 = double(lResultMax2-1);
	double ds = sin(double(nArgVal)*ScaledPi);
	return lEvResult(long(lrm2*ds)+lResultMax2);
}

long CImCosGene::Eval(long x, long y, CBitmap* pBitmap)
{
	long nArgVal = m_pArg1->Eval(x,y,pBitmap);
	double lrm2 = double(lResultMax2-1);
	double ds = cos(double(nArgVal)*ScaledPi);
	return lEvResult(long(lrm2*ds)+lResultMax2);
}

long CImLnGene::Eval(long x, long y, CBitmap* pBitmap)
{
	return lln(m_pArg1->Eval(x, y, pBitmap));
}

long CImNotGene::Eval(long x, long y, CBitmap* pBitmap)
{
	return lEvResult(labs(~(m_pArg1->Eval(x, y, pBitmap))));
}

long CImRandGene::Eval(long x, long y, CBitmap* pBitmap)
{   
	long rx = x+(rand()%50)-25;
	long ry = y+(rand()%50)-25;
	
	if((rx<0)||(rx>lResultMax)) rx=x;
	if((ry<0)||(ry>lResultMax)) ry=y;
	
	return m_pArg1->Eval(rx, ry, pBitmap);
}

long CImFiltGene::Eval(long x, long y, CBitmap* pBitmap)
{
	if((x==0)||(y==0)||(x==lResultMax)||(y==lResultMax)||(nSum==0))
	{
		// just use the argument.
		return m_pArg1->Eval(x, y, pBitmap);
	}
	else
	{
		// apply the filter to surrounding points.
		long sum = 0;
		for(long i = -1; i<=1; i+=2)
		{
			// TRACE("In filter (%ld, %ld, %ld): %i\n", nCorner, nEdge, nCenter, i);
			sum += nEdge*(m_pArg1->Eval(x+i, y, pBitmap));
			sum += nEdge*(m_pArg1->Eval(x, y+i, pBitmap));
			for(long j = -1; j<=1; j+=2)
				sum += nCorner*(m_pArg1->Eval(x+i, y+j, pBitmap));
		}
		sum += nCenter*(m_pArg1->Eval(x, y, pBitmap));
		return lEvResult(sum/nSum);
 	}
}

long CImFiltGene::Complexity()
{
	return 9*(m_pArg1->Complexity())+2;
}

long CImMaxGene::Eval (long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return(arg1 > arg2 ? arg1 : arg2);
}

long CImMinGene::Eval (long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return(arg1 < arg2 ? arg1 : arg2);
}

long CImAvgGene::Eval (long x, long y, CBitmap* pBitmap)
{
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	return (arg1+arg2)/2;
}

long CImRandRangeGene::Eval (long x, long y, CBitmap* pBitmap)
{
	// choose random value between arg1 and arg2.
	long arg1 = m_pArg1->Eval(x, y, pBitmap);
	long arg2 = m_pArg2->Eval(x, y, pBitmap);
	if(arg1==arg2) return arg1;
	
	long min = arg1<arg2 ? arg1 : arg2;
	long max = arg1<arg2 ? arg2 : arg1;
	return min+(rand()%(max-min));
}

long CImAtanGene::Eval(long x, long y, CBitmap* pBitmap)
{
	double arg1 = double(m_pArg1->Eval(x, y, pBitmap)) - dResultMax2; // -16000<arg<16000
	double sarg1 = dScale(arg1, dResultMax, 20.0);
	double r = atan(sarg1)+(pi/2);  // 0<r<pi
	return lEvResult(long(dScale(r, pi, dResultMax)));
}

long CImInvGene::Eval(long x, long y, CBitmap* pBitmap)
{
	// This has little do with, but is slightly based on,
	// Pickover's inversion (p. 157, "Computers & the Imagination")
	// scale x & y to (0,2).
	// invert wrt circle @ (0,0).
	// rescale & eval arg for new point.
	double x2 = dScale(double(x), dResultMax, 2.0);
	double y2 = dScale(double(y), dResultMax, 2.0);
	double xy = (x2*x2) + (y2*y2);
	long newx, newy;
	if(xy==0)
	{
	  	newx = x;
	  	newy = y;
	}
	else 
	{
		// newx = long(dScale(x2/xy, 1.0, dResultMax));
		// newy = long(dScale(y2/xy, 1.0, dResultMax));
		if(x2<xy)
			newx = long(x2*dResultMax/xy);
		else
			newx = long(xy*dResultMax/x2);
		
		if(y2<xy)
			newy = long(y2*dResultMax/xy);
		else
			newy = long(xy*dResultMax/y2);
	}
	return m_pArg1->Eval(newx, newy, pBitmap);
} 

long CImPolxGene::Eval(long rho, long theta, CBitmap* pBitmap)
{
	// Pickover's butterfly curve (p. 21).
	// Well, not really. Actually, we are assuming that (x, y) are
	// in polar co-ordinates, and we convert to rectangular.  Who knows
	// what this will do...
	// scale theta to (0, pi/2).
	// scale rho to (0, lResultMax/2).
	// compute rho * sin(theta), rho * cos(theta).
	double dtheta = dScale(double(theta), dResultMax, pi/2);
	double x = double(rho+1) * cos(theta);
	double y = double(rho+1) * sin(theta);
	return m_pArg1->Eval(long(x), long(y), pBitmap);
}

long CImPolyGene::Eval(long theta, long rho, CBitmap* pBitmap)
{
	// Pickover's butterfly curve (p. 21).
	// Well, not really. Actually, we are assuming that (x, y) are
	// in polar co-ordinates, and we convert to rectangular.  Who knows
	// what this will do...
	// scale theta to (0, pi/2).
	// scale rho to (0, lResultMax/2).
	// compute rho * sin(theta), rho * cos(theta).
	double dtheta = dScale(double(theta), dResultMax, pi/2);
	double x = double(rho+1) * cos(theta);
	double y = double(rho+1) * sin(theta);
	return m_pArg1->Eval(long(x), long(y), pBitmap);
}

long CImPolarGene::Eval(long x, long y, CBitmap* pBitmap)
{
	// convert from rectangular to polar & recalculate.
	// origin is at (m_dx, m_dy).
	// rho = hypot(x-m_dx, y-m_dy)
	// theta = asin(y/r)
	double dx = double(x)-m_dx;
	double dy = double(y)-m_dy;
	double rho = hypot(dx, dy);
	double theta = asin(dy/rho)+(pi/2); // between 0 and pi
	long srho = long(rho)*lMaxDist/lResultMax;
	long stheta = long((theta * dResultMax)/pi);
	if(m_bRhoX)
		return m_pArg1->Eval(srho, stheta, pBitmap);
	else
		return m_pArg1->Eval(stheta, srho, pBitmap);
} 

CImPolarGene::CImPolarGene()
{
	m_dx = rand()%lResultMax;
	m_dy = rand()%lResultMax;
	m_bRhoX = rand()%2;
}
	  

// print to string
char* CImGene::PrintToString()
{
	char* buf = new char[17];
	strcpy(buf, "Not a real gene!");
	return buf;
}

char* CImGene0::PrintToString()
{
	char* buf = new char[10];
	strcpy(buf, Name());
	return buf;
} 

char* CImGene1::PrintToString()
{
	char* buf = new char[250];
	char* sArg1 = m_pArg1->PrintToString();
	_snprintf(buf, 249, "(%s %s)", Name(), sArg1);
	delete[] sArg1;
	return buf;
}

char* CImGene2::PrintToString()
{
	char* buf = new char[250];
	char* sArg1 = m_pArg1->PrintToString();
	char* sArg2 = m_pArg2->PrintToString();
	_snprintf(buf, 249, "(%s %s %s)",  Name(), sArg1, sArg2);
	delete[] sArg1;
	delete[] sArg2;
	return buf;
}

#ifdef _DEBUG
void CImGene::Trace()
{
	TRACE("** Not a real gene!");
}

void CImGene0::Trace()
{
	TRACE("%s", Name());
}

void CImGene1::Trace()
{
	TRACE("(%s ", Name());
	m_pArg1->Trace();
	TRACE(")");
}

void CImGene2::Trace()
{
	TRACE("(%s ", Name());
	m_pArg1->Trace();
	TRACE(" ");
	m_pArg2->Trace();
	TRACE(")");
}

#endif // _DEBUG

// Mutations...

CImGene* CImTakeArg1::Mutate(CImEvolution*, CImGene1* pGene)
{   
	CImGene* pArg;
	CImGene* pResult;
	pArg = pGene->GetArg1();
	pResult = pArg->Copy();
	delete pGene;
	return pResult;
}

CImGene* CImTakeArg2::Mutate(CImEvolution*, CImGene2* pGene)
{   
	CImGene* pResult;
	CImGene* pArg;
	if((rand()%2)==0)
	{
		pArg = pGene->GetArg2();
	}
	else
	{
		pArg = pGene->GetArg1();
	}                         
	
	pResult = pArg->Copy();
	delete pGene;
	return pResult;
}

CImGene* doAddLevel(CImEvolution* pEvolution, CImGene* pGene)
{  
	// create a new gene one of whose arguments is pGene.
	CImGene* pNewArg = pEvolution->CreateRandomGenome();
	int nArgs = pNewArg->NArgs();
	switch(nArgs)
	{
		case 1:
		{
			// created a 1-arg gene.
			CImGene1* pNewArg1 = (CImGene1*)pNewArg;
			pNewArg1->SetArg1(pGene);
			return pNewArg1;
		}
			
		case 2:
		{
			// created a 2-arg gene.
			CImGene2* pNewArg2 = (CImGene2*)pNewArg;
			CImGene* pOtherArg = pEvolution->CreateRandomGenome(1);
			if((rand()%2)==0)
			{
				pNewArg2->SetArg1(pGene);
				pNewArg2->SetArg2(pOtherArg);
			}
			else
			{
				pNewArg2->SetArg1(pOtherArg);
				pNewArg2->SetArg2(pGene);
			}
			return pNewArg2;
		}
			
		default:
			TRACE("** Bad value for nArgs: %d\n", nArgs);
			return NULL;
	}
}

CImGene* CImAddLevel1::Mutate(CImEvolution* pEvolution, CImGene1* pGene)
{
	return doAddLevel(pEvolution, pGene);
}


CImGene* CImAddLevel2::Mutate(CImEvolution* pEvolution, CImGene2* pGene)
{
	return doAddLevel(pEvolution, pGene);
}


CImGene* CImReplaceFn1::Mutate(CImEvolution* pEvolution, CImGene1* pGene)
{
	// create a new gene to replace this whose arguments contain the arguments of pGene.
	CImGene* pArg1 = pGene->GetArg1();
	CImGene* pNewArg = pEvolution->CreateRandomGenome();
	int nArgs = pNewArg->NArgs();
	switch(nArgs)
	{
		case 1:
		{
			// created a 1-arg gene.
			CImGene1* pNewArg1 = (CImGene1*)pNewArg;
			pNewArg1->SetArg1(pArg1->Copy());
			delete pGene;
			return pNewArg1;
		}
			
		case 2:
		{
			// created a 2-arg gene.
			CImGene2* pNewArg2 = (CImGene2*)pNewArg;
			CImGene* pOtherArg = pEvolution->CreateRandomGenome(1);
			if((rand()%2)==0)
			{
				pNewArg2->SetArg1(pArg1->Copy());
				pNewArg2->SetArg2(pOtherArg);
			}
			else
			{
				pNewArg2->SetArg1(pOtherArg);
				pNewArg2->SetArg2(pArg1->Copy());
			}
			delete pGene;
			return pNewArg2;
		}
			
		default:
			TRACE("** Bad value for nArgs: %d\n", nArgs);
			return NULL;
	}
}

CImGene* CImReplaceFn2::Mutate(CImEvolution* pEvolution, CImGene2* pGene)
{
	// create a new gene to replace this whose arguments contain the arguments of pGene.
	CImGene* pArg1 = pGene->GetArg1();
	CImGene* pArg2 = pGene->GetArg2();
	CImGene* pNewArg = pEvolution->CreateRandomGenome();
	int nArgs = pNewArg->NArgs();
	switch(nArgs)
	{
		case 1:
		{
			// created a 1-arg gene.
			CImGene1* pNewArg1 = (CImGene1*)pNewArg;
			if((rand()%2)==0)
			{
				pNewArg1->SetArg1(pArg1->Copy());
			}
			else
			{
				pNewArg1->SetArg1(pArg2->Copy());
			}
			delete pGene;
			return pNewArg1;
		}
			
		case 2:
		{
			// created a 2-arg gene.
			CImGene2* pNewArg2 = (CImGene2*)pNewArg;
			if((rand()%2)==0)
			{
				pNewArg2->SetArg1(pArg1->Copy());
				pNewArg2->SetArg2(pArg2->Copy());
			}
			else
			{
				pNewArg2->SetArg1(pArg2->Copy());
				pNewArg2->SetArg2(pArg1->Copy());
			}
			delete pGene;
			return pNewArg2;
		}
			
		default:
			TRACE("** Bad value for nArgs: %d\n", nArgs);
			return NULL;
	}
}

// Constructors and destructors

CImGene::CImGene()
{
	m_nIndex = -1;
}

CImGene1::CImGene1()
{
	m_pArg1 = NULL;
}

CImGene2::CImGene2()
{
	m_pArg1 = NULL;
	m_pArg2 = NULL;
}

CImConstGene::CImConstGene()
{ 
	m_nConst = lEvResult(rand());
	
	m_sConst = new char[8];
	sprintf(m_sConst, "%ld", m_nConst);
}

CImConstGene::~CImConstGene()
{
	delete[] m_sConst;
}

CImDistxyGene::CImDistxyGene()
{
	nX = lEvResult(rand());
	nY = lEvResult(rand());
}

const char *FindBMPFile();

void CImBitmapGene::CreateRandom(CImEvolution* pEvolution, int)
{
	char *old_dir = _getcwd(NULL, 0);
	_chdir(pEvolution->GetBitmapLib());
	m_sFile = FindBMPFile();
	TRACE(" Reading bitmap from %s\n", m_sFile);
	HANDLE hBitmap = OpenDIB((char *)m_sFile);
	if (!hBitmap)
	{   
		CString err = CString("Couldn't read from ") + CString(m_sFile);
		AfxMessageBox(err, MB_OK | MB_ICONSTOP);
		exit(-1);
	}
	
	// extract DIB information.
	LPBITMAPINFOHEADER lpbi; 
	DibInfo(hBitmap, lpbi);
	ASSERT(((lpbi->biCompression)==0));
	m_nWidth = (long)lpbi->biWidth;
	m_nHeight = (long)lpbi->biHeight;
    
    // Get a handle on the image array, ignoring the palette.
	LPSTR pBits = (LPSTR)lpbi+(WORD)(lpbi->biSize)+PaletteSize(lpbi);
	
	// Fill in the array.
	HDC ic = CreateDC("DISPLAY", NULL, NULL, NULL);
	HDC hdc = CreateCompatibleDC(ic);
	HBITMAP hbm = CreateDIBitmap(hdc, lpbi, CBM_INIT, pBits, 
								 (LPBITMAPINFO)lpbi, DIB_RGB_COLORS);
	BITMAP bm;
	SelectObject(hdc, hbm);
	GetObject(hbm, sizeof(BITMAP), (LPSTR)&bm);
	DWORD size = (DWORD)bm.bmWidthBytes * bm.bmHeight * bm.bmPlanes;
	m_pBits = GlobalAlloc(GPTR, size);
	LPSTR bits = (LPSTR)GlobalLock(m_pBits);
	GetBitmapBits(hbm, size, bits);
	GlobalUnlock(m_pBits);
	DeleteObject(hbm);
	DeleteDC(hdc);
	DeleteDC(ic);
				
	_chdir(old_dir);
	free(old_dir);
}

CImBitmapGene::~CImBitmapGene()
{
	free((void *)m_sFile);
	GlobalFree(m_pBits);
}

const char *FindBMPFile()
// Returns the name of a random BMP file in the current directory.
{
	_find_t fileinfo;
	CStringArray files;
    unsigned int filep = _dos_findfirst("*.BMP", _A_NORMAL, &fileinfo);
    if (filep != 0)
    {
    	AfxMessageBox("No .BMP files in bitmap library.", MB_OK | MB_ICONSTOP);
    	exit(-1);
    }
    
    int nfiles = 1;
    TRACE("Found #%d: %s.\n", nfiles, fileinfo.name);
    files.Add(CString(fileinfo.name));
    
    while( _dos_findnext(&fileinfo) == 0)
    {
    	nfiles++;
    	TRACE("Found #%d: %s.\n", nfiles, fileinfo.name);
    	files.Add(CString((const char *)fileinfo.name));
    }
    
    int ret = rand()%nfiles;
    char *rets = (char *)malloc(14);
    strcpy(rets, (const char *)files.GetAt(ret));
    TRACE("Returning #%d: %s", ret, rets); 
    return (const char *)rets;
}
    
	
	

CImGene1::~CImGene1()
{
	delete m_pArg1;
}

CImGene2::~CImGene2()
{
	delete m_pArg1;
	delete m_pArg2;
}
