/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*@ GENETIC.LIB (c) Jochen Ruhland */
#include "version.inc"
/* Genetic Operators - Alle Funktione zum Verwalten einer rlist */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#ifdef _MSDOS
#include <search.h>
#endif

#include "rlist.h"
#include "g0.h"
#include "support.h"
#include "debug.h"
#include "portrnd.h"

#define UPTO2( x )      (((x)+1)&0xfffe)
/* Bringt eine beliebige USHORT auf das naechstgroessere Vielfache von 2 */

const struct auslese_param def_auslese = {0.5F};


/*
 * *  Funktionsname : fetch_free *
 *
 *  Beschreibung  : Gibt den Index eines freien Popelementes *  Input        
 * : struct r_list *r Zeiger auf Rangliste *                  number Anzahl
 * der Elemente die ueberhaupt da sind *  Output        : Nummer die Frei ist,
 * sonst NOMOREFREE 
 */
int             fetch_free(struct r_list * r, int number)
{
	int             i;

	DEBUG_call(fetch_free);
	DEBUG_int(number);
	for (i = 0; i < number; i++)
	{
		if (r[i].is_free == ISFREE)
		{
			DEBUGr_int(i);
			return i;
		}
	}
	DEBUGr_const(NOMOREFREE);
	return NOMOREFREE;
}

/*
 * *  Funktionsname : dump_ranklist *
 *
 *  Beschreibung  : Erstellt ein Grafisches Abbild der Rangliste *                 
 * Geht von einem Screen mit 80 Zeichen Breite aus *  Input         : struct
 * r_list *r Die Rangliste *                  number Die Anzahl *  Output       
 * : nope 
 */
void            dump_ranklist(struct r_list * r, int number)
{
	int             i, row, line;

	DEBUG_call(dump_ranklist);
	DEBUG_int(number);
	row = number / 80;
	DEBUGc('\n');
	for (line = 0; line <= row; line++)
	{
		for (i = 0; i + 80 * line < number; i++)
			DEBUGc((r[i + 80 * line].is_calculated == ISCALC ? 'C' : '-'));
		DEBUGc('\n');
		for (i = 0; i + 80 * line < number; i++)
			DEBUGc((r[i + 80 * line].is_free == ISFREE ? 'F' : '-'));
		DEBUGc('\n');
	}
}

/*
**  Funktionsname : debug_ranklist
**
**  Beschreibung  : Gibt die wesentlichen Parameter einer Rangliste aus
**  Input         : number Die Anzahl
**					r_list *r die Rangliste
**  Output        : 
*/
void            debug_ranklist(struct r_list * r, int number)
{
	int i;
	char buffer[256];
	sprintf(buffer,"Ranklist with %d Positions\n",number );
	DEBUGs(buffer);
	for( i=0; i<number; i++ )
	{
		sprintf( buffer, "%d: %f |%d|%c|%c|%d %d\n", i,r[i].pop_error,
			r[i].age,
			(r[i].is_calculated==ISCALC?'C':'-'),
			(r[i].is_free==ISFREE?'F':'-'),
			r[i].num_in_pop, r[i].size_of_pop );
		DEBUGs(buffer);
	}
}

/*
 * *  Function name : auslese *
 *
 *  Description :  Fuehrt eine Auslese unter den Elementen einer *                
 * Rangliste durch. Hierbei wird ein fester Anteil *                 der
 * Population entfernt *  Input : struct r_list *r Der Zeiger auf die
 * auszusortirende, schon *          bewertete Rangliste *          int
 * number Die Anzahl der Elemente ueberhaupt *          in auslese_param:
 * float rate Die Rate der Entfernten *  Ouput : Die Anzahl der entfernten 
 */

int             auslese(struct r_list * r, int number, struct auslese_param * a)
{
	int             live;	/* Wieviele sollen denn ueberleben? */
	int             i, out;

	DEBUG_call(auslese);
	DEBUG_int(number);

	assert(r != NULL);
	assert(a->rate > 0);
	assert(a->rate < 1.0);
	assert(number > 0);

#if(GENLIB_DEBUG>2)
	dump_ranklist(r, number);
#endif
	live = (int) ((float) number * a->rate + 0.999);

	if (live > number)
		live = number;
	DEBUG_int(live);

	for (out = 0, i = 0; i < number; i++)
	{
		if (r[i].is_free == ISFREE)
		{

			/*
			 * Wenn einer schon frei ist, haben wir weniger
			 * Arbeit 
			 */
			out++;
		}
		else
		{
			if (live > 0)
			{
				/* Wir muessen noch welche behalten */
				live--;
			}
			else
			{
				/* Der kommt raus */
				r[i].is_free = ISFREE;
				r[i].is_calculated = ISNOTCALC;
				r[i].age = 0;
				out++;
			}
		}
	}
	DEBUGr_int(out);
	return out;
}

/*
 * *      Funktionsname : init_rank_list *
 *
 *      Beschreibung  : Legt eine Rangliste fuer eine Population an *     
 * Input         : *r der Zeiger auf eine Rangliste *                      np
 * die Anzahl der Elemente in der Rangliste *      Output        : nope 
 */
void            init_rank_list(struct r_list * r, int np)
{
	int             i;

	DEBUG_call(init_rank_list);
	DEBUG_int(np);

	assert(r != NULL);
	assert(np > 0);

	for (i = 0; i < np; i++)
	{
		r[i].is_calculated = ISNOTCALC;
		r[i].is_free = ISFREE;
		r[i].pop_error = 0.0F;
		r[i].age = 0;
		r[i].bit_error = 0;
		r[i].p = (USHORT *) NULL;
		r[i].their_p = (USHORT *) NULL;
		r[i].num_in_pop = 0;
		r[i].size_of_pop = 0;
		r[i].spare = NULL;
	}
}


static float    bitwert = 0.0F;	/* Wertigkeit des Bitfehlers */

/*
 *  Funktionsname : comp_rank
 *
 *  Beschreibung  : Hilfsfunktion zum Sortieren einer Rangliste
 *  Ohne Beachtung des Bitfehlers
 *  Input         : r1, r2 Zeiger auf zwei Ranglisten Elemente
 *  Output        : -1 wenn r1 < r2 +1 wenn r1 > r2
 * ansonsten 0 
 */
int
#ifdef MSDOS
                __cdecl
/* cdecl wird benoetigt, damit die Funktion an qsort uebergeben werden kann */
#endif
                comp_rank(const void *r1, const void *r2)
{
	float           err1, err2;

	err1 = ((struct r_list *) r1)->pop_error +
	    bitwert * (float) ((struct r_list *) r1)->bit_error;
	err2 = ((struct r_list *) r2)->pop_error +
	    bitwert * (float) ((struct r_list *) r2)->bit_error;

	if (err1 < err2)
		return -1;
	else
	if (err1 > err2)
		return +1;
	else
		return 0;
}

/*
 * *  Funktionsname : sort_rank *
 *
 *  Beschreibung  : Sortiert eine Rangliste
 *  Input         : struct r_list
 * 		    *r Zeiger auf Liste
 *                : int size Anzahl der Elemente in r * 
 * Output         : nope 
 */
void            sort_rank(struct r_list * r, int size)
{
	                bitwert = 0.0F;
	qsort((void *) r, (size_t) size,
	      sizeof(struct r_list), comp_rank);

}

/*
 * *  Funktionsname : __set_rank *
 *
 *  Beschreibung  : Berechnet die Fitness der Populationen und sortiert dann
 *  die Rangliste
 *  Input         : rank Zeiger auf die Rangliste
 *		    number Die Anzahl der Elemente in der Liste
 *                  ges_error eine Funktion die den Fehler errechnet
 *                  t ein Zeiger auf die Targets
 *                  s ein Zeiger auf Populationsspezifische Info oder
 *                  NULL wenn solche nicht vorliegt
 *  Output        : nope 
 */
void            __set_rank(struct r_list * rank, int number,
			                   float (*ges_error) (),
			                   void *t, void *s)
{
	unsigned int    i;

	DEBUG_call(__set_rank);
	DEBUG_int(number);
	assert(rank != NULL);
	assert(ges_error != NULL);
	assert(t != NULL);
	/* Kein Test auf s!= NULL */
	dump_ranklist(rank, number);
	for (i = 0; i < (unsigned int) number; i++)
	{
		DEBUG_int(i);
		if (rank[i].is_calculated != ISCALC)
		{
			/* Nur die unkalkulierten berechnen */
			rank[i].pop_error = (s == NULL ?
					     (*ges_error) (rank[i].p, t) :
					     (*ges_error) (rank[i].p, t, s));
			rank[i].is_calculated = ISCALC;
		}
		else
			rank[i].age++;
		/* War rastet, der rostet - wer einfach nur fortgescribene
		   wird, der wird halt aelter */
	}

	/*
	 * for( i=0;i<number_in_pop;i++ ) printf(
	 * "Error[%u]=%f\n",rank[i].pop_index,rank[i].pop_error ); 
	 */
	/* Now do some sorting */
	/* Ohne beruecksichtigung des Bitwerts */
	bitwert = 0.0F;
	qsort((void *) rank, (size_t) number,
	      sizeof(struct r_list), comp_rank);
}

/*
 * *  Funktionsname : __compute_hamming *
 *
 *  Beschreibung  : Berechnet die Hammingdistantz der ersten Elemente der
 * Rangliste. *  Input         : r Zeiger auf die Rangliste number Anzahel
 * der Elemente, fuer die berechnet wird size Groesse der Elemente *  Output       
 * : Die durchshnittliche Hammingdistanz der Elemente 
 */
double          __compute_hamming(struct r_list * r, int number, int size)
{
	int             i, j;
	unsigned int    ham = 0;
	int             berechnet = 0;
	double          d;

	DEBUG_call(__compute_hamming);
	DEBUG_int(number);
	assert(r != NULL);
	assert(number > 0);
#if(GENLIB_DEBUG>2)
	dump_ranklist(r, number);
#endif
	for (i = 0; i - 1 < number; i++)
		for (j = i + 1; j < number; j++)
		{
			if ((r[i].is_calculated == ISCALC) &&
			    (r[j].is_calculated == ISCALC) &&
			    (r[i].is_free == ISNOTFREE) &&
			    (r[j].is_free == ISNOTFREE))
			{
				ham += __hamming_diff(r[i].p, r[j].p, size);
				berechnet++;
			}
		}
	assert(berechnet != 0);
	/* 2* mal weil ja h(p,q)==h(q,p) */
	d = (double) (2 * ham) / (double) (berechnet * berechnet);
	DEBUGr_double(d);
	return d;
}

static USHORT  *and_maske = NULL, *or_maske = NULL;


/*
 * *  Funktionsname : __dimension *
 *
 *  Beschreibung  : Berechnet die Anzahl der in den ersten number Elementen
 * noch vorhandenen Dimensionen des Suchraums. *  Input         : r Zeiger
 * auf die Rangliste number Anzahl der Elemente, fuer die berechnet wird int
 * size Groesse der Elemente *  Output        : Die Dimension 
 */
int             __dimension(struct r_list * r, int number, int size)
{
	int             dim, pop, sz;
	int             i, imax;

	DEBUG_call(__dimension);
	DEBUG_int(number);
	DEBUG_int(size);
	sz = UPTO2(size);
	DEBUG_int(sz);

	assert(r != NULL);

	and_maske = malloc((size_t) sz);
	or_maske = malloc(sz);

	assert(and_maske != NULL);
	assert(or_maske != NULL);

	imax = UPTO2(size) / sizeof(USHORT);

	for (i = 0; i < imax; i++)
	{
		and_maske[i] = 0xffff;
		or_maske[i] = 0x0;
	}
	for (pop = 0; pop < number; pop++)
	{
		for (i = 0; i < imax; i++)
		{
			and_maske[i] &= r[pop].p[i];
			/* Nur da 1 wo alle 1 sind */
			or_maske[i] |= r[pop].p[i];
			/* Nur da 0 wo alle 0 sind */
		}
	}
	for (dim = size * 8, i = 0; i < imax; i++)
	{
		and_maske[i] |= ~or_maske[i];
		/* Nur da 1 wo alle konstant sind */
		dim -= numbit_ushort(and_maske[i]);
	}

	free(and_maske);
	free(or_maske);

	DEBUGr_int(dim);
	return dim;
}

