/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*@ GENETIC.LIB (c) Jochen Ruhland */
#include "version.inc"
/* Nachbildung eines kompletten Genetischen Zyklus */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <memory.h>

#ifdef _MSDOS
#include <search.h>
#endif


#include "support.h"
#include "g0.h"
#include "portrnd.h"
#include "rlist.h"
#include "mgenops.h"


/*
 * Funktionsname : __mgenetic_step
 *
 * Beschreibung  : Fuehrt einen Meta-Generationsschritt durch
 * Input         : struct r_list *r Der Zeiger auf die Populationsliste
 *                 ges_error eine Funktion die den Fehler errechnet
 *                 t ein Zeiger auf die Targets
 *                 s ein Zeiger auf Populationsspezifische Info oder
 *                   NULL wenn solche nicht vorliegt
 *                 lese eine Funktion die aussondert
 *       	       sub_number Die Anzahl der Elemente in der Sub-Pop
 *                 *ap Ein Zeiger auf das Parameterarray fuer die Auslese
 *		           make_spare eine int-wertige Funktion die aus dem Genom
 *		                      des 1GA die Laenge des Genoms des 2GA errechnet
 *                 void (*destroy_spare)() Funktion um anhaengende SUB-Pop
 *                                         zu loeschen
 *                 c1_rate, c2_rate, cn_rate Die Rate der jeweiligen
 *                                           Crossovertypen
 *                 m_rate Mutationsrate
 *                 size Groesse des Meta-Genops in Bytes
 * Output        : Der jetzt erzielte Fehler 
 */
float __mgenetic_step(struct r_list * rank,
							float (*pop_error) (),
							/* Parameter fuer die Auslesefunktion */
							int (*lese) (),
							int sub_number,
							void *ap,
							int (*make_spare)(),
							void (*destroy_spare)(),
							float c1_rate, float c2_rate, float cn_rate,
							float m_rate,
							int size
)
{
	unsigned short  drin, raus, nmut;
	unsigned short  c1, c2, cn;
	int i;

	DEBUG_call(__genetic_step);
	DEBUG_float(c1_rate);
	DEBUG_float(c2_rate);
	DEBUG_float(cn_rate);
	DEBUG_float(m_rate);
	DEBUG_int(size);

	/* Zunaechst die gegebene Popliste bewerten und sortieren */
	set_rank_meta( pop_error );
	/* Diese Funktion ruft automatisch eval_sub_population auf */

	/* Nun die unguten rauswerfen */
	drin = (unsigned short) (get_number_meta() -
								(raus = (unsigned short) (*lese) (rank,
												get_number_meta(), ap)));
	/* Die Freigestellten loeschen */
	for( i=0; i<get_number_meta(); i++ )
	{
		if(rank[i].is_free == ISFREE)
			destroy_multi_population( i,destroy_spare );
	}

	DEBUG_ushort(drin);
	DEBUG_ushort(raus);

	assert(raus > 0);
	assert(drin > 0);
	assert(drin + raus == (unsigned short) get_number_meta());

	/* Nun muessen raus Elemente ersetzt werden */
	c1 = (unsigned short) ((float) raus * c1_rate);
	c1 = (unsigned short)(c1 > raus ? raus : c1);
	c2 = (unsigned short) ((float) raus * c2_rate);
	c2 = (unsigned short)(c2 > raus ? raus : c2);
	cn = (unsigned short) ((float) raus * cn_rate);
	cn = (unsigned short)(cn > raus ? raus : cn);

	/* Checks der Parameter */
	/* Es darf nur ein Wert c1,c2,cn != 0 sein */

	DEBUG_ushort(c1);
	DEBUG_ushort(c2);
	DEBUG_ushort(cn);

	/*
	 * Der Fall, dass mehr als eine Variable c1,c2,cn != 0 ist wird nicht
	 * gesondert abgeprueft 
	 */

	nmut = (unsigned short) (raus - c1 - c2 - cn);
	DEBUG_ushort(nmut);

	/* Zunaechst mal c1-mal mit Crossover arbeiten */
	for (i = 0; i < (int)c1; i++)
	{
		USHORT         *v, *m, *s;
		int             sohn;

		v = rank[rand16(drin)].p;
		m = rank[rand16(drin)].p;
		sohn = fetch_free(rank, get_number_meta());
		assert(sohn != NOMOREFREE);

		s = rank[sohn].p;
		rank[sohn].is_calculated = ISNOTCALC;

		__crossover(v, m, rand16((unsigned short) (size * 8)), s, (USHORT) size);
		make_multi_population( sohn, sub_number, make_spare );
	}
	/* Dann c2-mal mit Crossover2 arbeiten */
	for (i = 0; i < (int)c2; i++)
	{
		USHORT         *v, *m, *s;
		int             sohn;

		v = rank[rand16(drin)].p;
		m = rank[rand16(drin)].p;
		sohn = fetch_free(rank, get_number_meta());
		assert(sohn != NOMOREFREE);

		s = rank[sohn].p;
		rank[sohn].is_calculated = ISNOTCALC;

		__crossover2(v, m, rand16((unsigned short) (size * 8)),
		     rand16((unsigned short) (size * 8)), s, (USHORT) size);

		make_multi_population( sohn, sub_number, make_spare );
	}
	/* Zuletzt cn-mal mit Crossovern arbeiten */
	for (i = 0; i < (int)cn; i++)
	{
		USHORT         *v, *m, *s;
		int             sohn;

		v = rank[rand16(drin)].p;
		m = rank[rand16(drin)].p;
		sohn = fetch_free(rank, get_number_meta());
		assert(sohn != NOMOREFREE);

		s = rank[sohn].p;
		rank[sohn].is_calculated = ISNOTCALC;

		__crossovern(v, m, 0.5, s, size);

		make_multi_population( sohn, sub_number, make_spare );
	}
	/* Nun den Rest mittels Mutation */
	for (i = 0; i < (int)nmut; i++)
	{
		USHORT         *v, *s;
		int             sohn;

		v = rank[rand16((unsigned short) (drin + c1 + c2 + cn))].p;
		sohn = fetch_free(rank, get_number_meta());
		assert(sohn != NOMOREFREE);

		rank[sohn].is_calculated = ISNOTCALC;

		s = rank[sohn].p;
		__copy_popelem(v, s, size);
		__mutate(s, m_rate, (USHORT) size);

		make_multi_population( sohn, sub_number, make_spare );
	}
	/* Nun ist die Population wieder auf Sollstaerke, wir geben den f-Wert */
	DEBUGr_double(rank[0].pop_error);
	return rank[0].pop_error;
}
