/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*@ GENETIC.LIB (c) Jochen Ruhland */
#include "version.inc"
/* Genetic Operators */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <memory.h>

#ifdef _MSDOS
#include <search.h>
#endif

#include "g0.h"
#include "rlist.h"
#include "support.h"
#include "mgenops.h"
#include "mgenstep.h"
#include "portrnd.h"
#include "genstep.h"
#include "debug.h"

#define UPTO2( x )      (((x)+1)&0xfffe)
/* Bringt eine beliebige USHORT auf das naechstgroessere Vielfache von 2 */



/* Variablen nur fuer Multi-Pop Modus */
static          number_of_pops = 0;	/* Wieviele Populationen gibt es */

static int      max_size_in_pop;

/* Von allen Populationen die Groesse des groessten Elementes */
int      		meta_size_of_pop;

/* Die Liste der Populationen */
static struct r_list *POPS = NULL;

/* *POPS ist der globale Anker fuer alle Populationen */

/* Da haengen die einzelnen Populationen dran */
static USHORT  *POP_ELEM;

/*
 * Funktionsname : get_number_meta

 * Beschreibung  : Anzahl der Elemente in der Meta-Population
 * gleichzeitig Anzahl der verschiedenen Sup-pops
 * Input         : void
 * Output        : int 
 */
int get_number_meta(void)
{
	return number_of_pops;
}


/*
 * Funktionsname : get_number_multi

 * Beschreibung  : Gibt Anzahl der Elemente in der numerierten Pop
 * Input         : int number - welche?
 * Output        : Anzahl oder POPNONVALID 
 */
int get_number_multi(int number)
{
	if ((number < 0) || (number > get_number_meta()))
		return (POPNONVALID);
	else if (POPS[number].is_free == ISFREE)
		return (POPNONVALID);
	else
		return POPS[number].num_in_pop;
}


/*
 * Funktionsname : get_size_meta

 * Beschreibung  : Groesse der Elemente in der Meta-Population
 * Input         : void
 * Output        : int 
 */
int get_size_meta(void)
{
	return meta_size_of_pop;
}


/*
 * Funktionsname : get_size_multi

 * Beschreibung  : Groesse der Elemente in der numerierten Pop
 * Input         : int number - welche ?
 * Output        : Groesse oder POPNONVALID 
 */
int get_size_multi(int number)
{
	if ((number < 0) || (number > get_number_meta()))
		return (POPNONVALID);
	else if (POPS[number].is_free == ISFREE)
		return (POPNONVALID);
	else
		return POPS[number].size_of_pop;
}

/*
 * Funktionsname : get_spare

 * Beschreibung  : Holt die Information des spare-Zeigers fuer eine Pop
 * Input         : num Nummer der Population
 * Output        : void *spare oder NULL 
 */
void *get_spare(int num)
{
	DEBUG_call(get_spare);
	DEBUG_int(num);
	if ((num < 0) || (num >= get_number_meta()) || (POPS == NULL))
		return NULL;
	else
		return POPS[num].spare;
}

/*
 *  Funktionsname : get_rlist_multi
 *
 *  Beschreibung  : Gibt den Zeiger auf die Rangliste der angegebenen Pop
 *  Input         : int npop - welche?
 *  Output        : struct rlist *
*/
struct r_list *get_rlist_multi( int num )
{
	DEBUG_call(get_rlist_multi);
	DEBUG_int(num);
	if ((num < 0) || (num >= get_number_meta()) || (POPS == NULL))
		return NULL;
	else
		return POPS[num].r;
}

/*
**  Funktionsname : get_rlist_meta
**
**  Beschreibung  : Gibt den Zeiger auf die Meta-Rangliste zurueck
**  Input         : nope
**  Output        : struct roeist *
*/
struct r_list *get_rlist_meta( void )
{
	DEBUG_call(get_rlist_meta);
	return POPS;
}

/*
 * Funktionsname : init_multi

 * Beschreibung  : Legt die Liste der Populationen und die zugehoerige
 * Super-Rangliste an. Initialisiet den Zufallszahlengenerator
 * Input         : number Anzahl der Populationen
 *				   meta Groesse eines Elementes des 1.GA
 *                 rseed Initiator fuer Zufallszahlengenerator
 * Output        : nope 
 */
void init_multi(int number, int meta, USHORT rseed)
{
	unsigned int    i;

	DEBUG_call(init_multi);
	DEBUG_int(number);
	DEBUG_int(meta);
	DEBUG_ushort(rseed);

	assert(number > 0);
	assert(meta > 0);

	meta_size_of_pop = UPTO2(meta);
	number_of_pops = number;
	/* Speicher fuer Super-Rangliste */
	POPS = calloc(number, sizeof(struct r_list));
	assert(POPS != NULL);

	/* Speicher fuer die Pop-Elemente des 1.GA */
	POP_ELEM = calloc(number, get_size_meta());
	assert(POP_ELEM != NULL);
	init_rank_list(POPS, number);
	__randomize_popelem(POP_ELEM, number * get_size_meta());
	set_mask(get_size_meta());
	for (i = 0; i < (unsigned int) number; i++)
	{

		POPS[i].p = &(POP_ELEM[(i * get_size_meta()) / sizeof(USHORT)]);
		DEBUG_pointer(POPS[i].p);
	}

	seed16(rseed);

	max_size_in_pop = 0;
}

/*
 * Funktionsname : make_multi_population

 * Beschreibung  : Legt eine einzelne Poulation in der Super-Population an
 * Hierbei wird die Information ueber den Aufbau der Pop aus den Daten der
 * Super-Pop abgeleitet
 * Input         : npop Um welche Population handelt es sich
 *				   number Um wieviele Elemente handelt es sich
 *                 make_spare Funktion die zum einen die Anzahl der
 *                            Bytes in der Population angibt, zum anderen
 *                            die den spare- Bereich ausfuellt
 * Output        : nope 
 */
void make_multi_population(int npop, int number, int make_spare())
{
	int             size;

	DEBUG_call(make_multi_population);
	DEBUG_int(npop);
	DEBUG_int(number);

	size = (*make_spare) (&(POPS[npop].spare), POPS[npop].p);
	/* Diese Funktion liefert die Groesse des Genoms */
	init_multi_population(npop, number, size);
}

/*
 * Funktionsname : init_multi_population

 * Beschreibung  : Legt eine einzelne Population in der Super-Population an
 * Input         : npop Um welche Population habdelt es sich
 *                 number Anzahl der Elemente in der Population
 *                 sizepop Anzahl der Bytes in der Population
 * Output        : nope 
 */
void init_multi_population(int npop, int number, int sizepop)
{
	int             i;
	USHORT        **p;
	struct r_list **r;

	DEBUG_call(init_multi_population);
	DEBUG_int(npop);
	DEBUG_uint(number);
	DEBUG_int(sizepop);
	assert(npop < get_number_meta());
	assert(POPS[npop].is_free == ISFREE);

	p = &(POPS[npop].their_p);	/* Zugriff auf die Population */
	r = &(POPS[npop].r);    	/* Zugriff auf die Rangliste */

	POPS[npop].is_free = ISNOTFREE;
	POPS[npop].is_calculated = ISNOTCALC;
	POPS[npop].pop_error = 0.0F;

	POPS[npop].num_in_pop = number;	/* Anzahl der Populationselemente */
	assert(get_number_multi(npop) > 0);

	POPS[npop].size_of_pop = UPTO2(sizepop);	/* Groesse der einzelnen
							 * Elemente */
	assert(get_size_multi(npop) > 0);

	/* Der Speicher fuer die Populationselemente */
	*p = calloc(number, sizepop);
	assert(*p != NULL);

	/* Die Populationsspezifische Rangliste */
	*r = calloc(number, sizeof(struct r_list));
	assert(*r != NULL);

	/* p ist der Zeiger auf den Zeiger des gesamten Genoms */
	/* r ist der zeiger auf den Zeiger der Sub-Rangliste */

	set_mask(sizepop);
	init_rank_list(*r, number);
	__randomize_popelem(*p, number * sizepop);
	for (i = 0; i < number; i++)
	{
		/* Der persoenliche Pointer der Rangliste */
		(*r)[i].p = &((*p)[i * sizepop/ sizeof(USHORT)]);
		(*r)[i].is_free = ISNOTFREE;
		DEBUG_pointer((*r)[i].p);
	}
}

/*
 * Funktionsname : destroy_multi_population

 * Beschreibung  : Loest eine Population des 2.GA wieder auf
 * Input         : npop Welche Population ist's
 *                 destroy_spare eine (User-supplied) Fuktion zum Aufloesen
 *                               einer spare-Struktur
 * Output        : nope 
 */
void destroy_multi_population(int npop, void destroy_spare())
{
	DEBUG_call(destroy_multi_population);
	DEBUG_int(npop);

	assert(npop >= 0);
	assert(npop < get_number_meta());
	/* destro darf nur auf bereits freien Ausgefuehrt werden */
	assert(POPS[npop].is_free == ISFREE);

	(*destroy_spare)((void *)POPS[npop].spare);
	/* Den Speicherbereich von spare loeschen */
	POPS[npop].spare = NULL;

	free(POPS[npop].r);
	POPS[npop].r = NULL;
	free(POPS[npop].their_p);
	POPS[npop].their_p =NULL;
	POPS[npop].is_calculated = ISNOTCALC;
	POPS[npop].num_in_pop = POPS[npop].size_of_pop = 0;
}


/*
 * Funktionsname : set_rank_multi

 * Beschreibung  : Berechnet die Fitness der angegebenen Population und
 * sortiert dann die Rangliste
 * Input         : npop ummer der Population
 * 				   ges_error eine Funktion die den Fehler errechnet
 *                 t ein Zeiger auf die Targets
 * Output        : nope 
 */
void set_rank_multi(int npop, float (*ges_error) (), void *t)
{
	DEBUG_call(set_rank_multi);
	DEBUG_int(npop);

	assert(npop >= 0);
	assert(npop < get_number_meta());
	assert(POPS[npop].is_free == ISNOTFREE);

	__set_rank(POPS[npop].r, get_number_multi(npop),
		   ges_error, t, POPS[npop].spare);
}

/*
 * Funktionsname : eval_sub_population

 * Beschreibung  : Bewertet eine Subpopulation indem die Rangliste
 * ausgewertet wird
 * Inputi        : int number - welche Sub-Pop solls denn sein?
 *                 pop_error Die Funktion, die die Rangliste der
 *                           Sub-Population abklappert
 * Output        : nope
 */
void eval_sub_population(int number, float (*pop_error) ())
{
	DEBUG_call(eval_sup_population);
	DEBUG_int(number);

	assert(number >= 0);
	assert(number < get_number_meta());
	assert(POPS[number].is_free == ISNOTFREE);
	assert(POPS[number].r!=NULL);

	POPS[number].pop_error = (*pop_error) (POPS[number].r,
						get_number_multi(number), get_spare(number));

	/*
	 * Die Funktion pop_error bekommt zum einen einen Pointer auf die
	 * Rangliste, und die Anzahl der Elemente gegeben 
	 * und zum anderen einen Zeiger auf den spare-Bereich dieser Pop
	 */
	POPS[number].is_calculated = ISCALC;
}

/*
 * Funktionsname : a_dummy_function

 * Beschreibung  : Diese Funktion dient nur als Dummy, da zum einem
 * __set_rank einen Funktionszeiger != NULL benoetigt, und zum anderen
 * ja alle Elemente der meta-Liste mittels eval_sub_population bereits
 * bewertet sind.
 * Input         : nimmt einem Pointer entgegen, der Muell ist
 * Output        : angeblich ein float um den Compiler zu beruhigen 
 */
static float    a_dummy_function(void *a_pointer)
{
	DEBUG_call(a_dummy_function);
	assert(a_pointer != NULL);
	abort();
	return 0.0F;
}

/*
 * Funktionsname : set_rank_meta

 * Beschreibung  : Berechnet die Fitness der Meta-Population und sortiert
 * dann die Rangliste
 * Input         : pop_error eine Funktion die den Fehler errechnet, aus 
 *                           dem Fehler der Sub-Populationen, die jeweils
 *                           schon berechnet sein muessen
 * Output        : nope 
 */
void set_rank_meta(float (*pop_error) ())
{
	int             i;
	USHORT          a_dummy_value;

	DEBUG_call(set_rank_meta);

	for (i = 0; i < get_number_meta(); i++)
		eval_sub_population(i, pop_error);
	__set_rank(POPS, get_number_meta(),
		   a_dummy_function, &a_dummy_value, NULL);
}

/*
 * Funktionsname : compute_hamming_multi

 * Beschreibung  : Berechnet die Hammingdistanz einer Multi-Population
 * Input         : int number - welche denn?
 * Output        : Die durchschnittliche Hammingdistanz der Elemente 
 */
double compute_hamming_multi(int number)
{
	DEBUG_call(compute_hamming_multi);
	DEBUG_int(number);

	assert(number >= 0);
	assert(number < get_number_meta());
	assert(POPS[number].is_free == ISNOTFREE);

	return __compute_hamming(POPS[number].r, get_number_multi(number),
							get_size_multi(number));
}

/*
 * Funktionsname : compute_hamming_meta

 * Beschreibung  : Berechnet die Hammingdistanz der Meta-Population

 * Input         : nope
 * Output        : Die durchschnittliche Hammingdistanz der Elemente 
 */
double compute_hamming_meta(void)
{
	DEBUG_call(compute_hamming_meta);
	return __compute_hamming(POPS, get_number_meta(), get_size_meta());
}



/*
 * Funktionsname : dimension_multi

 * Beschreibung  : Berechnet die Anzahl der in der number-ten Pop noch
 * vorhandenen Dimensionen des Suchraums.
 * Input         : number - welche?
 * Output        : Die Dimension 
 */
int dimension_multi(int number)
{
	DEBUG_call(dimension_multi);
	DEBUG_int(number);

	assert(number >= 0);
	assert(number < get_number_meta());
	assert(POPS[number].is_free == ISNOTFREE);

	return __dimension(POPS[number].r, get_number_multi(number),
			   get_size_multi(number));
}

/*
 * Funktionsname : dimension_meta

 * Beschreibung  : Anzahl der Dimensionen die in der Meta-Population
 * noch vertreten sind
 * Input         : nope
 * Output        : Anzahl der Dimensionen 
 */
extern int dimension_meta(void)
{
	DEBUG_call(dimension_multi);
	return __dimension(POPS, get_number_meta(), get_size_meta());
}


/*
 * Funktionsname : genstep_c1_multi

 * Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C1 durch
 * ( bewerten, ausduennen, auffuellen )
 * Input         : npop welche denn?
 *                 ges_error Fehlerfunktion
 *                 t Targets
 *                 c1p Die Wahrscheinlichkeit fuer c1 crossover
 *                 mp Die Wahrscheinlichkeit der Mutation
 * Output        : Der momentane Fehler 
 */
float genstep_c1_multi(int npop, float (*ges_error) (), void *t,
			                   float c1_rate, float m_rate)
{
	DEBUG_call(genstep_c1_multi);
	DEBUG_int(npop);
	DEBUG_float(c1_rate);
	DEBUG_float(m_rate);

	assert(npop >= 0);
	assert(npop < get_number_meta());
	assert(POPS[npop].is_free == ISNOTFREE);

	return __genetic_step(POPS[npop].r, ges_error, t, get_spare(npop),
				   auslese, get_number_multi(npop), (void *) &def_auslese,
					 c1_rate, 0.0F, 0.0F, m_rate, get_size_multi(npop));
}

/*
 * Funktionsname : genstep_c2_multi

 * Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C2 durch
 * ( bewerten, ausduennen, auffuellen )
 * Input         : npop welche denn?
 *                 ges_error Fehlerfunktion
 *                 t Targets
 *                 c2p Die Wahrscheinlichkeit fuer c2 crossover
 *                 mp Die Wahrscheinlichkeit der Mutation
 * Output        : Der momentane Fehler 
 */
float genstep_c2_multi(int npop, float (*ges_error) (), void *t,
						   float c2_rate, float m_rate)
{
	DEBUG_call(genstep_c2_multi);
	DEBUG_int(npop);
	DEBUG_float(c2_rate);
	DEBUG_float(m_rate);

	assert(npop >= 0);
	assert(npop < get_number_meta());
	assert(POPS[npop].is_free == ISNOTFREE);

	return __genetic_step(POPS[npop].r, ges_error, t, get_spare(npop),
		   auslese, get_number_multi(npop), (void *) &def_auslese,
			 0.0F, c2_rate, 0.0F, m_rate, get_size_multi(npop));
}

/*
 * Funktionsname : genstep_cn_multi

 * Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit CN durch
 * ( bewerten, ausduennen, auffuellen )
 * Input         : npop welche denn?
 *                 ges_error Fehlerfunktion
 *                 t Targets
 *                 cn_rate Die Wahrscheinlichkeit fuer cn crossover
 *                 mp Die Wahrscheinlichkeit der Mutation
 * Output        : Der momentane Fehler 
 */
float genstep_cn_multi(int npop, float (*ges_error) (), void *t,
			                   float cn_rate, float m_rate)
{
	DEBUG_call(genstep_cn_multi);
	DEBUG_int(npop);
	DEBUG_float(cn_rate);
	DEBUG_float(m_rate);

	assert(npop >= 0);
	assert(npop < get_number_meta());
	assert(POPS[npop].is_free == ISNOTFREE);

	return __genetic_step(POPS[npop].r, ges_error, t, get_spare(npop),
				   auslese, get_number_multi(npop), (void *) &def_auslese,
					 0.0F, 0.0F, cn_rate, m_rate, get_size_multi(npop));
}

/*
 * Funktionsname : genstep_c1_meta
 *
 * Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C1 durch
 * ( bewerten, ausduennen, auffuellen )
 * Input         : pop_error Fehlerfunktion
		   make_spare Funktion zum Anlegen einer Subpopulation
		   destroy_spare Funktion zum Loeschen
 *                 c1_rate Die Wahrscheinlichkeit fuer c1 crossover
 *                 mp Die Wahrscheinlichkeit der Mutation
 * Output        : Der momentane Fehler 
 */
float genstep_c1_meta( float (*pop_error) (), int(*make_spare)(), void (*destroy_spare)(), 
			                   float c1_rate, float m_rate)
{
	DEBUG_call(genstep_c1_meta);
	DEBUG_float(c1_rate);
	DEBUG_float(m_rate);

	return __mgenetic_step(POPS, pop_error,
			   auslese, get_number_multi(0), (void *) &def_auslese,
			   make_spare, destroy_spare,
			     c1_rate, 0.0F, 0.0F, m_rate, get_size_meta());
}

/*
 * Funktionsname : genstep_cn_meta
 *
 * Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit Cn durch
 * ( bewerten, ausduennen, auffuellen )
 * Input         : pop_error Fehlerfunktion
		   make_spare Funktion zum Anlegen einer Subpopulation
		   destroy_spare Funktion zum Loeschen
 *                 cn_rate Die Wahrscheinlichkeit fuer cn crossover
 *                 mp Die Wahrscheinlichkeit der Mutation
 * Output        : Der momentane Fehler 
 */
float genstep_cn_meta( float (*pop_error) (), int(*make_spare)(), void (*destroy_spare)(), 
			                   float cn_rate, float m_rate)
{
	DEBUG_call(genstep_cn_meta);
	DEBUG_float(cn_rate);
	DEBUG_float(m_rate);

	return __mgenetic_step(POPS, pop_error,
			   auslese, get_number_multi(0), (void *) &def_auslese,
			   make_spare, destroy_spare,
			     0.0F, 0.0F, cn_rate, m_rate, get_size_meta());
}
