/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*@ GENETIC.LIB (c) Jochen Ruhland */
#include "version.inc"
/* Genetic Operators */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <memory.h>

#ifdef _MSDOS
#include <search.h>
#endif

#include "g0.h"
#include "support.h"
#include "rlist.h"
#include "genops.h"
#include "genstep.h"
#include "portrnd.h"

#define UPTO2( x )      (((x)+1)&0xfffe)
/* Bringt eine beliebige USHORT auf das naechstgroessere Vielfache von 2 */

/* Variablen nur fuer Single-Pop Modus */
unsigned int    size_of_pop;	/* Wieviele BYTES enthaelt ein Popel */

static unsigned int number_in_pop;	/* Wieviele Popelem gibts es */

/* Der Basisanker fuer die Rangliste der Population */

static struct r_list *POP;

static USHORT  *POP_GENOM;

/*
 *  Funktionsname : get_number 
 *
 *  Beschreibung  : Anzahl der Elemente der einzel-Population
 *  Input         : nope
 *  Output        : int Anzahl 
 */
int get_number(void)
{
	return number_in_pop;
}
/*
 *  Funktionsname : get_population
 *
 *  Beschreibung  : Gibt einen Zeiger auf die rangliste der Einzel-Population zurck
 *
 *  Input         : nope
 *  Output        : struct r_list *
 */
struct r_list *get_population( void )
{
	return POP;
}

/*
 *  Funktionsname : get_size 
 *
 *  Beschreibung  : Groesse der Elemente der einzel-Population 
 *  Diese Groesse kann um 1 zu gross sein, da es die im 
 *  Speicher reservierte Groesse angibt
 *  Input         : nope
 *  Output       : int Groesse in Bytes 
 */
int get_size(void)
{
	return size_of_pop;
}

/*
 *  Funktionsname : __destroy_population 
 *
 *  Beschreibung  : Zerstoert die einzel-Population
 *  Input         : * nope
 *  Output        : nope 
 */
void __destoy_population(void)
{
	free(POP);
	free(POP_GENOM);
}

/*
 *  Funktionsname : init_population 
 *
 *  Beschreibung  : Legt eine einzelne Population mit zugehoeriger
 *  Rangliste an, und initialisiert den Zufallszahlengenerator
 * Speicherplatz fuer POP wird von der Funktion allokiert
 * Speicher fuer die Bitmasken wird allokiert 
 *  Input         : number Anzahl der Elemente in der Population
 *                  sizepop Groesse eines Elementes der Popukation in Bytes
 *                  rseed Initiator fuer Zufallszahlengenerator
 *  Output        : nope
 */
void init_population(int number, int sizepop, USHORT rseed)
{
	unsigned int    i;

	DEBUG_call(__init_population);
	DEBUG_int(number);
	DEBUG_int(sizepop);
	DEBUG_ushort(rseed);

	number_in_pop = number;	/* Populationsgroesse */
	size_of_pop = UPTO2(sizepop);	/* Groesse eines Elementes */
	assert(size_of_pop > 0);

	/* Speicher fuer die Rangliste */
	POP = calloc(number_in_pop, sizeof(struct r_list));
	assert(POP != NULL);

	/* Speicher fuer die genome */
	POP_GENOM = calloc(number_in_pop, size_of_pop);
	assert(POP_GENOM != NULL);

	/* Der Speicherbereich fuer die Masken */
	set_mask(size_of_pop);

	/* Rangliste als unkalkuliert und frei kennzeichnen */
	init_rank_list(POP, number_in_pop);

	/* Zufalsszahlengenerator setzten */
	seed16(rseed);

	/* Alle Elemente randomisieren */
	__randomize_popelem(POP_GENOM, size_of_pop * number_in_pop);

	/* In der Rangliste den Speicherbereich dazulinken */
	for (i = 0; i < number_in_pop; i++)
	{
		POP[i].is_free = ISNOTFREE;
		POP[i].p = &((POP_GENOM)[i * size_of_pop / sizeof(USHORT)]);
	}
}

/*
 *  Funktionsname : set_rank 
 *
 *  Beschreibung  : Berechnet die Fitness der Population und sortiert dann die
 *  Rangliste
 *  Input         : ges_error eine Funktion die den Fehler errechnet
 *                  t ein Zeiger auf die Targets
 *  Output        : nope 
 */
void set_rank(float (*ges_error) (), void *t)
{
	__set_rank(POP, get_number(), ges_error, t, NULL);
}

/*
 *  Funktionsname : compute_hamming 
 *
 *  Beschreibung  : Berechnet die Hammingdistantz der ersten Elemente der
 *  Rangliste.
 *  Input         : nope
 *  Output        : Die durchshnittliche Hammingdistanz der Elemete 
 */
double compute_hamming(void)
{
	return __compute_hamming(POP, get_number(), get_size());
}


/*
 *  Funktionsname : dimension 
 *
 *  Beschreibung  : Berechnet die Anzahl der in den ersten number Elementen
 * noch vorhandenen Dimensionen des Suchraums.
 *  Input         : nope 
 *  Output        : Die Dimension 
 */
int dimension(void)
{
	return __dimension(POP, get_number(), get_size());
}

/*
 *  Funktionsname : genstep_c1 
 *
 *  Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C1 durch 
 *  (bewerten, ausduennen, auffuellen)
 *  Input         : ges_error Fehlerfunktion
 *                  t Targets
 *                  c1 Die Wahrscheinlichkeit fuer c1 crossover
 *                  mp Die Wahrscheinlichkeit der Mutation
 *  Output        : Der momentae Fehler 
 */
float genstep_c1(float (*ges_error) (), void *t,
				   float c1_rate, float m_rate)
{
	                return __genetic_step(POP, ges_error, t, NULL,
	                       auslese, get_number(), (void
 *) &def_auslese,
		                     c1_rate, 0.0, 0.0, m_rate, get_size());

}

/*
 *  Funktionsname : genstep_c2 
 
 *  Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C1 durch 
 * ( bewerten, ausduennen, auffuellen ) *  Input         : ges_error
 * Fehlerfunktion
 *              t Targets *                  c2p Die
 * Wahrscheinlichkeit fuer c2 crossover
 *              mp Die
 * Wahrscheinlichkeit der Mutation
 *  Output        : Der momentae Fehler 
 */
float           genstep_c2(float (*ges_error) (), void
 *t,
			                   float c2_rate, float m_rate)
{
			return __genetic_step(POP, ges_error, t, NULL,
				   auslese, get_number(), (void
*) &def_auslese,
					 0.0, c2_rate, 0.0, m_rate, get_size());
}

/*
 *  Funktionsname : genstep_cn 
 
 *  Beschreibung  : Fuehrt einen vollstaendigen Gen-Step mit C1 durch 
 * ( bewerten, ausduennen, auffuellen ) *  Input         : ges_error
 * Fehlerfunktion
 *              t Targets *                  cnp Die
 * Wahrscheinlichkeit fuer cn crossover
 *              mp Die
 * Wahrscheinlichkeit der Mutation
 *  Output        : Der momentae Fehler 
 */
float           genstep_cn(float (*ges_error) (), void
 *t,
			                   float cn_rate, float m_rate)
{
		return __genetic_step(POP, ges_error, t, NULL,
			   auslese, get_number(), (void
*) &def_auslese,
				 0.0, 0.0, cn_rate, m_rate, get_size());
}
