/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*@ GENETIC.LIB (c) Jochen Ruhland */
#include "version.inc"
/*  Genetic Operators */
/*  Low-Level Funktionen */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <memory.h>

#ifdef _MSDOS
#include <search.h>
#endif

#undef GENLIB_DEBUG
#define GENLIB_DEBUG 0

#include "support.h"
#include "g0.h"
#include "portrnd.h"
#include "debug.h"


/*  Masken fuer die binaere Verarbeitung von Genomen */
static int      masksize = 0;

static USHORT  *and_maske = NULL, *or_maske = NULL;

static USHORT  *bit_maske = NULL;

/*  Erstellt die Bitmasken fuer maximal sizemask USHORTs */

/*
 *  Funktionsname : set_mask 
 
 *  Beschreibung  : Legt Speicherbereich fuer die Bitmasken an. Bei
 *  Bedarf wird der Speicherbereich vergroessert
 *  Input         : sizemask Anzahl der benoetigten Bits in den Masken
 *  Output        : nope 
 */
void set_mask(int sizemask)
{
	DEBUG_call(set_mask);
	DEBUG_int(sizemask);
	/*  Damit die bit-Masken immer genuegend Platz haben */
	if (masksize < sizemask)
	{
		/*  Die schon definierten werden freigegeben */
		if (bit_maske != NULL)
		{
			free(bit_maske);
			free(and_maske);
			free(or_maske);
		}
		bit_maske = calloc(sizemask, sizeof(USHORT));
		assert(bit_maske != NULL);
		and_maske = calloc(sizemask, sizeof(USHORT));
		assert(and_maske != NULL);
		or_maske = calloc(sizemask, sizeof(USHORT));
		assert(or_maske != NULL);
		masksize = sizemask;
	}
}


/*  Belegt popelem mit zufaelligen Belegungen */

/*
 *  Funktionsname : __randomize_popelem 
 
 *  Beschreibung  : Eine Populationselement wird mit zufaelligen Werten
 *  gefuellt
 *  Input         : p Zeiger auf den Speicherbereich
 *				    size Groesse des Bereiches in Bytes
 *  Output        : nope 
 */
void __randomize_popelem(USHORT *p, int size)
{
	unsigned int    j;

	DEBUG_call(__randomize_popelem);
	DEBUG_int(size);
	assert(p != NULL);
	for (j = 0; j < size / sizeof(USHORT); j++)
		*p++ = rands();
}

/*
 *  Funktionsname : __copy_popelem 
 
 *  Beschreibung  : Kopiert ein Populationselement auf's andere 
 *  Input         : source von wo destination nach wo size wieviel 
 *  Output        : nope 
 */
void __copy_popelem(USHORT *source, USHORT *destination, int size)
{
	DEBUG_call(__copy_popelem);
	DEBUG_int(size);
	assert(source != NULL);
	assert(destination != NULL);
	assert(size > 0);
	memcpy(destination, source, size);
}

/*
 *  Funktionsname : __mutate 
 
 *  Beschreibung  : Kippt mit der Wahrscheinlichekit p_kipp die Bits in p 
 *  Input         : p Zeiger auf Element
 *                  p_kipp Wahrscheinlichkeit size Groesse eines Elementes
 *  Output        : nope 
 */
void __mutate(USHORT *p, float p_kipp, int size)
{
	USHORT          s, b, mask;

	DEBUG_call(__mutate);
	DEBUG_float(p_kipp);
	DEBUG_int(size);
	assert(p != NULL);
	assert((p_kipp >= 0.0) && (p_kipp <= 1.0));
	assert(size > 0);

	for (s = 0; s < size / sizeof(short); s++, p++)
		/*  Die aeussere Schleife geht ueber alle USHORTS drueber */
		for (mask = 0x1, b = 0; b < 16; b++, mask <<= 1)
			/*  Die innere Schleife ueber die Bits eines USHORTs */
			if (p_kipp > frand16())
				*p ^= mask;

}


#pragma auto_inline( off )

/*
 *  Funktionsname : mask_op 
 
 *  Beschreibung  : Mittels der Bitmaske wird aus v und m s
 *  Input         : v vater-Genom
 *                  m Mutter-Genom
 *                  s Sohn-Genom, da kommts rein
 *                  size Anzahl der Bytes
 *  Output        : nope 
 */
static void mask_op(USHORT *v, USHORT *m, USHORT *s, int size)
{
	/*  Mit Hilfe von BitMaske werden v und m nach s gebracht */
	/*  dabei kommen gesetzte Bits in bit_maske von v */

	unsigned int    i;

	DEBUG_call(mask_op);
	DEBUG_int(size);
	for (i = 0; i < size / sizeof(USHORT); i++)
		s[i] = (USHORT) ((v[i] & bit_maske[i]) |
		 (m[i] & ((USHORT) (~bit_maske[i]))));
}

#pragma auto_inline( on )

/*
 *  Funktionsname : __crossover 
 
 *  Beschreibung  : 1-Punkt Crossover, die ersten numbits kommen von v 
 *  Input         : v Vater-Genom
 *                  m Mutter-Genom
 *                  numbits Anzahl der Bits
 *                  s Sohn_genom, da kommts rein
 *                  size Anzahl der Bytes
 *  Output        : nope 
 */
void __crossover(USHORT *v, USHORT *m, USHORT numbits,
			                    USHORT *s, USHORT size)
{
	USHORT          maske;
	USHORT          numshort, bitcount, i;

	DEBUG_call(__crossover);
	DEBUG_ushort(numbits);
	DEBUG_ushort(size);
	assert(v != NULL);
	assert(m != NULL);
	assert(s != NULL);
	assert(size > 0);

	bitcount = (USHORT) (numbits & (USHORT) 0xf);
	/*  Wieviele Bits haben wir */
	numshort = (USHORT) (numbits >> (USHORT) 4);
	/*  Wieviele shorts haben wir */

	assert(numshort <= size);

	/*  Zunaechst die vollen shorts von v nach s */
	for (i = 0; i < numshort; i++)
		bit_maske[i] = 0xffff;

	/*  Dann das gesplittete short */
	maske = (USHORT) ((USHORT) 0xffff << (16 - bitcount));
	bit_maske[numshort] = maske;

	/*  Den Rest von m nach s */
	for (i = (USHORT) (numshort + 1);
			 i < (USHORT) (size / sizeof(USHORT));
				 i++)
		bit_maske[i] = 0;

	mask_op(v, m, s, size);

}

/*
 *  Funktionsname : __crossover2 
 
 *  Beschreibung  : 2-Punkt Crossover, die Bits n1 bis n2-1 kommen von m 
 *  Input         : v Vater-Genom
 *                  m Mutter-Genom
 *                  n1,n2 Bitbereich
 *                  s Sohn_genom, da kommts rein
 *                  size Anzahl der Bytes
 *  Output        : nope 
 */
void __crossover2(USHORT *v, USHORT *m, USHORT n1, USHORT n2,
			                     USHORT *s, USHORT size)
{
	/*  Teile Bits nach folgendem Muster auf : */
	/* 0..n1-1 von v n1..n2-1 von m n2..ende von v */
	USHORT          maske;
	USHORT          numshort, bitcount, i;

	DEBUG_call(__crossover2);
	DEBUG_ushort(n1);
	DEBUG_ushort(n2);
	DEBUG_ushort(size);

	assert(v != NULL);
	assert(m != NULL);
	assert(s != NULL);
	assert(size > 0);

	bitcount = (USHORT) (n1 & (USHORT) 0xf);
	numshort = (USHORT) (n1 >> (USHORT) 4);

	assert(numshort <= size);

	/*  Zunaechst die vollen shorts von v nach s */
	for (i = 0;
			 i < numshort;
				 i++)
		bit_maske[i] = 0xffff;

	/*  Dann das gesplittete short */
	maske = (USHORT) ((USHORT) 0xffff << (16 - bitcount));
	bit_maske[numshort] = maske;

	/*  Den Rest von m nach s */
	for (i = (USHORT) (numshort + 1);
			 i < (USHORT) (size / sizeof(USHORT));
				 i++)
		bit_maske[i] = 0;
	/*  Dat war die erste Operation  */

	bitcount = (USHORT) (n2 & (USHORT) 0xf);
	numshort = (USHORT) (n2 >> (USHORT) 4);

	assert(numshort <= size);

	/*  Zunaechst die vollen shorts von v nach s */
	for (i = 0; i < numshort; i++)
		bit_maske[i] ^= 0;

	/*  Dann das gesplittete short */
	maske = (USHORT) ((USHORT) 0xffff << (USHORT) (16 - bitcount));
	bit_maske[numshort] ^= ~maske;

	/*  Den Rest von m nach s */
	for (i = (USHORT) (numshort + 1);
			 i < (USHORT) (size / sizeof(USHORT));
				 i++)
		bit_maske[i] ^= 0xffff;

	mask_op(v, m, s, size);

}

/*
 *  Funktionsname : __crossovern 
 
 *  Beschreibung  : n-Punkt Crossover, die Bits kommen mit p_vat von v 
 *  Input         : v Vater-Genom
 *                  m Mutter-Genom
 *                  p_var Wahrscheinlichekit, dass die Bits von v kommen
 *                  s Sohn_genom, da kommts reini
 *                  size Anzahl der Bytes
 *  Output        : nope 
 */
void __crossovern(USHORT *v, USHORT *m, double p_vat, USHORT *s, int size)
{
	USHORT          i;

	DEBUG_call(__crossovern);
	DEBUG_double(p_vat);
	DEBUG_ushort(size);

	assert(v != NULL);
	assert(m != NULL);
	assert(s != NULL);
	assert(size > 0);
	assert((p_vat >= 0.0) && (p_vat <= 1.0));

	/*  Setze eine zufaellige Bitmaske */
	for (i = 0; i < size / sizeof(short); i++)
		bit_maske[i] = brand16(p_vat);

	mask_op(v, m, s, size);

}


#pragma auto_inline( off )

/*
 *  Funktionsname : __hamming_diff 
 
 *  Beschreibung  : Die Hammingdifferenz zwischen 2 Popelementen
 *  Input         : p1, p2 Zeiger auf die Popelemente
 *	                size die Groesse der beiden Elemente 
 *  Output        : Die Hammingdifferenz 
 */
int __hamming_diff(USHORT *p1, USHORT *p2, int size)
{
	unsigned int    i;
	int             df;

	DEBUG_call(__hamming_diff);
	DEBUG_int(size);

	assert(p1 != NULL);
	assert(p2 != NULL);
	for (i = 0, df = 0;
	     i < size / sizeof(USHORT);
	     i++)
		df += hamming_ushort(p1[i], p2[i]);
	return df;
}

#pragma auto_inline( on )
