/* ant.c */
/* 06-Mar-93 PMB : created: repeat John Muir Trail experiment */
/* 07-Mar-93 PMB : 1st version complete. */
/* 08-Mar-93 PMB : substantial modifications to gene definition. */
/* 11-Mar-93 PMB : fixed persistent memory bug */
/* 12-Mar-93 PMB : modified selection and mutation */
/* 14-Mar-93 PMB : improved examination tools somewhat */
/* 14-Mar-93 PMB : dist. to Dave Mac, then changed crossover */
/* 01-May-93 PMB : ant erases map space */
/* 13-Jul-93 PMB : express mutation rate as prob. per bit per generation;
                   express crossover as prob. per bit per generation. */
/* 19-Jul-93 PMB : write out state of experiment */
/* 02-Aug-93 PMB : don't overwrite good parents */
/* 15-Aug-93 PMB : improved instrumentation */
/* 22-Aug-93 PMB : more instruments */
/* 24-Aug-93 PMB : added some UNIX defines */
/* 05-Oct-93 PMB : improved random number handling */

#define TRUE 1
#define FALSE 0

#define DEBUG 0

#define MSDOS 1
#define UNIX  0

#define CHILD -1
#define PARENT -2

#include <stdio.h>

/* N O T E : The port to UNIX is incomplete as of this writing. */

#if MSDOS
#include <stdlib.h>
#include <conio.h>
#include <alloc.h>
#include <mem.h>
#include <time.h>
#include <dos.h>
#include <string.h>
#include <io.h>
#endif

#if UNIX
#include <sys/time.h>
#include <malloc.h>

#define huge
#endif

/* Necessary statics */
static char log_file_name[100];
static FILE *log_file = NULL;

static int keypressed = FALSE;

void error_crash(char *msg);

/* See Levy, Artificial Life, p166, and Jefferson et al., A-Life II, p549,
   for a complete description of the John Muir Trail experiment. */

/**************************************************************************/
void flush(FILE *stream)
{
     int duphandle;

     /* flush the stream's internal buffer */
     fflush(stream);

     /* make a duplicate file handle */
     duphandle = dup(fileno(stream));

     /* close the duplicate handle to flush the DOS buffer */
     close(duphandle);
}

/**************************************************************************/
/* Ant's gene is a state machine. */
/* Each state has associated with it :
   (1) an output code corresponding to each possible input code; and
   (2) the next state to assume. */

/* How many possible input codes are there? */
#define N_INPUT_CODES   2

/* How many possible output codes are there? */
#define N_OUTPUT_CODES  4
#define OUTPUT_MASK 192
#define OUTPUT_SHIFT 6

/* How many states are available? */
#define N_STATES        32
#define STATE_MASK      31

typedef struct td_state
{
  /* for each input code, there is an output code and a next state */
  char code_next_state [ N_INPUT_CODES ];
} State;

typedef struct td_state_machine
{
  State state [ N_STATES ];
} State_machine;

#define GENE_SIZE sizeof(State_machine)

/**************************************************************************/
void run_machine(State_machine huge *s, int this_state, int in,
                 int *r_out, int *r_next)
{
  int code = s->state[this_state].code_next_state[in];

  *r_out = (code & OUTPUT_MASK) >> OUTPUT_SHIFT ;
  *r_next = code & STATE_MASK ;
}

/**************************************************************************/
/* Bit patterns for output codes */
#define MOVE_MASK       0x3
#define NO_MOVE         0
#define MOVE_FWD        1
#define MOVE_RIGHT      2
#define MOVE_LEFT       3

/**************************************************************************/
void print_movename(int output_code)
{
  switch (output_code & MOVE_MASK)
  {
    case NO_MOVE    : printf("NOP"); break;
    case MOVE_FWD   : printf("FWD"); break;
    case MOVE_RIGHT : printf("RGT"); break;
    case MOVE_LEFT  : printf("LFT"); break;
  }
}

/**************************************************************************/
/* Print a gene: */
void print_gene(State_machine huge *p_gene)
{
  int i,j;
  int code, next;

  printf("Input =      0      1\n");
  printf("           ====== ======\n");

  for (i=0 ; i < N_STATES ; i++)
  {
    printf("State %02X : ", i);
    for ( j=0 ; j < 2 ; j++ )
    {
      run_machine(p_gene, i, j, &code, &next);
      print_movename(code);
      printf("/%02X ", (next));
    }
    printf("\n");
  }
}
    

/**************************************************************************/
/* Create a random gene */
void randomize_gene(char huge *p_gene)
{
  int i;

  for (i = 0 ; i < GENE_SIZE ; i++)
    p_gene[i] = rand() % 0xff;
}

/**************************************************************************/
/* Determine whether a random real number lies within a given threshold. */
int froll(double threshold)
{
  int inum;
  double dnum;

  inum = rand() & 0x3ff /* % 16384 */ ;
  dnum = ((double)inum)/16384.0;

  return(dnum < threshold);
}

/**************************************************************************/
/* Perform crossover for two given pairs of genes */
void crossover(char huge *p_gene1, char huge *p_gene2, double pcross)
{
  int index, bit_number;
  unsigned char char1, char2, mask1, mask2;
  int state = 0;
  int will_cross, where_cross;

  /* To improve speed, we are adopting the simplifying restriction that
     only one crossover occurs per byte in the gene.  Math majors and stats
     wonks will hate the next line : compute the prob. of crossover in one
     byte -- I know it's a bad computation, but it's a good approximation. */
  pcross *= 8.0 ;


  for (index = 0 ; index < GENE_SIZE ; index ++ )
  {
    char1 = p_gene1 [ index ];
    char2 = p_gene2 [ index ];
    mask1 = mask2 = 0;

    if ( will_cross = froll(pcross))
      where_cross = rand() % 8;
    else
      where_cross = -1;

    for ( bit_number = 0 ; bit_number < 8 ; bit_number ++ )
    {
      if (DEBUG)
        printf("gene # %4d bit # %d\n", index, bit_number);

      if ( bit_number == where_cross )
        state = !state ;  /* Switch state */
      mask1 += state << bit_number;
    }
    mask2 = mask1 ^ 0xff;
    p_gene1 [ index ] = (char1 & mask1) | (char2 & mask2);
    p_gene2 [ index ] = (char1 & mask2) | (char2 & mask1);
  }
}

/**************************************************************************/
/* Perform mutation on some given gene */
void mutate(char huge *p_gene)
{
  int stateno, bit;

  stateno = rand() % GENE_SIZE;
  bit = rand() % 8;

  p_gene [ stateno ] ^= ( 1 << bit);
}

/**************************************************************************/
/* A structure to associate a gene with its score. */
typedef struct td_gene_track
{
  int score;
  int n_children;
  char huge *p_gene;
} Gene_track;

#define GENE_TRACK_SIZE sizeof(Gene_track)

/**************************************************************************/
void print_gene_track( Gene_track huge *g )
{
  int i;

  printf("Score = %d\n", g->score);
  printf("n_children = %d\n", g->n_children);
  printf("Gene = ");
  for (i=0 ; i<GENE_SIZE ; i++)
    printf("%02X", (char)(g->p_gene[i]) & 0xff);
  printf("\n");
}

/**************************************************************************/
void print_pop_stats( Gene_track huge *gene_track_array, int n_pop)
{
  int score_bin[90];
  int i;

  /* Initialize */
  for ( i=0 ; i<90 ; i++ )
    score_bin[i] = 0;

  /* Count population with given score */
  for ( i=0 ; i<n_pop ; i++ )
    score_bin [ (gene_track_array[i].score) ] ++;

  /* Print out the results : */
  printf("\nPopulation Statistics :\n");
  printf("%8s %8s | %8s %8s | %8s %8s\n", "SCORE", "N_POP", "SCORE", "N_POP",
                                      "SCORE", "N_POP");
  for ( i=0 ; i<90 ; i += 3 )
    printf("%8d %8d | %8d %8d | %8d %8d\n", i, score_bin[i], 
                                        i+1, score_bin[i+1],
                                        i+2, score_bin[i+2]);
}

/**************************************************************************/
int find_ant_w_score( Gene_track huge *gene_track_array, int n_pop,
                      int score, int start_with )
{
  int i;

  for ( i=start_with ; i < n_pop ; i++ )
    if (gene_track_array[i].score == score)
      return(i);

  /* None found */
  return(-1);
}

/**************************************************************************/
/* Create a new random population. */
int initialize_gene_pool(Gene_track huge **gene_track_array_ptr, int n_pop)
{
  Gene_track huge *gtp;
  int i;

  if (gene_track_array_ptr == NULL)
    error_crash("Hey!  Bad pointer.\n");

  gtp = *gene_track_array_ptr = calloc(n_pop, GENE_TRACK_SIZE);

  if (gtp == NULL)
  {
    fprintf(stderr, "Can't allocate gene tracking array!\n");
    return(1);
  }

  for (i=0 ; i<n_pop ; i++, gtp++)
  {
    gtp->score = 0;
    gtp->n_children = 0;
    gtp->p_gene = (char huge *)malloc(GENE_SIZE);
    if (gtp->p_gene == NULL)
    {
      fprintf(stderr, "Can't allocate gene!\n");
      return(1);
    }
    randomize_gene(gtp->p_gene);
  }
  return(0);
}

/**************************************************************************/
#define SELECT_STRATEGY_TRUNCATE   0  /* Truncation selection strategy */
#define SELECT_STRATEGY_ROULETTE   1  /* Roulette-wheel selection strategy */

/**************************************************************************/
/* Select the best genes from the population. */
/* Return average, minimum, and maximum scores. */
double select(Gene_track huge *gene_track_array, int n_pop, int n_select, 
              int selection_strategy, int *r_max, int *r_min)
{
  int i, max_score, min_score;
  int target_score;
  double avg_score;
  int children_to_allocate = n_pop - n_select; /* Keep the pop. constant */
  int children_per_selected = children_to_allocate / n_select ;

  /* Initialize */
  max_score = 0;
  min_score = 100;
  for (i=0, avg_score = 0.0 ; i<n_pop ; i++)
  {
    avg_score += gene_track_array[i].score;
    gene_track_array[i].n_children = 0;
    if (gene_track_array[i].score > max_score)
      max_score = gene_track_array[i].score;
    if (gene_track_array[i].score < min_score)
      min_score = gene_track_array[i].score;
  }

  avg_score /= n_pop;
  *r_max = max_score;
  *r_min = min_score;

  /* Now allocate children */

  while (children_to_allocate > 0)
  {
    if (kbhit())
    {
      getch();
      keypressed = TRUE;
      printf(" <%d> " ,children_to_allocate);
    }

    for ( target_score = max_score ; 
          (target_score > 0)&&(children_to_allocate > 0) ; 
          target_score-- )
    {
      for ( i = 0 ; i<n_pop ; i++)
      {
        if ( gene_track_array[i].score == target_score )
        {
          if ((selection_strategy == SELECT_STRATEGY_ROULETTE) &&
              (rand() % max_score > gene_track_array[i].score))
              continue;

          if (children_to_allocate > children_per_selected)
          {
            gene_track_array[i].n_children = children_per_selected;
            children_to_allocate -= children_per_selected;
          }
          else
          {
            gene_track_array[i].n_children = children_to_allocate;
            children_to_allocate = 0;
          }
        }
      }
    }
  }
return(avg_score);
}

/**************************************************************************/
/* Reproduce the population. */
int reproduce(Gene_track huge *gene_track_array, int n_pop,
              double pcross, double pmutate)
{
  int i,j,k,n;
  int nmutations = (int)(n_pop * pmutate * GENE_SIZE);

  /* Build all children. */
  for ( i=0, j=0 ; (i<n_pop)&&(j<n_pop) ; i++)
  {
    if (gene_track_array[i].n_children <= 0)
      continue;

    for (n = gene_track_array[i].n_children ; n > 0 ; n--)
    {
      /* Search for a place to build a child */
      for ( ; (gene_track_array[j].n_children != 0)&&(j<n_pop) ; 
            j++ );

      if ( j == n_pop ) continue;

      /* Copy it: */
      memcpy( (void *)(gene_track_array[j].p_gene), 
              (void *)(gene_track_array[i].p_gene), 
              (size_t)GENE_SIZE );

      /* Overwrite the child as reproduced: */
      gene_track_array[j].n_children = CHILD;

    }
    /* Overwrite the parent as reproduced: */
    gene_track_array[i].n_children = PARENT;
  }

  /* Perform crossover */
  for ( i=0 ; i<(n_pop/2) ; i++ )
  {

    /* Select the parent genes */
    while ((j = rand() % n_pop),(gene_track_array[j].n_children == PARENT));
    while ((k = rand() % n_pop),(gene_track_array[k].n_children == PARENT));


    /* Perform crossover */
    crossover(gene_track_array[j].p_gene, gene_track_array[k].p_gene, 
              pcross);

  }

  /* Perform mutation */
  for ( i=0 ; i<nmutations ; i++ )
  {
    while ((j = rand() % n_pop),(gene_track_array[j].n_children == PARENT));
    mutate(gene_track_array[ j ].p_gene );
  }

  return(0);
}

/**************************************************************************/
#define EAST    0
#define NORTH   1
#define WEST    2
#define SOUTH   3

/**************************************************************************/
typedef struct td_ant
{
  int score;            /* starts at zero */
  int cur_state;        /* starts at zero */
  int x,y;              /* position; starts at (0,0) */
  int heading;          /* E=0, N=1, W=2, S=3; starts at E */
} Ant;

/**************************************************************************/
void init_ant(Ant *p_ant)
{
  p_ant->score = 0;
  p_ant->cur_state = 0;
  p_ant->x = 0;
  p_ant->y = 0;
  p_ant->heading = EAST;
}

/**************************************************************************/
/* Define the ant's map */
#define MAP_ROWS        32
#define MAP_COLS        32

/**************************************************************************/
/* If the map has a nonzero number in it, it is "scented", else not */
static int map [ MAP_ROWS ] [ MAP_COLS ] =
{
/*col:    0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 */
/*row:*/
/* 00 */  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 01 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 02 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 03 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 04 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 05 */ 35,34,33,32, 0, 0, 0, 0, 0, 0,15, 0, 0, 0, 0,89, 0, 0, 0, 0, 0, 0, 0, 0, 0,42,41,40,39,38,37,36,
/* 06 */  0, 0, 0,31, 0, 0, 0, 0, 0, 0,16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,43, 0, 0, 0, 0, 0, 0, 0,
/* 07 */  0, 0, 0,30, 0, 0, 0, 0, 0, 0,17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,44, 0, 0, 0, 0, 0, 0, 0,
/* 08 */  0, 0, 0,29, 0, 0, 0, 0, 0, 0,18, 0, 0, 0,88, 0, 0, 0, 0, 0, 0, 0, 0, 0,45, 0, 0, 0, 0, 0, 0, 0,
/* 09 */  0, 0, 0,28, 0, 0, 0, 0, 0, 0,19, 0,87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,46, 0, 0, 0, 0, 0, 0, 0,
/* 10 */  0, 0, 0,27,26,25,24,23,22,21,20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,47, 0, 0, 0, 0, 0, 0, 0,
/* 11 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,52,51,50,49,48, 0, 0, 0, 0, 0, 0, 0, 0,
/* 12 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,86, 0, 0, 0, 0, 0, 0,53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 13 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,85, 0, 0, 0,54, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 14 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,55, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 15 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,84, 0, 0,56, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 16 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,83, 0, 0, 0, 0, 0,57, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 17 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,58, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 18 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,82, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 19 */  0, 0, 0, 0, 0, 0, 0, 0,81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 20 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 21 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,60, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 22 */  0, 0, 0, 0, 0, 0, 0,80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 23 */  0, 0, 0, 0, 0,79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 24 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,63, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 25 */  0, 0, 0, 0,78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 26 */  0, 0, 0, 0,77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 27 */  0, 0, 0, 0,76, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 28 */  0, 0, 0, 0,75, 0, 0,74,73,72,71, 0,70,69,68,67,66,65, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 29 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 30 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 31 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

/**************************************************************************/
void draw_trail(void)
{
  int i,j;

  textmode(C4350);
  textbackground(LIGHTGRAY);
  clrscr();

  for (j=0; j<MAP_ROWS; j++)
  {
    for (i=0; i<MAP_COLS; i++)
      if (map[j][i])
      {
        textcolor(BLUE);
        putch(254);
      }
      else
      {
        textcolor(BLACK);
        putch(250);
      }
    gotoxy(1,j+2);
  }
}

/**************************************************************************/
void draw_ant(Ant *p_ant)
{
  int ch;

  gotoxy(p_ant->x + 1, p_ant->y + 1);
  textcolor(RED);  /* It's a RED ANT! */
  switch(p_ant->heading)
  {
    case (NORTH) : ch = 30 ; break;
    case (EAST)  : ch = 16 ; break;
    case (SOUTH) : ch = 31 ; break;
    case (WEST)  : ch = 17 ; break;
  };
  putch(ch);

  gotoxy(35,3);
  printf("Score = (dec)%02d", p_ant->score);
  gotoxy(35,4);
  printf("State = (hex)%02x", p_ant->cur_state);
  gotoxy(35,5);
  printf("Posn  = (%02d,%02d)", p_ant->x, p_ant->y);
}

/**************************************************************************/
void erase_ant(Ant *p_ant)
{
  int x = p_ant->x;
  int y = p_ant->y;

  gotoxy(x + 1, y + 1);
  if (map[y][x])
  {
    textcolor(BLUE);
    putch(254);
  }
  else
  {
    textcolor(BLACK);
    putch(250);
  }
}

/**************************************************************************/
void geom_incr(int *px, int *py, int head)
{
  int x = *px;
  int y = *py;
  
  switch (head)
  {
    case (EAST):
        x++;
        if (x==MAP_COLS)
          x = 0;
        break;
    case (NORTH):
        y--;
        if (y<0)
          y = MAP_ROWS - 1;
        break;
    case (WEST):
        x--;
        if (x<0)
          x = MAP_COLS - 1;
        break;
    case (SOUTH):
        y++;
        if (y==MAP_ROWS)
          y = 0;
        break;
    default:
        error_crash("Confused about heading!");
        break;
  };
  *px = x;
  *py = y;

}

/**************************************************************************/
int ant_input(Ant *pant) /* Returns what ant sees */
{
  int x = pant->x;
  int y = pant->y;

  /* What is the square that the ant sees? */
  geom_incr(&x, &y, pant->heading);
  
  return(map[y][x] > 0);
}

/**************************************************************************/
void move_ant(Ant *pant, int output_code)
{
  int head = pant->heading;

  switch (output_code & MOVE_MASK)
  {
    case NO_MOVE:
        break;
    case MOVE_FWD:
        geom_incr(&(pant->x), &(pant->y), head);
        break;
    case MOVE_RIGHT:
        head++;
        if (head > SOUTH)
          head = EAST;
        pant->heading = head;
        break;
    case MOVE_LEFT:
        head--;
        if (head < EAST)
          head = SOUTH;
        pant->heading = head;
        break;
  }
}

/**************************************************************************/
/* Score the ant */
void score_ant(Ant *p_ant)
{
  int x = p_ant->x;
  int y = p_ant->y;

  if (p_ant->score + 1 == map[y][x])
    (p_ant->score)++;

  if (map[y][x] > 0)
    map[y][x] *= -1;
}

/**************************************************************************/
/* Update the ant, given its gene and its current state */
void update_ant( Ant *p_ant, State_machine huge *p_gene, int draw_it )
{
  int input_code = ant_input(p_ant);
  int output_code;
  int next_state;

  run_machine(p_gene, p_ant->cur_state, input_code, 
              &output_code, &next_state);

  if (draw_it)
    erase_ant(p_ant);

  /* Move the ant according to the output code */
  move_ant(p_ant, output_code);

  if (draw_it)
    draw_ant(p_ant);

  /* Score the ant */
  score_ant(p_ant);

  /* And reset its state */
  p_ant->cur_state = next_state;
}

/**************************************************************************/
/* 
   Given a gene:
   Create an ant, run it through a life cycle,
   and return the score.
*/
int ant_life(State_machine huge *p_gene, int n_cycles, int draw_it)
{
  Ant this_ant;
  int cycle;
  int x,y;
  int single_step = FALSE;

  if (draw_it)
  {
    printf("Do you want to single-step the ant through its life?");
    single_step = getyn();
  }

  /* Fix map */
  for ( x=0 ; x<MAP_COLS ; x++ )
    for ( y=0 ; y<MAP_ROWS ; y++ )
      if (map[y][x] < 0)
        map[y][x] *= -1;

  init_ant(&this_ant);

  for (cycle = 0 ; cycle < n_cycles ; cycle++)
  {
    update_ant(&this_ant, p_gene, draw_it);
    if (draw_it)
      delay(100);
    if (single_step)
    {
        while (!kbhit());
        getch();
    }
  }

  if (draw_it)
    gotoxy(1,35);

  return(this_ant.score);
}

/**************************************************************************/
void expose(Gene_track huge *gene_track_array, int n_pop, int n_steps)
{
  int i, score;

  for (i=0 ; i<n_pop ; i++)
  {
    if (kbhit())
    {
      keypressed = TRUE;
      getch();
      printf(" <%d> ", i);
    }

    score = ant_life((State_machine huge *)gene_track_array[i].p_gene, 
                      n_steps, 0);
    gene_track_array[i].score = score;
  }
}

/**************************************************************************/
int save_state(char *save_file_name, int n_pop, int n_generations, 
               int gen_number, int n_life_steps, int n_sel, 
               int selection_strategy, double pcross, double pmutate,
               Gene_track huge *gene_track_array)
{
  FILE *savefile;
  time_t t_now;
  char *time_string;
  int i,j;

  t_now = time(NULL);
  time_string = ctime(&t_now);

  savefile = fopen(save_file_name, "w");

  if (savefile == NULL)
  {
    fprintf(stderr, "Sorry, can't open save file\n");
    return(1);
  }

  fprintf(savefile, "# Record of GA Experiment performed on %s", 
          time_string);
  fprintf(savefile, "%d # Population Size\n", n_pop);
  fprintf(savefile, "%d # Number of Generations\n", n_generations);
  fprintf(savefile, "%d # Current Generation\n", gen_number);
  fprintf(savefile, "%d # Number of Steps in an Ant's Life\n", n_life_steps);
  fprintf(savefile, "%d # Selection Number\n", n_sel);
  fprintf(savefile, "%d # Selection Strategy 1=Roulette, 0=Truncation\n",
          selection_strategy);
  fprintf(savefile, "%lf # Crossover Rate\n", pcross);
  fprintf(savefile, "%lf # Mutation Rate\n", pmutate);
  fprintf(savefile, "%s # Log File Name\n", log_file_name);

  for ( i=0 ; i<n_pop ; i++ )
  {
    fprintf(savefile, "%d %d # Gene %d\n", 
            gene_track_array[i].score, gene_track_array[i].n_children, i);
    for ( j=0 ; j<GENE_SIZE ; j++)
      fprintf(savefile, "%02X", (char)(gene_track_array[i].p_gene[j]) & 0xff);
    fprintf(savefile, "\n");
  }

  fclose(savefile);

  return(0);
}

/**************************************************************************/
void examine(Gene_track huge *gene_track_array, int n_pop, int n_life_steps,
             int n_gen, int gen_number, int n_select, int selection_strategy,
             double pcross, double pmutate)
{
  int n;
  int s,a;
  char save_file_name[100];

  while (1)
  {
    printf("\nEXAMINE :\n");

    printf("0 : Return to evolution\n");
    printf("1 : View an ant in action\n");
    printf("2 : View an ant's gene\n");
    printf("3 : View an ant's tracking information\n");
    printf("4 : View the score distribution of the entire population\n");
    printf("5 : Find an ant with a particular score\n");
    printf("6 : Save the current state of the experiment\n");

ENTER_LOOP :
    printf("\nEnter your choice :");
    scanf("%d", &n);

    if ((n<0)||(n>6))
    {
      printf("Invalid\n");
      goto ENTER_LOOP;
    }

    switch(n)
    {
      case(0) :
        return;

      case(1) :
        printf("Enter ant's # (0-%d)", n_pop - 1);
        scanf("%d", &n);
        draw_trail();
        ant_life((State_machine *)(gene_track_array[n].p_gene), 
                 n_life_steps, 1);
        textcolor(BLACK);
        break;

      case(2) :
        printf("Enter ant's # (0-%d)", n_pop - 1);
        scanf("%d", &n);
        print_gene((State_machine *)(gene_track_array[n].p_gene));
        break;

      case(3) :
        printf("Enter ant's # (0-%d)", n_pop - 1);
        scanf("%d", &n);
        print_gene_track(&(gene_track_array[n]));
        break;

      case(4) :
        print_pop_stats( gene_track_array, n_pop);
        break;

      case(5) :
        printf("Enter ant's # to begin searching from (0-%d) : ", n_pop - 1);
        scanf("%d", &n);
        printf("Enter desired score : ");
        scanf("%d", &s);
        a = find_ant_w_score( gene_track_array, n_pop, s, n);
        if (a >= 0)
          printf("Ant #%d has score %d\n", a, s);
        else
          printf("No more ants in list with score %d\n", s);
        break;

      case (6) :
        printf("Enter the name of the save file : ");
        scanf("%s", save_file_name);
        save_state(save_file_name, n_pop, n_gen, gen_number, n_life_steps, 
                   n_select, selection_strategy, pcross, pmutate, 
                   gene_track_array);
        break;
    }
  }
}
      

/**************************************************************************/
int open_log(char *filename, int n_pop, int n_generations, 
             int n_life_steps, int n_select, double pcross, double pmutate,
             int selection_strategy, unsigned seed_number)
{
  extern FILE *log_file;
  time_t t_now;
  char *time_string;

  t_now = time(NULL);
  time_string = ctime(&t_now);

  strcpy(log_file_name, filename);
  log_file = fopen(filename, "w");
  if (log_file == NULL)
  {
    fprintf(stderr,"Couldn't open log file!");
    log_file_name[0] = 0;
    return(1);
  }

  fprintf(log_file, "Record of GA experiment performed on %s\n\n", 
          time_string);

  fprintf(log_file, "PARAMETERS :\n");
  fprintf(log_file, "Random number seed     = %u\n", seed_number);
  fprintf(log_file, "Population Size        = %d\n", n_pop);
  fprintf(log_file, "Number of Generations  = %d\n", n_generations);
  fprintf(log_file, "Lifespan               = %d\n", n_life_steps);
  fprintf(log_file, "Selection Fraction     = %d%%\n", n_select);
  fprintf(log_file, "Selection Strategy     = %s\n",
          selection_strategy ? "ROULETTE" : "TRUNCATE");
  fprintf(log_file, "Crossover Rate         = %.3lf\n", pcross);
  fprintf(log_file, "Mutation Rate          = %.3lf\n\n", pmutate);

  fprintf(log_file, "%8s %8s %8s %8s\n", "GEN#", "MIN", "AVG", "MAX");

  return(0);
}

/**************************************************************************/
void close_log(void)
{
  time_t t_now;
  char *time_string;

  if (log_file == NULL)
    return;

  t_now = time(NULL);
  time_string = ctime(&t_now);

  fprintf(log_file, "\nEnd of experiment at %s\n", time_string);
  fclose(log_file);
}

/**************************************************************************/
void log_gen(int gen, int min, int max, double avg)
{
  if (log_file == NULL)
    return;

  fprintf(log_file, "%8d %8d %8.4lf %8d\n", gen, min, avg, max);
  flush(log_file);
}

/**************************************************************************/
void random_search(unsigned long n_passes, int n_life_steps)
{
  unsigned long score_bins[ 90 ];
  unsigned long i;
  int score;
  State_machine rgene;

  /* Initialize the score bins */
  for ( i=0 ; i<90 ; i++)
    score_bins[i] = 0;

  /* Perform the loop */
  for ( i=0 ; i<n_passes ; i++ )
  {
    randomize_gene((char *)&rgene);
    score = ant_life(&rgene, n_life_steps, 0);
    score_bins[ score ] ++;
    if ( (i % 10) == 0 )
      printf("%lu ", i);
  }

}

/**************************************************************************/
void evolve(Gene_track huge *gene_track_array, int n_pop, int n_generations,
            int n_life_steps, int n_select, double pcross, double pmutate,
            int selection_strategy)
{
  int gen;
  double avg_score;
  int max_score;
  int min_score;

  fprintf(stderr, 
    "Press any key during evolution cycle.  Between select and reproduce,\n");
  fprintf(stderr, 
    "The program will pause and allow you to examine the population in detail.\n\n");

  for (gen = 1 ; gen <= n_generations ; gen++)
  {
    fprintf(stderr, "Gen %d/%d: ", gen, n_generations);

    fprintf(stderr, "expose-");
    expose(gene_track_array, n_pop, n_life_steps);

    fprintf(stderr, "select-");
    avg_score = select(gene_track_array, n_pop, n_select, 
                       selection_strategy, &max_score, &min_score);

    if (keypressed || kbhit())
    {
      keypressed = FALSE;
      examine(gene_track_array, n_pop, n_life_steps, n_generations,
              gen, n_select, selection_strategy, pcross, pmutate);
    }

    if (gen != n_generations)
    {
      fprintf(stderr, "reproduce");
      reproduce(gene_track_array, n_pop, pcross, pmutate);
      fprintf(stderr, "; ");
    }

    log_gen(gen, min_score, max_score, avg_score);
    fprintf(stderr, "Min/Avg/Max Score = %d / %.2lf / %d\n", min_score, avg_score, max_score);

  }
  close_log();
}

/**************************************************************************/
int getyn(void)
{
  char in;

  for ( in = 0; (in != 'y')&&(in != 'Y')&&(in != 'n')&&(in != 'N') ; )
  {
    while (!kbhit());
    in = getche();
    if (in == 0)  /* Function or cursor key */
      in = getche();
  }

  return( (in == 'y')||(in == 'Y') );
}

/**************************************************************************/
void main()
{
  int error;
  int n_pop, n_gen, n_steps;
  int psel;
  double pcross, pmutate;
  unsigned long mem;
  int n_sel;
  Gene_track huge *gene_track_array;
  char log_file_name[80];
  int selection_strategy;
  unsigned seed_number;

  textmode(C4350);
  clrscr();

  printf("*** WELCOME TO ANT 1.0 ***\n");
  printf("Patrick Brennan 1993\n\n");
  
  printf("Do you want the system to select a seed number?\n");
  printf("  (Press 'N' to enter your own seed number) : [y/n] ");

  if (getyn())
  {
    seed_number = (unsigned)time(NULL);
    printf("\nThe system seed number is %u\n", seed_number);
  }
  else
  {
    printf("\nEnter your seed number : ");
    scanf("%u", &seed_number);
    printf("You entered %u\n", seed_number);
  }

  /* Reset random number generator */
  srand(seed_number);

  printf("Size of gene = %d\n", GENE_SIZE);
  printf("Size of gene track = %d\n", GENE_TRACK_SIZE);

  mem = farcoreleft();  
  printf("Memory available = %lu\n", mem);

  printf("Enter the size of your population :");
  scanf("%d", &n_pop);

  printf("Building a new population...\n");
  error = initialize_gene_pool(&gene_track_array, n_pop);
  if (error)
  {
    printf("Couldn't build your population, sorry\n");
    exit(0);
  }

  mem = farcoreleft();  
  printf("Memory available = %lu\n", mem);

  printf("How many generations do you wish to run? :");
  scanf("%d", &n_gen);

  printf("How many steps does an ant get to prove itself? :");
  scanf("%d", &n_steps);

  for (psel = -1 ; (psel <= 0)||(psel >= 100) ;)
  {
    printf("What percent of population will reproduce? :");
    scanf("%d", &psel);
    if ((psel <= 0)||(psel >= 100))
    {
      printf("Sorry, allowable range is 0 < x < 100.  Try again.\n");
      continue;
    }
  }
  n_sel = (int)(((long)n_pop * psel) / 100);
  printf("(This means that %d individuals will survive each generation.)\n",
         n_sel);

  printf("Do you wish to use ROULETTE WHEEL selection strategy?\n");
  printf("   (Answer N to use TRUNCATION strategy) : [y/n] ");
  selection_strategy = getyn();
  printf("\n");

  for (pcross = -1.0 ; (pcross < 0.0)||(pcross > 1.0) ;)
  {
    printf("What is the crossover rate (P/bit)/generation ? :");
    scanf("%lf", &pcross);
    if ((pcross < 0.0)||(pcross > 1.0))
    {
      printf("Sorry, allowable range is 0.0 <= x <= 1.0  Try again.\n");
      continue;
    }
  }

  for (pmutate = -1.0 ; (pmutate < 0.0)||(pmutate > 1.0) ;)
  {
    printf("What is the mutation rate (P/bit)/generation ? :");
    scanf("%lf", &pmutate);
    if ((pmutate < 0.0)||(pmutate > 1.0))
    {
      printf("Sorry, allowable range is 0.0 <= x <= 1.0  Try again.\n");
      continue;
    }
  }

  printf("What is the name of the log file? ");
  scanf("%s", log_file_name);
  if (strlen(log_file_name) > 0)
  {
    open_log(log_file_name, n_pop, n_gen, n_steps, psel, pcross, pmutate,
             selection_strategy, seed_number);
  }
  else
    printf("No log file.");


  evolve(gene_track_array, n_pop, n_gen, n_steps, n_sel, pcross,
         pmutate, selection_strategy);

}
