function ds=pendmod(s,u)
% PENDMOD Dynamical model of inverted pendulum.
%   This function is referred to in Chapter 5.
%
% PENDMOD(X,U)
%   X - Current state of pendulum.
%   U - Force to be applied to pendulum base.
% Returns the derivative of state dX.
%
% The state X is = [base position;
%                   base velocity;
%                   pendulum angle;
%                   pendulum angular velocity]
%
% All angles are in radians.

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

% CONSTANTS
L = 1;          % Length of pendulum
M = 5;          % Mass of base
m = 1;	         % Mass of pendulum
F = 0.1;	       % Friction for base wheels
J = 0.1;	       % Friction in pendulum joint
g = 9.81;       % Gravity
deg = pi/180;   % Used to convert degrees to radians

% STATE
x = s(1);
dx = s(2);
phi = s(3);
dphi = s(4);

% INTERMEDIATE VALUES
c1 = dphi*sin(phi);
c2 = dphi*cos(phi);
c3 = u-F*dx;
c4 = (J/L)+(sin(phi)^2)*m*L;
c5 = sin(phi)*m*(g-L*c2);

% CALCULATE PENDULUM ACCELERATION
if abs(phi) > (89*deg)
	dphi=0; ddphi=0; c1=0; phi=90;
else
	ddphi = c5 + cos(phi)*(M/(M+m)*(c3+m*L*c1)-c3);
	ddphi = ddphi / (c4 + (cos(phi)^2)*M*m*L/(M+m));
end

% CALCULATE BASE ACCELERATION
ddx = (c3 - m*L*(ddphi*cos(phi) - c1))/(M+m);

% STATE DERIVATIVES
ds = [dx; ddx; dphi; ddphi];
