function [y,g,q] = frules(a,w,v,x,f2,f1,f3)
% FRULE Evaluates sets of fuzzy rules defined with shapes.
%
% [Y,G] = FRULES(A,W,Z,X)
%   A - Antecedant table.
%   W - Weight table.
%   Z - GROUPed input fuzzy set shapes.
%   X - Input (column) vector(s).
% Returns:
%   Y  - Output (column) vectors. Uses Centroid of Max-of-Min.
%   G  - Antecedant grades for each rule.
%
% [Y,G] = FRULES(A,W,Z,X,'F1','F2','F3')
%   F1 - name of union function (default = 'or').
%   F2 - name of intersection function (default = 'and').
%   F3 - name of defuzzification function (default = 'centroid').
% Returns:
%   Y - output vectors. Uses F3 of F1-of-F2.
%
% EXAMPLE: x1S = [0 10]; x1L = eqshape(3,x1S,3);
%          x2S = [0 10]; x2L = eqshape(3,x2S,3);
%          % "IF X1 IS LARGE AND X2 IS SMALL, THEN Y IS NEAR 5"
%          % "IF X1 IS MEDIUM AND X2 IS MEDIUM, THEN Y IS NEAR 9"
%          A = [3 1; 2 2];
%          W = [5; 9];
%          Z = group(x1L,x2L);
%          % "IF X1 IS 8 AND X2 IS 4, THEN WHAT IS Y?"
%          x = [8; 4];
%          y = frules(A,W,Z,x)
%
% SEE ALSO: group, flearns, ftrains, shape

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

if nargin < 7, f3 = 'centroid';
  if nargin < 6, f1 = 'and';
    if nargin < 5, f2 = 'or';
    end
  end
end

[ar,ac] = size(a);
[wr,wc] = size(w);
[xr,xc] = size(x);

g = zeros(ar,xc);
q = zeros(ar,xc);

for p=1:xc

  % FUZZIFY INPUTS
  memb = zeros(ar,ac);
  for i=1:ac
    vi = ungroup(v,i);
    fuzzyx = shapeval(vi,x(i,p));
    memb(:,i) = fuzzyx(a(:,i));
  end

  % ANTECEDANT GRADES
  g(:,p) = feval(f1,memb')';

  % INDICES (undocumented output of FRULES)
  if nargout > 2
    for i=1:ar
      ind = find(memb(i,:) >= g(i,p));
      q(i,p) = ind(1);
    end
  end

end

% OUTPUTS

y = (w'*g) ./ (ones(wc,1) * sumc(g));
