//---------------------------------------------------------------------------
//
// FZNUM.CC
//
//    Fuzzy number type implementation.
//
// Contents:
//
//    FzNum::FzNum( double value)
//    FzNum::FzNum( double left, double middle, double right)
//    FzNUm::FzNum( double lBase, double lTop, double rTop, double rBase)
//
//    double FzNum::operator[]( double value)
//
//    FzNum FzNum::operator-()
//
//    FzNum FzNum::operator+( double opnd2)
//    FzNum FzNum::operator+( FzNum opnd2)
//    FzNum FzNum::operator-( double opnd2)
//    FzNum FzNum::operator-( FzNum opnd2)
//    FzNum FzNum::operator*( double opnd2)
//    FzNum FzNum::operator*( FzNum opnd2)
//    FzNum FzNum::operator/( double opnd2)
//    FzNum FzNum::operator/( FzNum opnd2)
//
//    FzNum FzNum::operator+=( double opnd2)
//    FzNum FzNum::operator-=( double opnd2)
//    FzNum FzNum::operator*=( double opnd2)
//    FzNum FzNum::operator/=( double opnd2)
//
//    FzNum operator/( double opnd1, FzNum opnd2)
//
//    ostream& operator<<( ostream& smStream, FzNum item)
//
//    FzNum min( FzNum opnd1, FzNum opnd2)
//    FzNum max( FzNum opnd1, FzNum opnd2)
//
//    FzNum sin( FzNum opnd)
//    FzNum cos( FzNum opnd)
//
//    double fzSM( FzNum opnd1, FzNum opnd2, double lambda)
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------


#include "fznum.h"

#include <stdlib.h>


//---------------------------------------------------------------------------
//
// FzNum::FzNum( double value)
//
//    Constructor: convert a singleton into a fuzzy number.
//
// Arguments:
//
//    double value
//       singleton to convert
//
// Return value:
//
//    a new fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum::FzNum( double value)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] = value;
}


//---------------------------------------------------------------------------
//
// FzNum::FzNum( double left, double middle, double right)
// 
//    Triangular fuzzy number (T.F.N.) constructor.
//
// Arguments:
//
//    double left
//       T.F.N. lower bound
//
//    double middle
//       T.F.N. maximum of presumption value
//
//    double right
//       T.F.N. upper bound
//
// Return value:
//
//    a new fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 03.03.94)
//
//---------------------------------------------------------------------------

FzNum::FzNum( double left, double middle, double right)

{
   ciPLev[ 0] = ConfInt( left, right);
   ciPLev[ NPLEV] = ConfInt( middle);

   double tmp1 = middle - left;
   double tmp2 = right - middle;

   for ( int i = 1; i < NPLEV; i++) {

      double tmp3 = double( i) / NPLEV;
      ciPLev[ i] = ConfInt( left + tmp1 * tmp3,
                            right - tmp2 * tmp3);
   }
}


//---------------------------------------------------------------------------
//
// FzNum::FzNum( double lBase, double lTop, double rTop, double rBase)
// 
//    Trapezoidal fuzzy number constructor.
//
// Arguments:
//
//    double lBase
//    double rBase
//       confidence interval at presumption level 0
//
//    double lTop
//    double rTop
//       confidence interval at presumption level 1
//
// Return value:
//
//    a new fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.08.94)
//
//---------------------------------------------------------------------------

FzNum::FzNum( double lBase, double lTop, double rTop, double rBase)

{
   ciPLev[ 0] = ConfInt( lBase, rBase);
   ciPLev[ NPLEV] = ConfInt( lTop, rTop);

   double tmp1 = lTop - lBase;
   double tmp2 = rBase - rTop;

   for ( int i = 1; i < NPLEV; i++) {

      double tmp3 = double( i) / NPLEV;
      ciPLev[ i] = ConfInt( lBase + tmp1 * tmp3,
                            rBase - tmp2 * tmp3);
   }
}


//---------------------------------------------------------------------------
//
// double FzNum::operator[]( double value)
//
//    Overload the subscripting operator to compute the presumption level
//    used to assert that a given value is represented by a fuzzy number.
//
// Arguments:
//
//    double value
//       the value to work on
//
// Return value:
//
//    the computed presumption level
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 27.08.94)
//
//---------------------------------------------------------------------------

double FzNum::operator[]( double value)

{
   if (ciPLev[ 0][ value]) {

      double pLev;
      int i;

      for ( i = 1; i < NPLEV && ciPLev[ i][ value]; i++);

      if ( value < ciPLev[ i].GetLower())

         pLev = double( i - 1) / NPLEV 
                + ( value - ciPLev[ i - 1].GetLower()) 
                  / ( NPLEV 
                    * ( ciPLev[ i].GetLower() - ciPLev[ i - 1].GetLower()));

      else if ( value > ciPLev[ i].GetUpper())

         pLev = double( i) / NPLEV
                - ( value - ciPLev[ i].GetUpper())
                  / ( NPLEV
                    * ( ciPLev[ i].GetUpper() - ciPLev[ i - 1].GetUpper()));

      else

         pLev = 1;

      return pLev;
   }
   else
      return 0;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator-()
//
//    Apply the unary minus operator to a fuzzy number.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator-()

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = -ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator+( double opnd2)
//
//    Addiction between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to add to
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator+( double opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] + opnd2;

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator+( FzNum opnd2)
//
//    Addiction between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       FzNum number to add to
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator+( FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] + opnd2.ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator-( double opnd2)
//
//    Subtraction between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to subtract from
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator-( double opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] - opnd2;

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator-( FzNum opnd2)
//
//    Subtraction between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to subtract from 
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator-( FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] - opnd2.ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator*( double opnd2)
//
//    Multiplication between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to multiply for
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator*( double opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] * opnd2;

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator*( FzNum opnd2)
//
//    Multiplication between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to multiply for
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator*( FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] * opnd2.ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator/( double opnd2)
//
//    Division between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to divide by
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator/( double opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] / opnd2;

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator/( FzNum opnd2)
//
//    Division between two fuzzy numbers.
//
// Arguments:
//
//    double opnd2
//       fuzzy number to divide by 
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator/( FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = ciPLev[ i] / opnd2.ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator+=( double opnd2)
//
//    Addiction and assignment between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to add to
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator+=( double opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] += opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator+=( FzNum opnd2)
//
//    Addiction and assignment between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to add to
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator+=( FzNum opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] += opnd2.ciPLev[ i];

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator-=( double opnd2)
//
//    Subtraction and assignment between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to subtract from
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator-=( double opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] -= opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator-=( FzNum opnd2)
//
//    Subtraction and assignment between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to subtract from 
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator-=( FzNum opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] -= opnd2.ciPLev[ i];

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator*=( double opnd2)
//
//    Multiplication and assignment between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to multiply for
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator*=( double opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] *= opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator*=( FzNum opnd2)
//
//    Multiplication and assignment between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to multiply for 
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator*=( FzNum opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] *= opnd2.ciPLev[ i];

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator/=( double opnd2)
//
//    Division and assignment between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd2
//       singleton to divide by
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator/=( double opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] /= opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum FzNum::operator/=( FzNum opnd2)
//
//    Division and assignment between two fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd2
//       fuzzy number to divide by 
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    updates the current fuzzy number
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum FzNum::operator/=( FzNum opnd2)

{
   for ( int i = 0; i <= NPLEV; i++)
      ciPLev[ i] /= opnd2.ciPLev[ i];

   return *this;
}


//---------------------------------------------------------------------------
//
// FzNum operator/( double opnd1, FzNum opnd2)
//
//    Division between a singleton and a fuzzy number.
//
// Arguments:
//
//    double opnd1
//    FzNum opnd2
//       division operands
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 06.03.94)
//
//---------------------------------------------------------------------------

FzNum operator/( double opnd1, FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = opnd1 / opnd2.ciPLev[ i];

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// ostream& operator<<( ostream& smStream, FzNum item)
// 
//    Write a fuzzy number into an output stream.
//
// Arguments:
//
//    ostream& smStream
//       reference to stream to write on
//
//    FzNum item
//       fuzzy number to write
//
// Return value:
//
//    reference to the used output stream
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

ostream& operator<<( ostream& smStream, FzNum item)

{
   for ( int i = 0; i <= NPLEV; i++)
    smStream << "Presumption level = " << setw( 5) << double( i) / NPLEV
         << " => " << item.ciPLev[ i] << "\n";

   return smStream;
}


//---------------------------------------------------------------------------
//
// FzNum min( FzNum opnd1, FzNum opnd2)
//
//    Calculate the minimum fuzzy number from two given ones.
//
// Arguments:
//
//    FzNum opnd1
//    FzNum opnd2
//       the two given fuzzy numbers
//
// Return value:
//
//    the computed minimum
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum min( FzNum opnd1, FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = min( opnd1.ciPLev[ i], opnd2.ciPLev[ i]);

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum max( FzNum opnd1, FzNum opnd2)
//
//    Calculate the maximum fuzzy number from two given ones.
//
// Arguments:
//
//    FzNum opnd1
//    FzNum opnd2
//       the two given fuzzy numbers
//
// Return value:
//
//    the computed maximum
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 04.03.94)
//
//---------------------------------------------------------------------------

FzNum max( FzNum opnd1, FzNum opnd2)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = max( opnd1.ciPLev[ i], opnd2.ciPLev[ i]);

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum sin( FzNum opnd)
//
//    Calculate sine of fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd
//       a fuzzy number (expressed in radiants, not degrees)
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.04.94)
//
//---------------------------------------------------------------------------

FzNum sin( FzNum opnd)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = sin( opnd.ciPLev[ i]);

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// FzNum cos( FzNum opnd)
//
//    Calculate cosine of fuzzy numbers.
//
// Arguments:
//
//    FzNum opnd
//       a fuzzy number (expressed in radiants, not degrees)
//
// Return value:
//
//    the resulting fuzzy number
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.04.94)
//
//---------------------------------------------------------------------------

FzNum cos( FzNum opnd)

{
   FzNum fzTmp;

   for ( int i = 0; i <= NPLEV; i++)
      fzTmp.ciPLev[ i] = cos( opnd.ciPLev[ i]);

   return fzTmp;
}


//---------------------------------------------------------------------------
//
// double fzSM( FzNum opnd1, FzNum opnd2, double lambda)
//
//    Calculate a similarity measure between two fuzzy numbers.
//
//    This is a modified version of the dissemblance index to take account
//    of not only the dissemblance between the shapes of two fuzzy numbers
//    but also the separation of their centers, according to the choice of
//    the appropriate 'lambda' value.
//
//    Based on "Uncertain robot environment modelling using fuzzy numbers"
//    by W. J. Kim et al. in "Fuzzy Sets and Systems" n. 61 (1994),
//    North-Holland, pp. 53-62.
//
// Arguments:
//
//    FzNum opnd1
//    FzNum opnd2
//       the two fuzzy numbers to compare
//
//    double lambda
//       the value to balance the weight of the center values and of the
//       shapes of the fuzzy numbers (0 <= lambda <= 1, default value = 0.5)
//
// Return value:
//
//    the computed similarity measure
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 19.04.94)
//
//---------------------------------------------------------------------------


double fzSM( FzNum opnd1, FzNum opnd2, double lambda)

{
   if ( lambda < 0 || lambda > 1) {

      cerr << "\n\nfzSM: 'lambda' value not in [ 0, 1] ( lambda = "
           << lambda << ").\n";

      abort();
   }

   double normFactor = 2 * (
      (( opnd1.ciPLev[ 0].GetUpper() > opnd2.ciPLev[ 0].GetUpper())
         ? opnd1.ciPLev[ 0].GetUpper()
         : opnd2.ciPLev[ 0].GetUpper()) -
      (( opnd1.ciPLev[ 0].GetLower() < opnd2.ciPLev[ 0].GetLower())
         ? opnd1.ciPLev[ 0].GetLower()
         : opnd2.ciPLev[ 0].GetLower()));

   double sm = 0;

   for ( int i = 0; i <= NPLEV; i++)
      sm += ciDelta( opnd1.ciPLev[ i], opnd2.ciPLev[ i]);

   sm = lambda * (( sm / normFactor) / ( NPLEV + 1));

   sm += (( 1 - lambda) *
            ( ciDelta( opnd1.ciPLev[ NPLEV], opnd2.ciPLev[ NPLEV]) /
              normFactor));

   return sm;
}


//---------------------------------------------------------------------------
//
// void FzNum::print()
//
//    Print the fuzzy number internal representation into the standard
//    output.
//
// Arguments:
//
//    none
//
// Return value:
//
//    none
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 08.07.94)
//
//---------------------------------------------------------------------------

void FzNum::print()

{
   int i;
   float x;
   float delta = float( 1) / NPLEV;

   cout << "\n\n";

   for ( i = 0, x = 0; i <= NPLEV; i++, x += delta)
      cout << setprecision( 6) << ciPLev[ i].GetLower() << "\t"
           << setw( 5) <<x << "\n";

   for ( i = NPLEV, x-=delta; i >= 0; i--, x -= delta)
      cout << setprecision( 6) << ciPLev[ i].GetUpper() << "\t"
           << setw( 5) << x << "\n";
}
