The purpose of these benchmarks is to try to identify strengths 
and weaknesses in the basic engine of a Prolog system.

Some of the benchmarks are inspired on a benchmark suite developed at
ICOT for their SIM project, and other benchmark choices were
influenced by discussions with ICOT researchers on the relative
performance of SIM-I vs.  Prolog-20.

File #1, driver.pl:

%   File   : driver.pl
%   Author : Richard O'Keefe based on earlier versions due to
%            Paul Wilk, Fernando Pereira, David Warren et al.
%   Updated: 29 December 1986
%   Defines: from/3 and get_cpu_time/1.
%   Version: Dec-10 Prolog & Quintus Prolog.

:- public
        from/3,
        get_cpu_time/1.

:- mode
        from(+, +, -),
        get_cpu_time(-).

%   from(LowerBound, UpperBound, I)
%   binds I to successive integers in the range LowerBound..UpperBound.
%   It is designed solely for use in this application; for a general
%   way of doing this use the standard library predicate between/3, or
%   perhaps repeat/1.

from(I, I, I) :- !.
from(L, U, I) :- M is (L+U) >> 1,       from(L, M, I).
from(L, U, I) :- M is (L+U) >> 1 + 1,   from(M, U, I).


%   get_cpu_time(T)
%   unifies T with the run time since start-up in milliseconds.
%   (We can't use the second element of the list, as some of the
%   tests will call statistics/2 and reset it.)

get_cpu_time(T) :-
        statistics(runtime, [T,_]).


%   report(N, T0, T1, T2)
%   takes the three times yielded by get_cpu_time and the number
%   of iterations and prints the total, overhead, and average.

report(N, T0, T1, T2) :-
        TestTime is T1-T0,
        OverHead is T2-T1,
        Average  is (TestTime-OverHead)/N,
        write((TestTime-OverHead)/N=Average),
        write(' milli-seconds/iteration'), nl.


%   bench_mark(Name)
%   is the new top level.  It calls bench_mark/4 to find out
%   how many Iterations of the Action and its Control to perform.
%   To get the old effect, do something like
%   bench_mark(nrev, 50, nrev(L), dummy(L)) :- data(L).

bench_mark(Name) :-
        bench_mark(Name, Iterations, Action, Control),
        get_cpu_time(T0),
        (   repeat(Iterations), call(Action), fail
        ;   get_cpu_time(T1)
        ),
        (   repeat(Iterations), call(Control), fail
        ;   get_cpu_time(T2)
        ),
        write(Name), write(' took '),
        report(Iterations, T0, T1, T2).


%   repeat(N)
%   succeeds precisely N times.

repeat(N) :-
        N > 0,
        from(1, N).

from(I, I) :- !.
from(L, U) :- M is (L+U)>>1,   from(L, M).
from(L, U) :- M is (L+U)>>1+1, from(M, U).

File #2, benches.pl

% File   : benches.pl
% Author : Fernando Pereira
% Updated: 29 December 1986
% Defines: benches/0, bench_mark/1
% Purpose:
% Here are all the benchmarks. Some are based on the ICOT benchmark set
% (version of January 24, 1985), others are different. All the benchmarks
% attempt to measure just one thing, eg. determinate procedure call, list
% construction, list destruction.
% To run the whole set, call 'benches'.

% Do all the benchmarks

:- public benches/0, bench_mark/1.

benches :-
    bench_mark(Name, _, _, _),
    bench_mark(Name),
    fail.
benches.

% Trivial predicates for use in controls.

:- public dummy/0, dummy/1, dummy/2, dummy/3.

dummy.

dummy(_).

dummy(_, _).

dummy(_, _, _).

% The actual benchamarks

% 1. 100 determinate tail calls

bench_mark(tail_call_atom_atom, 2000, p1(a), dummy(a)).

:- public p1/1.

p1(a) :- p2(a).
p2(a) :- p3(a).
p3(a) :- p4(a).
p4(a) :- p5(a).
p5(a) :- p6(a).
p6(a) :- p7(a).
p7(a) :- p8(a).
p8(a) :- p9(a).
p9(a) :- p10(a).
p10(a) :- p11(a).
p11(a) :- p12(a).
p12(a) :- p13(a).
p13(a) :- p14(a).
p14(a) :- p15(a).
p15(a) :- p16(a).
p16(a) :- p17(a).
p17(a) :- p18(a).
p18(a) :- p19(a).
p19(a) :- p20(a).
p20(a) :- p21(a).
p21(a) :- p22(a).
p22(a) :- p23(a).
p23(a) :- p24(a).
p24(a) :- p25(a).
p25(a) :- p26(a).
p26(a) :- p27(a).
p27(a) :- p28(a).
p28(a) :- p29(a).
p29(a) :- p30(a).
p30(a) :- p31(a).
p31(a) :- p32(a).
p32(a) :- p33(a).
p33(a) :- p34(a).
p34(a) :- p35(a).
p35(a) :- p36(a).
p36(a) :- p37(a).
p37(a) :- p38(a).
p38(a) :- p39(a).
p39(a) :- p40(a).
p40(a) :- p41(a).
p41(a) :- p42(a).
p42(a) :- p43(a).
p43(a) :- p44(a).
p44(a) :- p45(a).
p45(a) :- p46(a).
p46(a) :- p47(a).
p47(a) :- p48(a).
p48(a) :- p49(a).
p49(a) :- p50(a).
p50(a) :- p51(a).
p51(a) :- p52(a).
p52(a) :- p53(a).
p53(a) :- p54(a).
p54(a) :- p55(a).
p55(a) :- p56(a).
p56(a) :- p57(a).
p57(a) :- p58(a).
p58(a) :- p59(a).
p59(a) :- p60(a).
p60(a) :- p61(a).
p61(a) :- p62(a).
p62(a) :- p63(a).
p63(a) :- p64(a).
p64(a) :- p65(a).
p65(a) :- p66(a).
p66(a) :- p67(a).
p67(a) :- p68(a).
p68(a) :- p69(a).
p69(a) :- p70(a).
p70(a) :- p71(a).
p71(a) :- p72(a).
p72(a) :- p73(a).
p73(a) :- p74(a).
p74(a) :- p75(a).
p75(a) :- p76(a).
p76(a) :- p77(a).
p77(a) :- p78(a).
p78(a) :- p79(a).
p79(a) :- p80(a).
p80(a) :- p81(a).
p81(a) :- p82(a).
p82(a) :- p83(a).
p83(a) :- p84(a).
p84(a) :- p85(a).
p85(a) :- p86(a).
p86(a) :- p87(a).
p87(a) :- p88(a).
p88(a) :- p89(a).
p89(a) :- p90(a).
p90(a) :- p91(a).
p91(a) :- p92(a).
p92(a) :- p93(a).
p93(a) :- p94(a).
p94(a) :- p95(a).
p95(a) :- p96(a).
p96(a) :- p97(a).
p97(a) :- p98(a).
p98(a) :- p99(a).
p99(a) :- p100(a).
p100(a).

% 2. 63 determinate nontail calls, 64 determinate tail calls.

bench_mark(binary_call_atom_atom, 2000, q1(a), dummy(a)).

:- public q1/1.

q1(a) :- q2(a), q3(a).
q2(a) :- q4(a), q5(a).
q3(a) :- q6(a), q7(a).
q4(a) :- q8(a), q9(a).
q5(a) :- q10(a), q11(a).
q6(a) :- q12(a), q13(a).
q7(a) :- q14(a), q15(a).
q8(a) :- q16(a), q17(a).
q9(a) :- q18(a), q19(a).
q10(a) :- q20(a), q21(a).
q11(a) :- q22(a), q23(a).
q12(a) :- q24(a), q25(a).
q13(a) :- q26(a), q27(a).
q14(a) :- q28(a), q29(a).
q15(a) :- q30(a), q31(a).
q16(a) :- q32(a), q33(a).
q17(a) :- q34(a), q35(a).
q18(a) :- q36(a), q37(a).
q19(a) :- q38(a), q39(a).
q20(a) :- q40(a), q41(a).
q21(a) :- q42(a), q43(a).
q22(a) :- q44(a), q45(a).
q23(a) :- q46(a), q47(a).
q24(a) :- q48(a), q49(a).
q25(a) :- q50(a), q51(a).
q26(a) :- q52(a), q53(a).
q27(a) :- q54(a), q55(a).
q28(a) :- q56(a), q57(a).
q29(a) :- q58(a), q59(a).
q30(a) :- q60(a), q61(a).
q31(a) :- q62(a), q63(a).
q32(a) :- q64(a), q65(a).
q33(a) :- q66(a), q67(a).
q34(a) :- q68(a), q69(a).
q35(a) :- q70(a), q71(a).
q36(a) :- q72(a), q73(a).
q37(a) :- q74(a), q75(a).
q38(a) :- q76(a), q77(a).
q39(a) :- q78(a), q79(a).
q40(a) :- q80(a), q81(a).
q41(a) :- q82(a), q83(a).
q42(a) :- q84(a), q85(a).
q43(a) :- q86(a), q87(a).
q44(a) :- q88(a), q89(a).
q45(a) :- q90(a), q91(a).
q46(a) :- q92(a), q93(a).
q47(a) :- q94(a), q95(a).
q48(a) :- q96(a), q97(a).
q49(a) :- q98(a), q99(a).
q50(a) :- q100(a), q101(a).
q51(a) :- q102(a), q103(a).
q52(a) :- q104(a), q105(a).
q53(a) :- q106(a), q107(a).
q54(a) :- q108(a), q109(a).
q55(a) :- q110(a), q111(a).
q56(a) :- q112(a), q113(a).
q57(a) :- q114(a), q115(a).
q58(a) :- q116(a), q117(a).
q59(a) :- q118(a), q119(a).
q60(a) :- q120(a), q121(a).
q61(a) :- q122(a), q123(a).
q62(a) :- q124(a), q125(a).
q63(a) :- q126(a), q127(a).
q64(a).
q65(a).
q66(a).
q67(a).
q68(a).
q69(a).
q70(a).
q71(a).
q72(a).
q73(a).
q74(a).
q75(a).
q76(a).
q77(a).
q78(a).
q79(a).
q80(a).
q81(a).
q82(a).
q83(a).
q84(a).
q85(a).
q86(a).
q87(a).
q88(a).
q89(a).
q90(a).
q91(a).
q92(a).
q93(a).
q94(a).
q95(a).
q96(a).
q97(a).
q98(a).
q99(a).
q100(a).
q101(a).
q102(a).
q103(a).
q104(a).
q105(a).
q106(a).
q107(a).
q108(a).
q109(a).
q110(a).
q111(a).
q112(a).
q113(a).
q114(a).
q115(a).
q116(a).
q117(a).
q118(a).
q119(a).
q120(a).
q121(a).
q122(a).
q123(a).
q124(a).
q125(a).
q126(a).
q127(a).

% 3. Construct one 100 element list, nonrecursively.

bench_mark(cons_list, 2000, r1(L), dummy(L)).

:- public r1/1.

% 4. Walk down a 100 element list, nonrecursively

bench_mark(walk_list, 2000, r1(L), dummy(L)) :- r1(L).

% 5. Walk down a 100 element list, recursively

bench_mark(walk_list_rec, 2000, wlr(L), dummy(L)) :- r1(L).

:- public wlr/1.

% 6. Walk down N 100 copies of the same 100 element list, recursively.

bench_mark(args(N), 2000, args(N, L), dummy(N, L)) :- args(N), r1(L).

:- public args/2.

args(1).  args(2).  args(4).  args(8).  args(16).

args(1, L) :- wlr(L).
args(2, L) :- wlr(L, L).
args(4, L) :- wlr(L, L, L, L).
args(8, L) :- wlr(L, L, L, L, L, L, L, L).
args(16, L) :- wlr(L, L, L, L, L, L, L, L, L, L, L, L, L, L, L, L).

wlr([]).
wlr([_|L]) :- wlr(L).

wlr([], []).
wlr([_|L1], [_|L2]) :- wlr(L1, L2).

wlr([], [], [], []).
wlr([_|L1], [_|L2], [_|L3], [_|L4]) :- wlr(L1, L2, L3, L4).

wlr([], [], [], [], [], [], [], []).
wlr([_|L1], [_|L2], [_|L3], [_|L4], [_|L5], [_|L6], [_|L7], [_|L8]) :-
   wlr(L1, L2, L3, L4, L5, L6, L7, L8).

wlr([], [], [], [], [], [], [], [], [], [], [], [], [], [], [], []).
wlr([_|L1], [_|L2], [_|L3], [_|L4], [_|L5], [_|L6], [_|L7], [_|L8],
    [_|L9], [_|L10], [_|L11], [_|L12], [_|L13], [_|L14], [_|L15], [_|L16]) :-
   wlr(L1, L2, L3, L4, L5, L6, L7, L8, L9, L10, L11, L12, L13, L14, L15, L16).

% Nonrecursive list cruncher

r1([1|R]) :- r2(R).
r2([2|R]) :- r3(R).
r3([3|R]) :- r4(R).
r4([4|R]) :- r5(R).
r5([5|R]) :- r6(R).
r6([6|R]) :- r7(R).
r7([7|R]) :- r8(R).
r8([8|R]) :- r9(R).
r9([9|R]) :- r10(R).
r10([10|R]) :- r11(R).
r11([11|R]) :- r12(R).
r12([12|R]) :- r13(R).
r13([13|R]) :- r14(R).
r14([14|R]) :- r15(R).
r15([15|R]) :- r16(R).
r16([16|R]) :- r17(R).
r17([17|R]) :- r18(R).
r18([18|R]) :- r19(R).
r19([19|R]) :- r20(R).
r20([20|R]) :- r21(R).
r21([21|R]) :- r22(R).
r22([22|R]) :- r23(R).
r23([23|R]) :- r24(R).
r24([24|R]) :- r25(R).
r25([25|R]) :- r26(R).
r26([26|R]) :- r27(R).
r27([27|R]) :- r28(R).
r28([28|R]) :- r29(R).
r29([29|R]) :- r30(R).
r30([30|R]) :- r31(R).
r31([31|R]) :- r32(R).
r32([32|R]) :- r33(R).
r33([33|R]) :- r34(R).
r34([34|R]) :- r35(R).
r35([35|R]) :- r36(R).
r36([36|R]) :- r37(R).
r37([37|R]) :- r38(R).
r38([38|R]) :- r39(R).
r39([39|R]) :- r40(R).
r40([40|R]) :- r41(R).
r41([41|R]) :- r42(R).
r42([42|R]) :- r43(R).
r43([43|R]) :- r44(R).
r44([44|R]) :- r45(R).
r45([45|R]) :- r46(R).
r46([46|R]) :- r47(R).
r47([47|R]) :- r48(R).
r48([48|R]) :- r49(R).
r49([49|R]) :- r50(R).
r50([50|R]) :- r51(R).
r51([51|R]) :- r52(R).
r52([52|R]) :- r53(R).
r53([53|R]) :- r54(R).
r54([54|R]) :- r55(R).
r55([55|R]) :- r56(R).
r56([56|R]) :- r57(R).
r57([57|R]) :- r58(R).
r58([58|R]) :- r59(R).
r59([59|R]) :- r60(R).
r60([60|R]) :- r61(R).
r61([61|R]) :- r62(R).
r62([62|R]) :- r63(R).
r63([63|R]) :- r64(R).
r64([64|R]) :- r65(R).
r65([65|R]) :- r66(R).
r66([66|R]) :- r67(R).
r67([67|R]) :- r68(R).
r68([68|R]) :- r69(R).
r69([69|R]) :- r70(R).
r70([70|R]) :- r71(R).
r71([71|R]) :- r72(R).
r72([72|R]) :- r73(R).
r73([73|R]) :- r74(R).
r74([74|R]) :- r75(R).
r75([75|R]) :- r76(R).
r76([76|R]) :- r77(R).
r77([77|R]) :- r78(R).
r78([78|R]) :- r79(R).
r79([79|R]) :- r80(R).
r80([80|R]) :- r81(R).
r81([81|R]) :- r82(R).
r82([82|R]) :- r83(R).
r83([83|R]) :- r84(R).
r84([84|R]) :- r85(R).
r85([85|R]) :- r86(R).
r86([86|R]) :- r87(R).
r87([87|R]) :- r88(R).
r88([88|R]) :- r89(R).
r89([89|R]) :- r90(R).
r90([90|R]) :- r91(R).
r91([91|R]) :- r92(R).
r92([92|R]) :- r93(R).
r93([93|R]) :- r94(R).
r94([94|R]) :- r95(R).
r95([95|R]) :- r96(R).
r96([96|R]) :- r97(R).
r97([97|R]) :- r98(R).
r98([98|R]) :- r99(R).
r99([99|R]) :- r100(R).
r100([100|R]) :- r101(R).
r101([]).

% 7. Construct a term with 100 nodes, nonrecursively

bench_mark(cons_term, 2000, s1(T), dummy(T)).

:- public s1/1.

% 8. Walk down a term with 100 nodes, nonrecursively.

bench_mark(walk_term, 2000, s1(T), dummy(T)) :- s1(T).

% 9. Walk down a term with 100 nodes, recursively.

bench_mark(walk_term_rec, 2000, wtr(T), dummy(T)) :- s1(T).

:- public wtr/1.

wtr(nil).
wtr(f(_,R)) :- wtr(R).

% Nonrecursive term cruncher

s1(f(1, R)) :- s2(R).
s2(f(2, R)) :- s3(R).
s3(f(3, R)) :- s4(R).
s4(f(4, R)) :- s5(R).
s5(f(5, R)) :- s6(R).
s6(f(6, R)) :- s7(R).
s7(f(7, R)) :- s8(R).
s8(f(8, R)) :- s9(R).
s9(f(9, R)) :- s10(R).
s10(f(10, R)) :- s11(R).
s11(f(11, R)) :- s12(R).
s12(f(12, R)) :- s13(R).
s13(f(13, R)) :- s14(R).
s14(f(14, R)) :- s15(R).
s15(f(15, R)) :- s16(R).
s16(f(16, R)) :- s17(R).
s17(f(17, R)) :- s18(R).
s18(f(18, R)) :- s19(R).
s19(f(19, R)) :- s20(R).
s20(f(20, R)) :- s21(R).
s21(f(21, R)) :- s22(R).
s22(f(22, R)) :- s23(R).
s23(f(23, R)) :- s24(R).
s24(f(24, R)) :- s25(R).
s25(f(25, R)) :- s26(R).
s26(f(26, R)) :- s27(R).
s27(f(27, R)) :- s28(R).
s28(f(28, R)) :- s29(R).
s29(f(29, R)) :- s30(R).
s30(f(30, R)) :- s31(R).
s31(f(31, R)) :- s32(R).
s32(f(32, R)) :- s33(R).
s33(f(33, R)) :- s34(R).
s34(f(34, R)) :- s35(R).
s35(f(35, R)) :- s36(R).
s36(f(36, R)) :- s37(R).
s37(f(37, R)) :- s38(R).
s38(f(38, R)) :- s39(R).
s39(f(39, R)) :- s40(R).
s40(f(40, R)) :- s41(R).
s41(f(41, R)) :- s42(R).
s42(f(42, R)) :- s43(R).
s43(f(43, R)) :- s44(R).
s44(f(44, R)) :- s45(R).
s45(f(45, R)) :- s46(R).
s46(f(46, R)) :- s47(R).
s47(f(47, R)) :- s48(R).
s48(f(48, R)) :- s49(R).
s49(f(49, R)) :- s50(R).
s50(f(50, R)) :- s51(R).
s51(f(51, R)) :- s52(R).
s52(f(52, R)) :- s53(R).
s53(f(53, R)) :- s54(R).
s54(f(54, R)) :- s55(R).
s55(f(55, R)) :- s56(R).
s56(f(56, R)) :- s57(R).
s57(f(57, R)) :- s58(R).
s58(f(58, R)) :- s59(R).
s59(f(59, R)) :- s60(R).
s60(f(60, R)) :- s61(R).
s61(f(61, R)) :- s62(R).
s62(f(62, R)) :- s63(R).
s63(f(63, R)) :- s64(R).
s64(f(64, R)) :- s65(R).
s65(f(65, R)) :- s66(R).
s66(f(66, R)) :- s67(R).
s67(f(67, R)) :- s68(R).
s68(f(68, R)) :- s69(R).
s69(f(69, R)) :- s70(R).
s70(f(70, R)) :- s71(R).
s71(f(71, R)) :- s72(R).
s72(f(72, R)) :- s73(R).
s73(f(73, R)) :- s74(R).
s74(f(74, R)) :- s75(R).
s75(f(75, R)) :- s76(R).
s76(f(76, R)) :- s77(R).
s77(f(77, R)) :- s78(R).
s78(f(78, R)) :- s79(R).
s79(f(79, R)) :- s80(R).
s80(f(80, R)) :- s81(R).
s81(f(81, R)) :- s82(R).
s82(f(82, R)) :- s83(R).
s83(f(83, R)) :- s84(R).
s84(f(84, R)) :- s85(R).
s85(f(85, R)) :- s86(R).
s86(f(86, R)) :- s87(R).
s87(f(87, R)) :- s88(R).
s88(f(88, R)) :- s89(R).
s89(f(89, R)) :- s90(R).
s90(f(90, R)) :- s91(R).
s91(f(91, R)) :- s92(R).
s92(f(92, R)) :- s93(R).
s93(f(93, R)) :- s94(R).
s94(f(94, R)) :- s95(R).
s95(f(95, R)) :- s96(R).
s96(f(96, R)) :- s97(R).
s97(f(97, R)) :- s98(R).
s98(f(98, R)) :- s99(R).
s99(f(99, R)) :- s100(R).
s100(f(100, R)) :- s101(R).
s101(nil).

% 10. 99 shallow failures; assumes no indexing on 2nd argument

bench_mark(shallow_backtracking, 2000, shallow, dummy).

:- public shallow/0.

% 11. 99 deep failures; assumes no indexing on 2nd argument

bench_mark(deep_backtracking, 2000, deep, dummy).

:- public deep/0.

shallow :- b(_X, 100).
deep :- b(_X, Y), Y = 100.

b(_X, 1).
b(_X, 2).
b(_X, 3).
b(_X, 4).
b(_X, 5).
b(_X, 6).
b(_X, 7).
b(_X, 8).
b(_X, 9).
b(_X, 10).
b(_X, 11).
b(_X, 12).
b(_X, 13).
b(_X, 14).
b(_X, 15).
b(_X, 16).
b(_X, 17).
b(_X, 18).
b(_X, 19).
b(_X, 20).
b(_X, 21).
b(_X, 22).
b(_X, 23).
b(_X, 24).
b(_X, 25).
b(_X, 26).
b(_X, 27).
b(_X, 28).
b(_X, 29).
b(_X, 30).
b(_X, 31).
b(_X, 32).
b(_X, 33).
b(_X, 34).
b(_X, 35).
b(_X, 36).
b(_X, 37).
b(_X, 38).
b(_X, 39).
b(_X, 40).
b(_X, 41).
b(_X, 42).
b(_X, 43).
b(_X, 44).
b(_X, 45).
b(_X, 46).
b(_X, 47).
b(_X, 48).
b(_X, 49).
b(_X, 50).
b(_X, 51).
b(_X, 52).
b(_X, 53).
b(_X, 54).
b(_X, 55).
b(_X, 56).
b(_X, 57).
b(_X, 58).
b(_X, 59).
b(_X, 60).
b(_X, 61).
b(_X, 62).
b(_X, 63).
b(_X, 64).
b(_X, 65).
b(_X, 66).
b(_X, 67).
b(_X, 68).
b(_X, 69).
b(_X, 70).
b(_X, 71).
b(_X, 72).
b(_X, 73).
b(_X, 74).
b(_X, 75).
b(_X, 76).
b(_X, 77).
b(_X, 78).
b(_X, 79).
b(_X, 80).
b(_X, 81).
b(_X, 82).
b(_X, 83).
b(_X, 84).
b(_X, 85).
b(_X, 86).
b(_X, 87).
b(_X, 88).
b(_X, 89).
b(_X, 90).
b(_X, 91).
b(_X, 92).
b(_X, 93).
b(_X, 94).
b(_X, 95).
b(_X, 96).
b(_X, 97).
b(_X, 98).
b(_X, 99).
b(_X, 100).


% 14. Unify terms that are small in space but textually large.

bench_mark(medium_unify, 2000, equal(Term1, Term2), dummy(Term1, Term2)) :-
   term64(Term1),
   term64(Term2).
bench_mark(deep_unify, 100, equal(Term1, Term2), dummy(Term1, Term2)) :-
   term4096(Term1),
   term4096(Term2).

:- public equal/2.

equal(X, X).

term64(X1) :-
   X1 = f(X2, X2),
   X2 = f(X4, X4),
   X4 = f(X8, X8),
   X8 = f(X16, X16),
   X16 = f(X32, X32),
   X32 = f(X64, X64).

term4096(X1) :-
   X1 = f(X2, X2),
   X2 = f(X4, X4),
   X4 = f(X8, X8),
   X8 = f(X16, X16),
   X16 = f(X32, X32),
   X32 = f(X64, X64),
   X64 = f(X128, X128),
   X128 = f(X256, X256),
   X256 = f(X512, X512),
   X512 = f(X1024, X1024),
   X1024 = f(X2048, X2048),
   X2048 = f(X4096, X4096).

% 15. Do 100 integer additions nonrecursively,
% avoiding obvious compiler optimizations.

bench_mark(integer_add, 1000, a1(0, 1, R), dummy(0, 1, R)).

:- public a1/3.

a1(M, K, P) :- N is M + K, a2(N, 2, P).
a2(M, K, P) :- N is M + K, a3(N, 3, P).
a3(M, K, P) :- N is M + K, a4(N, 4, P).
a4(M, K, P) :- N is M + K, a5(N, 5, P).
a5(M, K, P) :- N is M + K, a6(N, 6, P).
a6(M, K, P) :- N is M + K, a7(N, 7, P).
a7(M, K, P) :- N is M + K, a8(N, 8, P).
a8(M, K, P) :- N is M + K, a9(N, 9, P).
a9(M, K, P) :- N is M + K, a10(N, 10, P).
a10(M, K, P) :- N is M + K, a11(N, 11, P).
a11(M, K, P) :- N is M + K, a12(N, 12, P).
a12(M, K, P) :- N is M + K, a13(N, 13, P).
a13(M, K, P) :- N is M + K, a14(N, 14, P).
a14(M, K, P) :- N is M + K, a15(N, 15, P).
a15(M, K, P) :- N is M + K, a16(N, 16, P).
a16(M, K, P) :- N is M + K, a17(N, 17, P).
a17(M, K, P) :- N is M + K, a18(N, 18, P).
a18(M, K, P) :- N is M + K, a19(N, 19, P).
a19(M, K, P) :- N is M + K, a20(N, 20, P).
a20(M, K, P) :- N is M + K, a21(N, 21, P).
a21(M, K, P) :- N is M + K, a22(N, 22, P).
a22(M, K, P) :- N is M + K, a23(N, 23, P).
a23(M, K, P) :- N is M + K, a24(N, 24, P).
a24(M, K, P) :- N is M + K, a25(N, 25, P).
a25(M, K, P) :- N is M + K, a26(N, 26, P).
a26(M, K, P) :- N is M + K, a27(N, 27, P).
a27(M, K, P) :- N is M + K, a28(N, 28, P).
a28(M, K, P) :- N is M + K, a29(N, 29, P).
a29(M, K, P) :- N is M + K, a30(N, 30, P).
a30(M, K, P) :- N is M + K, a31(N, 31, P).
a31(M, K, P) :- N is M + K, a32(N, 32, P).
a32(M, K, P) :- N is M + K, a33(N, 33, P).
a33(M, K, P) :- N is M + K, a34(N, 34, P).
a34(M, K, P) :- N is M + K, a35(N, 35, P).
a35(M, K, P) :- N is M + K, a36(N, 36, P).
a36(M, K, P) :- N is M + K, a37(N, 37, P).
a37(M, K, P) :- N is M + K, a38(N, 38, P).
a38(M, K, P) :- N is M + K, a39(N, 39, P).
a39(M, K, P) :- N is M + K, a40(N, 40, P).
a40(M, K, P) :- N is M + K, a41(N, 41, P).
a41(M, K, P) :- N is M + K, a42(N, 42, P).
a42(M, K, P) :- N is M + K, a43(N, 43, P).
a43(M, K, P) :- N is M + K, a44(N, 44, P).
a44(M, K, P) :- N is M + K, a45(N, 45, P).
a45(M, K, P) :- N is M + K, a46(N, 46, P).
a46(M, K, P) :- N is M + K, a47(N, 47, P).
a47(M, K, P) :- N is M + K, a48(N, 48, P).
a48(M, K, P) :- N is M + K, a49(N, 49, P).
a49(M, K, P) :- N is M + K, a50(N, 50, P).
a50(M, K, P) :- N is M + K, a51(N, 51, P).
a51(M, K, P) :- N is M + K, a52(N, 52, P).
a52(M, K, P) :- N is M + K, a53(N, 53, P).
a53(M, K, P) :- N is M + K, a54(N, 54, P).
a54(M, K, P) :- N is M + K, a55(N, 55, P).
a55(M, K, P) :- N is M + K, a56(N, 56, P).
a56(M, K, P) :- N is M + K, a57(N, 57, P).
a57(M, K, P) :- N is M + K, a58(N, 58, P).
a58(M, K, P) :- N is M + K, a59(N, 59, P).
a59(M, K, P) :- N is M + K, a60(N, 60, P).
a60(M, K, P) :- N is M + K, a61(N, 61, P).
a61(M, K, P) :- N is M + K, a62(N, 62, P).
a62(M, K, P) :- N is M + K, a63(N, 63, P).
a63(M, K, P) :- N is M + K, a64(N, 64, P).
a64(M, K, P) :- N is M + K, a65(N, 65, P).
a65(M, K, P) :- N is M + K, a66(N, 66, P).
a66(M, K, P) :- N is M + K, a67(N, 67, P).
a67(M, K, P) :- N is M + K, a68(N, 68, P).
a68(M, K, P) :- N is M + K, a69(N, 69, P).
a69(M, K, P) :- N is M + K, a70(N, 70, P).
a70(M, K, P) :- N is M + K, a71(N, 71, P).
a71(M, K, P) :- N is M + K, a72(N, 72, P).
a72(M, K, P) :- N is M + K, a73(N, 73, P).
a73(M, K, P) :- N is M + K, a74(N, 74, P).
a74(M, K, P) :- N is M + K, a75(N, 75, P).
a75(M, K, P) :- N is M + K, a76(N, 76, P).
a76(M, K, P) :- N is M + K, a77(N, 77, P).
a77(M, K, P) :- N is M + K, a78(N, 78, P).
a78(M, K, P) :- N is M + K, a79(N, 79, P).
a79(M, K, P) :- N is M + K, a80(N, 80, P).
a80(M, K, P) :- N is M + K, a81(N, 81, P).
a81(M, K, P) :- N is M + K, a82(N, 82, P).
a82(M, K, P) :- N is M + K, a83(N, 83, P).
a83(M, K, P) :- N is M + K, a84(N, 84, P).
a84(M, K, P) :- N is M + K, a85(N, 85, P).
a85(M, K, P) :- N is M + K, a86(N, 86, P).
a86(M, K, P) :- N is M + K, a87(N, 87, P).
a87(M, K, P) :- N is M + K, a88(N, 88, P).
a88(M, K, P) :- N is M + K, a89(N, 89, P).
a89(M, K, P) :- N is M + K, a90(N, 90, P).
a90(M, K, P) :- N is M + K, a91(N, 91, P).
a91(M, K, P) :- N is M + K, a92(N, 92, P).
a92(M, K, P) :- N is M + K, a93(N, 93, P).
a93(M, K, P) :- N is M + K, a94(N, 94, P).
a94(M, K, P) :- N is M + K, a95(N, 95, P).
a95(M, K, P) :- N is M + K, a96(N, 96, P).
a96(M, K, P) :- N is M + K, a97(N, 97, P).
a97(M, K, P) :- N is M + K, a98(N, 98, P).
a98(M, K, P) :- N is M + K, a99(N, 99, P).
a99(M, K, P) :- N is M + K, a100(N, 100, P).
a100(M, K, P) :- P is M + K.

# end of benchmarks


